/*
 *	webcam.c
 *	Streaming webcam using jpeg images over a multipart/x-mixed-replace stream
 *	Copyright (C) 2002 Jeroen Vreeken (pe1rxq@amsat.org)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <errno.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <string.h>

#include "picture.h"
#include "webcam.h"
#include "motion.h"

int http_bindsock(int port)
{
	int sl;
	struct sockaddr_in sin;

	if ((sl=socket(PF_INET, SOCK_STREAM, 0))==-1) {
		perror("socket()");
		exit(1);
	}

	memset(&sin, 0, sizeof(struct sockaddr_in));
	sin.sin_family=AF_INET;
	sin.sin_port=htons(port);
	sin.sin_addr.s_addr=INADDR_ANY;

	if (bind(sl, (struct sockaddr *)&sin, sizeof(struct sockaddr_in))==-1) {
		perror("bind()");
		exit(1);
	}

	if (listen(sl, 10)==-1) {
		perror("listen()");
		exit(1);
	}
	printf("Webcam server listning on port: %d\n", port);

	return sl;
}

int http_acceptsock(int sl)
{
	int sc, i;
	struct sockaddr_in sin;
	int addrlen=sizeof(struct sockaddr_in);

	if ((sc=accept(sl, (struct sockaddr *)&sin, &addrlen))!=1) {
		i=1;
		ioctl(sc, FIONBIO, &i);
		return sc;
	}

	perror("accept()");

	return -1;
}

void webcam_close(struct webcam *list)
{
	struct webcam *next=list->next;
	
	while (next) {
		list=next;
		next=list->next;
		if (list->tmpfile) {
			fclose(list->tmpfile->fd);
			free(list->tmpfile);
		}
		close(list->socket);
		free(list);
	}
}

void webcam_flush(struct webcam *list)
{
	char buffer[2048];
	int readcount=2048;
	void *tmp;

	while (list->next) {
		list=list->next;
		if (!list->tmpfile)
			continue;
		fseek(list->tmpfile->fd, list->filepos, SEEK_SET);
		readcount=2048;
		while (readcount==2048 && list->tmpfile) {
			readcount=fread(buffer, 1, 2048, list->tmpfile->fd);
			if (readcount>0) {
				readcount=write(list->socket, buffer, readcount);
				if (readcount>0)
					list->filepos+=readcount;
				if (list->filepos>=list->tmpfile->size || 
				    (readcount<0 && errno!=EAGAIN)) {
					list->tmpfile->ref--;
					if (list->tmpfile->ref<=0) {
						fclose(list->tmpfile->fd);
						free(list->tmpfile);
					}
					list->tmpfile=NULL;
				}
				if (readcount<0 && errno!=EAGAIN) {
					close(list->socket);
					if (list->next)
						list->next->prev=list->prev;
					list->prev->next=list->next;
					tmp=list;
					list=list->prev;
					free(tmp);
				}
			}
		}
	}
}

struct webcam_tmpfile *webcam_tmpfile(void)
{
	struct webcam_tmpfile *file=malloc(sizeof(struct webcam_tmpfile));
	file->ref=0;
	file->fd=tmpfile();
	if (!file->fd) {
		free(file);
		return NULL;
	}
	return file;
}

void webcam_add_client(struct webcam *list, int sc)
{
	struct webcam *new=malloc(sizeof(struct webcam));
	new->socket=sc;
	new->fwrite=fdopen(sc, "a");
	new->tmpfile=NULL;
	new->filepos=0;
	new->prev=list;
	new->next=list->next;
	if (new->next)
		new->next->prev=new;
	list->next=new;
	new->last=0;

	fprintf(new->fwrite, "HTTP/1.1 200 OK\n");
	fprintf(new->fwrite, "Max-Age: 0\n");
	fprintf(new->fwrite, "Expires: 0\n");
	fprintf(new->fwrite, "Cache-Control: no-cache\n");
	fprintf(new->fwrite, "Cache-Control: private\n");
	fprintf(new->fwrite, "Pragma: no-cache\n");
	fprintf(new->fwrite, "Content-type: multipart/x-mixed-replace;boundary=BoundaryString\n");
	fflush(0);
}

void webcam_add_write(struct webcam *list, struct webcam_tmpfile *tmpfile, int fps)
{
	struct timeval curtimeval;
	unsigned long int curtime;

	gettimeofday(&curtimeval, NULL);
	curtime=curtimeval.tv_usec+1000000L*curtimeval.tv_sec;
	while (list->next) {
		list=list->next;
		if (list->tmpfile==NULL && (curtime-list->last)>=1000000L/fps) {
			list->last=curtime;
			list->tmpfile=tmpfile;
			tmpfile->ref++;
			list->filepos=0;
			fprintf(list->fwrite, "\n--BoundaryString\n");
			fprintf(list->fwrite, "Content-type: image/jpeg\n");
			fprintf(list->fwrite, "Content-Length: %ld\n\n", list->tmpfile->size-1);
			fflush(list->fwrite);
		}
	}
	if (tmpfile->ref<=0) {
		fclose(tmpfile->fd);
		free(tmpfile);
	}
}

int webcam_check_write(struct webcam *list)
{
	int write=0;
	while (list->next) {
		list=list->next;
		if (list->tmpfile==NULL)
			write=1;
	}
	return write;
}

int webcam_init(struct context *cnt)
{
	cnt->webcam.socket=http_bindsock(cnt->conf.webcam_port);
	cnt->webcam.next=NULL;
	cnt->webcam.prev=NULL;
	signal(SIGPIPE, SIG_IGN);
	return 0;
}

void webcam_stop(struct context *cnt)
{
	printf("Closing webcam listen socket\n");
	close(cnt->webcam.socket);
	printf("Closing active webcam sockets\n");
	webcam_close(&cnt->webcam);
}

int webcam_put(struct context *cnt, char *image)
{
	struct timeval timeout;
	struct webcam_tmpfile *tmpfile;
	fd_set fdread;
	int sl=cnt->webcam.socket;
	int sc;

	timeout.tv_sec=0;
	timeout.tv_usec=0;
	FD_ZERO(&fdread);
	FD_SET(cnt->webcam.socket, &fdread);
	if (select(sl+1, &fdread, NULL, NULL, &timeout)>0) {
		sc=http_acceptsock(sl);
		webcam_add_client(&cnt->webcam, sc);
	}
	webcam_flush(&cnt->webcam);
	if (webcam_check_write(&cnt->webcam)) {
		tmpfile=webcam_tmpfile();
		if (tmpfile) {
			put_picture_fd(cnt, tmpfile->fd, image, cnt->conf.webcam_quality);
			fwrite("\n", 1, 1, tmpfile->fd);
			fseek(tmpfile->fd, 0, SEEK_END);
			tmpfile->size=ftell(tmpfile->fd);
			webcam_add_write(&cnt->webcam, tmpfile, cnt->conf.webcam_maxrate);
		} else {
			perror("Error creating tmpfile!");
		}
	}
	webcam_flush(&cnt->webcam);
	return 0;
}
