/*
 * Mooix proxy wrapper library.
 *
 * Copyright 2001-2003 by Joey Hess <joey@mooix.net>
 * under the terms of the modified BSD license given in full in the
 * file COPYRIGHT.
 * 
 * This file holds the bodies of wrapped functions, which in turn generally
 * call the real_ or proxy_ functions. It is split off from wrap.c, since
 * wrap.c cannot #include all the symbols all of these function bodies
 * need. For speed, the functions in here are generally inlined.
 * XXX unfortunatly, gcc probably does not inline them since they're in a
 * different file
 * 
 * Note that for the regression tests to work, each function body
 * must call mooix_debug.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <libgen.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/un.h>

#include "libmooproxy.h"
#include "proxy.h"
#include "real.h"
#include "client.h"

inline int do_open (const char *name, int flags, mode_t mode) {
	mooix_debug(PRIORITY_DEBUG, "open mode %i %s", mode, name, NULL);
	if (proxy_enabled) {
		char *field = is_field(name, 1);
		if (field) {
			int ret = proxy_open(field, flags, mode);
			free(field);
			return ret;
		}
	}
	return real_open(name, flags, mode);
}

inline int do_creat (const char *pathname, mode_t mode) {
	return open(pathname, O_CREAT|O_WRONLY|O_TRUNC, mode);
}

static inline int fopen_flags_parse (const char *mode) {
	int flags;
	#define extended_mode mode[1] == '+' || (mode[1] == 'b' && mode[2] == '+')
	switch (mode[0]) {
		case 'r':
			if (extended_mode)
				flags = O_RDWR;
			else
				flags = O_RDONLY;
			break;
		case 'w':
			if (extended_mode)
				flags = O_RDWR | O_CREAT | O_TRUNC;
			else
				flags = O_WRONLY | O_CREAT | O_TRUNC;
			break;
		case 'a':
			if (extended_mode)
				flags = O_RDWR | O_APPEND | O_CREAT;
			else
				flags = O_WRONLY | O_APPEND | O_CREAT;
			break;
		default:
			errno = EINVAL;
			return 0;
	}
	return flags;
}

inline FILE *do_fopen (const char *path, const char *mode) {
	int flags, fd;
	
	mooix_debug(PRIORITY_DEBUG, "open mode %s %s", mode, path, NULL);
	if (proxy_enabled) {
		char *field = is_field(path, 1);
		if (field) {
			/* Get a fd open to the file using the right flags.
			 * For the mode, always use 0666; any of the fopen
			 * modes that can create files should make them
			 * that mode. */
			errno = 0;
			flags = fopen_flags_parse(mode);
			if (errno != 0)
				return NULL;
			fd = proxy_open(field, flags, 0666);
			free(field);
			if (fd == -1)
				return NULL;
			/* Now associate it with a stream. */
			return fdopen(fd, mode);
		}
	}
/* Work around MIPS horkage. */
#if #cpu (mips)
#warning Using MIPS SYS_open syscall workaround for fopen
	errno = 0;
	flags = fopen_flags_parse(mode);
	if (errno != 0)
		return NULL;
	fd = real_open(path, flags, 0666);
	if (fd == -1)
		return NULL;
	return fdopen(fd, mode);
#else
	return real_fopen(path, mode);
#endif
}

inline mode_t do_umask (mode_t themask) {
	int ret = real_umask(themask);
	mooix_debug(PRIORITY_DEBUG, "umask", NULL);
	mask = themask;
	return ret;
}

inline int do_chdir (const char *path) {
	mooix_debug(PRIORITY_DEBUG, "chdir %s", path, NULL);
	if (proxy_enabled) {
		char *this = getenv("THIS");
		if (! in_object && this && strcmp(path, this) == 0) {
			/* Magic chdir to the object's directory, via THISFD.
			 * This is done to make sure they end up in the same object
			 * the method was run on, even if the path to it is
			 * controlled by unsavory characters who try to move a
			 * different one into place. */
			mooix_debug(PRIORITY_DEBUG, "magic chdir");
			return fchdir(THISFD);
		}
		else {
			/* Watch where the chdir goes. */
			int ret = real_chdir(path);
			if (ret == 0)
				check_in_object();
			mooix_debug(PRIORITY_DEBUG, "ret: %i", ret);
			return ret;
		}
	}
	else {
		return real_chdir(path);
	}
}

inline int do_fchdir (int fd) {
	/* Watch where the chdir goes. */
	int ret = real_fchdir(fd);
	mooix_debug(PRIORITY_DEBUG, "fchdir %i", fd, NULL);
	if (proxy_enabled && ret == 0)
		check_in_object();
	return ret;
}

inline int do_unlink (const char *pathname) {
	mooix_debug(PRIORITY_DEBUG, "unlink %s", pathname, NULL);
	if (proxy_enabled) {
		char *field = is_field(pathname, 0);
		if (field) {
			int ret = proxy_unlink(field);
			free(field);
			return ret;
		}
	}
	return real_unlink(pathname);
}

inline int do_symlink (const char *oldpath, const char *newpath) {
	mooix_debug(PRIORITY_DEBUG, "symlink %s %s", oldpath, newpath, NULL);
	if (proxy_enabled) {
		char *field = is_field(newpath, 1);
		if (field) {
			int ret = proxy_symlink(oldpath, field);
			free(field);
			return ret;
		}
	}
	return real_symlink(oldpath, newpath);
}

inline int do_mkdir (const char *pathname, mode_t mode) {
	mooix_debug(PRIORITY_DEBUG, "mkdir %s", pathname, NULL);
	if (proxy_enabled) {
		char *field = is_field(pathname, 1);
		if (field) {
			int ret = proxy_mkdir(field, mode);
			free(field);
			return ret;
		}
	}
	return real_mkdir(pathname, mode);
}

inline int do_rmdir (const char *pathname) {
	mooix_debug(PRIORITY_DEBUG, "rmdir %s", pathname, NULL);
	if (proxy_enabled) {
		char *field = is_field(pathname, 1);
		if (field) {
			int ret = proxy_rmdir(field);
			free(field);
			return ret;
		}
	}
	return real_rmdir(pathname);
}

inline int do_chmod (char *path, mode_t mode) {
	mooix_debug(PRIORITY_DEBUG, "chmod %s %i", path, mode, NULL);
	if (proxy_enabled) {
		char *field = is_field(path, 1);
		if (field) {
			int ret = proxy_chmod(field, mode);
			free(field);
			return ret;
		}
	}
	return real_chmod(path, mode);
}

inline int do_rename (const char *oldpath, const char *newpath) {
	mooix_debug(PRIORITY_DEBUG, "rename %s %s", oldpath, newpath, NULL);
	if (proxy_enabled) {
		char *oldfield = is_field(oldpath, 1);
		char *newfield = is_field(newpath, 1);
		if (oldfield && newfield) {
			int ret = proxy_rename(oldfield, newfield);
			free(oldfield);
			free(newfield);
			return ret;
		}
	}
	return real_rename(oldpath, newpath);
}

inline int do_connect (int sockfd, const struct sockaddr_un *serv_addr, socklen_t addrlen) {
	mooix_debug(PRIORITY_DEBUG, "connect", NULL);
	if (proxy_enabled) {
		/* Only unix socket connects can be proxied. */
		if (serv_addr->sun_family == AF_UNIX) {
			char *path = is_field(serv_addr->sun_path, 1);
			if (path) {
				int ret = proxy_connect(sockfd, serv_addr, addrlen);
				free(path);
				return ret;
			}
		}
	}
	
	return real_connect(sockfd, (const struct sockaddr *) serv_addr, addrlen);
}

inline int do_kill (pid_t pid, int sig) {
	mooix_debug(PRIORITY_DEBUG, "kill %i", pid, NULL);
	if (proxy_enabled && pid <= 0) {
		if (getenv("MOOIX_NONSTANDARD")) {
			return proxy_kill(pid, sig);
		}
	}
	
	return real_kill(pid, sig);
}

inline pid_t do_fork (void) {
	pid_t ret;
	int parentconn, childconn;
	
	mooix_debug(PRIORITY_DEBUG, "fork", NULL);
	
	if (! proxy_enabled)
		return real_fork();
	
	/* To avoid races, this must be done before the fork. Open a
	 * connection to mood for the child (and ensure the parent has one
	 * open). */
	parentconn = getproxy();
	proxysock = -1;
	childconn = getproxy();

	ret = real_fork();
	
	if (ret == 0) {
		proxysock = childconn;
		close(parentconn);
	}
	else {
		proxysock = parentconn;
		close(childconn);
	}
	return ret;
}

/* 
 * Only execve is passed through to the privledged process, so I must
 * translate the other functions in the exec family into calls to execve,
 * which is quite ugly since it means re-implementing them, and some are
 * complex. It's a pity I can't call the ones in libc, which do call
 * execve in the end, but they call the _wrong_ execve (or rather, they
 * call __execve, which is a weak_alias to execve -- symbol versioning may
 * be involved too).
 */

extern char **environ;

inline int do_execve (const char *filename, char * const argv[], char * const envp[]) {
	mooix_debug(PRIORITY_DEBUG, "execve %s", filename, NULL);
	if (proxy_enabled && mooix_obj(filename))
		return proxy_execve(filename, argv, envp);
	else
		return real_execve(filename, argv, envp);
}

/* Takes a va_list, otherwise is like execle. */
inline int do_vexecle (const char *path, const char *arg, va_list ap) {
	size_t argv_max = 1024;
	const char **argv = malloc(argv_max * sizeof (const char *));
	const char * const *envp;
	unsigned int i;
	
	argv[0] = arg;
	i = 0;
	while (argv[i++] != NULL) {
		if (i == argv_max)
			realloc(argv, (argv_max *= 2) * sizeof (const char *));
		argv[i] = va_arg(ap, const char *);
	}
	envp = va_arg(ap, const char *const *);
	return execve(path, (char *const *) argv, (char *const *) envp);
}

inline int do_vexecl (const char *path, const char *arg, va_list ap) {
	size_t argv_max = 1024;
	const char **argv = malloc(argv_max * sizeof (const char *));
	unsigned int i;
	
	argv[0] = arg;
	i = 0;
	while (argv[i++] != NULL) {
		if (i == argv_max)
			realloc(argv, (argv_max *= 2) * sizeof (const char *));
		argv[i] = va_arg(ap, const char *);
	}
	return execve(path, (char *const *) argv, environ);
}

static void script_execute (const char *file, char *const argv[]) {
	int argc = 0;
	while (argv[argc++]); 
	
	{
		char *new_argv[argc + 1];
		new_argv[0] = (char *) "/bin/sh";
		new_argv[1] = (char *) file;
		while (argc > 1) {
			new_argv[argc] = argv[argc - 1];
			--argc;
		}

		execve(new_argv[0], new_argv, environ);
	}
}

inline int do_execvp (const char *file, char *const argv[]) {
	if (*file == '\0') {
		errno = ENOENT;
		return -1;
	}
	if (strchr(file, '/') != NULL) {
		execve(file, argv, environ);

		if (errno == ENOEXEC)
			script_execute (file, argv);
	}
	else {
		int got_eacces = 0;
		char *path, *p, *name;
		size_t len;
		size_t pathlen;

		path = getenv ("PATH");
		if (path == NULL) {
			len = confstr (_CS_PATH, (char *) NULL, 0);
			path = (char *) malloc (1 + len);
			path[0] = ':';
			(void) confstr (_CS_PATH, path + 1, len);
		}

		len = strlen(file) + 1;
		pathlen = strlen(path);
		name = malloc(pathlen + len + 1);
		/* Copy the file name at the top. */
		name = (char *) memcpy(name + pathlen + 1, file, len);
		/* And add the slash. */
		*--name = '/';

		p = path;
		do {
			char *startp;
			path = p;
			p = strchr(path, ':');

			if (p == NULL) {
				p = path + strlen(path);
			}
			
			if (p == path)
				/* Search current directory. */
				startp = name + 1;
			else
				startp = (char *) memcpy (name - (p - path), path, p - path);

			execve(startp, argv, environ);

			if (errno == ENOEXEC)
				script_execute (startp, argv);
			switch (errno) {
				case EACCES:
					got_eacces = 1;
				case ENOENT:
				case ESTALE:
				case ENOTDIR:
					break;
				default:
					return -1;
			}
		} while (*p++ != '\0');
		if (got_eacces) {
			errno = EACCES;
		}
	}
	return -1;
}

inline int do_vexeclp (const char *file, const char *arg, va_list ap) {
	size_t argv_max = 1024;
	const char **argv = malloc(argv_max * sizeof (const char *));
	unsigned int i;
	
	argv[0] = arg;
	i = 0;
	while (argv[i++] != NULL) {
		if (i == argv_max)
			realloc(argv, (argv_max *= 2) * sizeof (const char *));
		argv[i] = va_arg(ap, const char *);
	}
	va_end(ap);
	return execvp(file, (char *const *) argv);
}

inline int do_execv (const char *path, char *const argv[]) {
	return execve(path, argv, environ);
}
