#!/usr/bin/zsh
#
# Mooix shell language binding. Allows object oriented access to mooix
# objects from Z-shell scripts.
#
# How to use it:
#
# 	#!/bin/zsh -e
# 	source /usr/share/mooix/mooix.zsh
#	this . field = value
#	this . list = "value" "value2" "value3"
#	this . counter = $(( $(this . counter) + 1 ))
#	result=$(this . method param param2 param3)
#	this . super
#	this . get otherobj /some/other/object
#	this . field = $(otherobj . field)
#
# Not shown in the above example is how to get at named parameters passed
# to the method (on stdin).  These parameters are stored in the associative
# array $param.

# For example
#	this . tell message "Your avatar is ${param[avatar]}"

# You may optionally call the namedparams method of the 'this'
# object, and they will be made available in $param_<name> variables.
# If the parameter is an object reference, an object named param_<name> is
# automatically set up.
#
# To return an object reference, just invoke the name of the object without
# any parameters. To return text, echo the text.
#
# Note that this is:
#   - slow
#   - insecure (it should never, ever, be used in a stackless method)

typeset -AA param

# All object field/method calls are handled in here.
_handle () {
	local OBJ USEDIR FULLDIR WHAT FIELDFILE
	OBJ=$1
	USEDIR=$2
	FULLDIR=$3
	THIS=$3
	shift 3
	if [[ -n "$2" ]]; then
		WHAT=$2
		FIELDFILE="$USEDIR/$2"
		shift 2
	fi

	case "$WHAT" in
	get)
		# Set up an object.
		local DIR
		DIR=${2#mooix:}
		pushd "$DIR"
		eval "$1 () { _handle $1 $DIR $(pwd) \"\$@\"; }"
		popd
		return
		;;
	id)
		# Get the directory of an object.
		print -- $FULLDIR
		return
		;;
	fieldfile)
		# Returns the filename where a field exists. Takes
		# inheritence into account.
		pushd $USEDIR

		if [[ "$1" != parent ]]; then
			local DIR
			DIR="./parent"
			while [[ -e "$DIR/.mooix" ]]; do
				if [[ -e "$DIR/$1" ]]; then
					print -- "$DIR/$1"
					popd
					return
				fi
				DIR="$DIR/parent"
			done
		fi

		# Mixin support.
		if [[ "$1" == *_* ]]; then
			local MIXIN FIELD DIR
			MIXIN=${1%_*}
			FIELD=${1#${1%_*}_}
			DIR=./
			# Try to find the mixin object, recursing
			# to parents as necessary.
			while : ; do
				if [[ -e "$DIR/$MIXIN/.mooix" ]]; then
					# Now find the field in the
					# mixin or one of its
					# parents.
					if [[ -e "$DIR/$MIXIN/$FIELD" || -L "$DIR/$MIXIN/$FIELD" ]]; then
						print -- "$DIR/$MIXIN/$FIELD"
						popd
						return
					fi
					DIR="$DIR/$MIXIN/parent"
					while [[ -e "$DIR/.mooix" ]]; do
						if [[ -e "$DIR/$FIELD" ]]; then
							print -- "$DIR/$FIELD"
							popd
							return
						fi
						DIR="$DIR/$PARENT"
					done
	
					# If the mixin doesn't have
					# it, failure (don't recurse
					# to other parents).
					popd
					return
				fi
			done
		fi

		return
		;;
	super)
		# Call parent's implementation of the currently
		# running method. $0 provides state about which
		# method is running now.
		FIELDFILE=''
		local METHOD PREFIX
		METHOD="${0:t}" # basename
		PREFIX="$USEDIR/${0:h}/parent" # dirname
		while [[ -e "$PREFIX/.mooix" ]]; do
			if [[ -f "$PREFIX/$METHOD" ]]; then
				if [[ -x "$PREFIX/$METHOD" ]]; then
					FIELDFILE="$PREFIX/$METHOD"
					break
				fi
				prefix="$PREFIX/parent"
			fi
		done
		if [[ -z "$FIELDFILE" ]]; then
			return
		fi
		# Fall through to the code that actually runs the
		# method (or accesses the field, if super is a field..)
		;;
	namedparams)
		local i

		for i in ${(k)param}; do
			if [[ "${param[$i]}" == mooix:* ]]; then
				_handle '' '' '' . get "param_${i}" "${param[$i]}"
			fi
			eval param_$i='$param[$i]'
		done
		return
		;;
	'')
		# "Stringify" an object.
		print -- "mooix:$FULLDIR"
		return
		;;
	esac

	if [[ ! -e "$FIELDFILE" ]]; then
		FIELDFILE=$(_handle $OBJ $USEDIR $FULLDIR . fieldfile $WHAT)
		if [[ -z "$FIELDFILE" ]]; then
			return
		fi
	fi

	pushd $USEDIR

	if [[ "$1" == "=" ]]; then
		shift
		if [[ -x "$FIELDFILE" && ! -d "$FIELDFILE" ]]; then
			# It was set like a field, but it's really a
			# method. That's fine, just run it like a method.
			# below.
		elif [[ "$1" != mooix:* ]]; then
			# Set a field of an object. Support array setting by
			# setting multiple lines.
			print -l -- "$@" > $USEDIR/$WHAT
		else
			# Set a field to refer to an object. Requires a
			# symlink.
			rm -f $USEDIR/$WHAT
			ln -sf "${1#mooix:}" $USEDIR/$WHAT
		fi
	fi

	if [[ -d "$FIELDFILE" ]]; then
		# For an object reference, return the filename the
		# directory or link points to, absolute path. There is no
		# way to instantiate the object and return that, sadly, in
		# shell script.
		pushd "$FIELDFILE"
		local DIR=$(pwd)
		popd
		print -- mooix:$DIR
		popd
		return
	elif [[ -x "$FIELDFILE" ]]; then
		[[ -n "$FULLDIR" ]] && THIS="$FULLDIR"
		if [ "$THIS" ]; then
			pushd $THIS
			print -l "$@" | "$FIELDFILE"
			popd
		else
			pushd $THIS
			print -l "$@" | runmeth "$FIELDFILE"
			popd
		fi
	else
		# Get field.
		cat "$FIELDFILE"
	fi
	popd
}

# When the language binding is sourced, if running inside mooix,
# create an object called "this" which is the object the method is running
# on. Also, deal with parameters.
if [[ -n "$THIS" ]]; then
	# Go to the object directory, so relative paths can be used; that
	# is marginally faster and safer.
	cd "$THIS"
	# Set up the 'this' object.
	_handle '' '' '' . get this .
	
	local NAME VALUE
	while : ; do
		read NAME || return 0
		NAME=${${NAME%\"}#\"}
		read VALUE || return
		VALUE=${${VALUE%\"}#\"}
		# the single quotes delay evaluation until
		# the second go round and close a security
		# hole in $VALUE. $NAME is still exploitable
		# though.
		param[$NAME]="$VALUE"
	done
	return
fi
