"""util.py
This module implements some utility functions necessary for parsing mooix
input.

This file is part of the Python 2.2.1+ binding for mooix
Copyright (c)2002,2003 Nicholas D. Borko.  All Rights Reserved.
The author can be reached at nick@dd.revealed.net

The Python binding for mooix is free software; you can redistribute
it and/or modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

The Python binding for mooix is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with mooix; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA."""

import os, types
import Thing
from variables import *

def convert_mooix_ref(line):
    """convert_mooix_ref(line)  -> Thing object or None

       line: string containing something that may look like ^mooix:.*

       If the passed parameter looks like a mooix reference, return a
       Thing, otherwise return None."""

    if line[:6] == 'mooix:':
        return Thing.Thing(line[6:])
    else:
        return None

def parse_mooix(line, strip_quotes = True, dereference_things = False):
    """parse_mooix(line [,strip_quotes [,dereference_things]]) ->
                                 unquoted string or Thing object from mooix

       line: string
       strip_quotes: if true, then leading and trailing quotes will be
                     stripped
       dereference_things: turn mooix references into a Thing

       Get rid of quotes, and convert anything that looks like mooix: to
       an actual Thing to use in the code."""

    if line[-1] == '\n':
        line = line[:-1]
    line = line.replace('\\n', '\n').replace('\\\\', '\\')

    if line and strip_quotes and line[0] == '"':
        line = line[1:]
        if line[-1] == '"':
            line = line[:-1]
    elif dereference_things:
        # convert anything that looks like ^mooix:.* to a Thing object
        thing = convert_mooix_ref(line)
        if isinstance(thing, Thing.Thing):
            return thing

    return line

def mooix_parse(line, raw = False):
    """mooix_parse(line [,raw]) -> string parsed into a mooix parameter

    line: string or Thing object
    raw: if true, then don't put quotation marks ("") around values

    Convert things into a format that can be used by methods on stdin
    for arguments."""

    if isinstance(line, Thing.Thing):
        return str(line)
    else:
        line = str(line).replace('\\', '\\\\').replace('\n', '\\n')
        if raw:
            return line
        else:
            return '"%s"' % line

class stat:
    """A class to contain a single call to os.lstat et. al. so file tests
    need not call these functions everytime you need one on the same file."""
    def __init__(self, filename):
        """Do the os.lstat call and come up with the results to relevant tests
        that will be needed in our code."""
        try:
            stat_obj = os.lstat(filename)
            # keep the bits around in case the user wants to use them for
            # something more fancy than we've provided
            self.bits = stat_obj.st_mode
            # we use the effective uid and gid since this is basically
            # running in a suid/sgid environment
            self.isowner = os.geteuid() == stat_obj.st_uid
            self.isgroup = os.getegid() == stat_obj.st_gid
            self.exists = True
            # what we would do if we called these as os functions, but
            # do it this one time and don't call stat over and over
            self.islink = (self.bits & 0170000) == 0120000
            self.isfile = (self.bits & 0170000) == 0100000
            self.isdir = (self.bits & 0170000) == 0040000
            self.issticky = (self.bits & 0001000) == 0001000
            self.isread = (self.bits & 00004) or \
                          (self.isgroup and self.bits & 00040) or \
                          (self.isowner and self.bits & 00400)
            self.iswrite = (self.bits & 00002) or \
                           (self.isgroup and self.bits & 00020) or \
                           (self.isowner and self.bits & 00200)
            self.isexec = (self.bits & 00001) or \
                          (self.isgroup and self.bits & 00010) or \
                          (self.isowner and self.bits & 00100)
        except OSError:
            # Probably the file doesn't exist, since you can always
            # stat a file regardless of the owner and mode, unless you
            # can't search the path in the first place.  In that case,
            # this is probably still okay behavior.
            self.bits = 0
            for attr in ('isowner', 'isgroup', 'exists', 'islink', 'isfile',
                         'isdir', 'issticky', 'isread', 'iswrite', 'isexec'):
                setattr(self, attr, False)

__all__ = [ 'convert_mooix_ref', 'mooix_parse', 'mooix_parse', 'parse_mooix',
            'stat' ]
