#!/usr/bin/perl -T
use warnings;
use strict;
use POSIX q{ttyname};
use Mooix::Thing;
use Mooix::Conf;
use Mooix::Root;
use Mooix::Session::Socket;
use UNIVERSAL q{isa};
use Time::HiRes qw(gettimeofday);

# Use the SAFEPATH, so tainting is happy.
$ENV{PATH}=$Mooix::Conf::field{safepath};
# Make %ENV safer
delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};   

my $session;
my $sessionparent=$Mooix::Root->sessions->socket;
my $sessionmanager=$Mooix::Root->system->sessionmanager;
$SIG{HUP} = \&cleanup;
$SIG{INT} = $SIG{TERM} = sub {};
$SIG{TSTP} = sub { kill(&POSIX::SIGCONT, 0) };

# Log in.
my @sessargs=(
	sessionparent => $sessionparent,
	auth_username => getpwuid($<),
	quiet => 1,
);
# See if there is a preferred name, and sanitize it.
my $name=shift;
if (defined $name) {
	if ($name=~m/^([a-zA-Z0-9-_ ]+)$/) {
		push @sessargs, name => $name;
	}
	else {
		die "Badly formed name, \"$name\"\n";
	}
}
$session = $sessionmanager->login(@sessargs);
if (! isa($session, "Mooix::Thing")) {
	print "Login failure.\n";
	cleanup();
	exit;
}

# This requires the program be running as mooadmin or root, since it writes
# into the object.
my $socket = makesocket($session->untaint);
# Now drop those perms.
$>=$<;

# Run the parse method in the background as child.
my $child = fork;
if ($child == 0) {
	close $socket;
	$session->avatar->untaint->exec->parser_parse(session => $session);
}

# When the parse method exits, we're done.
$SIG{CHLD} = \&cleanup;

while (accept(Client, $socket)) {
	my $command=<Client>;
	my @data=<Client>;
	chomp $command;
	if ($command eq 'o') {
		output(@data);
	}
	elsif ($command eq 'i') {
		# Feed it one line from stdin.
		if (eof()) {
			print Client "exit\n";
		}
		else {
			my $line;
			do {
				$line=<>;
			} while ($line=~/^\s*#/);
			output("prompt> $line");
			print Client $line;
		}
	}
	close Client;
}

cleanup();
my $cleaning=0;
sub cleanup {
	return if $cleaning;
	$cleaning=1;
	$sessionmanager->logout(quiet => 1, nolog => 1, session => $session)
		if $session and -d $session->id;
}

sub output {
	my $time=gettimeofday;
	foreach (@_) {
		print $time.(" " x (20 - length $time))."\t$_";
	}
}
