/*
 * Call stacks are lists of objects.
 * They are stored in memory as linked lists, and can be serialized to disk.
 */

#include <dirent.h>
#include <sys/types.h>
#include <stdio.h>

/* A callstack. */
struct callstack {
	dev_t dev;		/* device object's directory is on */
	ino_t inode;		/* inode of object's directory */
	char *method;		/* the method name */
	int boundry;		/* set to indicate stack boundry */
	struct callstack *next;	/* pointer to next item on stack */
};

/* Given a method sandbox uid, returns the filename of its stack. */
char *callstack_file (uid_t uid);

/* Given a method sandbox uid, removes its stack file. */
int callstack_unlink (uid_t uid);

/* Walks over all the call stack files, and runs a function on the
 *  * uid of each. */
void callstack_walk ();

/* This trio is used to iterate over call stack files one at a time. */
typedef DIR callstack_iter; /* the iterator type */
callstack_iter *callstack_iter_start (void);
/* Returns method sandbox uid of the next callstack, 0 when no more. */
uid_t callstack_iter_next (callstack_iter *iter);
void callstack_iter_end (callstack_iter *iter);

/* Free the passed stack. */
void callstack_free (struct callstack *stack);

/* Returns the number of items in a stack. */
int callstack_size (struct callstack *stack);

/* Pushes an object onto the top of a stack. Returns the new stack. */
/* The method parameter is the method that is responsible for adding this
 * object onto the stack. Optional. */
struct callstack *callstack_push (struct callstack *stack, int objectfd, char *method);

/* Makes and returns a copy of the stack. */
struct callstack *callstack_copy (struct callstack *s);

/* The first stack is expanded, with the extention stack placed at its end.
 * The resulting stack is returned.
 *
 * Note: Don't stack_free() the extention stack, since it's now also part
 *       of the first stack. And freeing the first stack will now free the
 *       second as well. */
struct callstack *callstack_extend (struct callstack *stack, struct callstack *extention);

/* Removes any duplicate items from the passed stack. This is expensive,
 * but can be worth it for oft-used stacks. Returns the number of
 * items removed. Does not touch the topmost element of the stack. */
int callstack_dedup (struct callstack *s);

/* Inverts the stack, and returns a pointed to the new top of the stack. */
struct callstack *callstack_invert (struct callstack *s);

/* Returns true if one stack is a subset of the other; that is, if every
 * entry in the subset is also present in the superset. Duplicate entries
 * are fine. Note that it stops looking through the subset once it reaches
 * a stack boundry, so the subset can contain stuff after the boundry that
 * is not in the superset, and it will still return true.
 *
 * Also note that it's currently O(n^2). */
int callstack_is_subset(struct callstack *subset, struct callstack *superset);

/* Dumps a stack out to a FILEhandle. */
void callstack_dump (struct callstack *s, FILE *out);

/* Writes a stack to disk. The format is pairs of lines, for object and
 * then method, repeated. Stack boundries take the form of an empty line. */
int callstack_save (uid_t dest, struct callstack *s);

/* Reads a stack in from disk. It can read the whole stack, or, if onlytop
 * is set, reads only the topmost object on the stack, and does not even
 * read its method field. 
 *
 * This is used to read in user-supplied stacks as well, so should be
 * robust. */
struct callstack *callstack_load (const char *filename, int onlytop);

/* Just like callstack_load but reads from a fd. */
struct callstack *callstack_fromfd (FILE *in, int onlytop);

struct callstack *callstack_next (struct callstack *stack);
void callstack_setnext (struct callstack *stack, struct callstack *next);
char *callstack_method (struct callstack *stack);
void callstack_setmethod (struct callstack *stack, char *method);
int callstack_boundry (struct callstack *stack);
void callstack_setboundry (struct callstack *stack, int boundry);
dev_t callstack_dev (struct callstack *stack);
void callstack_setdev (struct callstack *stack, dev_t dev);
ino_t callstack_inode (struct callstack *stack);
void callstack_setinode (struct callstack *stack, ino_t inode);
