/*
 * Copyright (C), 2000-2005 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 *  System dependent device methods.
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *  @author Martin Pala, <martinp@tildeslash.com>
 *
 *  @version \$Id: sysdep_HPUX.c,v 1.11 2005/01/06 20:51:50 martinp Exp $
 *
 *  @file
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_SYS_VFS_H
# include <sys/vfs.h>
#endif

#ifdef HAVE_MNTENT_H
#include <mntent.h>
#endif

#include "monitor.h"
#include "device_sysdep.h"


/**
 * HPUX special block device mountpoint method. Filesystem must be mounted.
 * In the case of success, mountpoint is stored in device information
 * structure for later use.
 *
 * @param inf  Information structure where resulting data will be stored
 * @param blockdev Identifies block special device
 * @return         NULL in the case of failure otherwise mountpoint
 */
char *device_mountpoint_sysdep(Info_T inf, char *blockdev) {

  struct mntent *mnt;
  FILE          *mntfd;

  ASSERT(inf);
  ASSERT(blockdev);


  if((mntfd= setmntent("/etc/mnttab", "r")) == NULL) {
    log("%s: Cannot open /etc/mnttab file", prog);
    return NULL;
  }

  /* First match is significant */
  while((mnt= getmntent(mntfd)) != NULL) {

    if(IS(blockdev, mnt->mnt_fsname)) {

      endmntent(mntfd);
      return strncpy(inf->mntpath, mnt->mnt_dir, sizeof(inf->mntpath));

    }

  }

  endmntent(mntfd);

  return NULL;

}


/**
 * HPUX filesystem usage statistics. In the case of success result is stored in
 * given information structure.
 *
 * @param inf Information structure where resulting data will be stored
 * @return        TRUE if informations were succesfully read otherwise FALSE
 */
int device_usage_sysdep(Info_T inf) {

  struct statfs usage;

  ASSERT(inf);

  if(statfsdev(inf->mntpath, &usage) != 0) {
    log("%s: Error getting usage statistics for device '%s' -- %s\n",
        prog, inf->mntpath, STRERROR);
    return FALSE;
  }

  inf->f_bsize=           usage.f_bsize;
  inf->f_blocks=          usage.f_blocks;
  inf->f_blocksfree=      usage.f_bavail;
  inf->f_blocksfreetotal= usage.f_bfree;
  inf->f_files=           usage.f_files;
  inf->f_filesfree=       usage.f_ffree;

  return TRUE;

}

