/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mlist.c,v 1.26 2003/01/06 10:55:27 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mconfig.h"

mlist* mlist_sort_full_by_string (mlist *l) {
	return mlist_sort_limited_by_string(l, -1);
}

mlist* mlist_sort_limited_by_string (mlist *l, int count) {
	mlist *act, *first;
	int i = 0;

	if (l == NULL || count == 0) return l;

	first = l;
	act = l;

	if (count < 0) count = 0;

	for ( i = 0; (!count || (i < count)) && act; i++) {
		mlist *w;
		mlist *s = act;
		
		for (w = act->next; w; w = w->next) {
			if (strcmp(w->data->key, s->data->key) < 0 ) {
				s = w;
			}
		}

		if ( s != act ) {
		/* insert in the sorted part of the list */

		/* take 'act' from the list */
			s->prev->next = s->next;

			if (s->next)
				s->next->prev = s->prev;

			/* insert it at the right place */

			if (act == first) {
				s->prev = NULL;
				s->next = first;

				first->prev = s;

				first = s;
			} else {
				s->prev = act->prev;
				s->next = act;

				act->prev->next = s;
				act->prev = s;
			}
		} else {
			act = act->next;
		}
	}

	return first;
}

mlist *mlist_init () {
	mlist *l = malloc(sizeof(mlist));
	assert(l);

	l->prev = NULL;
	l->next = NULL;
	l->data = NULL;

	return l;
}

int mlist_free_entry(mlist *l) {
	if (!l)	return -1;
	if (l->data)	mdata_free(l->data);

	free(l);

	l = NULL;

	return 0;
}

int mlist_free(mlist *l) {
	mlist *act;
	if (!l) return 0;

	/* rewind to the beginnig */
	while (l->prev) {
		l = l->prev;
	}

	/* erase the list */
	while (l) {
		act = l;
		l = l->next;

		mlist_free_entry(act);

	}

	return 0;
}

int mlist_is_empty(mlist *l) {
	if (!l) return 1;

	return (l->data == NULL);
}

/*
	einfgen (eher ein append)
	bei gleichheit versuch des anhngens
*/

int mlist_insert (mlist *l, mdata *ins_data) {
	int inserted = 0;

	if (!l) return -1;
	if (!ins_data) return -1;
	
	if (NULL == ins_data->key) {
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INTERNALS, M_DEBUG_LEVEL_ERRORS,
			 "ins_data = %p, but its key is NULL\n", ins_data);
		return -1;
	}

	for (; l && l->data; l = l->next) {
		mdata *data = l->data;
		int r = 0;

		if (NULL == data->key) {
			fprintf(stderr, "%s.%d: no key -> %d\n", __FILE__, __LINE__, data->type);
		}
			
		r = strcmp(ins_data->key, data->key);
		
		if (0 == r) {
			if (mdata_append(data, ins_data) == M_DATA_APPENDED) {
				inserted = 1;

			/* the memory isn't used anymore */
				mdata_free(ins_data);
				ins_data = NULL;
				break;
			}
		}

		if (!l->next) break;
	}

	if (!inserted) {
		if (mlist_is_empty(l)) {
			l->data = ins_data;
		} else {
			mlist *n = mlist_init();

			n->data = ins_data;
			n->prev = l;
			l->next = n;
		}
	}

	return 0;
}

/*
	einfgen (d.h. append)
	bei gleichheit ersetzen
*/

int mlist_insert_replace (mlist *l, mdata *ins_data) {
	int inserted = 0;

	if (!l) return -1;
	if (!ins_data) return -1;
	
	for (; l && l->data;  l = l->next) {
		mdata *data = l->data;
		
		if (data->key == NULL) {
			fprintf(stderr, "%s.%d: no key -> %d\n", __FILE__, __LINE__, data->type);
		}

		if (0 == strcmp(ins_data->key, data->key)) {
			/* Free old Data */
			mdata_free(data);
			l->data = ins_data;
			ins_data = NULL;
			inserted = 1;
			break;
		}

		if (!l->next) break;
	}

	if (!inserted) {
		if (mlist_is_empty(l)) {
			l->data = ins_data;
		} else {
			mlist *n = mlist_init();

			n->data = ins_data;
			n->prev = l;
			l->next = n;
		}
	}

	return 0;
}

/**
 * sortiert einfgen, aufsteigened
 * bei gleichheit versuchen anzuhngen
 *
 * @return -1 one error, 0 if appended, 1 if inserted
 */

int mlist_insert_sorted (mlist *l, mdata *ins_data) {
	int ins_me = 0;

	if (!l) return -1;
	if (!ins_data) return -1;
	
	for (; l && l->data; l = l->next) {
		mdata *data = l->data;
		int res = 0;
		
		if (data->type != ins_data->type) {
			fprintf(stderr, "%s.%d: die()\n", __FILE__, __LINE__);
			return -1;
		}
		
		res = strcmp(ins_data->key, data->key);

		if (res == 0) {
			if (mdata_append(data, ins_data) == M_DATA_APPENDED) {
				ins_me = -1;
			/* the memory isn't used anymore */
				mdata_free(ins_data);
				ins_data = NULL;
				break;
			}
		} else if (res < 0) {
			ins_me = 1;
			break;
		}

		if (!l->next) break;
	}

	/* every thing is done, just leave the building */
	if (ins_me == -1) {

	} else if (!l->data) {
		l->data = ins_data;
	} else if (ins_me) {
		mlist *n = mlist_init();

		n->data = l->data;
		l->data = ins_data;
		n->next = l->next;
		n->prev = l;
		if (n->next)
			n->next->prev = n;
		l->next = n;
	} else {
		mlist *n = mlist_init();

		n->data = ins_data;
		n->prev = l;
		n->next = l->next;
		if (n->next)
			n->next->prev = n;
		l->next = n;
	}

	return (ins_me == -1 ? 0 : 1);
}

int mlist_append (mlist *l, mdata *ins_data) {
	int inserted = 0;

	if (!l) return -1;
	if (!ins_data) return -1;

	while(l && l->data && l->next) {
		l = l->next;
	}
	if (!inserted) {
		if (mlist_is_empty(l)) {
			l->data = ins_data;
		} else {
			mlist *n = mlist_init();

			n->data = ins_data;
			n->prev = l;
			l->next = n;
		}
	}

	return 0;
}

int mlist_write(gzFile *fd, mlist *l) {
	while (l) {
		if (l->data) {
			mdata_write(fd, l->data);
		}
		l = l->next;
	}
	return 0;
}

int mlist_count(mlist *l) {
	int c = 0;
	if (!l) return 0;

	while (l) {
		if (l->data) c++;
		l = l->next;
	}

	return c;
}

mdata *mlist_get_data(mlist *l, const char *str) {
	mdata *data = NULL;

	for (; l && l->data; l = l->next) {
		data = l->data;

		if (!strcmp(str, data->key)) {
			break;
		}

		data = NULL;

		if (!l->next) break;
	}

	return data;
}

int mlist_in_list(mlist *l, const char *str) {
	mdata *data = NULL;

	for (; l && l->data; l = l->next) {
		data = l->data;

		if (!strcmp(str, data->key)) {
			return 1;
		}

		data = NULL;

		if (!l->next) break;
	}

	return 0;
}

int mlist_sumup(mlist *l) {
	int c = 0;
	if (!l) return 0;

	for (; l && l->data; l = l->next) {
		c += mdata_get_count(l->data);
	}

	return c;
}

double mlist_sumup_vcount(mlist *l) {
	double c = 0;
	if (!l) return 0;

	while (l) {
		if (l->data) {
			c += mdata_get_vcount(l->data);
		}
		l = l->next;
	}

	return c;
}
