/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mdatatypes.h,v 1.49 2003/01/06 10:52:24 ostborn Exp $
*/

#ifndef _M_DATATYPES_H_
#define _M_DATATYPES_H_

#include <pcre.h>
#include <zlib.h>
#include <stdio.h>

#include "config.h"
#ifdef HAVE_LIBADNS
#include <adns.h>
#endif

#include "mrecord.h"

#ifdef HAVE_LIBXML
#include <libxml/tree.h>
#elif HAVE_EXPAT
#include <expat.h>
/* emulate libxml2 */
#define xmlChar XML_Char
#else
# error no XML parser
#endif

/* #define DEBUG_DATATYPES */

typedef enum { M_HIST_TYPE_UNSET,
		M_HIST_TYPE_WEB,
		M_HIST_TYPE_TELECOM,
		M_HIST_TYPE_TRAFFIC,
		M_HIST_TYPE_MAIL
} mdata_hist_type;

enum { M_SORTBY_KEY, M_SORTBY_COUNT, M_SORTBY_VCOUNT,M_SORTBY_QUOTIENT };
enum { M_SORTDIR_ASC, M_SORTDIR_DESC };
enum { M_DATA_KEY_TYPE_STRING, M_DATA_KEY_TYPE_NDX };

/**
 * return type of mdata_append
 */

typedef enum {
	M_DATA_FAILED           =-1,
	M_DATA_APPENDED		= 0,
	M_DATA_NOT_APPENDED	= 1
} mdata_append_return;

typedef enum {
	M_DATA_FIELDTYPE_UNSET,	M_DATA_FIELDTYPE_HASH,                               /* 0 */
		M_DATA_FIELDTYPE_LONG, M_DATA_FIELDTYPE_LIST,                        /* 2 */
		M_DATA_FIELDTYPE_STRING, M_DATA_FIELDTYPE_DOUBLE,                    /* 4 */
		M_DATA_FIELDTYPE_WEB_ARRAY, M_DATA_FIELDTYPE_MAIL_ARRAY,             /* 6 */
		M_DATA_FIELDTYPE_TELECOM_ARRAY,                                      /* 8 */

		/*M_DATA_TYPE_UNSET, */M_DATA_TYPE_COUNT, M_DATA_TYPE_VISITED,       /* 9 */
		M_DATA_TYPE_BROKENLINK, M_DATA_TYPE_TRAFFIC, M_DATA_TYPE_VISITELEM,  /* 11 */
		M_DATA_TYPE_VISIT, M_DATA_TYPE_WEBHIST, M_DATA_TYPE_STRING,          /* 14 */
		M_DATA_TYPE_STATE, M_DATA_TYPE_QUERY, M_DATA_TYPE_MATCH,             /* 17 */
		M_DATA_TYPE_RECORD, M_DATA_TYPE_SUBLIST, M_DATA_TYPE_SPLIT,          /* 20 */
		M_DATA_TYPE_NETMASK, M_DATA_TYPE_PLUGIN, M_DATA_TYPE_MAILHIST,       /* 23 */
		M_DATA_TYPE_LOCATION                                                 /* 26 */

} mdata_field_type;


#define mdata_type mdata_field_type
#define M_DATA_TYPE_UNSET M_DATA_FIELDTYPE_UNSET


/* xml read */
typedef enum {M_TAG_UNSET, M_TAG_BEGIN, M_TAG_END, M_TAG_TEXT
} m_tag;

#define M_STATE_ST_ELEM 128

typedef struct {
	int (*function)(void *m, m_tag tagtype, const xmlChar *value, const xmlChar **attrs);
	void *data;
	mdata_field_type type;

} mstate_stack_elem;

typedef struct {
	void *ext_conf;
	void *state;
	mstate_stack_elem st[M_STATE_ST_ELEM];
	int st_depth;
	
	char *textbuf;
	int textbuf_size;
} mstate_stack;

/**
 * values for 'grouped'
 *   (no enum 'cause grouped is used for other values too)
 */
#define	M_DATA_STATE_PLAIN	0
#define	M_DATA_STATE_GROUPED	1

typedef enum {
	M_DATA_VISIT_USER, M_DATA_VISIT_ROBOT
} mdata_visit;

typedef enum {
	M_SPLIT_FIELD_UNSET, M_SPLIT_FIELD_DEFAULT, M_SPLIT_FIELD_SRVHOST,
		M_SPLIT_FIELD_SRVPORT, M_SPLIT_FIELD_REQURL,
		M_SPLIT_FIELD_REQUSER, M_SPLIT_FIELD_REQHOST,
		M_SPLIT_FIELD_REFURL
} mdata_split;

typedef struct {
	signed int	count;
	unsigned int	grouped;
} data_Count;

typedef struct {
	signed int	count;
	double	        vcount; /* is also used for traffic */
	unsigned int	grouped;
} data_Visited;

typedef struct {
	unsigned int	id;
	unsigned long	netmask;
	unsigned long	hostmask;
} data_Netmask;

typedef struct {
	signed int	count;
	unsigned int	grouped;
	time_t		timestamp;
	char		*referrer;
} data_BrokenLink;

/**
	carries the incomming and outgoing traffic for one station
	( host, network, router, ...)

	key	: name (ip, phonenumber)
	count	: number of accesses
*/

typedef struct {
	signed int	count;
	char            *src;
	char            *dst;
	int             src_port;
	int             dst_port;
	int             src_as;
	int             dst_as;
	int             src_interface;
	int             dst_interface;


	unsigned long xfer_in;
	unsigned long xfer_out;
} data_Traffic;

typedef struct {
	time_t		timestamp;
	int		vid;
} data_VisitElement;

typedef struct {
	signed int	count;
	mdata_visit	type;                 /**< robot/user */
	time_t		timestamp;
	time_t		timediff;

	void            *hits;                /**< typecast to mlist* */

	char 		*useragent;
} data_Visit;

typedef struct {
	unsigned long	hits;
	unsigned long	files;
	unsigned long	pages;
	unsigned long	visits;
	unsigned long	hosts;
	double		xfersize;
} data_WebHistory;

typedef struct {
	long incoming_mails;
	long outgoing_mails;

	long incoming_bytes;
	long outgoing_bytes;
} data_MailHistory;

typedef struct {
/* web */
	union {
		data_WebHistory web;
		data_MailHistory mail;
	} data;

/* general */
	unsigned int	year;
	unsigned int	month;
	unsigned int	week;

	unsigned int	days_passed;

	mdata_hist_type type;
} data_History;

typedef struct {
	char *string;
} data_String;

typedef struct {
	/* typecast to mstate */
	void	*state;
	/* typecast to mlist */
	void	*history;
} data_State;

typedef struct {
	pcre *match;
	pcre_extra *study;
} data_Match;

typedef struct {
	pcre		*match;
	mdata_split	fieldtype;
} data_Split;

typedef struct {
	mlogrec	*record;
} data_Record;

typedef struct {
	/* typecast to mlist */
	void *sublist;

	int count;
} data_SubList;

#ifdef HAVE_LIBADNS
typedef struct {
	adns_query *query;
	char *hostname;
} data_Query;
#endif

typedef struct {
	int count;
	
	char *city;
	char *province;
	char *country;
	
	char *provider;
} data_Location;

/**
 * the internal datastructure for all datatypes which should be included
 * in a hash or a list
 */

typedef struct {
	char *key;

	mdata_type	type;/**< type specifier */
#if 0
	long            id; /**< data id (debugging) */
#endif
	union {
		/* the rule for a '*' is simple: 
		 * 
		 * if (sizeof(data_...) > 16) use_'*' 
		 * else dont_use_'*';
		 * 
		 * we use that the reduce the memory consumption
		 */
		data_Count	count;
		data_Visited	visited;
		data_BrokenLink	brokenlink;
		data_Traffic	* traffic;
		data_VisitElement visitelem;
		data_Visit	* visit;
		data_History    * hist;
		data_String	string;
		data_Match	match;
		data_Split	split;
	/* use by the sorter */
		data_Record	record;
		data_State	state;
		data_SubList	sublist;
		data_Netmask    netmask;
#ifdef HAVE_LIBADNS
		data_Query	query;
#endif
		data_Location   *location;
	} data;
} mdata;

/**
 * the internal structure for the xml parsing process
 */
typedef struct {
	char	*string;
	int	type;
} mdata_values;

mdata  *mdata_init();
void	mdata_free(mdata *data);
int	mdata_write(gzFile *fd, mdata *data);
int     mdata_read(void *user_data, m_tag tagtype, const xmlChar *value, const xmlChar **attrs);
int     mdata_show(const mdata *data);

mdata_append_return	mdata_append(mdata *dst, mdata *src);
mdata  *mdata_copy(mdata *data);
int	mdata_get_count(mdata *data);
double  mdata_get_vcount(const mdata *data);
int	mdata_set_count(mdata *data, int count);
int     mdata_insert_value(void *user_data, m_tag tagtype, const xmlChar *value, const xmlChar **attrs);

void    mdata_array_msort(mdata **md, int *a, int *b, int l, int r, int sortby, int sortdir);
int     mdata_is_grouped(mdata *data);

int     mdata_get_datatype_from_key(const char *key);
int     mdata_get_key_type(const mdata *data);

const char *mdata_get_key(mdata *data, void *state);

#endif
