/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.14 2002/10/23 15:05:25 le_zas Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <zlib.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "mconfig.h"
#include "misc.h"

/* begin of Visited */

int mdata_Visited_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.visited.x, #x);

	WR(count, d);
	WR(grouped, d);
	WR(vcount, .0f);
#undef WR
	return 0;
}

int mdata_Visited_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_VISITED) return -1;

	return 0;
}

int mdata_Visited_setdata(mdata *data, const char *str, int count, int type, double vcount) {
	if (data == NULL || str == NULL) {
		fprintf(stderr, "%s.%d: (visited) either data (%p) or str (%p) is NULL\n",
			__FILE__,
			__LINE__,
			data,
			str);

		return -1;
	}


	data->key = strdup(str);
	assert(data->key);

	if (data->type == M_DATA_TYPE_UNSET)
		data->type		= M_DATA_TYPE_VISITED;

	data->data.visited.count	= count;
	data->data.visited.grouped	= type;
	data->data.visited.vcount	= vcount;

	return 0;
}
int mdata_Visited_from_xml(void *user_data, int tagtype, const xmlChar *value, const xmlChar **attrs) {
	int i;
	const mdata_values data_values[] = {
		{ "grouped", M_DATA_FIELDTYPE_LONG },
		{ "count", M_DATA_FIELDTYPE_LONG  },
		{ "vcount", M_DATA_FIELDTYPE_DOUBLE  },

		{ NULL, M_DATA_FIELDTYPE_UNSET }
	};

	mstate_stack *m = user_data;

//	M_WP();

	switch(tagtype) {
	case M_TAG_BEGIN:
		for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
			;

		if (data_values[i].string == NULL) {
			M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "unknown tag '%s'\n",
				 value);
			return -1;
		}

#define SET_DATA(x) \
	m->st[m->st_depth].data = &(((mdata *)(m->st[m->st_depth-1].data))->x);
		switch(i) {
		case 0: SET_DATA(data.visited.grouped); break;
		case 1: SET_DATA(data.visited.count); break;
		case 2: SET_DATA(data.visited.vcount); break;
		default:
			return -1;
		}
#undef SET_DATA
		/* the same for all 4 values */
		m->st[m->st_depth].function = mdata_insert_value;
		m->st[m->st_depth].type = data_values[i].type;

		break;
	case M_TAG_END:
		((mdata *)(m->st[m->st_depth-1].data))->type = M_DATA_TYPE_VISITED;

		switch(m->st[m->st_depth-2].type) {
		case M_DATA_FIELDTYPE_HASH: {
			mhash *h = m->st[m->st_depth-2].data;
			mdata *data = m->st[m->st_depth-1].data;

			mhash_insert_sorted(h, data);
			break;
		}
		default:
			M_WP();
		}

		break;
	case M_TAG_TEXT:
		M_WP();
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n",
			 tagtype);
		return -1;
	}

	return 0;
}



int mdata_Visited_append(mdata *dst, mdata *src) {
	dst->data.visited.count += src->data.visited.count;
	dst->data.visited.vcount += src->data.visited.vcount;

	return 0;
}

mdata *mdata_Visited_init() {
	mdata *data = mdata_init();
	assert(data);

	data->key		= NULL;
	data->type		= M_DATA_TYPE_VISITED;

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */
	data->data.visited.count	= 0;
	data->data.visited.grouped	= M_DATA_STATE_PLAIN;
	data->data.visited.vcount	= 0;

	return data;
}

mdata *mdata_Visited_copy(mdata *src) {
	mdata *data = mdata_Visited_init();

	mdata_Visited_setdata(data,
		src->key,
		src->data.visited.count,
		src->data.visited.grouped,
		src->data.visited.vcount);

	return data;
}

mdata *mdata_Visited_create(const char *str, int count, int type, double vcount) {
	mdata *data = mdata_Visited_init();

	mdata_Visited_setdata(data, str, count, type, vcount);

	return data;
}

/* end of Visited */
