/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.24 2002/10/23 15:05:20 le_zas Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "misc.h"

/* begin of BrokenLink */

int mdata_BrokenLink_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.brokenlink.x, #x);
#define WRS(x,f) \
	if (data->data.brokenlink.x != NULL) { \
		char *s = url_encode(data->data.brokenlink.x); \
		gzprintf(fd, "<%s>", #x); \
		gzwrite(fd, s, strlen(s)); \
		gzprintf(fd, "</%s>", #x); \
		free(s); \
	} else { \
		gzprintf(fd, "<%s />", #x); \
	}

	WR(count, d);
	WR(grouped, d);
	WR(timestamp, ld);
	WRS(referrer, s);
#undef WRS
#undef WR

	return 0;
}

int mdata_BrokenLink_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_BROKENLINK) return -1;

	if (data->data.brokenlink.referrer)	free(data->data.brokenlink.referrer);

	return 0;
}

int mdata_BrokenLink_setdata(mdata *data, const char *str, int count, int type,
	time_t timestamp, const char *referrer) {

	assert(str);

	data->key	= strdup(str);
	assert(data->key);

	if (data->type == M_DATA_TYPE_UNSET)
		data->type		= M_DATA_TYPE_BROKENLINK;

	data->data.brokenlink.count	= count;
	data->data.brokenlink.grouped	= type;
	data->data.brokenlink.timestamp = timestamp;


	if (referrer) {
		data->data.brokenlink.referrer = strdup(referrer);
		assert(data->data.brokenlink.referrer);
	} else {
		data->data.brokenlink.referrer = NULL;
	}

	return 0;
}

int mdata_BrokenLink_from_xml(void *user_data, m_tag tagtype, const xmlChar *value, const xmlChar **attrs) {
	int i;
	const mdata_values data_values[] = {
		{ "grouped", M_DATA_FIELDTYPE_LONG },
		{ "count", M_DATA_FIELDTYPE_LONG },
		{ "timestamp", M_DATA_FIELDTYPE_LONG },
		{ "referrer", M_DATA_FIELDTYPE_STRING },

		{ NULL, M_DATA_FIELDTYPE_UNSET }
	};

	mstate_stack *m = user_data;

//	M_WP();

	switch(tagtype) {
	case M_TAG_BEGIN:
		for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
			;

		if (data_values[i].string == NULL) {
			M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "unknown tag '%s'\n",
				 value);
			return -1;
		}

#define SET_DATA(x) \
	m->st[m->st_depth].data = &(((mdata *)(m->st[m->st_depth-1].data))->x);

		switch(i) {
		case 0: SET_DATA(data.brokenlink.grouped); break;
		case 1: SET_DATA(data.brokenlink.count); break;
		case 2: SET_DATA(data.brokenlink.timestamp); break;
		case 3: SET_DATA(data.brokenlink.referrer); break;
		default:
			return -1;
		}
#undef SET_DATA
		/* the same for all 4 values */
		m->st[m->st_depth].function = mdata_insert_value;
		m->st[m->st_depth].type = data_values[i].type;

		break;
	case M_TAG_END:
		((mdata *)(m->st[m->st_depth-1].data))->type = M_DATA_TYPE_BROKENLINK;

		switch(m->st[m->st_depth-2].type) {
		case M_DATA_FIELDTYPE_HASH: {
			mhash *h = m->st[m->st_depth-2].data;
			mdata *data = m->st[m->st_depth-1].data;

			mhash_insert_sorted(h, data);
			break;
		}
		case M_DATA_FIELDTYPE_LIST: {
			mlist *h = m->st[m->st_depth-2].data;
			mdata *data = m->st[m->st_depth-1].data;

			mlist_append(h, data);
			break;
		}
		default:
			M_WP();
		}
#if 0
		fprintf(stderr, "Brokenlink: %s, %d, %d, %ld, %s\n",
			((mdata *)(m->st[m->st_depth-1].data))->key,
			((mdata *)(m->st[m->st_depth-1].data))->data.brokenlink.grouped,
			((mdata *)(m->st[m->st_depth-1].data))->data.brokenlink.count,
			((mdata *)(m->st[m->st_depth-1].data))->data.brokenlink.timestamp,
			((mdata *)(m->st[m->st_depth-1].data))->data.brokenlink.referrer
			);
#endif
		break;
	case M_TAG_TEXT:
		M_WP();
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n",
			 tagtype);
		return -1;
	}

	return 0;
}



int mdata_BrokenLink_append(mdata *dst, mdata *src) {
	dst->data.brokenlink.count += src->data.brokenlink.count;
	return M_DATA_APPENDED;
}

mdata *mdata_BrokenLink_init() {
	mdata *data = mdata_init();
	assert(data);

	data->key		= NULL;
	data->type		= M_DATA_TYPE_BROKENLINK;

#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */
	data->data.brokenlink.count	= 0;
	data->data.brokenlink.grouped	= M_DATA_STATE_PLAIN;
	data->data.brokenlink.timestamp	= 0;
	data->data.brokenlink.referrer	= NULL;


	return data;
}

int mdata_BrokenLink_show(const mdata *data) {
	if (!data) return -1;
	if (!data->type == M_DATA_TYPE_BROKENLINK) return -1;

	        fprintf(stderr, "* datatype: brokenlink\n");

	if (data->key == NULL) {
		fprintf(stderr, "* key     : (unset)\n");
	} else {
		fprintf(stderr, "* key     : %s\n", data->key);
		fprintf(stderr, "* count   : %d\n", data->data.brokenlink.count);
		fprintf(stderr, "* grouped : %d\n", data->data.brokenlink.grouped);
		fprintf(stderr, "* tstamp  : %ld\n", data->data.brokenlink.timestamp);
		fprintf(stderr, "* referrer: %s\n", data->data.brokenlink.referrer);
	}

	return 0;
}

mdata *mdata_BrokenLink_copy(mdata *src) {
	mdata *data = mdata_BrokenLink_init();
	assert(data);
	mdata_BrokenLink_setdata(data, src->key,
		src->data.brokenlink.count,
		src->data.brokenlink.grouped,
		src->data.brokenlink.timestamp,
		src->data.brokenlink.referrer
	);

	return data;
}

mdata *mdata_BrokenLink_create(const char *str,
	int count, int grouped,
	time_t timestamp, const char *referrer) {
	mdata *data = mdata_BrokenLink_init();
	assert(data);
	mdata_BrokenLink_setdata(data, str, count, grouped, timestamp, referrer);

	return data;
}

/* end of BrokenLink */
