/* libc */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <getopt.h>
/* other libs */
#include <db.h>
#include <apr.h>
#include <apr_pools.h>
/* libbtt */
#include <libbttracker.h>

const static struct option opts[] = {
 { "master", 0, 0, 'm' },
 { "help", 0, 0, 'h' },
 { 0, 0, 0, 0 }
};

static int run = 1;
static int stopped = 0;

static void sigINT(int signum) {
 run = 0;
 stopped = 1;
 return;
}

static void usage(FILE *fh) {
 fprintf(fh,
  "btt_db2xml mod_bt/"
  VERSION
  " - http://www.crackerjack.net/mod_bt/\n"
  "* Prints a tracker's infohash information to standard output in XML.\n\n"
  "Usage: btt_db2xml [--master] [--help] homedir\n"
  "\thomedir\t\t\tmod_bt data directory (eg; /var/lib/mod_bt)\n\n"
  "\t-m\t--master\tAct as mod_bt master\n"
  "\t\t\t\t(Use when mod_bt is not running)\n"
  "\t-h\t--help\t\tThis help screen\n\n"
 );

 return;
}

int main(int argc, char** argv) {
 int master = 0;
 int rv = 0;
 int ret = 0;
 btt_tracker* tracker;
 DBT hash_key;
 DBT hash_val;
 DBC* hash_cur = NULL;
 DB_TXN* txn = NULL;
 int odone = 0;
 int oindex = 0;
 xmlDocPtr document;
 xmlNodePtr docnode;
    
 bzero(&hash_key, sizeof(hash_key));
 bzero(&hash_val, sizeof(hash_val));

 while(odone != -1) {
  odone = getopt_long(argc, argv, "mhp", opts, &oindex);
  switch(odone) {
   case -1:
    break;
   case 'm':
    master++;
    break;
   case 'h':
    usage(stdout);
    exit(0);
    break;
   case '?':
    usage(stderr);
    exit(1);
    break;
   default:
    fprintf(stderr, "unknown option code 0x%x\n", odone);
    break;
  }
 }

 if(argc - optind != 1) {
  usage(stderr);
  exit(2);
 }
 
 if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
  fprintf(stderr, "apr_app_initialize() failed!\n");
  fflush(stderr);
  exit(20);
 }
 
 atexit(apr_terminate);
 
 if(!(tracker = btt_tracker_alloc(NULL, argv[optind], master))) {
  fprintf(stderr, "bt_tracker_alloc() failed!\n");
  fflush(stderr);
  exit(5);
 }
 
 if(master)
  tracker->s->start_t = time(NULL);

 signal(SIGINT, sigINT);
 signal(SIGTERM, sigINT);
 signal(SIGPIPE, sigINT);

 btt_tracker_refresh_stats(tracker);

 document = xmlNewDoc(BAD_CAST "1.0");
 docnode = xmlNewDocNode(document, NULL, BAD_CAST "Infohashes", NULL);

 if((ret = btt_txn_start(tracker, NULL, &txn, 0)) != 0) {
  tracker->db.env->err(tracker->db.env, ret, "start");
  goto err;
 }
 
 if((ret = tracker->db.hashes->cursor(tracker->db.hashes, txn, &hash_cur, 0)) != 0) {
  tracker->db.hashes->err(tracker->db.hashes, ret, "cursor");
  goto err;
 }

 while(!ret)
  if((ret = hash_cur->c_get(hash_cur, &hash_key, &hash_val, DB_NEXT)) == 0)
   xmlAddChild(docnode, btt_infohash2nodes(tracker->p, document, (btt_infohash*)hash_val.data));

 if((ret != 0) && (ret != DB_NOTFOUND)) {
  rv = 12;
  goto err;
 }
 
 xmlDocSetRootElement(document, docnode);
 xmlDocFormatDump(stdout, document, 1);

 err:
 
 xmlFreeDoc(document);

 if(hash_cur)
  hash_cur->c_close(hash_cur);
 
 if(txn) {
  if(rv) {
   txn->abort(txn);
  } else if((ret = txn->commit(txn, 0)) != 0) {
   rv = 13;
   tracker->db.env->err(tracker->db.env, ret, "commit");
  }
 }

 if(!btt_tracker_free(&tracker, master)) {
  fprintf(stderr, "bt_tracker_free() failed!\n");
  exit(9);
 }
 
 exit(rv);
}
