/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include <libbtutil.h>
#include <libbtpeer.h>

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>

#include <apr.h>
#include <apr_pools.h>
#include <apr_errno.h>
#include <apr_strings.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>

static const char* banner = "\
btp_makemetafile "VERSION" - generate BitTorrent metainfo files\n\
\n\
";

static apr_status_t progress(btp_torrent* t, btp_peer* p, void* data) {
    printf(
        "Generating... %.02f%% complete      \r",
        btp_torrent_percent_complete(t)
    );
    fflush(stdout);
    return APR_EAGAIN;
}

int main(int argc, char **argv) {
    apr_pool_t*         pool = NULL;
    btp_file_pool*      file_pool = NULL;
    apr_status_t        ret;
    bt_metainfo* info;
    btp_torrent* torrent;
    char buf[255];
    char* pp;
    char* torrent_file;
    int i;

    printf(banner);
    
    if(argc < 3) {
        printf(
            "usage: %s file announce_url\n",
            argv[0]
        );
        exit(3);
    }
    
    if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_app_initialize() failed!\n");
        fflush(stderr);
        exit(4);
    }
 
    atexit(apr_terminate);
 
    if(apr_pool_initialize() != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_initialize() failed!\n");
        fflush(stderr);
        exit(4);
    }
 
    if(apr_pool_create(&pool, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_pool_create() failed!\n");
        fflush(stderr);
        exit(4);
    }

    apr_hook_global_pool = pool;
    
    file_pool = btp_file_pool_create(pool, 2);

    info = apr_pcalloc(pool, sizeof(bt_metainfo));
    info->piece_size = 1048576; /* 2^20 */
    
    if(
        (
            ret = bt_url2args(
                argv[2], info->tracker_address, &(info->tracker_port),
                info->tracker_announce
            )
        )
        != APR_SUCCESS
    ) {
        fprintf(stderr, "Bad announce url \"%s\".\n", argv[2]);
        exit(3);
    }
    
    while(argv[1][strlen(argv[1])-1] == '/') {
        argv[1][strlen(argv[1])-1] = 0;
    }
    
    if((ret = bt_metainfo_init_files(info, pool, argv[1])) != APR_SUCCESS) {
        fprintf(
            stderr, "failed to scan %s: %s\n", argv[1],
            apr_strerror(ret, buf, sizeof(buf))
        );
        exit(4);
    }
    
    info->creation_date = time(NULL);
    torrent = btp_torrent_create(pool, info, file_pool);
    
    if(strlen(argv[1]) > BT_PATH_LEN - 1) {
        fprintf(stderr, "%s: path too long\n", argv[1]);
        exit(3);
    }
    
    if((pp = rindex(argv[1], '/'))) {
        BT_STRCPY(torrent->destination, argv[1]);
        *(torrent->destination + (pp - argv[1])) = 0;
    } else {
        BT_STRCPY(torrent->destination, ".");
    }
    
    printf(
        "Generating a torrent for %i file(s) in \"%s\"...\n",
        info->file_count, torrent->destination
    );

    for(i=0; i<info->file_count; i++) {
        printf(" %s (%lld)\n", info->files[i].name, info->files[i].length);
    }
    
    printf(
        "\nTotal Size = %llu, Piece Size = %u, Piece Count = %i\n",
        info->total_size, info->piece_size, info->piece_count
    );
    
    torrent->pieces = btp_torrent_pieces_init(torrent);
    
    btp_hook_metainfo_sha1(progress, NULL, NULL, APR_HOOK_LAST);
    
    if((ret = btp_metainfo_generate_hash(torrent, NULL)) != APR_SUCCESS) {
        fprintf(
            stderr, "failed to generate hash for %s: %s\n", argv[1],
            apr_strerror(ret, buf, sizeof(buf))
        );
        exit(4);
    }
    
    torrent_file = apr_pstrcat(pool, argv[1], ".torrent", NULL);
    
    if((ret = bt_metainfo_save(info, pool, torrent_file)) != APR_SUCCESS) {
        fprintf(
            stderr, "failed to save torrent %s: %s\n", torrent_file,
            apr_strerror(ret, buf, sizeof(buf))
        );
        exit(4);
    }
    
    printf("\n\n");
    
    
    exit(0);
}
