/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
#include <stdlib.h>
/* other libs */
#include <db.h>
#include <apr.h>
#include <apr_pools.h>
/* local */
#include <libbttracker.h>

/* data is set to a bt_iter_content struct */
/* modifies peer, saves */
int btt_iter_make_peerlist_string(
    apr_pool_t* p, DB_TXN* txn, DBC* cursor, DBT* key, DBT* val, void* data
) {
    btt_iter_content* c = (btt_iter_content*) data;
    btt_peer* peer = (btt_peer*) val->data;
    btt_peer* out_peer = c->hashandpeer.peer;
    btt_tracker* tracker = c->hashandpeer.tracker;
    char* content;
    int len;
    int ret;

    if(out_peer->flags & BTT_PEER_COMPACT)
        len = btt_peer2compact(c->pool, peer, &content);
    else if(out_peer->flags & BTT_PEER_NOPEERID)
        len = btt_peer2nopeerid(c->pool, peer, &content);
    else if(out_peer->flags & BTT_PEER_HTML)
        len = btt_peer2tr(c->pool, peer, &content);
    else
        len = btt_peer2bencode(c->pool, peer, &content);

    if(len) {
        if(!(out_peer->flags & BTT_PEER_HTML)) {
            peer->last_serve_t = time(NULL);
            peer->serves++;
            if(
                (ret = tracker->db.peers->put(
                    tracker->db.peers, txn, key, val, 0
                )) != 0
            ) {
                tracker->db.env->err(
                    tracker->db.env, ret,
                    "bt_iter_make_peerlist_string(): peers->put()"
                );
                return ret;
            }
            out_peer->num_got++;
      }


        if(c->buffer_length < (c->content_length + len + BT_PEERSTR_LEN)) {
            fprintf(
                stderr, "btt_iter_make_peerlist string(): buffer too short "
                "(%u bytes), allocating %u more\n", c->buffer_length,
                (len*2) + BT_PEERSTR_LEN
            );

            c->buffer_length += (len*2) + BT_PEERSTR_LEN;

            if(!(c->content = realloc(c->content, c->buffer_length))) {
                fprintf(
                    stderr, "btt_iter_make_peerlist string(): "
                    "Couldn't allocate %u bytes!\n", c->buffer_length
                );
                return APR_ENOMEM;
            }
        }
  
        memcpy(&(c->content[c->content_length]), content, len);
        c->content_length += len;
        c->content[c->content_length] = 0;
        return 0;
    }

    return BT_CALLBACK_STOP;
}

/* data is set to a bt_iter_hashandpeer struct with requesting peer + infohash */
/* modifies nothing */
int btt_iter_make_peerlist_check(
    apr_pool_t* p, DB_TXN* txn, DBC* cursor, DBT* key, DBT* val, void* data
) {
    btt_peer* out_peer = ((btt_iter_hashandpeer*)data)->peer;
    btt_infohash* infohash = ((btt_iter_hashandpeer*)data)->oldhash;
    btt_peer* peer = (btt_peer*) val->data;
    int factor;

    if(out_peer->num_got >= out_peer->num_want) {
        if(((btt_iter_hashandpeer*)data)->updating)
            return BT_CALLBACK_STOP;
        else
            return BT_CALLBACK_STOP_ALL;
    }

    if(!memcmp(out_peer->peerid, peer->peerid, BT_PEERID_LEN))
        return BT_CALLBACK_STOP;

    if((BTT_PEER_IS_SHIELD(peer)) && (infohash->shields < infohash->seeds))
        return BT_CALLBACK_STOP;
   
    if((BTT_PEER_IS_SHIELD(out_peer)) && (infohash->shields < infohash->seeds))
        return BT_CALLBACK_STOP;

    if(peer->flags & BTT_PEER_SHUNNED)
        return BT_CALLBACK_STOP;

    if(out_peer->flags & BTT_PEER_SHUNNED)
        return BT_CALLBACK_STOP;

    if(infohash->peers <= out_peer->num_want)
        return 0;

    if((!peer->num_want) || (!(infohash->peers)))
        return 0;

    factor = infohash->peers / peer->num_want;

    if(factor <= 2)
        factor = 1;

    if(factor == 1)
        return 0;

    if(!(rand() % factor))
        return 0;
 
    return BT_CALLBACK_STOP;
}
