/* libc */
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <getopt.h>
#include <time.h>
/* other libs */
#include <db.h>
#include <apr.h>
#include <apr_pools.h>
/* libbtt */
#include <libbttracker.h>
#include <libbtutil.h>

#define IOPT_MASTER     0x01
#define IOPT_DELETE     0x02
#define IOPT_FORCE      0x04
#define IOPT_CREATE     0x08
#define IOPT_VERBOSE    0x10
#define IOPT_REGISTER   0x20
#define IOPT_UNREGISTER 0x40

const static struct option opts[] = {
    { "help",       0, 0, 'h' },
    { "master",     0, 0, '!' },
    { "verbose",    0, 0, 'v' },
    { "delete",     0, 0, 'd' },
    { "force",      0, 0, 'f' },
    { "create",     0, 0, 'c' },
    { "metainfo",   1, 0, 'm' },
    { "register",   1, 0, 'r' },
    { "name",       1, 0, 'n' },
    { "size",       1, 0, 's' },
    { 0,            0, 0, 0 }
};

static void usage(FILE *fh) {
    fprintf(
        fh,
        "btt_infohash mod_bt/"
        VERSION
        " - http://www.crackerjack.net/mod_bt/\n"
        "* View/edit infohash data in tracker database\n\n"
        "Usage:\n"
        "\tbtt_infohash [--help] [--master] homedir [--verbose]\n"
        "\t\t[--delete [--force]| --create]\n"
        "\t\t[--metainfo file.torrent] [--register 0|1] [--name=name]\n"
        "\t\t[--size=size] infohash\n"
        "\thomedir\t\t\tmod_bt data directory (eg; /var/lib/mod_bt)\n"
        "\tinfohash\t\tinfohash to manipulate if not using --metainfo\n\n"
        "\t-h\t--help\t\tThis help screen\n"
        "\t\t--master\tAct as mod_bt master\n"
        "\t\t\t\t(Use when mod_bt is not running)\n"
        "\t-v\t--verbose\tDisplay details about torrent\n"
        "\t-d\t--delete\tDelete this infohash\n"
        "\t-f\t--force\t\tDelete this infohash even if there peers\n"
        "\t-c\t--create\tCreate this infohash if it doesn't exist\n"
        "\t-m\t--metainfo\tRead torrent details from metainfo file\n"
        "\t-r\t--register\t-r1: Register torrent\t-r0: Unregister torrent\n"
        "\t-n\t--name\t\tName of torrent\n"
        "\t-s\t--size\t\tSize of torrent in bytes\n"
        "\n"
    );

    return;
}

static btt_infohash hash_details;
static unsigned int flag_opts;
static char metainfo[BT_SHORT_STRING];

static void set_options(int argc, char** argv) {
    int odone = 0;
    int oindex = 0;
    int nopts = 0;

    hash_details = new_btt_infohash;
    bzero(&metainfo, sizeof(metainfo));
        
    while(odone != -1) {
        odone = getopt_long(argc, argv, "hvdfcm:r:n:s:", opts, &oindex);
        switch(odone) {
            case -1:
                break;
            case '?':
                usage(stderr);
                exit(1);
                break;
            case 'h':
                usage(stdout);
                exit(0);
                break;
            case '!':
                flag_opts |= IOPT_MASTER;
                break;
            case 'v':
                flag_opts |= IOPT_VERBOSE;
                break;
            case 'd':
                flag_opts |= IOPT_DELETE;
                break;
            case 'f':
                flag_opts |= IOPT_FORCE;
                break;
            case 'c':
                flag_opts |= IOPT_CREATE;
                break;
            case 'm':
                BT_STRCPY(metainfo, optarg);
                break;
            case 'r':
                if(strlen(optarg) != 1) {
                    usage(stderr);
                    exit(1);
                }
                
                switch(*optarg) {
                    case '0':
                        flag_opts |= IOPT_UNREGISTER;
                        break;
                    case '1':
                        flag_opts |= IOPT_REGISTER;
                        break;
                    default:
                        usage(stderr);
                        exit(1);
                }

                break;
            case 'n':
                BT_STRCPY(hash_details.filename, optarg);
                break;
            case 's':
                if(!(hash_details.filesize = atol(optarg))) {
                    fprintf(stderr, "invalid size %s\n", optarg);
                    usage(stderr);
                    exit(1);
                }
                break;
            default:
                fprintf(stderr, "unknown option code 0x%x\n", odone);
            break;
        }
    }

    nopts = argc - optind;
    
    if(metainfo[0]) {
        if(nopts != 1) {
            fprintf(
                stderr, "btt_infohash --metainfo takes 1 additional argument\n"
            );
            usage(stderr);
            exit(2);
        }
    } else {
        if(nopts != 2) {
            fprintf(stderr, "btt_infohash takes 2 additional arguments\n");
            usage(stderr);
            exit(2);
        }
    }
    
    if((flag_opts & IOPT_FORCE) && !(flag_opts & IOPT_DELETE)) {
        fprintf(stderr, "--force only makes sense with --delete\n");
        usage(stderr);
        exit(2);
    }
    
    if(
        (flag_opts & (IOPT_CREATE | IOPT_DELETE)) ==
        (IOPT_CREATE | IOPT_DELETE)
    ) {
        fprintf(stderr, "--create and --delete cannot be used together\n");
        usage(stderr);
        exit(2);
    }
    
    if(
        metainfo[0] &&
        (memcmp(&hash_details, &new_btt_infohash, sizeof(hash_details)))
    ) {
        fprintf(stderr, "Cannot combine metainfo and commandline details\n");
        usage(stderr);
        exit(2);
    }
}

static btt_tracker* startup(const char* dir, int master) {
    btt_tracker* tracker;
    
    if(apr_app_initialize(NULL, NULL, NULL) != APR_SUCCESS) {
        fprintf(stderr, "apr_app_initialize() failed!\n");
        fflush(stderr);
        exit(20);
    }
 
    atexit(apr_terminate);
 
    if(!(tracker = btt_tracker_alloc(NULL, dir, master))) {
        fprintf(stderr, "bt_tracker_alloc() failed!\n");
        fflush(stderr);
        exit(5);
    }
 
    if(master)
        tracker->s->start_t = time(NULL);

    return tracker;
}

static int update_infohash(
    btt_tracker* tracker, DB_TXN* txn, DBC* cur, DBT* hash_key, DBT* hash_val
) {
    btt_infohash *dbhash = (btt_infohash*)hash_val->data;
    int rv = 0;

    if(hash_details.filesize) dbhash->filesize = hash_details.filesize;
    if(hash_details.filename)
        BT_STRCPY(dbhash->filename, hash_details.filename);
    
    if((flag_opts & IOPT_REGISTER) && !dbhash->register_t)
        dbhash->register_t = time(NULL);
    else if(flag_opts & IOPT_UNREGISTER)
        dbhash->register_t = 0;
    
    dbhash->last_t = time(NULL);
    
    if((rv = cur->c_put(cur, hash_key, hash_val, DB_CURRENT)) != 0) {
        tracker->db.hashes->err(
            tracker->db.hashes, rv, "failure saving infohash record"
        );
        rv = 5;
    }
    
    return rv;
}

static int iter_delete_peer(
    apr_pool_t* p, DB_TXN* txn, DBC* cursor, DBT* key, DBT* val, void* data
) {
    int rv = 0;
    if((rv = cursor->c_del(cursor, 0)) != 0)
        return rv;
    return BT_CALLBACK_STOP;
}

const static btt_txn_iterator delete_peer_iterator[] = {
  { iter_delete_peer, NULL },
  { NULL, NULL }
};

static int delete_infohash(
    btt_tracker* tracker, DB_TXN* txn, DBC* cur, DBT* hash_key, DBT* hash_val
) {
    int rv = 0;
    btt_infohash *dbhash = (btt_infohash*)hash_val->data;
    
    if(!(flag_opts & IOPT_FORCE)) {
        if(dbhash->peers) {
            fprintf(
                stderr, "not deleting hash: %i peers still reported\n",
                dbhash->peers
            );
            rv = 6;
        } else {
            if((rv = cur->c_del(cur, 0)) != 0) {
                tracker->db.hashes->err(
                    tracker->db.hashes, rv, "failure deleting infohash"
                );
                rv = 5;
            }
        }
    } else {
        if((
            rv = btt_txn_iterate_peerlist(
                tracker, tracker->p, dbhash, txn, 0, DB_RMW,
                delete_peer_iterator
            )
        ) != 0) {
            tracker->db.hashes->err(
                tracker->db.hashes, rv, "failure deleting peers"
            );
            rv = 5;
        } else {
            if((rv = cur->c_del(cur, 0)) != 0) {
                tracker->db.hashes->err(
                    tracker->db.hashes, rv, "failure deleting infohash"
                );
                rv = 5;
            }
        }
    }
    
    return rv;
}

int main(int argc, char** argv) {
    int rv = 0;
    int ret = 0;
    btt_tracker* tracker = NULL;
    bt_metainfo* info = NULL;
    DBT hash_key;
    DBT hash_val;
    DBC* cur = NULL;
    DB_TXN* txn = NULL;
    
    set_options(argc, argv);
    tracker = startup(argv[optind], (flag_opts & IOPT_MASTER) ? 1 : 0);

    if(metainfo[0]) {
        if(!(info = bt_metainfo_parse(tracker->p, metainfo))) {
            fprintf(
                stderr, "Fatal: failed to parse metainfo file %s\n", metainfo
            );
            exit(4);
        }
        
        BT_STRCPY(hash_details.filename, info->name);
        memcpy(hash_details.infohash, info->hash, BT_INFOHASH_LEN);
        hash_details.filesize = info->total_size;
    } else {
        char* temp = bt_infohash_str(tracker->p, argv[optind + 1]);
        memcpy(hash_details.infohash, temp, BT_INFOHASH_LEN);
    }

    bzero(&hash_key, sizeof(hash_key));
    bzero(&hash_val, sizeof(hash_val));
    
    hash_key.data = apr_palloc(tracker->p, BT_INFOHASH_LEN);
    memcpy(hash_key.data, hash_details.infohash, BT_INFOHASH_LEN);
    hash_key.size = hash_key.ulen = BT_INFOHASH_LEN;
    hash_key.flags = DB_DBT_USERMEM;

    hash_val.ulen = sizeof(btt_infohash);
    hash_val.data = apr_pcalloc(tracker->p, sizeof(btt_infohash));
    hash_val.flags = DB_DBT_USERMEM;
    
    if((ret = btt_txn_start(tracker, NULL, &txn, 0)) != 0) {
        fprintf(stderr, "Failed to start a database transaction!\n");
        exit(9);
    }
    
    if((ret = btt_txn_load_hashcursor(
        tracker, tracker->p, txn, &hash_key, &hash_val, &cur,
        BTT_WRITE_CURSOR(tracker), DB_RMW, (flag_opts & IOPT_CREATE) ? 1 : 0
    )) != 0) {
        tracker->db.hashes->err(
            tracker->db.hashes, ret, "failure loading infohash record"
        );
        exit(9);
    }
    
    if(flag_opts & IOPT_DELETE)
        rv = delete_infohash(tracker, txn, cur, &hash_key, &hash_val);
    else
        rv = update_infohash(tracker, txn, cur, &hash_key, &hash_val);

    if(cur)
        cur->c_close(cur);

    if(txn) {
        if(rv) {
            txn->abort(txn);
        } else if((ret = txn->commit(txn, 0)) != 0) {
            rv = 9;
            tracker->db.env->err(tracker->db.env, ret, "commit");
        }
    }

    if((!rv) && (flag_opts & IOPT_VERBOSE)) {
        char* buf = NULL;
        
        btt_infohash2text(tracker->p, (btt_infohash*)hash_val.data, &buf);
        printf(buf);
        
        if(flag_opts & IOPT_DELETE) {
            printf("This infohash has been deleted from the tracker.\n\n");
        }        
    }

    if(!btt_tracker_free(&tracker, (flag_opts & IOPT_MASTER) ? 1 : 0)) {
        fprintf(stderr, "bt_tracker_free() failed!\n");
        exit(9);
    }
 
    
    exit(rv);
}
