#include <libbtutil.h>

#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apr_network_io.h>
#include <apr_errno.h>

/* never download more than this */
#define MAX_HTTP_RECV   1048576
#define MAX_HTTP_SLOT   MAX_HTTP_RECV / BT_SHORT_STRING

#define SF APR_SIZE_T_FMT

static inline apr_status_t bt_http_slurp(
    apr_pool_t* p, apr_pool_t* temp, apr_socket_t* s,
    char** buf, apr_size_t* buflen
);

apr_status_t bt_http_get(
    apr_pool_t* p, const char *host, apr_port_t port, const char *resource,
    char** buf, apr_size_t* buflen
) {
    const char*         path = *resource ? resource : "/";
    char*               req;
    apr_pool_t          *temp;
    apr_sockaddr_t      *sa;
    apr_socket_t        *s;
    apr_status_t        ret;
    apr_size_t          inlen;
    int                 in_content = 0;
    int                 crlfs = 0;
    
    apr_pool_create(&temp, p);
    
    if(
        (ret = apr_sockaddr_info_get(&sa, host, APR_UNSPEC, port, 0, temp))
        != APR_SUCCESS
    ) {
        fprintf(
            stderr, "bt_http_get: sockaddr_info_get(%s:%i) failed\n",
            host, port
        );
        apr_pool_destroy(temp);
        return ret;
    }
    
    if(
        (ret = apr_socket_create(
            &s, sa->family, SOCK_STREAM, APR_PROTO_TCP, temp
        ))
        != APR_SUCCESS
    ) {
        fprintf(stderr, "bt_http_get: socket_create failed\n");
        apr_pool_destroy(temp);
        return ret;
    }

    if((ret = apr_socket_connect(s, sa)) != APR_SUCCESS) {
        fprintf(stderr, "bt_http_get: socket_connect failed\n");
        apr_pool_destroy(temp);
        return ret;
    }
    
    req = apr_psprintf(
        temp, "GET %s HTTP/1.0\nHost: %s\nUser-Agent: %s\n\n",
        path, host, BT_HTTP_USER_AGENT
    );
    inlen = strlen(req);
    
    if((ret = apr_socket_send(s, req, &inlen)) != APR_SUCCESS) {
        fprintf(stderr, "bt_http_get: socket_send failed\n");
        apr_pool_destroy(temp);
        return ret;
    }

    inlen=0;
    while(!in_content) {
        char c;
        apr_size_t cl = 1;
        inlen++;
        
        if((ret = apr_socket_recv(s, &c, &cl)) != APR_SUCCESS) {
            fprintf(stderr, "bt_http_get: socket_recv(%"SF") failed\n", inlen);
            apr_pool_destroy(temp);
            return ret;
        }
        
        if(c == '\r' || c == '\n') {
            crlfs++;
            if(crlfs>=4)
                in_content = 1;
        } else {
            crlfs=0;
        }
    }

    if((ret = bt_http_slurp(p, temp, s, buf, buflen)) != APR_SUCCESS)
        fprintf(stderr, "bt_http_get: http_slurp failed\n");
        
    apr_pool_destroy(temp);
    return ret;
}

static inline apr_status_t bt_http_slurp(
    apr_pool_t* p, apr_pool_t* temp, apr_socket_t* s,
    char** buf, apr_size_t* buflen
) {
    char*               slots[MAX_HTTP_SLOT];
    int                 slot = 0;
    apr_size_t          len = 0;
    apr_size_t          inlen = 1;
    apr_status_t        ret = APR_SUCCESS;

    while(ret == APR_SUCCESS) {
        apr_size_t alen = 0;
        ret = APR_SUCCESS;
        char* sp = slots[slot] = apr_palloc(temp, BT_SHORT_STRING);
        inlen = BT_SHORT_STRING;
        
        while(ret == APR_SUCCESS && inlen) {
            ret = apr_socket_recv(s, sp, &inlen);
            sp += inlen;
            alen += inlen;
            inlen = BT_SHORT_STRING - (sp - slots[slot]);
        }
            
        len += alen;
        slot++;
    }
    
    if(len) {
        int slotpos = 0;
        char* bufpos;
        *buf = apr_palloc(p, len);
        bufpos = *buf;
        *buflen = len;
        
        while(len) {
            apr_size_t cpy = len > BT_SHORT_STRING ? BT_SHORT_STRING : len;
            memcpy(bufpos, slots[slotpos], cpy);
            slotpos++;
            len-=cpy;
            bufpos+=cpy;
        }
    }
    
    if(ret == APR_EOF)
        ret = APR_SUCCESS;
    
    return ret;
}
