#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <libbtutil.h>

static const char *bt_bcode_errors[] = {
    "Success",
    "Bad Integer",
    "Bad List",
    "Bad Dictionary",
    "Uneven Dictionary",
    "Bad String",
    "Bad String Length",
    NULL
};

const char* bt_bcode_error(int val) {
    return bt_bcode_errors[val];
}

int bt_bcode_decode(apr_pool_t* p, char* buf, bt_bcode* val, char** end) {
    char* pos;

    val->begin = buf;

    if(buf[0] == 'i') {
        /* Integer: i1242e */

        val->type  = BT_BCODE_TYPE_INT;
        val->val.i = strtoll(&buf[1], &pos, 10);

        if(pos == &buf[1] || pos[0] != 'e')
            return BT_BCODE_ERR_BAD_INT;

        val->end = pos+1;
    } else if(buf[0] == 'l' || buf[0] == 'd') {
        /* List: l<item1><item2>e
         * Dict: d<string1><item1><string2><item2>e
         */

        char*   cur;
        char    is_dict, str_expected;

        is_dict          = (buf[0] == 'd');
        val->type        = is_dict ? BT_BCODE_TYPE_DICT : BT_BCODE_TYPE_LIST;
        val->val.l.alloc = BT_BCODE_LIST_SIZE;
        val->val.l.count = 0;
        val->val.l.vals  = apr_palloc(p, BT_BCODE_LIST_SIZE * sizeof(bt_bcode));
        cur              = &buf[1];
        str_expected     = 1;
        
        while(cur[0] != 'e') {
            if(val->val.l.count == val->val.l.alloc) {
                /* grow our pool. APR doesn't have realloc() so we have
                 * to fake it. this means the old memory is left laying
                 * around, but pools for bdecodes should be temporary
                 * anyway.
                 */
                void* oldmem = val->val.l.vals;
                val->val.l.vals = apr_palloc(
                    p, (val->val.l.alloc+BT_BCODE_LIST_SIZE) * sizeof(bt_bcode)
                );
                memcpy(
                    val->val.l.vals, oldmem,
                    val->val.l.alloc * sizeof(bt_bcode)
                );
                val->val.l.alloc += BT_BCODE_LIST_SIZE;
            }
            
            if(bt_bcode_decode(
                p, cur, &val->val.l.vals[val->val.l.count], &pos
            ))
                return BT_BCODE_ERR_BAD_LIST;

            if(
                is_dict && str_expected &&
                val->val.l.vals[val->val.l.count].type != BT_BCODE_TYPE_STR
            )
                return BT_BCODE_ERR_BAD_DICT;

            str_expected = !str_expected;

            val->val.l.count++;
            cur = pos;
        }

        if(is_dict && (val->val.l.count & 1))
            return BT_BCODE_ERR_DICT_UNEVEN;

        val->end = cur + 1;
    } else {
        /* String: 12:whateverword */
        val->type    = BT_BCODE_TYPE_STR;
        val->val.s.i = strtol(buf, &pos, 10);

        if(pos == buf) return BT_BCODE_ERR_BAD_STRING_LENGTH;
        if(pos[0] != ':') return BT_BCODE_ERR_BAD_STRING;
        
        val->val.s.s = apr_palloc(p, val->val.s.i+1);
        val->val.s.s[val->val.s.i] = 0;
        memcpy(val->val.s.s, pos+1, val->val.s.i);

        val->end = pos + val->val.s.i + 1;
    }

    if(end)
        *end = val->end;

    return BT_BCODE_SUCCESS;
}

bt_bcode* bt_bcode_find(bt_bcode* val, char* key) {
    int i;
    
    if(val->type != BT_BCODE_TYPE_DICT)
        return NULL;
    
    for(i=0; i<val->val.l.count; i+=2 )
        if(!strcmp(val->val.l.vals[i].val.s.s,key))
            return &val->val.l.vals[i+1];

    return NULL;
}

int bt_bcode_encode(
    apr_pool_t* p, bt_bcode* val, char** rv, apr_size_t* len,
    apr_pool_t** rpool
) {
    apr_pool_t* pool;
    char* result = NULL;
    
    apr_pool_create(&pool, p);
    
    if(val->type == BT_BCODE_TYPE_INT) {
        result = apr_psprintf(pool, "i%"PRIi64"e", val->val.i);
        *len = strlen(result);
    } else if(val->type == BT_BCODE_TYPE_STR) {
        if(!val->val.s.i) {
            apr_pool_destroy(pool);
            return BT_BCODE_ERR_BAD_STRING_LENGTH;
        }
        
        result = apr_palloc(pool, val->val.s.i + BT_TINY_STRING);
        sprintf(result, "%"APR_SIZE_T_FMT":", val->val.s.i);
        *len = strlen(result);
        memcpy(result + *len, val->val.s.s, val->val.s.i);
        *len += val->val.s.i;
    } else if(
        val->type == BT_BCODE_TYPE_LIST || val->type == BT_BCODE_TYPE_DICT
    ) {
        char* pos;
        int i, subrv;
        apr_size_t plen;
        char* item;
        int wantstr = 1;
        
        /* count is zero-base, and we need an even number of entries */
        if(!val->val.l.count % 2) {
            apr_pool_destroy(pool);
            return BT_BCODE_ERR_DICT_UNEVEN;
        }
        
        result = apr_palloc(pool, val->val.l.alloc + 2);
        *result = val->type == BT_BCODE_TYPE_LIST ? 'l' : 'd';
        *len = 1;
        pos = result + 1;
        
        for(i=0; i<val->val.l.count; i++) {
            if(
                wantstr &&
                val->type == BT_BCODE_TYPE_DICT &&
                val->val.l.vals[i].type != BT_BCODE_TYPE_STR
            ) {
                if(rpool) *rpool = NULL;
                apr_pool_destroy(pool);
                return BT_BCODE_ERR_BAD_DICT;
            }
            
            wantstr = !wantstr;
            
            /* this will cause *rpool to be assigned, but that's okay
             * since we're also going to reasisgn it down below.
             */
            if(
                (
                    subrv = bt_bcode_encode(
                        pool, &(val->val.l.vals[i]), &item, &plen, rpool
                    )
                )
                != BT_BCODE_SUCCESS
            ) {
                if(rpool) *rpool = NULL;
                apr_pool_destroy(pool);
                return subrv;
            }
            
            *len += plen;
            memcpy(pos, item, plen);
            pos += plen;
        }
        
        *pos = 'e';
        ++*len;
    }
    
    if(rpool) {
        val->begin = result;
        val->end = result + *len;
        *rv = result;
        *rpool = pool;
    } else {
        val->begin = val->end = NULL;
        *rv = apr_palloc(p, *len);
        memcpy(*rv, result, *len);
        apr_pool_destroy(pool);
    }

    return BT_BCODE_SUCCESS;
}

bt_bcode* bt_bcode_new_int(apr_pool_t* p, int64_t i) {
    bt_bcode* rv = apr_pcalloc(p, sizeof(bt_bcode));
    rv->type = BT_BCODE_TYPE_INT;
    rv->val.i = i;
    return rv;
}

bt_bcode* bt_bcode_new_strn(apr_pool_t* p, const uint8_t* buf, apr_size_t len) {
    bt_bcode* rv = apr_pcalloc(p, sizeof(bt_bcode));
    
    if(len == 0) {
        fprintf(
            stderr, "bt_bcode_new_strn: Attempt to create an empty string!\n"
        );
        return NULL;
    }
    
    rv->type = BT_BCODE_TYPE_STR;
    rv->val.s.i = len;
    rv->val.s.s = apr_pmemdup(p, buf, len);
    return rv;
}

bt_bcode* bt_bcode_new_str(apr_pool_t* p, const char* buf) {
    return bt_bcode_new_strn(p, (const uint8_t*)buf, strlen(buf));
}

bt_bcode* bt_bcode_new_list(apr_pool_t* p, unsigned int size) {
    bt_bcode* rv = apr_pcalloc(p, sizeof(bt_bcode));
    rv->type = BT_BCODE_TYPE_LIST;
    rv->val.l.count = size;
    rv->val.l.vals = apr_pcalloc(p, sizeof(bt_bcode) * size);
    return rv;
}

bt_bcode* bt_bcode_new_dict(apr_pool_t* p, unsigned int size) {
    bt_bcode* rv = apr_pcalloc(p, sizeof(bt_bcode));
    rv->type = BT_BCODE_TYPE_DICT;
    rv->val.l.count = size * 2;
    rv->val.l.vals = apr_pcalloc(p, sizeof(bt_bcode) * size * 2);
    return rv;
}

apr_status_t bt_bcode_list_add(
    bt_bcode* list, unsigned int pos, bt_bcode* entry
) {
    /* attempt to add a null entry */
    if(!entry)
        return APR_ENOENT;
    
    if(pos < list->val.l.count) {
        switch(entry->type) {
            case BT_BCODE_TYPE_INT:
                list->val.l.alloc += BT_TINY_STRING;
                break;
            case BT_BCODE_TYPE_STR:
                list->val.l.alloc += entry->val.s.i;
                break;
            case BT_BCODE_TYPE_DICT:
            case BT_BCODE_TYPE_LIST:
                list->val.l.alloc += entry->val.l.alloc;
                break;
            default:
                fprintf(
                    stderr, "bt_bcode_list_add: unknown type %d\n", entry->type
                );
                return APR_EINVAL;
                break;
        }
        
        list->val.l.alloc += BT_TINY_STRING;
        list->val.l.vals[pos] = *entry;        
        return APR_SUCCESS;
    } else {
        return APR_ENOMEM;
    }
}

apr_status_t bt_bcode_dict_add(
    apr_pool_t* p, bt_bcode* dict, unsigned int pos,
    const char* name, bt_bcode* entry
) {
    apr_status_t ret;
    bt_bcode* key = bt_bcode_new_str(p, name);
    
    if((ret = bt_bcode_list_add(dict, pos * 2, key)) == APR_SUCCESS)
        ret = bt_bcode_list_add(dict, (pos * 2) + 1, entry);
    
    return ret;
}
