#include <pglog.h>

#include "MenuManager.h"
#include "Configuration.h"
#include "PlayerConfiguration.h"

#include "GuiException.h"
#include "MainMenu.h"
#include "QuitMenu.h"
#include "NewGameMenu.h"
#include "OptionMenu.h"
#include "CreditsMenu.h"
#include "PlayerMenu.h"
#include "LevelSelectorMenu.h"
#include "IngameMenu.h"
#include "PlayGroundMenu.h"
#include "HighscoreMenu.h"
#include "GraphicsMenu.h"
#include "SoundMenu.h"
#include "ControlMenu.h"
#include "StatusBarMenu.h"

//----------------------------------------------------------------------
MenuManager* MenuManager::sm_instance = NULL;

//----------------------------------------------------------------------
MenuManager::MenuManager()
{
  m_app = new PG_Application(); 
  m_current = NULL;
  m_background = NULL;
}

//----------------------------------------------------------------------
MenuManager::~MenuManager()
{
  MainMenu::destroy();
  QuitMenu::destroy();
  NewGameMenu::destroy();
  OptionMenu::destroy();
  CreditsMenu::destroy();
  PlayerMenu::destroy();
  LevelSelectorMenu::destroy();
  PlayGroundMenu::destroy();
  HighscoreMenu::destroy();
  GraphicsMenu::destroy();
  SoundMenu::destroy();
  ControlMenu::destroy();
  StatusBarMenu::destroy();
  IngameMenu::destroy(); 

  ZAP_POINTER(m_background);
  ZAP_POINTER(m_app);
}

//----------------------------------------------------------------------
void MenuManager::init(const char* layout, const char* theme)
{
  if(sm_instance != NULL)
  {
    throw GuiException("MenuManager already initialized!");
  }

  PG_LogConsole::SetLogLevel(PG_LOG_WRN);
  
  sm_instance = new MenuManager();

  initScreen(
      PlayerConfiguration::getInstance()->getGraphics()->isFullScreen());

  // set search paths
  PG_Application::SetApplicationPath(
      std::string(Configuration::getInstance()->getDataDir())
      .append("/gfx/gui").c_str());
  PG_Application::SetApplicationPath(
      std::string(Configuration::getInstance()->getDataDir())
      .append("/gfx/statusbar").c_str());
  PG_Application::SetApplicationPath(
      std::string(Configuration::getInstance()->getDataDir())
      .append("/cfg").c_str());

  PG_Application::GetApp()->SetCaption("moagg", NULL);
  
  if (PG_Application::GetApp()->LoadTheme(theme) == NULL)
  {
    throw GuiException("Error loading theme!");
  }

  if (PG_Application::LoadLayout(layout) == false)
  {
    throw GuiException("Error loading layout!");
  }

  // show first screen
  getInstance()->hideAll();
  getInstance()->changeTo(MainMenu::getInstance());
}

//----------------------------------------------------------------------
void MenuManager::initScreen(bool fullScreen)
{
    Uint32 flags = SDL_HWSURFACE | SDL_ANYFORMAT;
    if (fullScreen)
    {
        flags |= SDL_FULLSCREEN;
    }

    if (!PG_Application::GetApp()->InitScreen(640, 480, 32, flags))
    {
        throw GuiException("Error initializing screen!");
    }

    if (PG_Application::GetScreen()->flags & SDL_HWSURFACE == 0)
    {
        std::cout << "Warning: Cannot use a hardware surface for the screen."
                  << std::endl;
    }
}

//----------------------------------------------------------------------
void MenuManager::hideAll()
{
  getWidgetById(ID_CREDITS_MENU)->Hide();
  getWidgetById(ID_MAIN_MENU)->Hide();
  getWidgetById(ID_QUIT_MENU)->Hide();
  getWidgetById(ID_NEW_MENU)->Hide();
  getWidgetById(ID_OPTION_MENU)->Hide();
  getWidgetById(ID_INGAME_MENU)->Hide();
  getWidgetById(ID_HIGHSCORE_MENU)->Hide();
  getWidgetById(ID_GRAPHICS_MENU)->Hide();
  getWidgetById(ID_SOUND_MENU)->Hide();
  getWidgetById(ID_CONTROL_MENU)->Hide();
  getWidgetById(ID_STATUSBAR_MENU)->Hide();

  if(m_background == NULL)
  {
    m_background = new PG_ThemeWidget(NULL, PG_Rect(0,0,640,480), false);
    m_background->SetTransparency(255);
    m_background->Show();
    PG_Application::GetApp()->EnableBackground(true);
    PG_Application::GetApp()->SetBackground("background.bmp");
  }
  
  getWidgetById(ID_BACKGROUND_LABEL)->Show();
}


//----------------------------------------------------------------------
void MenuManager::destroy()
{
  if(sm_instance != NULL)
  {
    ZAP_POINTER(sm_instance);
  }
}

//----------------------------------------------------------------------
void MenuManager::changeTo(Menu* menu, bool background, bool overridePrevious)
{
  try
  {
    if(m_current != NULL)
      m_current->Hide();

    if(overridePrevious)
      menu->setPrevious(m_current);
    m_current = menu;

    if(background)
      redrawBackground();
  
    m_current->Show();
  }
  catch(Exception e)
  {
    std::cerr << e << std::endl;
    PG_Application::GetApp()->Quit();
  }
}

//----------------------------------------------------------------------
void MenuManager::redrawBackground()
{
  try
  {
    m_background->Redraw();
  }
  catch(GuiMenuException &e)
  {
    std::cout << e << std::endl;
    PG_Application::GetApp()->Quit();
  }
}

//----------------------------------------------------------------------
PG_Widget* MenuManager::getWidgetById(int id)
{
    PG_Widget* w = PG_Application::GetWidgetById(id);
    if(w == NULL)
        throw GuiMenuException("Widget ID not found in layout", id);
    return w;
}

