#ifndef PLATFORM_H
#define PLATFORM_H

#include <map>

#include "Tools.h"
#include "StaticDecorationBase.h"

//----------------------------------------------------------------------------
class PlatformSurfaces : public SurfacesBase
{
    SURFACES_SINGLETON_OBJECT(PlatformSurfaces);

  public:

    //------------------------------------------------------------------------
    /**
     * An enum, which values define the available color of beacons.
     */
    enum BeaconColor
    {
        BC_NO_BEACON = -1,
        BC_RED = 0,
        BC_YELLOW = 1,
        BC_GREEN = 2,
        BC_CYAN = 3,
        BC_BLUE = 4,
        BC_MAGENTA = 5,
        BC_TOTAL_NUMBER = 6
    };

  private:

    //------------------------------------------------------------------------
    class SurfaceKey
    {
      public:
        SurfaceKey(BeaconColor leftBeacon, BeaconColor rightBeacon,
                   unsigned number, unsigned frame, unsigned size)
            : m_leftBeacon(leftBeacon), m_rightBeacon(rightBeacon),
              m_number(number), m_frame(frame), m_size(size) {}
        ~SurfaceKey() {}

        friend bool operator<(const SurfaceKey &k1, const SurfaceKey &k2)
        {
            COMPARE_SURFACE_KEY_MEMBER(m_leftBeacon);
            COMPARE_SURFACE_KEY_MEMBER(m_rightBeacon);
            COMPARE_SURFACE_KEY_MEMBER(m_number);
            COMPARE_SURFACE_KEY_MEMBER(m_frame);
            COMPARE_SURFACE_KEY_MEMBER(m_size);
            return false;
        }

      private:
        BeaconColor m_leftBeacon;
        BeaconColor m_rightBeacon;
        unsigned m_number;
        unsigned m_frame;
        unsigned m_size;
    };

    //------------------------------------------------------------------------
    typedef std::map<SurfaceKey, SDL_Surface*> SurfaceMap;
    typedef SurfaceMap::iterator SurfaceIter;
    typedef SurfaceMap::const_iterator SurfaceCIter;

  public:

    //------------------------------------------------------------------------
    const SDL_Surface *getSurface(BeaconColor leftBeacon,
                                  BeaconColor rightBeacon,
                                  unsigned number, unsigned frame,
                                  unsigned size);

  private:

    //------------------------------------------------------------------------
    SDL_Surface *createSurface(BeaconColor leftBeacon, BeaconColor rightBeacon,
                               unsigned number, unsigned frame,
                               unsigned size) const;


    /// All number blocks for the left bound.
    SDL_Surface *m_leftNumber[10];

    /// All beacon frames for the left bound.
    SDL_Surface *m_leftBeacon[BC_TOTAL_NUMBER][2];

    /// The platform frame for the inner parts.
    SDL_Surface *m_center;

    /// All number blocks for the right bound.
    SDL_Surface *m_rightNumber[10];

    /// All beacon frames for the right bound.
    SDL_Surface *m_rightBeacon[BC_TOTAL_NUMBER][2];


    /// A map for all dynamically created magnet surfaces via getSurface().
    SurfaceMap m_surfaceMap;
};


//----------------------------------------------------------------------------
class Platform : public StaticDecorationBase
{
  public:
    //------------------------------------------------------------------------
    class InitializationData : public StaticDecorationBase::InitializationData
    {
      public:
        InitializationData(const XMLNode *platformNode)
            throw (XMLException);
        ~InitializationData();

        unsigned w;
        unsigned number;

        PlatformSurfaces::BeaconColor leftBeacon;
        PlatformSurfaces::BeaconColor rightBeacon;

      private:
        PlatformSurfaces::BeaconColor getBeaconColor(const XMLNode *node,
                                                     const char *prop) const
            throw (XMLException);
    };


    //------------------------------------------------------------------------
    virtual ~Platform();

    static Platform *create(const XMLNode *platformNode)
        throw (Exception);

    //------------------------------------------------------------------------
    inline unsigned getWidth() const
    {
        return m_width;
    }

    inline unsigned getNumber() const
    {
        return m_number;
    }

    inline const SDL_Rect &getLandingZone() const
    {
        return m_landingZone;
    }

    //------------------------------------------------------------------------
    void update();

  protected:
    //------------------------------------------------------------------------
    Platform(const InitializationData &init);

    /// Called by the constructor to initialize m_landingZone.
    void initLandingZone();


    //------------------------------------------------------------------------
    unsigned m_frameCounter;
    unsigned m_currentFrame;

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    unsigned m_width;
    unsigned m_number;
    PlatformSurfaces::BeaconColor m_leftBeaconColor;
    PlatformSurfaces::BeaconColor m_rightBeaconColor;

    /// A one pixel high rectangle on top of the center platform tiles.
    SDL_Rect m_landingZone;
};

#endif //PLATFORM_H
