#include "SDLCalls.h"
#include "SDLFrameRate.h"

#include "GameBridge.h"
#include "SoundInterface.h"

#include "PlayGround.h"
#include "PlayerStatus.h"
#include "PlayerConfiguration.h"

#include "MainMenu.h"
#include "PlayGroundMenu.h"
#include "IngameMenu.h"
#include "HighscoreMenu.h"
#include "StatusBarMenu.h"

MENU_SINGLETON_INIT(PlayGround);

//----------------------------------------------------------------------------
PlayGroundMenu::FullFrameRate PlayGroundMenu::FullFrameRate::sm_instance;

PlayGroundMenu::RelativeUpdate PlayGroundMenu::RelativeUpdate::sm_instance;
PlayGroundMenu::FullUpdateDelay PlayGroundMenu::FullUpdateDelay::sm_instance;

//----------------------------------------------------------------------------
void PlayGroundMenu::FullFrameRate::updateScreen(PlayGroundMenu *driver)
{
    if (driver->hasViewBoxChanged())
    {
        driver->do_playGroundFullUpdate();
    }
    else
    {
        driver->do_playGroundRelativeUpdate();
    }
}

//----------------------------------------------------------------------------
void PlayGroundMenu::RelativeUpdate::updateScreen(PlayGroundMenu *driver)
{
    if (driver->hasViewBoxChanged())
    {
        driver->setState(FullUpdateDelay::getInstance());
    }
    else
    {
        driver->do_playGroundRelativeUpdate();
    }
}

//----------------------------------------------------------------------------
void PlayGroundMenu::FullUpdateDelay::updateScreen(PlayGroundMenu *driver)
{
    driver->do_playGroundFullUpdate();
    driver->setState(RelativeUpdate::getInstance());
}


//----------------------------------------------------------------------
PlayGroundMenu::PlayGroundMenu() throw(GuiMenuException)
{
  m_quit = false;
  m_fireDown = false;
  m_backSurface = NULL;
  
  m_initialized = false;
  m_viewBox.x = 0;
  m_viewBox.y = 0;
  m_viewBox.w = 640;
  m_viewBox.h = 480 - 4*16;

  memset(&m_oldViewBox, 0, sizeof(m_oldViewBox));

  m_state = RelativeUpdate::getInstance();
  //m_state = FullFrameRate::getInstance();
}

//----------------------------------------------------------------------
PlayGroundMenu::~PlayGroundMenu()
{
}

//----------------------------------------------------------------------
bool PlayGroundMenu::eventKeyDown(const SDL_KeyboardEvent *key)
{
  TranslateNumpadKeys((SDL_KeyboardEvent*)key);

  GameControlBase *gameControl = GameControlBase::getInstance();
  PlayerConfiguration::Control::KeyConfig *kc =
      PlayerConfiguration::getInstance()->getControl()->getKeyConfig();
  SDLKey sym = key->keysym.sym;
  if (sym == kc->rotleft)
  {
    gameControl->updateShipControl(GameControlBase::ROT_LEFT);
  }
  else if (sym == kc->rotright)
  {
    gameControl->updateShipControl(GameControlBase::ROT_RIGHT);
  }
  else if (sym == kc->thrust)
  {
    gameControl->updateShipControl(GameControlBase::THRUST_ON);
  }
  else if (sym == kc->align)
  {
    gameControl->updateShipControl(GameControlBase::ALIGN_ON);
  }
  else if (sym == kc->fire)
  {
    if(!m_fireDown)
    {
      gameControl->updateShipControl(GameControlBase::FIRE);
      m_fireDown = true; // just to disable continuous fire
    }
  }
  else if (sym == kc->escape)
  {
    if(m_ingame)
      m_quit = true;
  }
  return true;
}

//----------------------------------------------------------------------
bool PlayGroundMenu::eventKeyUp(const SDL_KeyboardEvent *key)
{
  TranslateNumpadKeys((SDL_KeyboardEvent*)key);

  PlayerConfiguration::Control::KeyConfig *kc =
      PlayerConfiguration::getInstance()->getControl()->getKeyConfig();
  SDLKey sym = key->keysym.sym;

  GameControlBase *gameControl = GameControlBase::getInstance();
  const Ship *player = gameControl->getPlayerShip();
  if (player)
  {
      if (sym == kc->rotleft)
      {
          if (player->getRotation() == Ship::ROT_LEFT)
          {
              gameControl->updateShipControl(GameControlBase::ROT_OFF);
          }
      }
      else if (sym == kc->rotright)
      {
          if (player->getRotation() == Ship::ROT_RIGHT)
          {
              gameControl->updateShipControl(GameControlBase::ROT_OFF);
          }
      }
      else if (sym == kc->thrust)
      {
          if (player->isThrust())
          {
              gameControl->updateShipControl(GameControlBase::THRUST_OFF);
          }
      }
      else if (sym == kc->align)
      {
          if (player->getAlign())
          {
              gameControl->updateShipControl(GameControlBase::ALIGN_OFF);
          }
      }
  }

  // Fire key up must be accepted, even if the player's ship was destroyed.
  if (sym == kc->fire)
  {
      if (m_fireDown)
      {
          m_fireDown = false;
      }
  }

  return true;
}

//----------------------------------------------------------------------
bool PlayGroundMenu::eventQuit(int id, PG_MessageObject *widget, unsigned long data)
{
  m_quit = true;  
  return true;
}
//----------------------------------------------------------------------
bool PlayGroundMenu::eventMessage(MSG_MESSAGE *msg)
{
  GameControlBase *gameControl = GameControlBase::getInstance();
  bool finished = false;

  if(msg->type != MSG_USER_1)
  {
    return true;
  }

  if(gameControl->isFinished())
  {
    if (PlayerStatus::getInstance()->getLifes() > 0 &&
        SinglePlayerGame::getInstance()->hasMoreLevels())
    {
      SinglePlayerGame::getInstance()->nextLevel();
      initLevel();
    }
    else
    {
      m_initialized = false;
      finished = true;
    }
  }
  else
  {
    gameControl->update();
    StatusBarMenu::getInstance()->update();

    updateViewBox();
    updateScreen();
 
    SDLFrameRate::delay();
  }

  if(m_quit && m_ingame)
  {
    m_ingame = false; // prevent to change into menu if already there
    SDL_Surface *video = SDL_GetVideoSurface();
    m_backSurface = SDL_CALLS::ConvertSurface(
        video, video->format, video->flags);
    PG_Application::GetApp()->SetBackground(m_backSurface);
    MenuManager::getInstance()->changeTo(IngameMenu::getInstance());
  }
  else if(finished)
  {
    restoreBackground();
    m_initialized = false;

    HighscoreMenu *highscore = HighscoreMenu::getInstance();
    highscore->setInsertEntryOnShow(true);
    MenuManager::getInstance()->changeTo(highscore, true, false);
  }
  else
  {
    // trigger next frame
    trigger();
  }

  m_quit = false;
  return true;
}

//----------------------------------------------------------------------
void PlayGroundMenu::eventShow()
{
  SetInputFocus();
  PG_Application::ShowCursor(PG_CURSOR_NONE);
  m_ingame = true;

  if(!m_initialized)
  {
    try
    {
      SoundInterface::getInstance()->playInGameMusic();

      SinglePlayerGame::init(m_mission.c_str(), m_level);

      m_initialized = true;
    }
    catch(Exception &e)
    {
      throw GuiException(e.toString());
    }
  }
  
  StatusBarMenu::getInstance()->Show();

  // start frame rate handling
  setState(FullUpdateDelay::getInstance()); 
  trigger();
}

//----------------------------------------------------------------------
void PlayGroundMenu::eventHide()
{
  StatusBarMenu::getInstance()->Hide();
  ReleaseInputFocus();
  PG_Application::ShowCursor(PG_CURSOR_HARDWARE);
}

//----------------------------------------------------------------------
void PlayGroundMenu::trigger()
{
  MSG_MESSAGE* msg = new MSG_MESSAGE;

  msg->_to = this;
  msg->_from = this;
  msg->type = MSG_USER_1;
  msg->widget_id = 0;
  msg->data = 0;

  SDL_Event event;
  event.type = SDL_USEREVENT;
  event.user.code = 0;
  event.user.data1 = (void*)msg;
  event.user.data2 = NULL;

  try
  {
    SDL_CALLS::PushEvent(event);
  }
  catch(Exception &e)
  {
    std::cout << "Error triggering new event: " << e.toString() << std::endl;
  }
}

//----------------------------------------------------------------------
void PlayGroundMenu::initLevel() throw (Exception)
{
    m_game = SinglePlayerGame::getInstance();
    GameControlBase::init(m_game->getMission().c_str(), 
                          m_game->getLevel().c_str());
    getWidget(ID_STATUSBAR_MENU)->Redraw();

    setState(FullUpdateDelay::getInstance()); 
}
    
//----------------------------------------------------------------------------
void PlayGroundMenu::updateViewBox()
{
    m_oldViewBox = m_viewBox;

    const Ship *ship = GameControlBase::getInstance()->getPlayerShip();
    if (!ship)
    {
        return;
    }

    const SDL_Rect &shipPosition = ship->getPosition();
    const SDL_Rect &boundingBox = PlayGround::getInstance()->getBoundingBox();

    m_viewBox.x = shipPosition.x + shipPosition.w/2 - m_viewBox.w/2;
    if (m_viewBox.x < 0)
    {
        m_viewBox.x = 0;
    }
    else if (m_viewBox.x + m_viewBox.w > boundingBox.w)
    {
        m_viewBox.x = boundingBox.w - m_viewBox.w;
    }

    m_viewBox.y = shipPosition.y + shipPosition.h/2 - m_viewBox.h/2;
    if (m_viewBox.y < 0)
    {
        m_viewBox.y = 0;
    }
    else if (m_viewBox.y + m_viewBox.h > boundingBox.h)
    {
        m_viewBox.y = boundingBox.h - m_viewBox.h;
    }
}

//----------------------------------------------------------------------------
bool PlayGroundMenu::hasViewBoxChanged() const
{
    return m_viewBox.x != m_oldViewBox.x ||
           m_viewBox.y != m_oldViewBox.y;
}

#include "StopWatch.h"

//----------------------------------------------------------------------------
void PlayGroundMenu::do_playGroundRelativeUpdate()
{
    // For all update rectangles, three things will be done here:
    // 1. Clip the rectangles with the current view box.
    // 2. Blit the area of the clipped rectangle
    //    from the playground's shadow surface to the display.
    // 3. Change the rectangle's origin to that of the view box.

    PlayGround *playGround = PlayGround::getInstance();

    const PlayGround::UpdateRects &updateRects =
        playGround->getUpdateRects();

    std::vector<SDL_Rect> clippedUpdateRects;
    clippedUpdateRects.reserve(32);

    for (PlayGround::UpdateRects::const_iterator iter = updateRects.begin();
         iter != updateRects.end(); ++iter)
    {
        SDL_Rect r;
        if (SDL_TOOLS::intersect(*iter, m_viewBox, r))
        {
            SDL_Rect r2 = r;
            r2.x -= m_viewBox.x;
            r2.y -= m_viewBox.y;

            SDL_CALLS::BlitSurface(
                playGround->getShadowSurface(), &r,
                SDL_GetVideoSurface(), &r2);

            clippedUpdateRects.push_back(r2);
        }
    }

    // Finally, update the display.
    SDL_CALLS::UpdateRects(
        clippedUpdateRects.size(), &*clippedUpdateRects.begin());
}

//----------------------------------------------------------------------------
void PlayGroundMenu::do_playGroundFullUpdate()
{
    SDL_CALLS::BlitSurface(
        PlayGround::getInstance()->getShadowSurface(), &m_viewBox,
        SDL_GetVideoSurface(), 0);

    SDL_CALLS::UpdateRect(0, 0, m_viewBox.w, m_viewBox.h);
}

//----------------------------------------------------------------------------
void PlayGroundMenu::restoreBackground() throw(Exception)
{
  if(m_backSurface != NULL)
  {
      ZAP_SURFACE(m_backSurface);
      PG_Application::GetApp()->SetBackground("background.bmp");
  }
}

//----------------------------------------------------------------------------
const SDL_Surface* PlayGroundMenu::getLevelPreview()
    throw (Exception)
{
    SinglePlayerGame::init(m_mission.c_str(), m_level);
    initLevel();

    PlayGround *playGround = PlayGround::getInstance();

    playGround->updateForPreview();
    return playGround->getShadowSurface();
}
