#include <ctype.h>
#include <sstream>
#include <iomanip>

#include "Configuration.h"
#include "Tools.h"
#include "Tiles.h"


//----------------------------------------------------------------------------
TileException::TileException(Uint8 id)
{
    m_id = id;
}

//----------------------------------------------------------------------------
TileException::~TileException()
{
}

//----------------------------------------------------------------------------
std::string TileException::toString() const
{
    std::ostringstream s;
    s << "The tile '"
      << std::setfill('0') << std::setw(2) << std::hex << (int)m_id
      << "' doesn't exist" << std::ends;
    return s.str();
}


//----------------------------------------------------------------------------
TileCategoryException::TileCategoryException(Uint8 id)
{
    m_id = id;
}

//----------------------------------------------------------------------------
TileCategoryException::~TileCategoryException()
{
}

//----------------------------------------------------------------------------
std::string TileCategoryException::toString() const
{
    std::ostringstream s;
    s << "The tile category '"
      << std::setfill('0') << std::setw(2) << std::hex << (int)m_id
      << "' doesn't exist" << std::ends;
    return s.str();
}




//----------------------------------------------------------------------------
TileCategory::TileCategory()
{
}

//----------------------------------------------------------------------------
TileCategory::~TileCategory()
{
    for (TileMap::iterator iter = m_map.begin(); iter != m_map.end(); ++iter)
    {
        SDL_CALLS::FreeSurface(iter->second);
    }
    m_map.clear();
}

//----------------------------------------------------------------------------
SDL_Surface *TileCategory::getTile(Uint8 id) const
    throw (Exception)
{
    TileMap::const_iterator iter = m_map.find(id);
    if (iter == m_map.end())
    {
        throw TileException(id);
    }

    return iter->second;
}

//----------------------------------------------------------------------------
TileCategory *TileCategory::create(const Uint8 categoryId, const char *dir)
    throw (Exception)
{
    TileCategory *category = NULL;

    std::string full;
    full.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/tiles/").append(dir).append("/tiles.idx");

    File f(full.c_str(), "r");

    category = new TileCategory();
    while (!f.isEOF())
    {
        char buffer[128] = "";
        f.readline(buffer, 128);

        if (isxdigit(buffer[0]) && isxdigit(buffer[1]))
        {
            unsigned int id = 0;
            if (sscanf(buffer, "%x", &id) != 1)
            {
                continue;
            }

            std::string tileFile;
            tileFile.append(Configuration::getInstance()->getDataDir())
                .append("/gfx/tiles/").append(dir).append("/")
                .append(buffer+3);
            SDL_Surface *s = SDL_CALLS::LoadBMP(tileFile.c_str());
            SDL_CALLS::SetColorKey(
                s, SDL_SRCCOLORKEY, SDL_MapRGB(s->format, 0, 0, 0));
            if (categoryId == 0xff)
            {
                SDL_CALLS::SetAlpha(s, SDL_SRCALPHA, 128);
            }

            category->m_map[id] = s;
        }
    }

    return category;
}



//----------------------------------------------------------------------------
TileCategories *TileCategories::sm_instance = NULL;

//----------------------------------------------------------------------------
TileCategories::TileCategories()
{
    sm_instance = this;
}

//----------------------------------------------------------------------------
TileCategories::~TileCategories()
{
    sm_instance = NULL;
    for (CategoryMap::iterator iter = m_map.begin();
         iter != m_map.end(); ++iter)
    {
        delete iter->second;
    }
    m_map.clear();
}

//----------------------------------------------------------------------------
const TileCategory &TileCategories::getCategory(Uint8 id) const
    throw (Exception)
{
    CategoryMap::const_iterator iter = m_map.find(id);
    if (iter == m_map.end())
    {
        throw TileCategoryException(id);
    }

    return *iter->second;
}

//----------------------------------------------------------------------------
void TileCategories::init()
    throw (Exception)
{
    std::string file;
    file.append(Configuration::getInstance()->getDataDir())
        .append("/gfx/tiles/")
        .append("categories.idx");

    File f(file.c_str(), "r");

    TileCategories *categories = new TileCategories();
    while (!f.isEOF())
    {
        char buffer[128] = "";
        f.readline(buffer, 128);

        if (isxdigit(buffer[0]) && isxdigit(buffer[1]))
        {
            unsigned int id = 0;
            if (sscanf(buffer, "%x", &id) != 1)
            {
                continue;
            }

            TileCategory *category = TileCategory::create(id, buffer+3);
            if (category)
            {
                categories->m_map[id] = category;
            }
        }
    }

    sm_instance = categories;
}

//----------------------------------------------------------------------------
void TileCategories::destroy()
{
    ZAP_POINTER(sm_instance);
}
