#include <iostream>

#include "config.h"
#include "Directory.h"
#include "Configuration.h"
#include "XMLParser.h"

#include "SoundInterface.h"
#include "PlayerConfiguration.h"


//----------------------------------------------------------------------------
PlayerConfiguration *PlayerConfiguration::sm_instance = NULL;


//----------------------------------------------------------------------------
void PlayerConfiguration::ReadConfigVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "config")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "player")
    {
        Player::ReadVisitor v(m_config->getPlayer());
        n->accept(v);
    }
    else if (n->getName() == "control")
    {
        Control::ReadVisitor v(m_config->getControl());
        n->accept(v);
    }
    else if (n->getName() == "graphics")
    {
        Graphics::ReadVisitor v(m_config->getGraphics());
        n->accept(v);
    }
    else if (n->getName() == "sound")
    {
        Sound::ReadVisitor v(m_config->getSound());
        n->accept(v);
    }
    else if (n->getName() == "mission")
    {
        Mission::ReadVisitor v(m_config->getMission());
        n->accept(v);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::WriteConfigVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "config")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "player")
    {
        Player::WriteVisitor v(m_config->getPlayer());
        n->accept(v);
    }
    else if (n->getName() == "control")
    {
        Control::WriteVisitor v(m_config->getControl());
        n->accept(v);
    }
    else if (n->getName() == "graphics")
    {
        Graphics::WriteVisitor v(m_config->getGraphics());
        n->accept(v);
    }
    else if (n->getName() == "sound")
    {
        Sound::WriteVisitor v(m_config->getSound());
        n->accept(v);
    }
    else if (n->getName() == "mission")
    {
        Mission::WriteVisitor v(m_config->getMission());
        n->accept(v);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::ReadHighscoreVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "highscore")
    {
        Highscore::ReadVisitor v(m_config->getHighscore());
        n->accept(v);
    }
}


//----------------------------------------------------------------------------
void PlayerConfiguration::Player::ReadVisitor::do_visit(const XMLProperty *p)
{
    if (p->getName() == "name")
    {
        m_player->setName(p->getValue());
    }
    else if (p->getName() == "shiptype")
    {
        unsigned shipType = p->getAsUnsigned();
        if (shipType >= ShipSurfaces::S_TOTAL_NUMBER)
        {
            shipType = ShipSurfaces::S_TOTAL_NUMBER-1;
        }

        m_player->setShipType((ShipSurfaces::Ship)shipType);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Player::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "player")
    {
        n->acceptAllProperties(*this);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Player::WriteVisitor::do_visit(XMLProperty *p)
{
    if (p->getName() == "name")
    {
        p->setValue(m_player->getName());
    }
    else if (p->getName() == "shiptype")
    {
        p->setValue((unsigned)m_player->getShipType());
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Player::WriteVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "player")
    {
        n->acceptAllProperties(*this);
    }
}


//----------------------------------------------------------------------------
void PlayerConfiguration::Control::ReadVisitor::do_visit(const XMLProperty *p)
{
    const std::string &name = p->getName();
    SDLKey key = (SDLKey)p->getAsUnsigned();
    KeyConfig *kc = m_control->getKeyConfig();

    if (name == "thrust")
        kc->thrust = key;
    else if (name == "align")
        kc->align = key;
    else if (name == "rotleft")
        kc->rotleft = key;
    else if (name == "rotright")
        kc->rotright = key;
    else if (name == "fire")
        kc->fire = key;
    else if (name == "escape")
        kc->escape = key;
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Control::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "control")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "keyboard")
    {
        n->acceptAllProperties(*this);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Control::WriteVisitor::do_visit(XMLProperty *p)
{
    const std::string &name = p->getName();
    const KeyConfig *kc = m_control->getKeyConfig();

    if (name == "thrust")
        p->setValue(kc->thrust);
    else if (name == "align")
        p->setValue(kc->align);
    else if (name == "rotleft")
        p->setValue(kc->rotleft);
    else if (name == "rotright")
        p->setValue(kc->rotright);
    else if (name == "fire")
        p->setValue(kc->fire);
    else if (name == "escape")
        p->setValue(kc->escape);
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Control::WriteVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "control")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "keyboard")
    {
        n->acceptAllProperties(*this);
    }
}

//----------------------------------------------------------------------------
PlayerConfiguration::Control::Control()
{
    m_keyConfig.thrust = SDLK_UP;
    m_keyConfig.align = SDLK_DOWN;
    m_keyConfig.rotleft = SDLK_LEFT;
    m_keyConfig.rotright = SDLK_RIGHT;
    m_keyConfig.fire = SDLK_SPACE;
    m_keyConfig.escape = SDLK_ESCAPE;
}

//----------------------------------------------------------------------------
PlayerConfiguration::Control::~Control()
{
}


//----------------------------------------------------------------------------
void PlayerConfiguration::Graphics::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "graphics")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "fullscreen")
    {
        m_graphics->setFullScreen(n->getBoolProperty("enabled"));
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Graphics::WriteVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "graphics")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "fullscreen")
    {
        n->getMandatoryProperty("enabled")->setValue(
            m_graphics->isFullScreen());
    }
}


//----------------------------------------------------------------------------
void PlayerConfiguration::Sound::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "sound")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "sfx")
    {
        SoundInterface *si = SoundInterface::getInstance();

        if (n->getBoolProperty("enabled"))
        {
            si->enableSound();
        }
        else
        {
            si->disableSound();
        }

        unsigned volume = MIN(n->getUnsignedProperty("volume"), 100);
        si->setSoundVolume((Uint8)volume);
    }
    else if (n->getName() == "music")
    {
        SoundInterface *si = SoundInterface::getInstance();

        if (n->getBoolProperty("enabled"))
        {
            si->enableMusic();
        }
        else
        {
            si->disableMusic();
        }

        unsigned volume = MIN(n->getUnsignedProperty("volume"), 100);
        si->setMusicVolume((Uint8)volume);
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Sound::WriteVisitor::do_visit(XMLNode *n)
{
    SoundInterface *si = SoundInterface::getInstance();

    if (n->getName() == "sound")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "sfx")
    {
        n->getMandatoryProperty("enabled")->setValue(si->isSoundEnabled());
        n->getMandatoryProperty("volume")->setValue(si->getSoundVolume());
    }
    else if (n->getName() == "music")
    {
        n->getMandatoryProperty("enabled")->setValue(si->isMusicEnabled());
        n->getMandatoryProperty("volume")->setValue(si->getMusicVolume());
    }
}


//----------------------------------------------------------------------------
void PlayerConfiguration::Mission::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "mission")
    {
        m_mission->m_currentMission = n->getStringProperty("lastplayed", "");

        n->acceptAllChilds(*this);
    }
    else
    {
        Entry entry;
        entry.lastPlayed = n->getUnsignedProperty("lastplayed");
        entry.handicap = n->getUnsignedProperty("handicap");

        m_mission->m_missionMap.insert(
            MissionMap::value_type(n->getName(), entry));
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Mission::WriteVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "mission")
    {
        // Build new mission nodes.

        n->clear();

        n->addProperty(
            new XMLProperty("lastplayed", m_mission->m_currentMission));

        MissionMap::const_iterator iter;
        for (iter = m_mission->m_missionMap.begin();
             iter != m_mission->m_missionMap.end(); ++iter)
        {
            XMLNode *mission = new XMLNode(iter->first);
            mission->addProperty(
                new XMLProperty("lastplayed", iter->second.lastPlayed));
            mission->addProperty(
                new XMLProperty("handicap", iter->second.handicap));
            n->addNode(mission);
        }
    }
}

//----------------------------------------------------------------------------
unsigned PlayerConfiguration::Mission::getHandicap() const
{
    MissionMap::const_iterator iter = m_missionMap.find(m_currentMission);
    if (iter != m_missionMap.end())
    {
        return iter->second.handicap;
    }

    return 0;
}

//----------------------------------------------------------------------------
unsigned PlayerConfiguration::Mission::getLastPlayedLevel() const
{
    MissionMap::const_iterator iter = m_missionMap.find(m_currentMission);
    if (iter != m_missionMap.end())
    {
        return iter->second.lastPlayed;
    }

    return 0;
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Mission::updateLastPlayed(const std::string &mission,
                                                    const unsigned level)
{
    m_currentMission = mission;

    MissionMap::iterator iter = m_missionMap.find(mission);
    if (iter != m_missionMap.end())
    {
        if (level > 0)
        {
            iter->second.lastPlayed = level;
            if (level > iter->second.handicap)
            {
                iter->second.handicap = level;
            }
        }
    }
    else
    {
        Entry entry;
        entry.lastPlayed = MAX(1, level);
        entry.handicap = MAX(1, level);

        m_missionMap.insert(MissionMap::value_type(mission, entry));
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Mission::trimToRealNumberOfLevels(const std::string &mission,
                                                            const unsigned numberOfLevels)
{
    MissionMap::iterator iter = m_missionMap.find(mission);
    if (iter != m_missionMap.end())
    {
        iter->second.lastPlayed =
            MIN(iter->second.lastPlayed, numberOfLevels);
        iter->second.handicap =
            MIN(iter->second.handicap, numberOfLevels);
    }
}



//----------------------------------------------------------------------------
void PlayerConfiguration::Highscore::ReadVisitor::do_visit(const XMLNode *n)
{
    if (n->getName() == "highscore")
    {
        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "mission")
    {
        const std::string &mission = n->getStringProperty("name");
        m_currentTable = new HighscoreTable();
        m_highscore->m_highscoreMap[mission] = m_currentTable;

        n->acceptAllChilds(*this);
    }
    else if (n->getName() == "scoreentry")
    {
        const std::string &name = n->getStringProperty("name");
        unsigned score = n->getUnsignedProperty("score");
        time_t timeStamp = (time_t)n->getUnsignedProperty("timestamp");

        m_currentTable->insert(Entry(name, score, timeStamp));
    }
}

//----------------------------------------------------------------------------
unsigned PlayerConfiguration::Highscore::insertEntry(
    const std::string &mission,
    const std::string &name,
    const unsigned score)
{
    bool dirty = false;

    HighscoreMap::iterator iter = m_highscoreMap.find(mission);
    if (iter == m_highscoreMap.end())
    {
        // No highscore table for the given mission yet. Create one.
        iter = m_highscoreMap.insert(
            HighscoreMap::value_type(mission, new HighscoreTable())).first;
        dirty = true;
    }

    HighscoreTable *table = iter->second;
    HighscoreTable::iterator newEntry = table->end();

    if (table->size() < HIGHSCORE_TABLE_MAX_SIZE ||
        table->rbegin()->getScore() < score)
    {
        // Trim the table to HIGHSCORE_TABLE_MAX_SIZE-1 entries.
        while (table->size() >= HIGHSCORE_TABLE_MAX_SIZE)
        {
            table->erase(*(table->rbegin()));
        }

        newEntry = table->insert(Entry(name, score, time(NULL))).first;
        dirty = true;
    }

    if (dirty)
    {
        PlayerConfiguration::getInstance()->writeHighscore();
    }

    return std::distance(table->begin(), newEntry);
}

//----------------------------------------------------------------------------
void PlayerConfiguration::Highscore::updateHighscoreNode(XMLNode *highscoreNode) const
{
    highscoreNode->clear();

    for (HighscoreMap::const_iterator mapit = m_highscoreMap.begin();
         mapit != m_highscoreMap.end(); ++mapit)
    {
        XMLNode *missionNode = new XMLNode("mission");
        missionNode->addProperty(new XMLProperty("name", mapit->first));

        for (HighscoreTable::const_iterator tabit = mapit->second->begin();
             tabit != mapit->second->end(); ++tabit)
        {
            XMLNode *scoreentryNode = new XMLNode("scoreentry");
            scoreentryNode->addProperty(
                new XMLProperty("name", tabit->getName()));
            scoreentryNode->addProperty(
                new XMLProperty("score", tabit->getScore()));
            scoreentryNode->addProperty(
                new XMLProperty("timestamp", (unsigned)tabit->getTimestamp()));

            missionNode->addNode(scoreentryNode);
        }

        highscoreNode->addNode(missionNode);
    }
}



//----------------------------------------------------------------------------
/**
 * @todo: Remove this class, in one of the next releases.
 */
class GMigrateOldConfigControlVisitor : public XMLConstVisitor
{
public:
    GMigrateOldConfigControlVisitor(XMLNode *newKeyboard) 
            : m_newKeyboard(newKeyboard) {}
    ~GMigrateOldConfigControlVisitor() { m_newKeyboard = NULL; }

private:
    void do_visit(const XMLNode *n)
    {
        if (n->getName() == "keyboard")
        {
            n->acceptAllChilds(*this);
        }
        if (n->getName() == "key")
        {
            const std::string &name = n->getStringProperty("name");
            unsigned key = n->getUnsignedProperty("value");

            if (name == "thrust" || name == "align" ||
                name == "rotleft" || name == "rotright" ||
                name == "fire" || name == "escape")
            {
                m_newKeyboard->addProperty(new XMLProperty(name, key));
            }
        }
    }

    XMLNode *m_newKeyboard;
};

//----------------------------------------------------------------------------
/**
 * @todo: Remove this function, in one of the next releases.
 */
static bool g_migrateOldConfig(const Directory &homeMoaggDir) throw (Exception)
{
    const char *controlFile = "controls.xml";
    const char *playerFile = "player.xml";
    const char *soundFile = "sound.xml";

    if (!homeMoaggDir.hasFile(controlFile) ||
        !homeMoaggDir.hasFile(playerFile) ||
        !homeMoaggDir.hasFile(soundFile))
    {
        return false;
    }

    std::cout << "Migrating old config files to new format ... " << std::flush;

    // Parse the old config files.

    XMLNode player;
    XMLNode controls;
    XMLNode sound;

    XMLParser p;
    p.parse(std::string(homeMoaggDir.getName())
            .append("/").append(playerFile).c_str(), player);
    p.parse(std::string(homeMoaggDir.getName())
            .append("/").append(controlFile).c_str(), controls);
    p.parse(std::string(homeMoaggDir.getName())
            .append("/").append(soundFile).c_str(), sound);

    // Create the new config structure.

    XMLNode newRoot;
    XMLNode *newConfig = new XMLNode("config");
    newConfig->addProperty(new XMLProperty("version", PACKAGE_VERSION));

    XMLNode *newPlayer = new XMLNode("player");
    newPlayer->addProperty(
        new XMLProperty(
            "name", player.getMandatoryNode("player")->getMandatoryNode("name")
            ->getStringProperty("value")));
    newPlayer->addProperty(new XMLProperty("shiptype", 0));
    newConfig->addNode(newPlayer);

    XMLNode *newControl = new XMLNode("control");
    XMLNode *newKeyboard = new XMLNode("keyboard");
    GMigrateOldConfigControlVisitor v(newKeyboard);
    controls.getMandatoryNode("keyboard")->accept(v);
    newControl->addNode(newKeyboard);
    newConfig->addNode(newControl);

    XMLNode *newGraphics = new XMLNode("graphics");
    XMLNode *newFullScreen = new XMLNode("fullscreen");
    newFullScreen->addProperty(new XMLProperty("enabled", "0"));
    newGraphics->addNode(newFullScreen);
    newConfig->addNode(newGraphics);

    XMLNode *newSound = new XMLNode("sound");
    XMLNode *newSfx = new XMLNode("sfx");
    newSfx->addProperty(
        new XMLProperty(
            "enabled", sound.getMandatoryNode("soundconfig")
            ->getMandatoryNode("sound")->getBoolProperty("enabled")));
    newSfx->addProperty(
        new XMLProperty(
            "volume", sound.getMandatoryNode("soundconfig")
            ->getMandatoryNode("sound")->getUnsignedProperty("volume")));
    newSound->addNode(newSfx);
    XMLNode *newMusic = new XMLNode("music");
    newMusic->addProperty(
        new XMLProperty(
            "enabled", sound.getMandatoryNode("soundconfig")
            ->getMandatoryNode("music")->getBoolProperty("enabled")));
    newMusic->addProperty(
        new XMLProperty(
            "volume", sound.getMandatoryNode("soundconfig")
            ->getMandatoryNode("music")->getUnsignedProperty("volume")));
    newSound->addNode(newMusic);
    newConfig->addNode(newSound);

    XMLNode *newMission = new XMLNode("mission");
    newConfig->addNode(newMission);

    newRoot.addNode(newConfig);

    XMLWriteToFileVisitor w(
        std::string(homeMoaggDir.getName()).append("/config.xml").c_str());
    newRoot.accept(w);

    File::unlink(
        std::string(homeMoaggDir.getName())
        .append("/").append(playerFile).c_str());
    File::unlink(
        std::string(homeMoaggDir.getName())
        .append("/").append(controlFile).c_str());
    File::unlink(
        std::string(homeMoaggDir.getName())
        .append("/").append(soundFile).c_str());

    return true;
}

//----------------------------------------------------------------------------
static bool g_updateConfig(const Directory &homeMoaggDir,
                           const char *configFile) throw (Exception)
{
    bool updated = false;

    std::string file =
        std::string(homeMoaggDir.getName()).append("/").append(configFile);

    XMLParser p;
    XMLNode root;

    p.parse(file.c_str(), root);

    XMLNode *config = root.getMandatoryNode("config");

    if (config->getMandatoryProperty("version")->getValue() != "0.11")
    {
        std::cout << "Updating version in config file ... " << std::flush;

        config->getMandatoryProperty("version")->setValue("0.11");

        updated = true;
    }

    if (!config->hasNode("graphics"))
    {
        std::cout << "Adding graphics node to config file ... " << std::flush;

        XMLNode *graphics = new XMLNode("graphics");
        XMLNode *fullScreen = new XMLNode("fullscreen");
        fullScreen->addProperty(new XMLProperty("enabled", "0"));
        graphics->addNode(fullScreen);

        config->addNode(graphics);

        updated = true;
    }

    if (updated)
    {
        XMLWriteToFileVisitor w(file.c_str());
        root.accept(w);
    }

    return updated;
}


//----------------------------------------------------------------------------
PlayerConfiguration::PlayerConfiguration()
{
    const char *home = getenv("HOME");
    const char *dotMoagg = ".moagg";
    const char *configFile = "config.xml";
    const char *highscoreFile = "score.xml";

    if (home == NULL)
    {
        throw Exception(
            "The HOME environment is not set. "
            "Cannot initialize local .moagg directory");
    }

    std::string globalPath =
        std::string(Configuration::getInstance()->getDataDir()) + "/cfg/user/";
    std::string homePath = std::string(home) + "/" + dotMoagg;

    m_configFile = homePath + "/" + configFile;
    m_highscoreFile = homePath + "/" + highscoreFile;

    Directory d(home);
    if (!d.hasDirectory(dotMoagg))
    {
        Directory::mkdir(homePath.c_str(), 0700);
    }

    Directory homeMoaggDir(homePath.c_str());
    if (homeMoaggDir.hasFile(configFile))
    {
        g_updateConfig(homeMoaggDir, configFile);
    }
    else
    {
        if (!g_migrateOldConfig(homeMoaggDir))
        {
            std::cout << "Copying default config file to '"
                      << homePath << std::endl;
            std::string globalConfigFile = globalPath + configFile;
            File::copy(globalConfigFile.c_str(), m_configFile.c_str());
        }
    }

    if (!homeMoaggDir.hasFile(highscoreFile))
    {
        std::string globalHighscoreFile = globalPath + highscoreFile;
        File::copy(globalHighscoreFile.c_str(), m_highscoreFile.c_str());
    }
}

//----------------------------------------------------------------------------
PlayerConfiguration::~PlayerConfiguration()
{
}


//----------------------------------------------------------------------------
void PlayerConfiguration::loadConfiguration() throw (Exception)
{
    XMLParser p;
    p.parse(m_configFile.c_str(), m_configRoot);

    ReadConfigVisitor v(this);
    m_configRoot.getMandatoryNode("config")->accept(v);
}

//----------------------------------------------------------------------------
void PlayerConfiguration::loadHighscore() throw (Exception)
{
    XMLParser p;
    p.parse(m_highscoreFile.c_str(), m_highscoreRoot);

    ReadHighscoreVisitor v(this);
    m_highscoreRoot.getMandatoryNode("highscore")->accept(v);
}

//----------------------------------------------------------------------------
void PlayerConfiguration::writeConfiguration()
{
    try
    {
        WriteConfigVisitor v1(this);
        m_configRoot.getMandatoryNode("config")->accept(v1);

        XMLWriteToFileVisitor v2(m_configFile.c_str());
        m_configRoot.accept(v2);
    }
    catch (Exception &e)
    {
        std::cerr << "Error writing back configuration: " << e << std::endl;
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::writeHighscore()
{
    try
    {
        XMLNode *highscoreNode = m_highscoreRoot.getMandatoryNode("highscore");
        m_highscore.updateHighscoreNode(highscoreNode);

        XMLWriteToFileVisitor v(m_highscoreFile.c_str());
        m_highscoreRoot.accept(v);
    }
    catch (Exception &e)
    {
        std::cerr << "Error writing back highscore: " << e << std::endl;
    }
}


//----------------------------------------------------------------------------
void PlayerConfiguration::create() throw (Exception)
{
    if (sm_instance == NULL)
    {
        sm_instance = new PlayerConfiguration();
        sm_instance->loadConfiguration();
        sm_instance->loadHighscore();
    }
}

//----------------------------------------------------------------------------
void PlayerConfiguration::destroy()
{
    ZAP_POINTER(sm_instance);
}
