#ifndef PLAYER_CONFIGURATION_H
#define PLAYER_CONFIGURATION_H

#include <set>
#include <map>

#include "XMLElements.h"
#include "XMLVisitors.h"
#include "Ship.h"


//----------------------------------------------------------------------------
#define HIGHSCORE_TABLE_MAX_SIZE 10


//----------------------------------------------------------------------------
class PlayerConfiguration
{
  protected:
    //------------------------------------------------------------------------
    class ReadConfigVisitor : public XMLConstVisitor
    {
      public:
        //--------------------------------------------------------------------
        ReadConfigVisitor(PlayerConfiguration *config) : m_config(config) {}
        ~ReadConfigVisitor() { m_config = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(const XMLNode *n);

        //--------------------------------------------------------------------
        PlayerConfiguration *m_config;
    };

    //------------------------------------------------------------------------
    class WriteConfigVisitor : public XMLVisitor
    {
      public:
        //--------------------------------------------------------------------
        WriteConfigVisitor(const PlayerConfiguration *config) : m_config(config) {}
        ~WriteConfigVisitor() { m_config = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(XMLNode *n);

        //--------------------------------------------------------------------
        const PlayerConfiguration *m_config;
    };

    //------------------------------------------------------------------------
    class ReadHighscoreVisitor : public XMLConstVisitor
    {
      public:
        //--------------------------------------------------------------------
        ReadHighscoreVisitor(PlayerConfiguration *config) : m_config(config) {}
        ~ReadHighscoreVisitor() { m_config = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(const XMLNode *n);

        //--------------------------------------------------------------------
        PlayerConfiguration *m_config;
    };

  public:

    //------------------------------------------------------------------------
    class Player
    {
      public:
        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Player *player) : m_player(player) {}
            ~ReadVisitor() { m_player = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLProperty *p);
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Player *m_player;
        };
        
        //--------------------------------------------------------------------
        class WriteVisitor : public XMLVisitor
        {
          public:
            //----------------------------------------------------------------
            WriteVisitor(const Player *player) : m_player(player) {}
            ~WriteVisitor() { m_player = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(XMLProperty *p);
            void do_visit(XMLNode *n);

            //----------------------------------------------------------------
            const Player *m_player;
        };

      public:
        //--------------------------------------------------------------------
        Player() {}
        ~Player() {}

        //--------------------------------------------------------------------
        inline void setName(const std::string &name)
        {
            m_name = name;
        }

        inline const std::string &getName() const
        {
            return m_name;
        }

        //--------------------------------------------------------------------
        inline void setShipType(ShipSurfaces::Ship shipType)
        {
            assert(shipType >= 0);
            assert(shipType < ShipSurfaces::S_TOTAL_NUMBER);

            m_shipType = shipType;
        }

        inline ShipSurfaces::Ship getShipType() const
        {
            return m_shipType;
        }

      private:
        //--------------------------------------------------------------------
        std::string m_name;
        ShipSurfaces::Ship m_shipType;
    };


    //------------------------------------------------------------------------
    class Control
    {
      public:
        //--------------------------------------------------------------------
        struct KeyConfig
        {
            SDLKey thrust;
            SDLKey align;
            SDLKey rotleft;
            SDLKey rotright;
            SDLKey fire;
            SDLKey escape;
        };

        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Control *control) : m_control(control) {}
            ~ReadVisitor() { m_control = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLProperty *p);
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Control *m_control;
        };
        
        //--------------------------------------------------------------------
        class WriteVisitor : public XMLVisitor
        {
          public:
            //----------------------------------------------------------------
            WriteVisitor(const Control *control) : m_control(control) {}
            ~WriteVisitor() { m_control = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(XMLProperty *p);
            void do_visit(XMLNode *n);

            //----------------------------------------------------------------
            const Control *m_control;
        };

      public:
        //--------------------------------------------------------------------
        Control();
        ~Control();

        //--------------------------------------------------------------------
        inline const KeyConfig *getKeyConfig() const
        {
            return &m_keyConfig;
        }

        inline KeyConfig *getKeyConfig()
        {
            return &m_keyConfig;
        }
        
      private:
        //--------------------------------------------------------------------
        KeyConfig m_keyConfig;
    };


    //------------------------------------------------------------------------
    class Graphics
    {
      public:
        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Graphics *graphics) : m_graphics(graphics) {}
            ~ReadVisitor() { m_graphics = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Graphics *m_graphics;
        };
        
        //--------------------------------------------------------------------
        class WriteVisitor : public XMLVisitor
        {
          public:
            //----------------------------------------------------------------
            WriteVisitor(const Graphics *graphics) : m_graphics(graphics) {}
            ~WriteVisitor() { m_graphics = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(XMLNode *n);

            //----------------------------------------------------------------
            const Graphics *m_graphics;
        };

        //--------------------------------------------------------------------
        Graphics() { m_fullScreen = false; }
        ~Graphics() {}

        inline bool isFullScreen() const
        {
            return m_fullScreen;
        }

        inline void setFullScreen(bool fullScreen)
        {
            m_fullScreen = fullScreen;
        }

      private:
        //--------------------------------------------------------------------
        bool m_fullScreen;
    };


    //------------------------------------------------------------------------
    class Sound
    {
      public:
        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Sound *sound) : m_sound(sound) {}
            ~ReadVisitor() { m_sound = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Sound *m_sound;
        };
        
        //--------------------------------------------------------------------
        class WriteVisitor : public XMLVisitor
        {
          public:
            //----------------------------------------------------------------
            WriteVisitor(const Sound *sound) : m_sound(sound) {}
            ~WriteVisitor() { m_sound = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(XMLNode *n);

            //----------------------------------------------------------------
            const Sound *m_sound;
        };

        //--------------------------------------------------------------------
        Sound() {}
        ~Sound() {}
    };


    //------------------------------------------------------------------------
    class Mission
    {
        //--------------------------------------------------------------------
        struct Entry
        {
            unsigned lastPlayed;
            unsigned handicap;
        };

        //--------------------------------------------------------------------
        typedef std::map<std::string, Entry> MissionMap;

      public:
        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Mission *mission)
                : m_mission(mission) {}
            ~ReadVisitor() { m_mission = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Mission *m_mission;
        };

        //--------------------------------------------------------------------
        class WriteVisitor : public XMLVisitor
        {
          public:
            //----------------------------------------------------------------
            WriteVisitor(const Mission *mission)
                : m_mission(mission) {}
            ~WriteVisitor() { m_mission = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(XMLNode *n);

            //----------------------------------------------------------------
            const Mission *m_mission;
        };


        //--------------------------------------------------------------------
        Mission() {}
        ~Mission() {}

        //--------------------------------------------------------------------
        /**
         * @return The last played mission.
         */
        inline const std::string &getCurrentMission() const
        {
            return m_currentMission;
        }

        /**
         * @return The handicap for the current mission.
         */
        unsigned getHandicap() const;

        /**
         * @return The last played level of the current mission.
         */
        unsigned getLastPlayedLevel() const;

        /**
         * Updates the last played mission, level and handicap.
         * If the level parameter is 0, only m_lastPlayedMission
         * will be updated and an empty MissionMap entry will be created,
         * if the mission doesn't exist yet.
         */
        void updateLastPlayed(const std::string &mission,
                              const unsigned level);

        void trimToRealNumberOfLevels(const std::string &mission,
                                      const unsigned numberOfLevels);

      private:
        //--------------------------------------------------------------------
        std::string m_currentMission;
        MissionMap m_missionMap;
    };


    //------------------------------------------------------------------------
    class Highscore
    {
      protected:
        //--------------------------------------------------------------------
        class Entry
        {
          public:
            //----------------------------------------------------------------
            Entry(const std::string &name, unsigned score, time_t timestamp)
                : m_name(name), m_score(score), m_timestamp(timestamp) {}
            ~Entry() {}

            //----------------------------------------------------------------
            inline const std::string &getName() const { return m_name; }
            inline unsigned getScore() const { return m_score; }
            inline time_t getTimestamp() const { return m_timestamp; }

          private:
            //----------------------------------------------------------------
            std::string m_name;
            unsigned m_score;
            time_t m_timestamp;
        };

        //--------------------------------------------------------------------
        struct LessThan
        {
            //----------------------------------------------------------------
            bool operator()(const Entry &s1, const Entry &s2) const
            {
                if (s1.getScore() == s2.getScore())
                    return s1.getTimestamp() < s2.getTimestamp();
                else
                    return s1.getScore() > s2.getScore();
            }
        };
    
      public:

        //--------------------------------------------------------------------
        typedef std::set<Entry, LessThan> HighscoreTable;
        typedef std::map<std::string, HighscoreTable*> HighscoreMap;

        typedef HighscoreTable::iterator iterator;
        typedef HighscoreTable::const_iterator const_iterator;


        //--------------------------------------------------------------------
        class ReadVisitor : public XMLConstVisitor
        {
          public:
            //----------------------------------------------------------------
            ReadVisitor(Highscore *highscore)
                : m_highscore(highscore), m_currentTable(NULL) {}
            ~ReadVisitor() { m_highscore = NULL; m_currentTable = NULL; }

          private:
            //----------------------------------------------------------------
            void do_visit(const XMLNode *n);

            //----------------------------------------------------------------
            Highscore *m_highscore;
            HighscoreTable *m_currentTable;
        };


        //--------------------------------------------------------------------
        Highscore() {}
        ~Highscore() {}

        //--------------------------------------------------------------------
        /**
         * @return 0..HIGHSCORE_TABLE_MAX_SIZE-1 for an apropriate entry,
         *         or HIGHSCORE_TABLE_MAX_SIZE, if no entry was made.
         */
        unsigned insertEntry(const std::string &mission,
                             const std::string &name,
                             const unsigned score);

        //--------------------------------------------------------------------
        void updateHighscoreNode(XMLNode *highscoreNode) const;


        //--------------------------------------------------------------------
        inline bool hasMission(const std::string &mission) const
        {
            return m_highscoreMap.find(mission) != m_highscoreMap.end();
        }

        //--------------------------------------------------------------------
        inline size_t size(const std::string &mission) const
        {
            return m_highscoreMap.find(mission)->second->size();
        }

        inline bool empty(const std::string &mission) const
        {
            return m_highscoreMap.find(mission)->second->empty();
        }

        inline iterator begin(const std::string &mission)
        {
            return m_highscoreMap.find(mission)->second->begin();
        }

        inline iterator end(const std::string &mission)
        {
            return m_highscoreMap.find(mission)->second->end();
        }

        inline const_iterator begin(const std::string &mission) const
        {
            return m_highscoreMap.find(mission)->second->begin();
        }

        inline const_iterator end(const std::string &mission) const
        {
            return m_highscoreMap.find(mission)->second->end();
        }

      private:
        //--------------------------------------------------------------------
        HighscoreMap m_highscoreMap;
    };
    
  protected:
    //------------------------------------------------------------------------
    PlayerConfiguration();

  public:
    //------------------------------------------------------------------------
    ~PlayerConfiguration();


    //------------------------------------------------------------------------
    inline Player *getPlayer() { return &m_player; }
    inline const Player *getPlayer() const { return &m_player; }
    inline Control *getControl() { return &m_control; }
    inline const Control *getControl() const { return &m_control; }
    inline Graphics *getGraphics() { return &m_graphics; }
    inline const Graphics *getGraphics() const { return &m_graphics; }
    inline Sound *getSound() { return &m_sound; }
    inline const Sound *getSound() const { return &m_sound; }
    inline Mission *getMission() { return &m_mission; }
    inline const Mission *getMission() const { return &m_mission; }

    inline Highscore *getHighscore() { return &m_highscore; }
    inline const Highscore *getHighscore() const { return &m_highscore; }


    //------------------------------------------------------------------------
    void writeConfiguration();
    void writeHighscore();


    //------------------------------------------------------------------------
    static void create() throw (Exception);

    static void destroy();

    inline static PlayerConfiguration* getInstance()
    {
        return sm_instance;
    }

  private:

    //-----------------------------------------------------------------------
    void loadConfiguration() throw (Exception);
    void loadHighscore() throw (Exception);


    //------------------------------------------------------------------------
    std::string m_configFile;
    std::string m_highscoreFile;

    //------------------------------------------------------------------------
    XMLNode m_configRoot;
    XMLNode m_highscoreRoot;

    //------------------------------------------------------------------------
    Player m_player;
    Control m_control;
    Graphics m_graphics;
    Sound m_sound;
    Mission m_mission;
    Highscore m_highscore;

    //------------------------------------------------------------------------
    static PlayerConfiguration *sm_instance;
};

#endif // PLAYER_CONFIGURATION_H
