#include <cstring>
#include <iostream>

#include "MathTools.h"
#include "SinusTable.h"
#include "StopWatch.h"


//============================================================================
// Implementation of Tools.h
//============================================================================

//----------------------------------------------------------------------------
int myRand(int limit)
{
    return (int)(1.0 * limit * rand() / (RAND_MAX + 1.0));
}



//============================================================================
// Implementation of MathTools.h
//============================================================================

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::Vector()
{
    m_x = 0.0;
    m_y = 0.0;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::Vector(double x, double y)
{
    m_x = x;
    m_y = y;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector::~Vector()
{
}

//----------------------------------------------------------------------------
double MATH_TOOLS::Vector::getAbsValue() const
{
    return sqrt(m_x*m_x + m_y*m_y);
}

//----------------------------------------------------------------------------
double MATH_TOOLS::Vector::getAngle() const
{
    return sqrt(m_x*m_x + m_y*m_y);
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::setExponential(double absValue, double angle)
{
    m_x = absValue * SinusTable::sin((int)rint(angle));
    m_y = absValue * SinusTable::cos((int)rint(angle));
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::addExponential(double absValue, double angle)
{
    m_x += absValue * SinusTable::sin((int)rint(angle));
    m_y += absValue * SinusTable::cos((int)rint(angle));
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator+=(const Vector &other)
{
    m_x += other.m_x;
    m_y += other.m_y;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator-=(const Vector &other)
{
    m_x -= other.m_x;
    m_y -= other.m_y;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator*=(const double scale)
{
    m_x *= scale;
    m_y *= scale;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector &MATH_TOOLS::Vector::operator/=(const double scale)
{
    m_x /= scale;
    m_y /= scale;
    return *this;
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector MATH_TOOLS::operator+(const Vector &v1, const Vector &v2)
{
    return Vector(v1.m_x + v2.m_x, v1.m_y + v2.m_y);
}

//----------------------------------------------------------------------------
MATH_TOOLS::Vector MATH_TOOLS::operator-(const Vector &v1, const Vector &v2)
{
    return Vector(v1.m_x - v2.m_x, v1.m_y - v2.m_y);
}


//----------------------------------------------------------------------------
double MATH_TOOLS::getAngle(double x, double y)
{
    double angle = (atan2(y, -x) * 360.0 / (2.0*M_PI)) - 90.0;
    if (angle < 0.0)  angle += 360.0;
    return angle;
}



//============================================================================
// Implementation of SinusTable.h
//============================================================================

//----------------------------------------------------------------------------
SinusTable SinusTable::sm_instance;
double SinusTable::sm_sinTable[360];
double SinusTable::sm_cosTable[360];

//----------------------------------------------------------------------------
SinusTable::SinusTable()
{
    for (int i=0; i<360; i++)
    {
        double a = i * 2.0*M_PI/360.0;

        sm_sinTable[i] = ::sin(a);
        sm_cosTable[i] = ::cos(a);
    }
}

//----------------------------------------------------------------------------
SinusTable::~SinusTable()
{
}



//============================================================================
// Implementation of StopWatch.h
//============================================================================

//----------------------------------------------------------------------------
StopWatch::StopWatch(const char *name)
{
    m_name = name;

    reset();
}

//----------------------------------------------------------------------------
StopWatch::~StopWatch()
{
    m_name = NULL;
}

//----------------------------------------------------------------------------
void StopWatch::start()
{
    (void)gettimeofday(&m_start, NULL);
}

//----------------------------------------------------------------------------
void StopWatch::stop()
{
    (void)gettimeofday(&m_stop, NULL);

    m_total.tv_sec += m_stop.tv_sec - m_start.tv_sec;
    m_total.tv_usec += m_stop.tv_usec - m_start.tv_usec;

    if (m_total.tv_usec < 0)
    {
        m_total.tv_sec--;
        m_total.tv_usec += 1000000;
    }
    else if (m_total.tv_usec > 1000000)
    {
        m_total.tv_sec++;
        m_total.tv_usec -= 1000000;
    }
}

//----------------------------------------------------------------------------
void StopWatch::reset()
{
    memset(&m_start, 0, sizeof(m_start));
    memset(&m_stop, 0, sizeof(m_stop));
    memset(&m_total, 0, sizeof(m_total));
}

//----------------------------------------------------------------------------
int StopWatch::compare(const StopWatch &other) const
{
    if (m_total.tv_sec < other.m_total.tv_sec)
    {
        return -1;
    }
    if (m_total.tv_sec > other.m_total.tv_sec)
    {
        return 1;
    }
    if (m_total.tv_usec < other.m_total.tv_usec)
    {
        return -1;
    }
    if (m_total.tv_usec > other.m_total.tv_usec)
    {
        return 1;
    }

    return 0;
}

//----------------------------------------------------------------------------
std::ostream &operator<<(std::ostream &os, const StopWatch &s)
{
    return os << s.m_name << ": "
              << s.m_total.tv_sec << " sec, "
              << s.m_total.tv_usec << " usec";
}
