/* Copyright (C) 2000-2006 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL

/*
#define DEBUG_SQL
*/

#define DEBUG_ERR_QUERY



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"
#include "udm_xmalloc.h"

#if HAVE_IODBC
#include <sql.h>
#include <sqlext.h>
#endif

#if HAVE_EASYSOFT
#include <sql.h>
#include <sqlext.h>
#endif

#if HAVE_VIRT
#include <iodbc.h>
#include <isql.h>
#include <isqlext.h>
#endif

#if HAVE_UNIXODBC
#include <sql.h>
#include <sqlext.h>
#endif

#if HAVE_SAPDB
#include <WINDOWS.H>
#include <sql.h>
#include <sqlext.h>
#endif

#if HAVE_SOLID
#include <cli0cli.h>
#endif

#if HAVE_DB2
#include <sqlcli1.h>
#endif

#if HAVE_CTLIB
#include <ctpublic.h>
#endif

#if HAVE_SQLITE
#include <sqlite.h>
#endif

#if HAVE_SQLITE3
#include <sqlite3.h>
#endif

#ifdef WIN32
#include <process.h>
#endif


void DecodeHexStr (char *src, UDM_PSTR *dst, size_t size)
{
  char p1;
  char p2;
  size_t dst_size = size / 2 + 1;
  size_t i;
  size_t d = 0;

  dst->val = UdmMalloc(dst_size);

  if (!size)
    goto ex;
  
  for (i = 0; (size > 0) && (i < size - 1); i++)
  {
    if (src[i] >= '0' && src[i] <= '9')
      p1 = src[i] - '0';
    else if (src[i] >= 'A' && src[i] <= 'F')
      p1 = src[i] - 'A' + 10;
    else if (src[i] >= 'a' && src[i] <= 'f')
      p1 = src[i] - 'a' + 10;
    else break;

    if (src[i + 1] >= '0' && src[i + 1] <= '9')
      p2 = src[i + 1] - '0';
    else if (src[i + 1] >= 'A' && src[i + 1] <= 'F')
      p2 = src[i + 1] - 'A' + 10;
    else if (src[i + 1] >= 'a' && src[i + 1] <= 'f')
      p2 = src[i + 1] - 'a' + 10;
    else break;

    dst->val[d++] = (p1 << 4) | p2;
    i++;
  }
  
ex:
  dst->val[d] = 0;
  dst->len = d;
}


/***************************************************************/


static int udb_free_result(UDM_SQLRES *res){
  size_t i;
  size_t nitems;
  if(res)
  {
    if(res->Items)
    {
      nitems = res->nCols * res->nRows;
      for(i=0;i<nitems;i++)
        if(res->Items[i].val)
          UDM_FREE(res->Items[i].val);
      UDM_FREE(res->Items);
    }
  }
  return(0);
}

/************************** ODBC ***********************************/
#if (HAVE_ODBC)

typedef struct udm_odbc_conn_st
{
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB)
  HDBC  hDbc;
  HENV  hEnv;
  HSTMT hstmt;
#endif
#if HAVE_DB2
  SQLHANDLE hDbc;
  SQLHANDLE hEnv;
  SQLHANDLE hstmt;
#endif
} UDM_ODBC_CONN;


#define SQL_OK(rc)  ((rc==SQL_SUCCESS)||(rc==SQL_SUCCESS_WITH_INFO))

static int UdmODBCDisplayError(UDM_DB *db, const char *prefix)
{
  UCHAR szSqlState[10];
  UCHAR szErrMsg[400];
  SDWORD naterr;
  SWORD length;
  RETCODE rc=SQL_SUCCESS;
  size_t len= strlen(prefix);
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;

  strcpy(db->errstr, prefix);
  
  while (1)
  {   
    rc = SQLError(sdb->hEnv, sdb->hDbc, sdb->hstmt, szSqlState,
                  &naterr,szErrMsg,sizeof(szErrMsg)-1,&length);
    if ((SQL_SUCCESS != rc) && (SQL_SUCCESS_WITH_INFO != rc))
      break;
    szErrMsg[sizeof(szErrMsg)-1]='\0';
    len+= sprintf(db->errstr+len, "[SQLSTATE:%s]%s",szSqlState,szErrMsg);
  }
  return(0);
}


typedef struct udm_odbc_field_st
{
  SQLCHAR     Name[32];
  SQLSMALLINT NameLen;
  SQLSMALLINT Type;
  SQLUINTEGER Size;
  SQLSMALLINT Scale;
  SQLSMALLINT Nullable;
} UDM_ODBC_FIELD;


static int
UdmODBCTypeIsBinary(int Type)
{
  return  Type == SQL_BINARY ||
          Type == SQL_VARBINARY ||
          Type == SQL_LONGVARBINARY ||
          Type == -98/* DB2 SQL_BLOB */;
}


static int execDB(UDM_DB*db,UDM_SQLRES *result, const char *sqlstr)
{
  RETCODE rc;
  SWORD iResColumns;
  SDWORD iRowCount;
  static char  bindbuf[(int)(64L * 1024L - 16L)];
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;

  if(!strcmp(sqlstr,"COMMIT"))
  {
    rc=SQLTransact(sdb->hEnv, sdb->hDbc, SQL_COMMIT);
    if(!SQL_OK(rc))
    {
      db->errcode=1;
      return UDM_ERROR;
    }
    else
    {
      db->errcode=0;
      return UDM_OK;
    }
  }
  
  rc=SQLAllocStmt(sdb->hDbc, &sdb->hstmt);
  if (!SQL_OK(rc))
  {
    db->errcode=1;
    return UDM_ERROR;
  }
  
  rc=SQLExecDirect(sdb->hstmt,(SQLCHAR *)sqlstr, SQL_NTS);
  if (!SQL_OK(rc))
  {
    if(rc==SQL_NO_DATA) goto ND;
    db->errcode=1;
    return UDM_ERROR; 
  }
  
  rc=SQLNumResultCols(sdb->hstmt, &iResColumns);
  if(!SQL_OK(rc))
  {
    db->errcode=1;
    return UDM_ERROR;
  }
  
  if(!iResColumns)
  {
    rc=SQLRowCount(sdb->hstmt, &iRowCount);
    if (!SQL_OK(rc))
    {
      db->errcode=1;
      return UDM_ERROR;
    }
  }
  else
  {
    int res_count, col;
    size_t mRows;

    UDM_ODBC_FIELD Field[128];

    result->nRows = 0;
    result->nCols = iResColumns;
    result->Items = NULL;

    result->Fields=(UDM_SQLFIELD*)UdmMalloc(result->nCols*sizeof(UDM_SQLFIELD));
    bzero(result->Fields, result->nCols*sizeof(UDM_SQLFIELD));

    for(col=0; col < iResColumns; col++)
    {
      SQLDescribeCol(sdb->hstmt, col+1,
                     Field[col].Name, 
                     (SQLSMALLINT) sizeof(Field[col].Name) - 1,
                     &Field[col].NameLen,
                     &Field[col].Type,
                     &Field[col].Size,
                     &Field[col].Scale,
                     &Field[col].Nullable);
      result->Fields[col].sqlname= (char*)UdmStrdup((const char*)Field[col].Name);
    }

    rc= SQL_NO_DATA_FOUND;
    for (mRows=0, res_count=0;(db->res_limit?(res_count<db->res_limit):1);res_count++)
    {
      int i;
      rc=SQLFetch(sdb->hstmt);
      if (!SQL_OK(rc))
      {
        if (rc != SQL_NO_DATA_FOUND)
        {
          db->errcode=1;
          if (db->DBType == UDM_DB_DB2) /* FIXME: check with all databases */
            return UDM_ERROR;
        }
        break;
      }
      if (mRows <= (result->nRows+1) * iResColumns)
      {
        mRows+= 1024 * iResColumns;
        result->Items=(UDM_PSTR*)UdmXrealloc(result->Items,mRows*sizeof(UDM_PSTR));
      }
    
      for (i = 0; i < iResColumns; i++)
      {
        size_t offs=result->nRows*iResColumns+i;
        const char *p;
        char *tmp= NULL;
        SDWORD  pcbValue;
        SWORD   get_type;

        if (Field[i].Type == SQL_FLOAT)
        {
          get_type= SQL_FLOAT;
        }
        if (Field[i].Type == SQL_DOUBLE)
        {
          get_type= SQL_C_DOUBLE;
        }
        else if (UdmODBCTypeIsBinary(Field[i].Type))
        {
          /*
            PostgreSQL bytea
            MSSQL BLOB
            XTG Interbase/FireBird BLOB
            Oracle BLOB
            Sybase IMAGE & VARBINARY
          */
          if (db->DBType == UDM_DB_PGSQL ||
              db->DBType == UDM_DB_MSSQL ||
              db->DBType == UDM_DB_IBASE ||
              db->DBType == UDM_DB_ORACLE8 ||
              db->DBType == UDM_DB_SYBASE ||
              db->DBType == UDM_DB_DB2 ||
              db->DBType == UDM_DB_MYSQL)
            get_type= SQL_C_BINARY;
          else
            get_type= SQL_CHAR;
        }

#ifdef HAVE_UNIXODBC
        /*
           For some weird reasons, Sybase native (i.e. not FreeTDS)
           ODBC driver in conjunction with unixODBC always returns 4
           in pcbValue if get_type is SQL_CHAR. Let's fetch as SQL_C_SLONG.
        */
        else if (Field[i].Type == SQL_INTEGER && db->DBType == UDM_DB_SYBASE)
          get_type= SQL_C_SLONG;
#endif
        else
          get_type= SQL_CHAR;

        rc= SQLGetData(sdb->hstmt,i+1,get_type,bindbuf,sizeof(bindbuf),&pcbValue);
        if (rc == SQL_SUCCESS_WITH_INFO)
        {
          if (pcbValue > sizeof(bindbuf) &&
              (tmp= UdmMalloc(pcbValue)))
          {
            SDWORD tmp_pcbValue;
            memcpy(tmp, bindbuf, sizeof(bindbuf));
            rc= SQLGetData(sdb->hstmt,i+1,get_type,tmp+sizeof(bindbuf),pcbValue - sizeof(bindbuf),&tmp_pcbValue);
            p= tmp;
          }
          else
          {
            p= "";
            pcbValue= 0;
          }
        }
        else if (rc == SQL_SUCCESS)
        {
          if (pcbValue == SQL_NULL_DATA)
          {
            bindbuf[0]= '\0';
            pcbValue= 0;
          }
          else if (get_type == SQL_C_FLOAT)
          {
            float num;
            memcpy(&num, bindbuf, sizeof(num));
            pcbValue= sprintf(bindbuf, "%f", num);
          }
          else if (get_type == SQL_C_DOUBLE)
          {
            double num;
            memcpy(&num, bindbuf, sizeof(num));
            pcbValue= sprintf(bindbuf, "%f", num);
          }
          else if (get_type == SQL_C_SLONG)
          {
            long int num;
            memcpy(&num, bindbuf, sizeof(num));
            pcbValue= sprintf(bindbuf, "%ld", num);
          }
          p= bindbuf;
        }


        if (get_type != SQL_C_BINARY &&
            UdmODBCTypeIsBinary(Field[i].Type))
        {
          DecodeHexStr(p, &result->Items[offs], pcbValue);
        }
        else
        {
          result->Items[offs].val= (char*)malloc(pcbValue+1);
          memcpy(result->Items[offs].val, p, pcbValue);
          result->Items[offs].val[pcbValue]='\0';
          result->Items[offs].len= pcbValue;
        }
        if (tmp)
          UdmFree(tmp);
      }
      result->nRows++;
    }

  }
ND:

  SQLFreeStmt(sdb->hstmt, SQL_DROP);
  db->res_limit=0;
  db->errcode=0;
  return UDM_OK;
}

static int UdmODBCInitDB(UDM_DB *db)
{
  char DSN[512]="";
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);
  UDM_SQLRES SQLRes;
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) UdmMalloc(sizeof(UDM_ODBC_CONN));
  db->specific= (void*) sdb;

  sdb->hDbc=  SQL_NULL_HDBC;
  sdb->hEnv=  SQL_NULL_HENV;
  sdb->hstmt= SQL_NULL_HSTMT;

#if (HAVE_SOLID)
  udm_snprintf(DSN, sizeof(DSN)-1, "tcp %s %d", DBHost, DBPort?DBPort:1313);
#elif (HAVE_SAPDB)
  udm_snprintf(DSN, sizeof(DSN)-1, "%s:%s", DBHost, db->DBName?db->DBName:"");
#else
  strncpy(DSN, db->DBName?db->DBName:"", sizeof(DSN)-1);
#endif
  
  db->errcode = SQLAllocEnv(&sdb->hEnv);
  if( SQL_SUCCESS != db->errcode )return -2;
  db->errcode = SQLAllocConnect(sdb->hEnv, &sdb->hDbc);
  if( SQL_SUCCESS != db->errcode )return -3;
  db->errcode = SQLSetConnectOption(sdb->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_ON);
  if( SQL_SUCCESS != db->errcode )return -4;
  db->errcode = SQLConnect(sdb->hDbc, (SQLCHAR *)DSN, SQL_NTS, DBUser, SQL_NTS, DBPass, SQL_NTS);
  if( !SQL_OK(db->errcode)) return(-5);
  else db->errcode=0;
  db->connected=1;
  /*
  if (db->DBType == UDM_DB_DB2)
  {
    1253 = SQL_ATTR_LONGDATA_COMPAT
    int rc= SQLSetConnectAttr(sdb->hDbc, 1253, (SQLPOINTER) 1, 0);
  }
  */
  if (db->DBType == UDM_DB_ORACLE8)
    execDB(db, &SQLRes, "ALTER SESSION SET NLS_NUMERIC_CHARACTERS='. '");
  return 0;
}

static void UdmODBCClose(UDM_DB *db)
{
  if(db->connected)
  {
    UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
    db->errcode= SQLTransact(sdb->hEnv, sdb->hDbc, SQL_COMMIT);
    if(SQL_SUCCESS != db->errcode)
      goto ret;
    db->errcode= SQLDisconnect(sdb->hDbc );
    if(SQL_SUCCESS != db->errcode)
      goto ret;
    db->errcode= SQLFreeConnect(sdb->hDbc);
    if(SQL_SUCCESS != db->errcode)
      goto ret;
    db->errcode= SQLFreeEnv(sdb->hEnv);
    if(SQL_SUCCESS != db->errcode)
      goto ret;
ret:
    db->connected= 0;
    UDM_FREE(db->specific);
  }
}


static int UdmODBCQuery(UDM_DB *db,UDM_SQLRES *res, const char *qbuf)
{
  int rc;
   
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }

  if(!db->connected)
  {
    UdmODBCInitDB(db);
    if(db->errcode)
    {
      UdmODBCDisplayError(db,"");
      return(0);
    }
    else
    {
      db->connected=1;
    }
  }

  rc= execDB(db,res,qbuf);
  if((db->errcode))
  {
    UdmODBCDisplayError(db,"");
    if(strstr(db->errstr,"[SQLSTATE:23000]")) /* MS Access */
    {
      db->errcode=0;
      rc= UDM_OK;
    }
#if 0
    /* Cannot use S1000: it's returned by unixODBC in many cases */
    else if(strstr(db->errstr,"[SQLSTATE:S1000]"))
    { /* Oracle 8i */
      db->errcode=0;
      rc= UDM_OK;
    }
#endif
    else if(strstr(db->errstr,"uplicat")) /* PgSQL,MySQL*/
    {
      db->errcode=0;
      rc= UDM_OK;
    }
    else if(strstr(db->errstr,"nique")) /* Solid, Virtuoso */
    {
      db->errcode=0;
      rc= UDM_OK;
    }
    else if(strstr(db->errstr,"UNIQUE")) /* Mimer */
    {
      db->errcode=0;
      rc= UDM_OK;
    }
    else
    {
      db->errcode=1;
    }
    {
      UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
      SQLFreeStmt(sdb->hstmt, SQL_DROP);
      sdb->hstmt= SQL_NULL_HSTMT;
    }
    
  }
  return rc;
}

static int UdmODBCBegin(UDM_DB *db)
{
  UDM_ODBC_CONN *sdb;
  if(!db->connected)
  {
    UdmODBCInitDB(db);
    if(db->errcode)
    {
      UdmODBCDisplayError(db,"");
      return UDM_ERROR;
    }
    else
    {
      db->connected=1;
    }
  }
  sdb= (UDM_ODBC_CONN*) db->specific;
  db->errcode= SQLSetConnectOption(sdb->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF);
  if(SQL_SUCCESS != db->errcode)return UDM_ERROR;
  return UDM_OK;
}


static int UdmODBCCommit(UDM_DB *db)
{
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
  int rc;
  if (UDM_OK!=(rc=UdmODBCQuery(db,NULL,"COMMIT")))
    return UDM_ERROR;
  db->errcode= SQLSetConnectOption(sdb->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_ON);
  if(SQL_SUCCESS != db->errcode)
    return UDM_ERROR;
  return UDM_OK;
}

static int UdmODBCPrepare (UDM_DB *db, const char *query)
{
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
  int rc;
  db->errcode = 0;
  db->errstr[0] = 0;
  rc= SQLAllocStmt(sdb->hDbc, &sdb->hstmt);

  if (!SQL_OK(rc))
  {
    db->errcode = rc;
    UdmODBCDisplayError(db,"UdmODBCPrepare: ");
    return(UDM_ERROR);
  }
  rc= SQLPrepare(sdb->hstmt, query, SQL_NTS);
  if (!SQL_OK(rc))
  {
    db->errcode = rc;
    UdmODBCDisplayError(db, "UdmODBCPrepare: ");
    return(UDM_ERROR);
  }
  return(UDM_OK);
}


static int
UdmSQLType2ODBCType(int udm_type)
{
  switch (udm_type)
  {
    case UDM_SQLTYPE_LONGVARBINARY: return SQL_LONGVARBINARY;
    case UDM_SQLTYPE_LONGVARCHAR  : return SQL_LONGVARCHAR;
    case UDM_SQLTYPE_VARCHAR      : return SQL_VARCHAR;
    default                       : break;
  }
  return SQL_UNKNOWN_TYPE;
}


static int
UdmSQLParamLen2ODBC(int udm_len)
{
  switch (udm_len)
  {
    case UDM_SQL_NULL_DATA    : return SQL_NULL_DATA;
    case UDM_SQL_DATA_AT_EXEC : return SQL_DATA_AT_EXEC;
    case UDM_SQL_NTS          : return SQL_NTS;
  }
  return udm_len;
}


typedef struct udm_odbc_bind_st
{
  SQLINTEGER size;
  void *data;
} UDM_ODBC_BINDBUF;

static UDM_ODBC_BINDBUF BindBuf[64];

static int UdmODBCBind (UDM_DB *db, int position,
                        void *data, int size, int type)
{
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
  int rc;
  BindBuf[position].size= UdmSQLParamLen2ODBC(size);
  BindBuf[position].data= data;
  
  type= UdmSQLType2ODBCType(type);
  db->errcode = 0;
  db->errstr[0] = 0;

  rc= SQLBindParameter(sdb->hstmt, position, SQL_PARAM_INPUT, SQL_C_DEFAULT,
                       type, 0 /* Column size */, 0 /* Decimal digits */,
                       /*&BindBuf[position]*/ data, size < 0 ? 0 : size,
                       &BindBuf[position].size);

#ifdef WIN32  
/* s_size= SQL_DATA_AT_EXEC; */
#endif

  if (!SQL_OK(rc))
  {
    db->errcode= rc;
    UdmODBCDisplayError(db, "UdmOBDCBind: ");
    return(UDM_ERROR);
  }
  return(UDM_OK);
}

static int UdmODBCExec (UDM_DB *db)
{
  UDM_ODBC_CONN *sdb= (UDM_ODBC_CONN*) db->specific;
  int rc, put_data= 0;
  UDM_ODBC_BINDBUF *Buf;

  db->errcode = 0;
  db->errstr[0] = 0;
  rc= SQLExecute(sdb->hstmt);

#ifdef WIN32
  put_data= 1;
#endif

  if (!put_data || rc != SQL_NEED_DATA)
    goto check_err;

  for (rc= SQLParamData(sdb->hstmt, (void*)&Buf);
       rc == SQL_NEED_DATA;
       rc= SQLParamData(sdb->hstmt, (void*)&Buf))
  {
    if (SQL_ERROR == (rc= SQLPutData(sdb->hstmt, Buf->data, Buf->size)))
      goto check_err;
  }

check_err:

  if (!SQL_OK(rc))
  {
    UdmODBCDisplayError(db, "UdmODBCExec: ");
    db->errcode= rc;
  }
  SQLFreeStmt(sdb->hstmt, SQL_DROP);
  return SQL_OK(rc) ? UDM_OK : UDM_ERROR;
}

UDM_SQLDB_HANDLER udm_sqldb_odbc_handler =
{
  NULL,
  UdmODBCQuery,
  UdmODBCClose,
  UdmODBCBegin,
  UdmODBCCommit,
  UdmODBCPrepare,
  UdmODBCBind,
  UdmODBCExec,
  UdmSQLFetchRowSimple,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmODBCQuery
};

#endif

/***********************************************************************/

#if HAVE_CTLIB

#ifndef MAX
#define MAX(X,Y)  (((X) > (Y)) ? (X) : (Y))
#endif

#ifndef MIN
#define MIN(X,Y)  (((X) < (Y)) ? (X) : (Y))
#endif

#define MAX_CHAR_BUF  1024*64

typedef struct
{
  CS_CONTEXT	*ctx;
  CS_CONNECTION	*conn;
} UDM_CTLIB;


typedef struct _ex_column_data
{
  CS_INT        datatype;
  CS_CHAR         *value;
  CS_INT        valuelen;
  CS_SMALLINT  indicator;
} EX_COLUMN_DATA;

static char sybsrvmsg[2048]="";

static CS_RETCODE CS_PUBLIC
ex_clientmsg_cb(CS_CONTEXT *context, 
                CS_CONNECTION *connection,
                CS_CLIENTMSG *errmsg)
{
  fprintf(stderr, "\nOpen Client Message:\n");
  fprintf(stderr, "Message number: LAYER = (%ld) ORIGIN = (%ld) ",(long)CS_LAYER(errmsg->msgnumber), (long)CS_ORIGIN(errmsg->msgnumber));
  fprintf(stderr, "SEVERITY = (%ld) NUMBER = (%ld)\n",(long)CS_SEVERITY(errmsg->msgnumber), (long)CS_NUMBER(errmsg->msgnumber));
  fprintf(stderr, "Message String: %s\n", errmsg->msgstring);
  if (errmsg->osstringlen > 0)
  {
    fprintf(stderr, "Operating System Error: %s\n",errmsg->osstring);
  }
  return CS_SUCCEED;
}

static CS_RETCODE CS_PUBLIC
ex_servermsg_cb(CS_CONTEXT *context,CS_CONNECTION *connection,CS_SERVERMSG *srvmsg)
{
#ifdef DEBUG_SQL
  char msg[1024];
  udm_snprintf(msg, sizeof(msg)-1, "Server message: Message number: %ld, Severity %ld, State: %ld, Line: %ld, Msg: '%s'",(long)srvmsg->msgnumber, (long)srvmsg->severity, (long)srvmsg->state, (long)srvmsg->line,srvmsg->text);
  fprintf(stderr, "%s\n",msg);
#endif  

  /*
  if (srvmsg->svrnlen > 0){fprintf(stderr, "Server '%s'\n", srvmsg->svrname);}
  if (srvmsg->proclen > 0){fprintf(stderr, " Procedure '%s'\n", srvmsg->proc);}
  */
  
  if (srvmsg->msgnumber != 3621) /* for SYBASE*/
  if(srvmsg->severity>1)
    udm_snprintf(sybsrvmsg, sizeof(sybsrvmsg)-1, "Message number: %ld, Severity %ld, State: %ld, Line: %ld, Msg: '%s'",(long)srvmsg->msgnumber, (long)srvmsg->severity, (long)srvmsg->state, (long)srvmsg->line,srvmsg->text);
  
  return CS_SUCCEED;
}

static int
ex_execute_cmd(UDM_DB *db,CS_CHAR *cmdbuf)
{
  CS_RETCODE      retcode;
  CS_INT          restype;
  CS_COMMAND      *cmd;
  CS_RETCODE      query_code=CS_SUCCEED;
  UDM_CTLIB       *ct= (UDM_CTLIB*) db->specific;

  if ((retcode = ct_cmd_alloc(ct->conn, &cmd)) != CS_SUCCEED)
  {
    udm_snprintf(db->errstr, sizeof(db->errstr)-1,"ct_cmd_alloc() failed: %s", sybsrvmsg);
    query_code=retcode;
    goto unlock_ret;
  }
  
  if ((retcode = ct_command(cmd, CS_LANG_CMD, cmdbuf,
                            CS_NULLTERM,CS_UNUSED)) != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_command() failed: %s", sybsrvmsg);
    ct_cmd_drop(cmd);
    query_code=retcode;
    goto unlock_ret;
  }
  
  if ((retcode = ct_send(cmd)) != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_send() failed: %s",sybsrvmsg);
    ct_cmd_drop(cmd);
    query_code=retcode;
    goto unlock_ret;
  }
  
  while ((retcode = ct_results(cmd, &restype)) == CS_SUCCEED)
  {
    switch((int)restype)
    {
      case CS_CMD_SUCCEED:
      case CS_CMD_DONE:
        break;

      case CS_CMD_FAIL:
        query_code = CS_FAIL;
        break;

      case CS_STATUS_RESULT:
        retcode = ct_cancel(NULL, cmd, CS_CANCEL_CURRENT);
        if (retcode != CS_SUCCEED)
        {
          udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_cancel() failed: %s",sybsrvmsg);
          query_code = CS_FAIL;
        }
        break;

      default:
        query_code = CS_FAIL;
        break;
    }
    
    if (query_code == CS_FAIL)
    {
      retcode = ct_cancel(NULL, cmd, CS_CANCEL_ALL);
      if (retcode != CS_SUCCEED)
        udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_cancel() failed: %s",sybsrvmsg);
      break;
    }
  }
  
  if (retcode == CS_END_RESULTS)
  {
    retcode = ct_cmd_drop(cmd);
    if (retcode != CS_SUCCEED)
      query_code = CS_FAIL;
  }
  else
  {
    (void)ct_cmd_drop(cmd);
    query_code = CS_FAIL;
  }
       
unlock_ret:
  return query_code;
}


static int UdmCTLIBInitDB(UDM_DB *db)
{
  CS_RETCODE  retcode;
  CS_INT      netio_type = CS_SYNC_IO;
  CS_INT      len;
  CS_CHAR     *cmdbuf;
  UDM_CTLIB   *ct;
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", NULL);
  
  db->specific= malloc(sizeof(UDM_CTLIB)); 
  bzero(db->specific, sizeof(UDM_CTLIB));
  ct= (UDM_CTLIB*) db->specific;
  
  retcode = cs_ctx_alloc(CS_VERSION_100, &ct->ctx);
  if (retcode != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"cs_ctx_alloc() failed: %s",sybsrvmsg);
    db->errcode=1;
    goto unlock_ex;
  }

  retcode = ct_init(ct->ctx, CS_VERSION_100);
  if (retcode != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ex_init: ct_init() failed: %s",sybsrvmsg);
    cs_ctx_drop(ct->ctx);
    ct->ctx = NULL;
    db->errcode=1;
    goto unlock_ex;
  }

#ifdef EX_API_DEBUG
  retcode = ct_debug(db->ctx, NULL, CS_SET_FLAG, CS_DBG_API_STATES,NULL, CS_UNUSED);
  if (retcode != CS_SUCCEED)
  {
    sprintf(db->errstr,"ex_init: ct_debug() failed");
  }
#endif

  if (retcode == CS_SUCCEED)
  {
    retcode = ct_callback(ct->ctx, NULL, CS_SET, CS_CLIENTMSG_CB,(CS_VOID *)ex_clientmsg_cb);
    if (retcode != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_callback(clientmsg) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode == CS_SUCCEED)
  {
    retcode = ct_callback(ct->ctx, NULL, CS_SET, CS_SERVERMSG_CB,(CS_VOID *)ex_servermsg_cb);
    if (retcode != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_callback(servermsg) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode == CS_SUCCEED)
  {
    retcode = ct_config(ct->ctx, CS_SET, CS_NETIO, &netio_type, CS_UNUSED, NULL);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ct_config(netio) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode != CS_SUCCEED)
  {
    ct_exit(ct->ctx, CS_FORCE_EXIT);
    cs_ctx_drop(ct->ctx);
    ct->ctx = NULL;
    db->errcode=1;
    goto unlock_ex;
  }

  if (retcode==CS_SUCCEED)
    retcode = ct_con_alloc(ct->ctx, &ct->conn);

  if (retcode != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_alloc failed: %s",sybsrvmsg);
    db->errcode=1;
    goto unlock_ex;
  }

  if (retcode == CS_SUCCEED && DBUser != NULL)
  {
    if ((retcode = ct_con_props(ct->conn, CS_SET, CS_USERNAME, DBUser, CS_NULLTERM, NULL)) != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(username) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode == CS_SUCCEED && DBPass != NULL)
  {
    if ((retcode = ct_con_props(ct->conn, CS_SET, CS_PASSWORD, DBPass, CS_NULLTERM, NULL)) != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(password) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode == CS_SUCCEED)
  {
    if ((retcode = ct_con_props(ct->conn, CS_SET, CS_APPNAME, "indexer", CS_NULLTERM, NULL)) != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(appname) failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode == CS_SUCCEED)
  {
    len = DBHost ? CS_NULLTERM : 0;
    retcode = ct_connect(ct->conn, DBHost, len);
    if (retcode != CS_SUCCEED)
    {
      udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_connect failed: %s",sybsrvmsg);
      db->errcode=1;
    }
  }

  if (retcode != CS_SUCCEED)
  {
    ct_con_drop(ct->conn);
    ct->conn = NULL;
    db->errcode=1;
    goto unlock_ex;
  }

  db->connected=1;
  cmdbuf = (CS_CHAR *) UdmMalloc(1024);
  sprintf(cmdbuf, "use %s\n", db->DBName);
  
  if ((retcode = ex_execute_cmd(db, cmdbuf)) != CS_SUCCEED)
  {
    udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ex_execute_cmd(use db) failed: %s",sybsrvmsg);
    db->errcode=1;
  }
  UDM_FREE(cmdbuf);
  
unlock_ex:
  return retcode;
}


static void UdmCTLIBClose(UDM_DB *db)
{
  CS_RETCODE  retcode=CS_SUCCEED;
  CS_INT      close_option;
  CS_INT      exit_option;
  CS_RETCODE  status=CS_SUCCEED; /* FIXME: previous retcode was here */
  UDM_CTLIB  *ct= (UDM_CTLIB*) db->specific;

  if (!ct)
    return;
  
  if(ct->conn)
  {
    close_option = (status != CS_SUCCEED) ? CS_FORCE_CLOSE : CS_UNUSED;
    retcode = ct_close(ct->conn, close_option);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_con_cleanup: ct_close() failed");
      db->errcode=1;
      goto unlock_ret;
    }
    retcode = ct_con_drop(ct->conn);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_con_cleanup: ct_con_drop() failed");
      db->errcode=1;
      goto unlock_ret;
    }
  }
  
  if(ct->ctx)
  {
    exit_option = (status != CS_SUCCEED) ? CS_FORCE_EXIT : CS_UNUSED;
    retcode = ct_exit(ct->ctx, exit_option);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_ctx_cleanup: ct_exit() failed");
      db->errcode=1;
      goto unlock_ret;
    }
    retcode = cs_ctx_drop(ct->ctx);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_ctx_cleanup: cs_ctx_drop() failed");
      db->errcode=1;
      goto unlock_ret;
    }
  }
unlock_ret:
  UDM_FREE(db->specific);
  return;
}

static CS_INT CS_PUBLIC
ex_display_dlen(CS_DATAFMT * column)
{
  CS_INT    len;

  switch ((int) column->datatype)
  {
    case CS_CHAR_TYPE:
    case CS_VARCHAR_TYPE:
    case CS_LONGCHAR_TYPE:
    case CS_TEXT_TYPE:
    case CS_IMAGE_TYPE:
      len = MIN(column->maxlength*2+2, MAX_CHAR_BUF);
      break;

    case CS_BINARY_TYPE:
    case CS_VARBINARY_TYPE:
      len = MIN((2 * column->maxlength) + 2, MAX_CHAR_BUF);
      break;

    case CS_BIT_TYPE:
    case CS_TINYINT_TYPE:
      len = 3;
      break;

    case CS_SMALLINT_TYPE:
      len = 6;
      break;

    case CS_INT_TYPE:
      len = 11;
      break;

    case CS_REAL_TYPE:
    case CS_FLOAT_TYPE:
      len = 20;
      break;

    case CS_MONEY_TYPE:
    case CS_MONEY4_TYPE:
      len = 24;
      break;

    case CS_DATETIME_TYPE:
    case CS_DATETIME4_TYPE:
      len = 30;
      break;

    case CS_NUMERIC_TYPE:
    case CS_DECIMAL_TYPE:
      len = (CS_MAX_PREC + 2);
      break;

    default:
      len = 12;
      break;
  }
  return MAX((CS_INT)(strlen(column->name) + 1), len);
}

static
int ex_fetch_data(UDM_DB *db, UDM_SQLRES *res, CS_COMMAND *cmd)
{
  CS_RETCODE    retcode;
  CS_INT      num_cols;
  CS_INT      i;
  CS_INT      j;
  CS_INT      row_count = 0;
  CS_INT      rows_read;
  CS_DATAFMT    *datafmt;
  EX_COLUMN_DATA    *coldata;

  retcode = ct_res_info(cmd, CS_NUMDATA, &num_cols, CS_UNUSED, NULL);
  if (retcode != CS_SUCCEED)
  {
    sprintf(db->errstr,"ex_fetch_data: ct_res_info() failed");
    return UDM_ERROR;
  }
  if (num_cols <= 0)
  {
    sprintf(db->errstr,"ex_fetch_data: ct_res_info() returned zero columns");
    return UDM_ERROR;
  }

  coldata = (EX_COLUMN_DATA *)UdmMalloc(num_cols * sizeof (EX_COLUMN_DATA));
  datafmt = (CS_DATAFMT *)UdmMalloc(num_cols * sizeof (CS_DATAFMT));

  for (i = 0; i < num_cols; i++)
  {
    retcode = ct_describe(cmd, (i + 1), &datafmt[i]);
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_fetch_data: ct_describe() failed");
      break;
    }

    /* Remember the original datatype */
    coldata[i].datatype= datafmt[i].datatype;

    datafmt[i].maxlength = ex_display_dlen(&datafmt[i]) + 1;
    datafmt[i].datatype = CS_CHAR_TYPE;
#ifdef CS_CURRENT_VERSION
    if (CS_CURRENT_VERSION >= 15000)
      datafmt[i].datatype= CS_LONGCHAR_TYPE;
#endif
    datafmt[i].format   = CS_FMT_NULLTERM;

    /* Allocate data */
    coldata[i].value = (CS_CHAR *)UdmMalloc((size_t)datafmt[i].maxlength);
    
    retcode = ct_bind(cmd, (i + 1), &datafmt[i],coldata[i].value, &coldata[i].valuelen,&coldata[i].indicator);
    
    if (retcode != CS_SUCCEED)
    {
      sprintf(db->errstr,"ex_fetch_data: ct_bind() failed");
      break;
    }
  }
  
  if (retcode != CS_SUCCEED)
  {
    for (j = 0; j < i; j++)
    {
      UDM_FREE(coldata[j].value);
    }
    UDM_FREE(coldata);
    UDM_FREE(datafmt);
    return UDM_ERROR;
  }
  
  res->nCols=num_cols;
  if (res->nCols){
    res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
    bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
    for (i=0 ; i < res->nCols; i++)
    {
      res->Fields[i].sqlname = (char*)UdmStrdup(datafmt[i].name);
      res->Fields[i].sqllen= 0;
      res->Fields[i].sqltype= 0;
    }
  }
  
  while (((retcode = ct_fetch(cmd, CS_UNUSED, CS_UNUSED, CS_UNUSED,&rows_read)) == CS_SUCCEED) || (retcode == CS_ROW_FAIL))
  {
    row_count = row_count + rows_read;
    if (retcode == CS_ROW_FAIL)
    {
      sprintf(db->errstr, "Error on row %d.", (int)row_count);
      break;
    }
    res->Items=(UDM_PSTR*)UdmRealloc(res->Items,(res->nRows+1)*res->nCols*sizeof(UDM_PSTR));
    
    for (i = 0; i < num_cols; i++)
    {
      size_t offs= res->nRows*res->nCols+i;
      
      if (coldata[i].datatype == CS_IMAGE_TYPE ||
          coldata[i].datatype == CS_BINARY_TYPE ||
          coldata[i].datatype == CS_VARBINARY_TYPE ||
          coldata[i].datatype == CS_LONGBINARY_TYPE )
      {
        /*
          Unfortunately bind to CS_IMAGE_TYPE,
          CS_BINARY_TYPE, CS_VARBINARY_TYPE,
          CS_LONGBINARY_TYPE either didn't
          work or returned zero-padded values
          with FreeTDS ctlib. So, let's bind 
          these types as CHAR and convert back
          into BINARY. This is slower, but at
          least works as expected.
          
          Note, if valuelen is a positive number, then
          the value includes trailing '\0'.
        */
        size_t valuelen= coldata[i].valuelen;
        valuelen= valuelen > 0 ? valuelen - 1 : valuelen;
        DecodeHexStr(coldata[i].value,&res->Items[offs],valuelen);
      }
      else
      {
        /* valuelen includes trailing 0 */
        if (!coldata[i].indicator)
        {
          res->Items[offs].val= (char*)UdmMalloc(coldata[i].valuelen);
          memcpy(res->Items[offs].val,coldata[i].value,coldata[i].valuelen);
          res->Items[offs].len= coldata[i].valuelen-1;
        }
        else
        {
          res->Items[offs].val= NULL;
          res->Items[offs].len= 0;
        }
      }
    } 
    res->nRows++;
  }

  for (i = 0; i < num_cols; i++)
  {
    UDM_FREE(coldata[i].value);
  }
  UDM_FREE(coldata);
  UDM_FREE(datafmt);

  switch ((int)retcode)
  {
    case CS_END_DATA:
      retcode = CS_SUCCEED;
      db->errcode=0;
      break;

    case CS_FAIL:
      sprintf(db->errstr,"ex_fetch_data: ct_fetch() failed");
      db->errcode=1;
      return UDM_ERROR;
      break;

    default:
      sprintf(db->errstr,"ex_fetch_data: ct_fetch() returned an expected retcode (%d)",(int)retcode);
      db->errcode=1;
      return UDM_ERROR;
      break;
  }
  return UDM_OK;
}


static
int UdmCTLIBQuery(UDM_DB *db, UDM_SQLRES *res, const char * query)
{
  CS_RETCODE  retcode;
  CS_COMMAND  *cmd;
  CS_INT      res_type;
  UDM_CTLIB   *ct;
  
  db->errcode=0;
  db->errstr[0]='\0';
  
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }

  if(!db->connected)
  {
    UdmCTLIBInitDB(db);
    if(db->errcode)
      goto unlock_ret;
  }
  
  ct= (UDM_CTLIB*) db->specific;
  
  if ((retcode = ct_cmd_alloc(ct->conn, &cmd)) != CS_SUCCEED)
  {
    sprintf(db->errstr,"ct_cmd_alloc() failed");
    db->errcode=1;
    goto unlock_ret;
  }
  
  retcode = ct_command(cmd, CS_LANG_CMD, query, CS_NULLTERM, CS_UNUSED);
  if (retcode != CS_SUCCEED)
  {
    sprintf(db->errstr,"ct_command() failed");
    db->errcode=1;
    goto unlock_ret;
  }
  
  if (ct_send(cmd) != CS_SUCCEED)
  {
    sprintf(db->errstr,"ct_send() failed");
    db->errcode=1;
    goto unlock_ret;
  }
  
  while ((retcode = ct_results(cmd, &res_type)) == CS_SUCCEED)
  {
    switch ((int)res_type)
    {
    case CS_CMD_SUCCEED:
    case CS_CMD_DONE:
         break;

    case CS_STATUS_RESULT:
      break;

    case CS_CMD_FAIL:
      /*printf("here: %s\n",sybsrvmsg);*/
      if(!strstr(sybsrvmsg,"UNIQUE KEY") && 
         !strstr(sybsrvmsg,"PRIMARY KEY") &&
         !strstr(sybsrvmsg,"unique index")){
        udm_snprintf(db->errstr,sizeof(db->errstr)-1,"%s",sybsrvmsg);
        db->errcode=1;
      }
      break;

    case CS_ROW_RESULT:
      if (!res)
      {
        udm_snprintf(db->errstr,sizeof(db->errstr)-1,"CTLibQuery: Query returned result but called without 'res'");
        db->errcode=1;
        goto unlock_ret;
      }
      if(UDM_OK != ex_fetch_data(db,res,cmd))
      {
        db->errcode=1;
        goto unlock_ret;
      }
      break;

    case CS_COMPUTE_RESULT:
      /* FIXME: add checking */
      break; 

    default:
      sprintf(db->errstr,"DoCompute: ct_results() returned unexpected result type (%d)",(int)res_type);
      db->errcode=1;
      goto unlock_ret;
    }
  }

  switch ((int)retcode)
  {
    case CS_END_RESULTS:
      break;

    case CS_FAIL:
      sprintf(db->errstr,"DoCompute: ct_results() failed");
      db->errcode=1;
      goto unlock_ret;

    default:
      sprintf(db->errstr,"DoCompute: ct_results() returned unexpected result code");
      db->errcode=1;
      goto unlock_ret;
  }

  if ((retcode = ct_cmd_drop(cmd)) != CS_SUCCEED)
  {
    sprintf(db->errstr,"DoCompute: ct_cmd_drop() failed");
    db->errcode=1;
    udb_free_result(res);
    goto unlock_ret;
  }

unlock_ret:

  return db->errcode ? UDM_ERROR : UDM_OK;
}


static int UdmCTLibBegin(UDM_DB *db)
{
  return UdmCTLIBQuery(db,NULL,"BEGIN TRANSACTION");
}


static int UdmCTLibCommit(UDM_DB *db)
{
  return UdmCTLIBQuery(db,NULL,"COMMIT");
}


UDM_SQLDB_HANDLER udm_sqldb_ctlib_handler =
{
  NULL,
  UdmCTLIBQuery,
  UdmCTLIBClose,
  UdmCTLibBegin,
  UdmCTLibCommit,
  NULL,
  NULL,
  NULL,
  UdmSQLFetchRowSimple,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmCTLIBQuery
};

#endif


/*********************************** SQLITE *********************/

#if (HAVE_SQLITE)

static int UdmSQLiteInitDB(UDM_DB *db)
{
  char edbname[1024];
  char dbname[1024], *e, *errmsg;
  strncpy(edbname,db->DBName,sizeof(dbname));
  dbname[sizeof(edbname)-1]='\0';
  
  UdmUnescapeCGIQuery(dbname,edbname);
  /* Remove possible trailing slash */
  e= dbname+strlen(dbname);
  if (e>dbname && e[-1]=='/')
    e[-1]='\0';
  
  if (!(db->specific= (void*) sqlite_open(dbname, 0, &errmsg)))
  {
    sprintf(db->errstr, "sqlite driver: %s", errmsg ? errmsg : "<NOERROR>");
    UDM_FREE(errmsg);
    db->errcode=1;
    return UDM_ERROR;
  }
  db->connected=1;
  sqlite_busy_timeout((struct sqlite*)(db->specific),30*1000);
  return UDM_OK;
}

static int xCallBack(void *pArg, int argc, char **argv, char **name)
{
  UDM_SQLRES *res= (UDM_SQLRES*) pArg;
  int i;

  if (res->nCols == 0)
  {
    res->nCols= argc;
    for (i=0 ; i<argc; i++)
    {
      res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
      bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
      for (i=0 ; i < res->nCols; i++)
      {
        res->Fields[i].sqlname = (char*)UdmStrdup(name[i]);
        res->Fields[i].sqllen= 0;
        res->Fields[i].sqltype= 0;
      }
    }
  }
  res->nRows++;
  res->Items=(UDM_PSTR*)UdmRealloc(res->Items,res->nRows*res->nCols*sizeof(UDM_PSTR));
  
  for (i = 0; i < argc; i++)
  {
    size_t offs= (res->nRows-1)*res->nCols+(size_t)i;
    size_t len= strlen(argv[i]?argv[i]:"");
    res->Items[offs].len= len;
    res->Items[offs].val= (char*)UdmMalloc(len+1);
    memcpy(res->Items[offs].val,argv[i]?argv[i]:"",len+1);
  } 
  return 0;
}


static int UdmSQLiteQuery(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  char *errmsg;
  int rc;
  
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }
  
#ifdef WIN32
  /** Remove heading "/" from DBName (in case of absolute path) */
  if(db->DBName[2] == ':')
  {
    char *tmp_dbname = UdmStrdup(db->DBName + 1);
    UDM_FREE(db->DBName);
    db->DBName = tmp_dbname;
  }
#endif /* #ifdef WIN32 */

  db->errcode=0;
  db->errstr[0]='\0';
  if(!db->connected && UDM_OK != UdmSQLiteInitDB(db))
    return UDM_ERROR;
  
  if ((rc= sqlite_exec((struct sqlite *)db->specific, q, xCallBack, (void*)res, &errmsg)))
  {
    sprintf(db->errstr,"sqlite driver: %s",errmsg ? errmsg : "<NOERROR>");
    sqlite_freemem(errmsg);
    if (!strstr(db->errstr,"unique"))
    {
      db->errcode=1;
      return UDM_ERROR;
    }
  }
  return UDM_OK;
}

static void UdmSQLiteClose(UDM_DB *db)
{
  sqlite_close((struct sqlite*)(db->specific));
  db->specific = NULL;
}

static int UdmSQLiteBegin(UDM_DB *db)
{
  return UdmSQLiteQuery(db,NULL,"BEGIN TRANSACTION");
}

static int UdmSQLiteCommit(UDM_DB *db)
{
  return UdmSQLiteQuery(db,NULL,"COMMIT");
}


UDM_SQLDB_HANDLER udm_sqldb_sqlite_handler =
{
  NULL,
  UdmSQLiteQuery,
  UdmSQLiteClose,
  UdmSQLiteBegin,
  UdmSQLiteCommit,
  NULL,
  NULL,
  NULL,
  UdmSQLFetchRowSimple,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmSQLiteQuery
};

#endif  /* HAVE_SQLITE */


#if (HAVE_SQLITE3)

#include <sqlite3.h>

static int UdmSQLite3InitDB(UDM_DB *db)
{
  sqlite3 *ppDb;
  char edbname[1024];
  char dbname[1024], *e;
  strncpy(edbname,db->DBName,sizeof(dbname));
  dbname[sizeof(edbname)-1]='\0';
  
  UdmUnescapeCGIQuery(dbname,edbname);
  /* Remove possible trailing slash */
  e= dbname+strlen(dbname);
  if (e>dbname && e[-1]=='/')
    e[-1]='\0';
  
  if (SQLITE_OK != sqlite3_open(dbname, &ppDb))
  {
    const char *errmsg= sqlite3_errmsg(ppDb);
    udm_snprintf(db->errstr, sizeof(db->errstr),
                "sqlite3 driver: %s", errmsg ? errmsg : "<NOERROR>");
    db->errcode= 1;
    sqlite3_close(ppDb);
    return UDM_ERROR;
  }
  db->specific= (void*) ppDb;
  db->connected= 1;
  sqlite3_busy_timeout((sqlite3*)(db->specific),30*1000);
  return UDM_OK;
}

#if 0
static int SQLite3CallBack(void *pArg, int argc, char **argv, char **name)
{
  UDM_SQLRES *res= (UDM_SQLRES*) pArg;
  int i;

  if (res->nCols == 0)
  {
    res->nCols= argc;
    for (i=0 ; i<argc; i++)
    {
      res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
      bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
      for (i=0 ; i < res->nCols; i++)
      {
        res->Fields[i].sqlname = (char*)UdmStrdup(name[i]);
        res->Fields[i].sqllen= 0;
        res->Fields[i].sqltype= 0;
      }
    }
  }

  res->nRows++;
  res->Items=(UDM_PSTR*)UdmRealloc(res->Items,res->nRows*res->nCols*sizeof(UDM_PSTR));
  
  for (i = 0; i < argc; i++)
  {
    size_t offs= (res->nRows-1)*res->nCols+(size_t)i;
    size_t len= strlen(argv[i]?argv[i]:"");
    res->Items[offs].len= len;
    res->Items[offs].val= (char*)UdmMalloc(len+1);
    memcpy(res->Items[offs].val,argv[i]?argv[i]:"",len+1);
  } 
  return 0;
}
#endif

static int UdmSQLite3Query(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  int rc;
  sqlite3_stmt *pStmt;
  const char *Tail;
  int columns= 0;
  
  if (res)
  {
    bzero((void*) res, sizeof(UDM_SQLRES));
    res->db= db;
  }
  
#ifdef WIN32
  /** Remove heading "/" from DBName (in case of absolute path) */
  if(db->DBName[2] == ':')
  {
    char *tmp_dbname= UdmStrdup(db->DBName + 1);
    UDM_FREE(db->DBName);
    db->DBName= tmp_dbname;
  }
#endif /* #ifdef WIN32 */

  db->errcode= 0;
  db->errstr[0]= '\0';

  if (!db->connected && UDM_OK != UdmSQLite3InitDB(db))
    return UDM_ERROR;

  if ((rc= sqlite3_prepare((sqlite3*) db->specific, q, -1, &pStmt, &Tail)))
  {
    udm_snprintf(db->errstr, sizeof(db->errstr),
                 "sqlite3 driver: (%d) %s", 
                 sqlite3_errcode((sqlite3*) db->specific),
                 sqlite3_errmsg((sqlite3*) db->specific));
    db->errcode= 1;
    return UDM_ERROR;
  }
  
  for (rc= sqlite3_step(pStmt) ; ; rc= sqlite3_step(pStmt))
  {
    int i;
    switch (rc)
    {
      case SQLITE_DONE:
        goto fin;

      case SQLITE_ROW:
        if (!columns)
        {
          columns= sqlite3_column_count(pStmt);
          res->nCols= columns;
          res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
          bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
          for (i=0 ; i < res->nCols; i++)
          {
            const char *name= sqlite3_column_name(pStmt, i);
            res->Fields[i].sqlname= (char*)UdmStrdup(name);
            res->Fields[i].sqllen= 0;
            res->Fields[i].sqltype= 0;
          }
        }
        res->nRows++;
        res->Items=(UDM_PSTR*)UdmRealloc(res->Items,res->nRows*res->nCols*sizeof(UDM_PSTR));

        for (i= 0 ; i < columns; i++)
        {
          const unsigned char *value;
          int type, bytes;
          size_t offs= (res->nRows-1)*res->nCols+(size_t)i;
          UDM_PSTR *Item= &res->Items[offs];
          type= sqlite3_column_type(pStmt, i);
          if (type == SQLITE_BLOB)
            value= sqlite3_column_blob(pStmt, i);
          else
            value= sqlite3_column_text(pStmt, i);
          bytes= sqlite3_column_bytes(pStmt, i);
          Item->len= bytes;
          Item->val= (char*)UdmMalloc(bytes + 1);
          if (bytes)
            memcpy(Item->val, value, bytes);
          res->Items[offs].val[bytes]= '\0';
        }
        break;

      case SQLITE_ERROR:
        sqlite3_finalize(pStmt);
        udm_snprintf(db->errstr, sizeof(db->errstr),
                     "sqlite3 driver: (%d) %s",
                     sqlite3_errcode((sqlite3*) db->specific),
                     sqlite3_errmsg((sqlite3*) db->specific));
        if (!strstr(db->errstr,"unique"))
        {
          db->errcode=1;
          return UDM_ERROR;
        }
        return UDM_OK;
        break;
    
      case SQLITE_MISUSE:
      case SQLITE_BUSY:
      default:
        udm_snprintf(db->errstr, sizeof(db->errstr),
                     "sqlite3_step() returned MISUSE or BUSY");
        db->errcode= 1;
        sqlite3_finalize(pStmt);
        return UDM_ERROR;
        break;
      }
    }

fin:
  if ((rc= sqlite3_finalize(pStmt)))
    return UDM_ERROR;

  return UDM_OK;
}

static void UdmSQLite3Close(UDM_DB *db)
{
  sqlite3_close((sqlite3*)(db->specific));
  db->specific = NULL;
}

static int UdmSQLite3Begin(UDM_DB *db)
{
  return UdmSQLite3Query(db,NULL,"BEGIN TRANSACTION");
}

static int UdmSQLite3Commit(UDM_DB *db)
{
  return UdmSQLite3Query(db,NULL,"COMMIT");
}


UDM_SQLDB_HANDLER udm_sqldb_sqlite3_handler =
{
  NULL,
  UdmSQLite3Query,
  UdmSQLite3Close,
  UdmSQLite3Begin,
  UdmSQLite3Commit,
  NULL,
  NULL,
  NULL,
  UdmSQLFetchRowSimple,
  UdmSQLStoreResultSimple,
  UdmSQLFreeResultSimple,
  UdmSQLite3Query
};

#endif  /* HAVE_SQLITE3 */




/***********************************************************************/

/*
 *   Wrappers for different databases
 *
 *   UdmDBEscStr();
 *   UdmSQLQuery();
 *   UdmSQLValue();
 *   UdmSQLLen();
 *   UdmSQLFree();
 *   UdmSQLClose();
 */  


char * UdmDBEscStr(int DBType,char *to,const char *from,size_t len)
{
  char *s=to;

  if (DBType == UDM_DB_ORACLE8 || DBType == UDM_DB_SYBASE ||
      DBType == UDM_DB_MSSQL || DBType == UDM_DB_DB2 ||
      DBType == UDM_DB_IBASE || DBType == UDM_DB_SAPDB ||
      DBType == UDM_DB_SQLITE || DBType == UDM_DB_ACCESS ||
      DBType == UDM_DB_MIMER || DBType == UDM_DB_CACHE ||
      DBType == UDM_DB_SQLITE3)
   {
     while(*from)
     {
       switch(*from)
       {
         case '\'':
          /* Note that no break here!*/
          *to=*from;to++;
         default:
          *to=*from;
       }
       to++;from++;
      }
  }
  else
  {
    while(*from)
    {
#ifdef WIN32
      /*
        A workaround against a bug in SQLExecDirect
        in PostgreSQL ODBC driver. It produces
        an error when meets a question mark in a string
        constant. For some reasons question marks are
        considered as parameters. This should not happen
        with SQLExecDirect.
        Let's escape question mark using \x3F notation.
      */
      if (DBType == UDM_DB_PGSQL &&
          (*from == '?' ||
           *from == '{' ||
           *from == '}'))
      {
        *to++= '\\';
        *to++= 'x';
        *to++= '3';
        *to++= 'F';
        *from++;
        continue;
      }
#endif
      switch(*from)
      {
        case '\'':
        case '\\':
         *to='\\';to++;
        default:*to=*from;
      }
      to++;from++;
    }
  }
  *to=0;return(s);
}

char *UdmSQLEscStr(UDM_DB *db, char *to, const char *from, size_t len)
{
  size_t multiply= db->DBType == UDM_DB_PGSQL ? 4 : 2;
  if(!from)return(NULL);
  if(!to)to=(char*)UdmMalloc(len * multiply + 1);
  if(!to)return(NULL);
  return db->sql->SQLEscStr ? db->sql->SQLEscStr(db, to, from, len) : UdmDBEscStr(db->DBType, to, from, len);
}

#ifdef HAVE_DEBUG
int
UdmSQLBeginDebug(UDM_DB *db, const char *file, const int line)
{
  if (db->flags & UDM_SQL_DEBUG_QUERY)
    fprintf(stderr, "[%d] SQLBegin {%s:%d}\n", getpid(), file, line);
  return db->sql->SQLBegin(db);
}

int
UdmSQLCommitDebug(UDM_DB *db, const char *file, const int line)
{
  if (db->flags & UDM_SQL_DEBUG_QUERY)
    fprintf(stderr, "[%d] SQLCommit {%s:%d}\n", getpid(), file, line);
  return db->sql->SQLCommit(db);
}
int
UdmSQLExecDirectDebug(UDM_DB *db, UDM_SQLRES *R, const char *query,
                      const char *file, const int line)
{
  if (db->flags & UDM_SQL_DEBUG_QUERY)
  {
    int rc;
    unsigned long long ticks= UdmStartTimer();
    rc= db->sql->SQLExecDirect(db, R, query);
    ticks= UdmStartTimer() - ticks;
    fprintf(stderr,"[%d] %.2fs SQL {%s:%d}: %s\n",
            getpid(), (float)ticks/1000, file, line, query);
    return rc;
  }
  else
    return db->sql->SQLExecDirect(db, R, query);
}

#else
int
UdmSQLBegin(UDM_DB *db)
{
  return db->sql->SQLBegin(db);
}

int
UdmSQLCommit(UDM_DB *db)
{
  return db->sql->SQLCommit(db);
}

int
UdmSQLExecDirect(UDM_DB *db, UDM_SQLRES *R, const char *q)
{
  return db->sql->SQLExecDirect(db, R, q);
}
#endif


int __UDMCALL
_UdmSQLQuery(UDM_DB *db, UDM_SQLRES *SQLRes, const char * query,
             const char *file, const int line)
{
  UDM_SQLRES res;
  
#ifdef HAVE_DEBUG
  unsigned long ticks;
  if (db->flags & UDM_SQL_DEBUG_QUERY)
  {
    ticks= UdmStartTimer();
  }
#endif
  
  /* FIXME: call UdmSQLFree at exit if SQLRes = NULL */
  if (! SQLRes) SQLRes = &res;
  
  db->sql->SQLQuery(db, SQLRes, query);

#ifdef HAVE_DEBUG
  if (db->flags & UDM_SQL_DEBUG_QUERY)
  {
    ticks= UdmStartTimer() - ticks;
    fprintf(stderr,"[%d] %.2fs SQL {%s:%d}: %s\n", getpid(), (float)ticks/1000, file, line, query);
  }
#endif
  
  if (db->errcode && (db->flags & UDM_SQL_IGNORE_ERROR))
    db->errcode= 0;
  
#ifdef DEBUG_ERR_QUERY
  if (db->errcode)
    fprintf(stderr, "{%s:%d} Query: %s\n\n", file, line, query);
#endif

  return db->errcode ? UDM_ERROR : UDM_OK;
}


size_t __UDMCALL UdmSQLNumRows(UDM_SQLRES * res)
{
  return(res?res->nRows:0);
}

size_t UdmSQLNumCols(UDM_SQLRES * res)
{
  return(res?res->nCols:0);
}

const char * __UDMCALL UdmSQLValue(UDM_SQLRES * res,size_t i,size_t j){

#if HAVE_PGSQL
  if(res->db->DBDriver==UDM_DB_PGSQL && !res->Items)
    return(PQgetvalue(res->pgsqlres,(int)(i),(int)(j)));
#endif
  
  if (i<res->nRows)
  {
    size_t offs=res->nCols*i+j;
    return res->Items[offs].val;
  }
  else
  {
    return NULL;
  }
}


int __UDMCALL UdmSQLFetchRowSimple (UDM_DB *db, UDM_SQLRES *res, UDM_PSTR *buf) {
  size_t j;
  size_t offs = res->nCols * res->curRow;

  if (res->curRow >= res->nRows)
    return UDM_ERROR;
  
  for (j = 0; j < res->nCols; j++)
  {
    buf[j]= res->Items[offs + j];
  }
  res->curRow++;
  
  return(UDM_OK);
}


int __UDMCALL UdmSQLStoreResultSimple(UDM_DB *db, UDM_SQLRES *res)
{
  return UDM_OK;
}

int __UDMCALL UdmSQLFreeResultSimple(UDM_DB *db, UDM_SQLRES *res)
{
  if (res->Fields)
  {
    size_t i;
    for(i=0;i<res->nCols;i++)
    {
      /*
      printf("%s(%d)\n",res->Fields[i].sqlname,res->Fields[i].sqllen);
      */
      UDM_FREE(res->Fields[i].sqlname);
    }
    UDM_FREE(res->Fields);
  }
  
#if HAVE_PGSQL
  if(res->db->DBDriver==UDM_DB_PGSQL)
  {
    PQclear(res->pgsqlres);
    /*
      Continue to udb_free_results() to
      free allocated memory if we have bytea datatype
    */
  }
#endif

  udb_free_result(res);

  return UDM_OK;
}


void __UDMCALL UdmSQLFree(UDM_SQLRES * res)
{
  res->db->sql->SQLFreeResult(res->db, res);
}


size_t UdmSQLLen(UDM_SQLRES * res,size_t i,size_t j)
{
  size_t offs=res->nCols*i+j;
#if HAVE_PGSQL
  if(res->db->DBDriver==UDM_DB_PGSQL && !res->Items)
    return PQgetlength(res->pgsqlres, i, j);
#endif
  return res->Items[offs].len;
}


void UdmSQLClose(UDM_DB *db)
{
  if(!db->connected)
    return;
  db->sql->SQLClose(db);
  db->connected=0;
  return;
}


void
UdmSQLResListInit(UDM_SQLRESLIST *List)
{
  bzero((void*)List, sizeof(*List));
}


int
UdmSQLResListAdd(UDM_SQLRESLIST *List, UDM_SQLRES *Res)
{
  size_t nbytes= (List->nitems + 1) * sizeof(UDM_SQLRES);
  if (!(List->Item= (UDM_SQLRES*) UdmRealloc(List->Item, nbytes)))
    return UDM_ERROR;
  List->Item[List->nitems]= Res[0];
  List->nitems++;
  return UDM_OK;
}


void
UdmSQLResListFree(UDM_SQLRESLIST *List)
{
  size_t i;
  for (i= 0; i < List->nitems; i++)
  {
    UdmSQLFree(&List->Item[i]);
  }
  UdmFree(List->Item);
  UdmSQLResListInit(List);
}


#include "udm_mutex.h"

/*

  UdmSQLMonitor()
  
  - Executes SQL queries arriving from some virtual input stream
  - Displays query results to some virtual output streams
  
  The incoming strings are checked against:
    
    - SQL comments, i.e. lines starting with "--".
      The comment lines are skipped.
    
    - A set of end-of-query markers, i.e. 
      * MSSQL   style: 'GO' keyword written in the beginning
      * mysql   style: ';'  character written in the end of the string
      * minisql style: '\g' characters written in the end of the string
      
      As soon as a end-of-query marker has found, current collected query
      is executed.
    
    - In other case, the next line is concatenated to the previously
      collected data thus helping to run multi-line queries.
  
  
  The particular ways of handling input and output streams are
  passed in the UDM_SQLMON_PARAM argument. This approach makes 
  it possible to use UdmSQLMonitor() for many purposes:
  
  - executing a bundle of queries written in a text file.
  - running a command-line SQL monitor ("mysql", "pgsql" or "sqlplus" alike),
    i.e. executing queries arriving from the keyboard.
    
  - running a GUI SQL monitor,
    i.e. executing querues arriving from the keyboard in some GUI application.
  
  
  UDM_SQLMON_PARAM has got this structure:
  
  typedef struct udm_sqlmon_param_st
  {
    int flags;
    FILE *infile;
    FILE *outfile;
    char *(*gets)(struct udm_sqlmon_param_st *prm, char *str, size_t size);
    int (*display)(struct udm_sqlmon_param_st *, UDM_SQLRES *sqlres);
    int (*prompt)(struct udm_sqlmon_param_st *, const char *msg);
  void* tag_ptr;
  } UDM_SQLMON_PARAM;
  
  
  gets():     a function to get one string from the incoming stream.
  display():  a function to display an SQL query results
  prompt():   a function to display some prompts:
                - an invitation to enter data, for example: "SQL>".
                - an error message
  
  flags:      sets various aspects of the monitor behaviour:
              the only one so far:
                - whether to display field names in the output

  infile:     if the caller wants to input SQL queries from some FILE stream,
              for example, from STDIN or from a textual file opened via fopen(),
              it can use this field for this purpose
              
  outfile:    if the caller wants to output results to some FILE stream,
              for example, to STDOUT or to a textial file opened via fopen(),
              it can use this field for this purpose

  context_ptr:    extra pointer parameter for various purposes.
  
  To start using UdmSQLMonitor() one have to set
    - gets()
    - diplay()
    - prompt()
  
  If diplay() or prompt() use some FILE streams, the caller should
  also set infile and outfile with appropriative stream pointers.
  
*/


char str[64*1024];

__C_LINK int __UDMCALL UdmSQLMonitor(UDM_AGENT *A, UDM_ENV *Env, UDM_SQLMON_PARAM *prm){
  char *snd=str;
  int  rc=UDM_OK;
     
  str[sizeof(str)-1]='\0';
  while(1)
  {
    int  res;
    int  exec=0;
    char *send0;
    size_t  rbytes= (sizeof(str)-1) - (snd - str);

    if (!prm->gets(prm,snd,rbytes))
      break;

    if(snd[0]=='#' || !strncmp(snd, "--", 2))
      continue;
    
    send0 = snd;
    snd = snd + strlen(snd);
    while(snd > send0 && strchr(" \r\n\t", snd[-1]))
      *--snd='\0';

    if(snd == send0)
      continue;
          
    if(snd[-1]==';')
    {
      exec=1;
      *--snd='\0';
    }else if(snd - 2 >= str && snd[-1]=='g' && snd[-2]=='\\')
    {
      exec=1;
      snd-=2;
      *snd='\0';
    }
    else if(snd-2 >= str && strchr("oO", snd[-1]) && strchr("gG",snd[-2]))
    {
      exec=1;
      snd-=2;
      *snd='\0';
    }else if((size_t)(snd - str + 1) >= sizeof(str))
    {
      exec=1;
    }
          
    if(exec)
    {
      prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "'");
      prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, str);
      prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "'");
      prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
               
      if (!strncasecmp(str,"connection",10))
      {
        size_t newnum= atoi(str+10);
        char msg[255];
        if (newnum < Env->dbl.nitems)
        {
          Env->dbl.currdbnum= newnum;
          sprintf(msg,"Connection changed to #%d",Env->dbl.currdbnum);
          prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, msg);
          prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
        }
        else
        {
          sprintf(msg,"Wrong connection number %d",newnum);
          prm->prompt(prm, UDM_SQLMON_MSG_ERROR, msg);
          prm->prompt(prm, UDM_SQLMON_MSG_ERROR, "\n");
        }
      }
      else if (!strcasecmp(str,"fields=off"))
      {
        prm->flags=0;
      }
      else if (!strcasecmp(str,"fields=on"))
      {
        prm->flags=1;
      }
      else if (!strncasecmp(str,"colflags",8))
      {
        int colnum= atoi(str+8);
        int colval= atoi(str+10);
        if (colnum>=0 && colnum<10)
          prm->colflags[colnum]= colval;
      }
      else
      {
        UDM_SQLRES sqlres;
        UDM_DB *db= &Env->dbl.db[Env->dbl.currdbnum];
        prm->nqueries++;
        bzero((void*)&sqlres,sizeof(sqlres));
        UDM_GETLOCK(A, UDM_LOCK_DB);
        res=UdmSQLQuery(db,&sqlres,str);
        UDM_RELEASELOCK(A, UDM_LOCK_DB);
        if (res!=UDM_OK)
        {
          prm->nbad++;
          rc=UDM_ERROR;
          prm->prompt(prm, UDM_SQLMON_MSG_ERROR, db->errstr);
          prm->prompt(prm, UDM_SQLMON_MSG_ERROR, "\n");
        }
        else
        {
          prm->ngood++;
          res=prm->display(prm,&sqlres);
        }
        UdmSQLFree(&sqlres);
      }
      snd=str;
      str[0]='\0';
    }
    else
    {
      if(send0!=snd)
      {
        *snd=' ';
        snd++;
        *snd='\0';
      }
    }
  }
  prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
  return rc;
}


#endif
