/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#ifdef HAVE_SQL

/*
#define DEBUG_SQL
*/

#define DEBUG_ERR_QUERY



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"

#include "udm_xmalloc.h"
#ifdef WIN32
#include <process.h>
#endif


#if (HAVE_ORACLE8||HAVE_ODBC||HAVE_CTLIB)
static void DecodeHexStr (char *src, UDM_PSTR *dst, size_t size) {
	char p1;
	char p2;
	size_t dst_size = size / 2 + 1;
	size_t i;
	size_t d = 0;

	dst->val = UdmMalloc(dst_size);

	for (i = 0; i < size - 1; i++) {
                if (src[i] >= '0' && src[i] <= '9') p1 = src[i] - '0';
                else if (src[i] >= 'A' && src[i] <= 'F') p1 = src[i] - 'A' + 10;
                else if (src[i] >= 'a' && src[i] <= 'f') p1 = src[i] - 'a' + 10;
		else break;

                if (src[i + 1] >= '0' && src[i + 1] <= '9') p2 = src[i + 1] - '0';
                else if (src[i + 1] >= 'A' && src[i + 1] <= 'F') p2 = src[i + 1] - 'A' + 10;
                else if (src[i + 1] >= 'a' && src[i + 1] <= 'f') p2 = src[i + 1] - 'a' + 10;
		else break;

		dst->val[d++] = (p1 << 4) | p2;
		i++;
	}

	dst->val[d] = 0;
	dst->len = d;
}
#endif

/***************************************************************/


static int udb_free_result(UDM_SQLRES *res){
	size_t i;
	size_t nitems;
	if(res){
		if(res->Items){
			nitems = res->nCols * res->nRows;
			for(i=0;i<nitems;i++)
				if(res->Items[i].val)
					UDM_FREE(res->Items[i].val);
			UDM_FREE(res->Items);
		}
	}
	return(0);
}




/************************ MYSQL **************************************/
#if   (HAVE_MYSQL)

static int UdmMySQLInit(UDM_DB *db)
{
  const char* DBSock= UdmVarListFindStr(&db->Vars,"socket",NULL);
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);
  
  mysql_init(&(db->mysql));
  if(!(mysql_real_connect(&(db->mysql), DBHost, DBUser, DBPass,
  			  db->DBName?db->DBName:"mnogosearch",
  			  (unsigned)DBPort, DBSock, 0)))
  {
    db->errcode=1;
    sprintf(db->errstr, "MySQL driver: #%d: %s",
    	    mysql_errno(&db->mysql), mysql_error(&db->mysql));
    return UDM_ERROR;
  }
  db->connected=1;
  return UDM_OK;
}

static int UdmMySQLQuery(UDM_DB *db,UDM_SQLRES *R,const char *query){
	size_t	i;
	
	db->errcode=0;
	
	if(!db->connected){
		int rc=UdmMySQLInit(db);
		if(rc!=UDM_OK)return rc;
	}
	
	for(i=0;i<2;i++){
		if((mysql_query(&db->mysql,query))){
			if((mysql_errno(&db->mysql)==CR_SERVER_LOST)||
			   (mysql_errno(&db->mysql)==CR_SERVER_GONE_ERROR)||
			   (mysql_errno(&db->mysql)==ER_SERVER_SHUTDOWN)){
				UDMSLEEP(5);
			}else{
				sprintf(db->errstr,"MySQL driver: #%d: %s",mysql_errno(&db->mysql),mysql_error(&db->mysql));
				if((mysql_errno(&(db->mysql))!=ER_DUP_ENTRY) &&
				   (mysql_errno(&(db->mysql))!=ER_DUP_KEY)){
					db->errcode=1;
					return UDM_ERROR;
				}
				db->errcode=0;
				return UDM_OK;
			}
		}else{
			MYSQL_RES	*mysqlres;
			
			if((mysqlres=mysql_use_result(&db->mysql))){
				MYSQL_FIELD	*field;
				MYSQL_ROW	mysqlrow;
				size_t		mitems=0;
				size_t		nfields;
				
				R->nCols=mysql_num_fields(mysqlres);
				R->nRows=0;
				R->Items=NULL;
				R->Fields=(UDM_SQLFIELD*)UdmMalloc(R->nCols*sizeof(UDM_SQLFIELD));
				bzero(R->Fields,R->nCols*sizeof(UDM_SQLFIELD));
				
				for(nfields=0; (field=mysql_fetch_field(mysqlres)); nfields++){
					R->Fields[nfields].sqlname = (char*)UdmStrdup(field->name);
					R->Fields[nfields].sqllen=field->length;
				}
				
				while((mysqlrow=mysql_fetch_row(mysqlres))){
					size_t		col;
					unsigned long	*lengths=mysql_fetch_lengths(mysqlres);
					
					for(col=0;col<R->nCols;col++){
						size_t offs=R->nRows*R->nCols+col;
						size_t len;
						
						if(offs>=mitems){
							mitems+=256;
							R->Items=(UDM_PSTR*)UdmRealloc(R->Items,mitems*sizeof(UDM_PSTR));
						}
						
						len=R->Items[offs].len=lengths[col];
						R->Items[offs].val=(char*)UdmMalloc(len+1);
						memcpy(R->Items[offs].val,mysqlrow[col],len);
						R->Items[offs].val[len]='\0';
					}
					R->nRows++;
				}
				mysql_free_result(mysqlres);
			}
			return UDM_OK;
		}
	}
	db->errcode=1;
	sprintf(db->errstr,"MySQL driver: #%d: %s",mysql_errno(&db->mysql),mysql_error(&db->mysql));
	return UDM_ERROR;
}
#endif


/*********************************** POSTGRESQL *********************/
#if (HAVE_PGSQL)

static int UdmPgSQLInitDB(UDM_DB *db)
{
  char port[8];
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", NULL);
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);
  
  sprintf(port,"%d",DBPort);
  db->pgsql = PQsetdbLogin(DBHost,DBPort?port:0, 0, 0, db->DBName, DBUser, DBPass);
  if (PQstatus(db->pgsql) == CONNECTION_BAD)
  {
    db->errcode=1;
    return UDM_ERROR;
  }
  db->connected=1;
  return UDM_OK;
}

static int UdmPgSQLQuery(UDM_DB *db, UDM_SQLRES *res, const char *q){
	size_t i;
	PGresult *PGres;
	
	db->errcode=0;
	if(!db->connected){
		UdmPgSQLInitDB(db);
		if(db->errcode)
			return UDM_ERROR;
	}
	
	if(!(PGres= PQexec(db->pgsql,q))){
		sprintf(db->errstr, "%s", PQerrorMessage(db->pgsql));
		db->errcode=1;
		return UDM_ERROR;
	}
	if(PQresultStatus(PGres)==PGRES_COMMAND_OK){
		/* Free non-SELECT query */
		PQclear(PGres);
		return UDM_OK;
	}
	
	if(PQresultStatus(PGres) != PGRES_TUPLES_OK){
		PQclear(PGres);
		sprintf(db->errstr, "%s", PQerrorMessage(db->pgsql));
		if(strstr(db->errstr,"duplicate") ||
		   strstr(db->errstr, "") ||
		   strcasestr(db->errstr,"Duplizierter")) {
			return UDM_OK;
		}else{
			db->errcode = 1;
			return UDM_ERROR;
		}
	}
	
	if (!res){
		/* 
		 Don't allow to call UdmPgSQLQuery
		 returning data with NULL res pointer
		*/
		sprintf(db->errstr, "UdmPgSQLQuery executed with res=NULL returned result %d, %s",PQresultStatus(PGres),PQerrorMessage(db->pgsql));
		db->errcode=1;
		return UDM_ERROR;
	}
	
	res->pgsqlres= PGres;
	res->nCols=(size_t)PQnfields(res->pgsqlres);
	res->nRows=(size_t)PQntuples(res->pgsqlres);
	res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
	for(i=0;i<res->nCols;i++){
		res->Fields[i].sqlname = (char*)UdmStrdup(PQfname(res->pgsqlres,(int)i));
	}
	return UDM_OK;
}

#endif




/************************** ODBC ***********************************/
#if (HAVE_ODBC)


#define SQL_OK(rc)	((rc==SQL_SUCCESS)||(rc==SQL_SUCCESS_WITH_INFO))

static int UdmODBCDisplayError(UDM_DB *db){
	UCHAR szSqlState[10];
	UCHAR szErrMsg[256];
	SDWORD naterr;
	SWORD length;
	RETCODE rc=SQL_SUCCESS;
	size_t len=0;
	
	db->errstr[0]=0;
	while (1)
	{ 	
		rc = SQLError(db->hEnv,db->hDbc,db->hstmt,szSqlState,&naterr,szErrMsg,sizeof(szErrMsg),&length);
		if ((SQL_SUCCESS != rc) && (SQL_SUCCESS_WITH_INFO != rc))
			break;
		len+= sprintf(db->errstr+len, "[SQLSTATE:%s]%s",szSqlState,szErrMsg);
	}
	return(0);
}
static int execDB(UDM_DB*db,UDM_SQLRES *result, const char *sqlstr){
	RETCODE rc;
	SWORD iResColumns;
	SDWORD iRowCount;
	int i,res_count;
	UCHAR szColName[32];
	SWORD pcbColName;
	SWORD pfSQLType;
	UDWORD pcbColDef;
	SWORD pibScale;
	SWORD pfNullable;
	SDWORD	pcbValue;
	static char	bindbuf[(int)(32L * 1024L - 16L)];

	/* -------- 
	p=sqlstr;
	while(*p){
		if(*p=='?')*p='.';
		p++;
	}
	*/
	
	if(!strcmp(sqlstr,"COMMIT")){
		rc=SQLTransact(db->hEnv,db->hDbc,SQL_COMMIT);
		if(!SQL_OK(rc))	{
			db->errcode=1;
			return UDM_ERROR;
		} else {
			db->errcode=0;
			return UDM_OK;
		}
	}

	rc=SQLAllocStmt(db->hDbc, &(db->hstmt));
	if (!SQL_OK(rc)){
		db->errcode=1;
		return UDM_ERROR;
	}
	rc=SQLExecDirect(db->hstmt,(SQLCHAR *)sqlstr, SQL_NTS);
	if (!SQL_OK(rc)){
		if(rc==SQL_NO_DATA) goto ND;
		db->errcode=1;
		return UDM_ERROR; 
	}
	rc=SQLNumResultCols(db->hstmt, &iResColumns);
	if(!SQL_OK(rc)){
		db->errcode=1;
		return UDM_ERROR;
	}
	if(!iResColumns) {
		rc=SQLRowCount(db->hstmt, &iRowCount);
		if (!SQL_OK(rc)){
			db->errcode=1;
			return UDM_ERROR;
		}
	}else{
		result->nRows = 0;
		result->nCols = iResColumns;
		result->Items = NULL;

		rc = SQL_NO_DATA_FOUND;
		for (res_count=0;(db->res_limit?(res_count<db->res_limit):1);res_count++){
			rc=SQLFetch(db->hstmt);
			if (!SQL_OK(rc)) {
				if (rc!=SQL_NO_DATA_FOUND){
					db->errcode=1;
				}
				break;
			}
			result->Items=(UDM_PSTR*)UdmXrealloc(result->Items,((result->nRows+1)*iResColumns*sizeof(UDM_PSTR)));
			for (i = 0; i < iResColumns; i++) {
				size_t offs=result->nRows*iResColumns+i;
				const char *p;
				SQLDescribeCol(db->hstmt, i+1, szColName, sizeof(szColName),
					&pcbColName, &pfSQLType, &pcbColDef,&pibScale, &pfNullable);
				
				SQLGetData(db->hstmt,i+1,SQL_CHAR,bindbuf,sizeof(bindbuf),&pcbValue);
				p= (pcbValue==SQL_NULL_DATA) ? "" : bindbuf;
				pcbValue= (pcbValue==SQL_NULL_DATA) ? 0 : pcbValue;
				/* if(p)UdmRTrim(p," "); */
				if (pfSQLType == SQL_BINARY ||
				    pfSQLType == SQL_VARBINARY ||
				    pfSQLType == SQL_LONGVARBINARY ||
				    pfSQLType == -98 /* DB2 SQL_BLOB */)
				{
					DecodeHexStr(bindbuf,&result->Items[offs],pcbValue);
				}
				else
				{
					result->Items[offs].val= (char*)malloc(pcbValue+1);
					memcpy(result->Items[offs].val, p, pcbValue);
					result->Items[offs].val[pcbValue]='\0';
					result->Items[offs].len= pcbValue;
				}
			}
			result->nRows++;
		}
	}
ND:

	SQLFreeStmt(db->hstmt, SQL_DROP);
	
	db->res_limit=0;
	db->errcode=0;
	return UDM_OK;
}

static int UdmODBCInitDB(UDM_DB *db)
{
  char DSN[512]="";
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);
  
#if (HAVE_SOLID)
  udm_snprintf(DSN, sizeof(DSN)-1, "tcp %s %d", DBHost, DBPort?DBPort:1313);
#elif (HAVE_SAPDB)
  udm_snprintf(DSN, sizeof(DSN)-1, "%s:%s", DBHost, db->DBName?db->DBName:"");
#else
  strncpy(DSN, db->DBName?db->DBName:"", sizeof(DSN)-1);
#endif
  
  db->errcode = SQLAllocEnv( &(db->hEnv) );
  if( SQL_SUCCESS != db->errcode )return -2;
  db->errcode = SQLAllocConnect( db->hEnv, &(db->hDbc) );
  if( SQL_SUCCESS != db->errcode )return -3;
  db->errcode = SQLSetConnectOption( db->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_ON);
  if( SQL_SUCCESS != db->errcode )return -4;
  db->errcode = SQLConnect( db->hDbc, DSN, SQL_NTS, DBUser, SQL_NTS, DBPass, SQL_NTS);
  if( !SQL_OK(db->errcode)) return(-5);
  else db->errcode=0;
  db->connected=1;
  return 0;
}

static void UdmODBCCloseDB(UDM_DB *db){
	if(db->connected){
		db->connected=0;
		db->errcode = SQLTransact( db->hEnv, db->hDbc, SQL_COMMIT);
		if( SQL_SUCCESS != db->errcode )return;
		db->errcode = SQLDisconnect( db->hDbc );
		if( SQL_SUCCESS != db->errcode )return;
		db->errcode = SQLFreeConnect( db->hDbc );
		if( SQL_SUCCESS != db->errcode )return;
		else	db->hDbc = SQL_NULL_HDBC;
		db->errcode = SQLFreeEnv( db->hEnv );
		if( SQL_SUCCESS != db->errcode )return;
		else	db->hEnv = SQL_NULL_HENV;
	}
}
static int UdmODBCQuery(UDM_DB *db,UDM_SQLRES *res, const char *qbuf){
	int rc;
	 
	if(!db->connected){
		UdmODBCInitDB(db);
		if(db->errcode){
			UdmODBCDisplayError(db);
			return(0);
		}else{
			db->connected=1;
		}
	}
	rc= execDB(db,res,qbuf);
	if((db->errcode)){
		UdmODBCDisplayError(db);
		if(strstr(db->errstr,"[SQLSTATE:23000]")){ /* MS Access */
			db->errcode=0;
			rc= UDM_OK;
		}else
		if(strstr(db->errstr,"[SQLSTATE:S1000]")){ /* Oracle 8i */
			db->errcode=0;
			rc= UDM_OK;
		}else
                if(strstr(db->errstr,"uplicat")){ /* PgSQL,MySQL*/
			db->errcode=0;
			rc= UDM_OK;
		}else
		if(strstr(db->errstr,"nique")){ /* Solid, Virtuoso */
			db->errcode=0;
			rc= UDM_OK;
		}
		else if(strstr(db->errstr,"UNIQUE")){ /* Mimer */
			db->errcode=0;
			rc= UDM_OK;
		}else{
			db->errcode=1;
		}
		SQLFreeStmt(db->hstmt, SQL_DROP);
	}
	return rc;
}

#endif

/*******************************************************/

#if HAVE_IBASE

#define SQL_VARCHAR(len) struct {short vary_length; char vary_string[(len)+1];}
typedef struct {
	short	len;
	char	str[1];
} UDM_IBASE_VARY;

static void UdmIBaseDisplayError(UDM_DB *db){
	char * s = db->errstr;
	ISC_STATUS * ibstatus=db->status;
	
	while(isc_interprete(s ,&ibstatus)){
		strcat(s," ");
		s=db->errstr+strlen(db->errstr);
	}
}

static int UdmIBaseInitDB(UDM_DB *db)
{
	char dpb_buffer[256], *dpb, *e;
	const char *p;
	int dpb_length, len;
	char connect_string[256];
	const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
	const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
	const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
	
	dpb = dpb_buffer;
	*dpb++ = isc_dpb_version1;

	if (DBUser != NULL && (len = strlen(DBUser))) {
		*dpb++ = isc_dpb_user_name;
		*dpb++ = len;
		for (p = DBUser; *p;) {
			*dpb++ = *p++;
		}
	}
	if (DBPass != NULL && (len = strlen(DBPass))) {
		*dpb++ = isc_dpb_password;
		*dpb++ = len;
		for (p = DBPass; *p;) {
			*dpb++ = *p++;
		}
	}
	/*
	if (charset != NULL && (len = strlen(charset))) {
		*dpb++ = isc_dpb_lc_ctype;
		*dpb++ = strlen(charset);
		for (p = charset; *p;) {
			*dpb++ = *p++;
		}
	}
#ifdef isc_dpb_sql_role_name
	if (role != NULL && (len = strlen(role))) {
		*dpb++ = isc_dpb_sql_role_name;
		*dpb++ = strlen(role);
		for (p = role; *p;) {
			*dpb++ = *p++;
		}
	}
#endif
	*/

	dpb_length = dpb - dpb_buffer;
	
	if(strcmp(DBHost,"localhost"))
		udm_snprintf(connect_string,sizeof(connect_string)-1,"%s:%s",DBHost, db->DBName);
	else
		udm_snprintf(connect_string,sizeof(connect_string)-1,"%s",db->DBName);
	
	/* Remove possible trailing slash */
	e= connect_string+strlen(connect_string);
	if (e>connect_string && e[-1]=='/')
		e[-1]='\0';
	
#ifdef DEBUG_SQL
	fprintf(stderr, "SQL Connect to: '%s'\n",connect_string);
#endif	
	if(isc_attach_database(db->status, strlen(connect_string), connect_string, &(db->DBH), dpb_length, dpb_buffer)){
		db->errcode=1;
		return(1);
	}
	return(0);
}
static void UdmIBaseCloseDB(UDM_DB*db){
	if(db->connected){
		if (isc_detach_database(db->status, &(db->DBH))){
			db->errcode=1;
		}
	}
}
static int sql_ibase_query(UDM_DB *db, UDM_SQLRES *res, const char *query){
	ISC_STATUS	status[20]; /* To not override db->status */
	isc_stmt_handle	query_handle = NULL;
	char		query_info[] = { isc_info_sql_stmt_type };
	char		info_buffer[18];
	long		query_type;
	int		autocommit=1;
	char		shortdata[64];
	short		sqlind_array[128];
	int		rc= UDM_OK;
	
	if(!db->connected){
		UdmIBaseInitDB(db);
		if(db->errcode){
			UdmIBaseDisplayError(db);
			db->errcode=1;
			return UDM_ERROR;
		}else{
			db->connected=1;
		}
	}
	
	if(!strcmp(query,"BEGIN")){
		if(!db->tr_handle){
			if (isc_start_transaction(db->status, &db->tr_handle, 1, &(db->DBH), 0, NULL)) {
				db->errcode=1;
				rc= UDM_ERROR;
			}
		}else{
			db->errcode=1;
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"Wrong calls order: begin");
			rc= UDM_ERROR;
		}
		return rc;
	}
	
	if(!strcmp(query,"COMMIT")){
		if(db->tr_handle){
			if (isc_commit_transaction(db->status, &db->tr_handle)){
				db->errcode=1;
				rc= UDM_ERROR;
			}
			db->tr_handle=NULL;
		}else{
			db->errcode=1;
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"Wrong calls order: commit");
			rc= UDM_ERROR;
		}
		return rc;
	}
	
	if(!db->tr_handle){
		if (isc_start_transaction(db->status, &db->tr_handle, 1, &(db->DBH), 0, NULL)){
			db->errcode=1;
			return UDM_ERROR;
		}
		autocommit=1;
	}else{
		autocommit=0;
	}
	
	if (isc_dsql_allocate_statement(db->status, &(db->DBH), &query_handle)){
		db->errcode=1; 
		return UDM_ERROR;
	}
	if (isc_dsql_prepare(db->status, &db->tr_handle, &query_handle, 0, query, 1, NULL)){
		db->errcode=1; 
		return UDM_ERROR;
	}
	if (!isc_dsql_sql_info(db->status, &query_handle, sizeof(query_info), query_info, sizeof(info_buffer), info_buffer)) {
		short l;
		l = (short) isc_vax_integer((char ISC_FAR *) info_buffer + 1, 2);
		query_type = isc_vax_integer((char ISC_FAR *) info_buffer + 3, l);
	}
	
	/* Find out what kind of query is to be executed */
	if (query_type == isc_info_sql_stmt_select || query_type == isc_info_sql_stmt_select_for_upd) {
		XSQLDA *osqlda=NULL;
		long fetch_stat;
		int i;
		
		/*
		 * Select, need to allocate output sqlda and and prepare it for use.
		 */
		osqlda = (XSQLDA *) UdmXmalloc(XSQLDA_LENGTH(0));
		osqlda->sqln = 0;
		osqlda->version = SQLDA_VERSION1;
		
		/* Fetch column information */
		if (isc_dsql_describe(db->status, &query_handle, 1, osqlda)) {
			UDM_FREE(osqlda);
			db->errcode=1;
			isc_rollback_transaction(status, &db->tr_handle);
			return UDM_ERROR;
		}
		
		if (osqlda->sqld) {
			osqlda = (XSQLDA *) UdmXrealloc(osqlda, XSQLDA_LENGTH(osqlda->sqld));
			osqlda->sqln = osqlda->sqld;
			osqlda->version = SQLDA_VERSION1;
			if (isc_dsql_describe(db->status, &query_handle, 1, osqlda)) {
				UDM_FREE(osqlda);
				db->errcode=1;
				isc_rollback_transaction(status, &db->tr_handle);
				return UDM_ERROR;
			}
		}
		
		if (!res){
			db->errcode=1;
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ibase_query with empty 'res' returned rows");
			return UDM_ERROR;
		}
		
		res->nCols = osqlda->sqld;
		res->Fields= (UDM_SQLFIELD*)UdmXmalloc(res->nCols*sizeof(UDM_SQLFIELD));
		
		for (i = 0; i < osqlda->sqld; i++) {
			XSQLVAR *var=&osqlda->sqlvar[i];
			int coltype = osqlda->sqlvar[i].sqltype & ~1;
			
			osqlda->sqlvar[i].sqlind=&sqlind_array[i];
			osqlda->sqlvar[i].sqlind[0]=0;
			res->Fields[i].sqlname = (char*)UdmStrdup(osqlda->sqlvar[i].sqlname);
			res->Fields[i].sqllen  = osqlda->sqlvar[i].sqllen;
			
			
			switch(coltype){
				case SQL_SHORT:
					var->sqldata = (char*)UdmMalloc(sizeof(short));
					break;
				case SQL_LONG:
					var->sqldata = (char*)UdmMalloc(sizeof(long));
					break;
				case SQL_FLOAT:
					var->sqldata = (char*)UdmMalloc(sizeof(float));
					break;
				case SQL_DOUBLE:
					var->sqldata = (char*)UdmMalloc(sizeof(double));
					break;
				case SQL_DATE:
				case SQL_BLOB:
				case SQL_ARRAY:
					var->sqldata = (char*)UdmMalloc(sizeof(ISC_QUAD));
					break;
				case SQL_TEXT:
					var->sqldata = (char*)UdmMalloc((size_t)(osqlda->sqlvar[i].sqllen));
					break;
				case SQL_VARYING:
					osqlda->sqlvar[i].sqldata = (char*)UdmMalloc((size_t)(osqlda->sqlvar[i].sqllen+sizeof(short)));
					break;
			}
		}
		if (isc_dsql_execute(db->status, &db->tr_handle, &query_handle, 1, NULL)) {
			UDM_FREE(osqlda);
			db->errcode=1;
			isc_rollback_transaction(status, &db->tr_handle);
			return UDM_ERROR;
		}
		
		while ((fetch_stat = isc_dsql_fetch(db->status, &query_handle, 1, osqlda)) == 0){
			res->Items=(UDM_PSTR *)UdmRealloc(res->Items,(res->nRows+1)*(res->nCols)*sizeof(UDM_PSTR));
			
			for(i=0;i<osqlda->sqld; i++){
				UDM_IBASE_VARY *vary;
				XSQLVAR *var=osqlda->sqlvar+i;
				char *p=NULL;
				size_t len;
				
				if(*var->sqlind==-1)
				{
					/* NULL data */
					p= (char*)UdmStrdup("");
					len= 0;
				}
				else
				switch(var->sqltype & ~1){
				case SQL_TEXT:
					p=(char*)UdmMalloc((size_t)(var->sqllen+1));
					strncpy(p,(char*)var->sqldata,(size_t)(var->sqllen));
					p[var->sqllen]='\0';
					len= var->sqllen;
					break;
				case SQL_VARYING:
					vary=(UDM_IBASE_VARY*)var->sqldata;
					p=(char*)UdmMalloc((size_t)(vary->len+1));
					strncpy(p,vary->str,(size_t)(vary->len));
					p[vary->len]='\0';
					len= vary->len;
					break;
				case SQL_LONG:
					len= sprintf(shortdata,"%ld",*(long*)(var->sqldata));
					p = (char*)UdmStrdup(shortdata);
					break;
				case SQL_SHORT:
					len= sprintf(shortdata,"%d",*(short*)(var->sqldata));
					p = (char*)UdmStrdup(shortdata);
					break;
				case SQL_FLOAT:
					len= sprintf(shortdata,"%f",*(float*)(var->sqldata));
					p = (char*)UdmStrdup(shortdata);
					break;
				case SQL_DOUBLE:
					len= sprintf(shortdata,"%f",*(double*)(var->sqldata));
					p = (char*)UdmStrdup(shortdata);
					break;
				default:
					len= sprintf(shortdata,"Unknown SQL type");
					p = (char*)UdmStrdup(shortdata);
					break; 
				}
				/*UdmRTrim(p," ");*/
				res->Items[res->nRows*res->nCols+i].val= p;
				res->Items[res->nRows*res->nCols+i].len= len;
			}
			res->nRows++;
		}
		/* Free fetch buffer */
		for (i = 0; i < osqlda->sqld; i++) {
			UDM_FREE(osqlda->sqlvar[i].sqldata);
		}
		UDM_FREE(osqlda);
		
		if (fetch_stat != 100L){
			db->errcode=1; 
			isc_rollback_transaction(status, &db->tr_handle);
			return UDM_ERROR;
		}
		
	} else {
		/* Not select */
		if (isc_dsql_execute(db->status, &db->tr_handle, &query_handle, 1, NULL)) {
			db->errcode=1;
			isc_rollback_transaction(status, &db->tr_handle);
			isc_dsql_free_statement(status, &query_handle, DSQL_drop);
			return UDM_ERROR;
		}
	}
	if (isc_dsql_free_statement(db->status, &query_handle, DSQL_drop)){
		db->errcode=1; 
		isc_rollback_transaction(status, &db->tr_handle);
		return UDM_ERROR;
	}
	if(autocommit){
		if (isc_commit_transaction(db->status, &db->tr_handle)){
			db->errcode=1; 
			db->tr_handle=NULL;
			return UDM_ERROR;
		}
		db->tr_handle=NULL;
	}
	return rc;
}

static int UdmIBaseQuery(UDM_DB *db, UDM_SQLRES *res, const char *query){
	int rc;
	db->errcode= 0;
	if(UDM_OK != (rc= sql_ibase_query(db,res,query))){
		UdmIBaseDisplayError(db);
		
		if(strstr(db->errstr,"uplicat") || strstr(db->errstr,"UNIQUE")){
			db->errcode=0;
			rc= UDM_OK;
		}else{
			/*strcat(db->errstr," ");
			strcat(db->errstr,query);*/
		}
		return rc;
	}
	return rc;
}

#endif


/******************** Oracle8 OCI - native support driver ******************/

#if HAVE_ORACLE8

/* (C) copyleft 2000 Anton Zemlyanov, az@hotmail.ru */
/* TODO: efficient transactions, multi-row fetch, limits stuff */




#define SQL_OK(rc)	((rc==OCI_SUCCESS)||(rc==OCI_SUCCESS_WITH_INFO))

static int oci_free_result(UDM_SQLRES *res)
{
	int i;
	if(res){
		if(res->defbuff){
			for(i=0;i<res->nCols;i++)
				if(res->defbuff[i])
					UDM_FREE(res->defbuff[i]);
		}
	}
	return(0);
}

static int UdmOracle8DisplayError(UDM_DB *db)
{
	sb4	errcode=0;
	text	errbuf[512];
	char	*ptr;

	db->errstr[0]='\0';

	switch (db->errcode)
	{
	case OCI_SUCCESS:
		sprintf(db->errstr,"Oracle - OCI_SUCCESS");
		break;
	case OCI_SUCCESS_WITH_INFO:
		sprintf(db->errstr,"Oracle - OCI_SUCCESS_WITH_INFO");
		break;
	case OCI_NEED_DATA:
		sprintf(db->errstr,"Oracle - OCI_NEED_DATA");
		break;
	case OCI_NO_DATA:
		sprintf(db->errstr,"Oracle - OCI_NODATA");
		break;
	case OCI_ERROR:
		OCIErrorGet((dvoid *)db->errhp, (ub4) 1, (text *) NULL, &errcode,
			errbuf, (ub4) sizeof(errbuf), OCI_HTYPE_ERROR);
		ptr=errbuf;
		while(*ptr) {
			if(*ptr=='\n')
				*ptr='!';
			++ptr;
		}
		sprintf(db->errstr,"Oracle - %.*s", 512, errbuf);
		break;
	case OCI_INVALID_HANDLE:
		sprintf(db->errstr,"Oracle - OCI_INVALID_HANDLE");
		break;
	case OCI_STILL_EXECUTING:
		sprintf(db->errstr,"Oracle - OCI_STILL_EXECUTE");
		break;
	case OCI_CONTINUE:
		sprintf(db->errstr,"Oracle - OCI_CONTINUE");
		break;
	default:
		sprintf(db->errstr,"Oracle - unknown internal bug");
		break;
	}
	return 0;
}


static void UdmOracle8CloseDB(UDM_DB *db)
{
	if(!db->connected)
		return;
	db->errcode=OCILogoff(db->svchp,db->errhp);
	db->connected=0;
}

static int oci_bind(UDM_DB *db){
	int i,pos;
	sword   rc;

        for (i=0; i<MAX_BIND_PARAM; i++)
                db->bndhp[i] = (OCIBind *) 0;

        for( i=0; i<MAX_BIND_PARAM; i++){
                if ((pos=db->par->out_pos[i])>0){
                        if ( (rc = OCIBindByPos(db->stmthp, &db->bndhp[pos-1], db->errhp, (ub4) pos,
				(dvoid *) &db->par->out_pos_val[pos-1][0],
                                (sb4) sizeof(db->par->out_pos_val[pos-1][0]), SQLT_INT,
				(dvoid *) 0, (ub2 *)0, (ub2 *)0,
                                (ub4) 0, (ub4 *) 0, (ub4) OCI_DEFAULT))){

                                return rc;
                        }
                        /*  bind array  */
                        if ((rc = OCIBindArrayOfStruct(db->bndhp[pos-1], db->errhp,                                
				sizeof(db->par->out_pos_val[pos-1][0]),0, 0, 0))){

                                return rc;
                        }
                }
        }
        return 0;
}

static void param_free(UDM_DB *db){
        db->par->out_rec = 0;
}


static int UdmOracle8InitDB(UDM_DB *db)
{
  const char* DBUser= UdmVarListFindStr(&db->Vars, "DBUser", "");
  const char* DBPass= UdmVarListFindStr(&db->Vars, "DBPass", "");

  OCIInitialize( OCI_DEFAULT, NULL, NULL, NULL, NULL );
  OCIEnvInit( &(db->envhp), OCI_DEFAULT, 0, NULL );
  OCIHandleAlloc( db->envhp, (dvoid **)&(db->errhp), OCI_HTYPE_ERROR,0, NULL );
  db->errcode=OCILogon(db->envhp, db->errhp, &(db->svchp),
		       DBUser, strlen(DBUser),
		       DBPass, strlen(DBPass),
		       db->DBName, strlen(db->DBName));
  if(db->errcode!=OCI_SUCCESS)
    return -1;

  db->errcode=0;
  db->connected=1;
  db->commit_fl=0;
  param_free(db);

/*sql_query(((UDM_DB*)(db)), "ALTER SESSION SET SQL_TRACE=TRUE"); */

  return 0;
}

static int sql_oracle_query(UDM_DB *db, UDM_SQLRES *result, char *qbuf)
{
	sword	rc;
	ub2	stmt_type;
	int	cnt, buf_nRows, row;

	int	colcnt;
	ub2	coltype;
	ub2	colsize;
	int	oci_fl;
	sb4	errcode=0;
	text	errbuf[512];
	int	num_rec;

	db->errcode=0;
	rc=OCIHandleAlloc(db->envhp, (dvoid *)&(db->stmthp), OCI_HTYPE_STMT,0,NULL);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return(UDM_ERROR);
	}
	rc=OCIStmtPrepare(db->stmthp,db->errhp,qbuf,strlen(qbuf),
			OCI_NTV_SYNTAX,OCI_DEFAULT);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return(UDM_ERROR);
	}

        if (db->par->out_rec){
                oci_bind(db);
                num_rec = db->par->out_rec;
                param_free(db);
        }else{
                num_rec = 1;
        }
    
	rc=OCIAttrGet(db->stmthp,OCI_HTYPE_STMT,&stmt_type,0,
			OCI_ATTR_STMT_TYPE,db->errhp);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return(UDM_ERROR);
	}

	if(stmt_type!=OCI_STMT_SELECT) {
		/* non-select statements */
		/* COMMIT_ON_SUCCESS in inefficient */
		if (db->commit_fl)
			oci_fl=OCI_DEFAULT;
		else
			oci_fl=OCI_COMMIT_ON_SUCCESS;

		rc=OCIStmtExecute(db->svchp,db->stmthp,db->errhp, num_rec,0,
				NULL,NULL,oci_fl);

		if (num_rec>1)
			  (void) OCITransCommit(db->svchp, db->errhp, (ub4) 0);

		if(!SQL_OK(rc)){
			db->errcode=rc;
			OCIErrorGet((dvoid *)db->errhp, (ub4) 1, (text *) NULL, &errcode,
				errbuf, (ub4) sizeof(errbuf), OCI_HTYPE_ERROR);
			if(strncmp(errbuf,"ORA-00001",9)) /* ignore ORA-00001 */
				return(UDM_ERROR);
			else 
				db->errcode=OCI_SUCCESS;
		}
		/* OCI_ATTR_ROW_COUNT of db->stmthp - Rows affected */
		rc=OCIHandleFree(db->stmthp,OCI_HTYPE_STMT);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return(UDM_ERROR);
		}
		return(UDM_OK);
	}

	/* select statements */

	rc=OCIStmtExecute(db->svchp,db->stmthp,db->errhp,0,0,
			NULL,NULL,OCI_DEFAULT);
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return(UDM_ERROR);
	}

	/*describe the select list and define buffers for the select list */
	colcnt=0;
	while(OCIParamGet(db->stmthp,OCI_HTYPE_STMT,db->errhp,(dvoid *)&(db->param),
			colcnt+1 ) == OCI_SUCCESS) {
		
		ub4 str_len;
		text *namep;
		
		result->Fields=(UDM_SQLFIELD*)UdmRealloc(result->Fields,(colcnt+1)*sizeof(UDM_SQLFIELD));
		bzero((void*)&result->Fields[colcnt],sizeof(UDM_SQLFIELD));
						
		rc=OCIAttrGet(db->param,OCI_DTYPE_PARAM,(dvoid*) &namep, (ub4*) &str_len,
				(ub4) OCI_ATTR_NAME, db->errhp);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return(UDM_ERROR);
		}
		namep[str_len]= '\0';
		result->Fields[colcnt].sqlname = (char*)UdmStrdup(namep);
		
		rc=OCIAttrGet(db->param,OCI_DTYPE_PARAM,&(coltype),0,
				OCI_ATTR_DATA_TYPE,db->errhp);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return(UDM_ERROR);
		}
		rc=OCIAttrGet(db->param,OCI_DTYPE_PARAM,&colsize,0,
			OCI_ATTR_DATA_SIZE,db->errhp);
		if(!SQL_OK(rc)){
			db->errcode=rc;
			return(UDM_ERROR);
		}

		/* OCIStmtFetch do not terminate data with \0 -
		add a byte for terminator to define buffers - insurance*/

		result->col_type[colcnt] = coltype;
		switch(coltype) {
		case SQLT_CHR: /* variable length string */
		case SQLT_AFC: /* fixed length string */
		case SQLT_BLOB:
		case SQLT_CLOB:
		case SQLT_BIN:
                        result->col_size[colcnt]=colsize;
                        result->defbuff[colcnt]=(char *)UdmXmalloc(result->col_size[colcnt]*BUF_OUT_SIZE+1);
			break;
		case SQLT_NUM: /* numbers up to 14 digits now */
                        result->col_size[colcnt]=14;
                        result->defbuff[colcnt]=(char *)UdmXmalloc(result->col_size[colcnt]*BUF_OUT_SIZE+1);
			break;
		default:
			printf("<P>Unknown datatype: %d\n",coltype);
			return(UDM_ERROR);
		}
		rc=OCIDefineByPos(db->stmthp,&(db->defb[colcnt]),db->errhp,
			colcnt+1,result->defbuff[colcnt],result->col_size[colcnt],SQLT_CHR,
			&(result->indbuff[colcnt]),0,0,OCI_DEFAULT);
		if(!SQL_OK(rc)){
			db->errcode=rc;
	 		return(UDM_ERROR);
		}
                rc=OCIDefineArrayOfStruct(db->defb[colcnt], db->errhp,
                         result->col_size[colcnt], sizeof(result->indbuff[0][0]), 0, 0);
		if(!SQL_OK(rc)){
                        db->errcode=rc;
                        return(UDM_ERROR);
                }
		colcnt++;
	}
	result->nCols=colcnt;

	/* Now fetching the selected rows into the memory */
	while(1) {
		if (db->res_limit)
			if (db->res_limit == result->nRows)
				break;
                /*Fix me: Process of determine real fetched rows
                 Fill indicator buffer for value -6 to */
                for(row=0; row<BUF_OUT_SIZE; row++)
                        result->indbuff[result->nCols-1][row]=-6;

		rc=OCIStmtFetch(db->stmthp,db->errhp,BUF_OUT_SIZE,
			OCI_FETCH_NEXT,OCI_DEFAULT);

                if((rc!=OCI_NO_DATA) && !SQL_OK(rc)) {
                        db->errcode=rc;
                        return(UDM_ERROR);
                }

                /* Find number of fetched rows */
                for(buf_nRows=0; buf_nRows<BUF_OUT_SIZE; buf_nRows++){
                        if(result->indbuff[result->nCols-1][buf_nRows]==-6)
                                break;
                }

                if(!buf_nRows)
                        break;

		if(!result->nRows)  /* first row - allocate */
			result->Items=(UDM_PSTR *)UdmXmalloc(buf_nRows*result->nCols*sizeof(UDM_PSTR));
		else
			result->Items=(UDM_PSTR *)UdmXrealloc(result->Items,
				((result->nRows+buf_nRows)*result->nCols*sizeof(UDM_PSTR)));
		/* Limit rows */	
	        if (db->res_limit)
            		buf_nRows=result->nRows+buf_nRows<=db->res_limit?buf_nRows:(db->res_limit-result->nRows);
		for(row=0; row<buf_nRows; row++){
			for(cnt=0; cnt<result->nCols; ++cnt) {
                    		if(result->indbuff[cnt][row]==OCI_IND_NULL) {
					result->Items[(result->nRows + row) * result->nCols + cnt].val = (char *)UdmStrdup("");
					result->Items[(result->nRows + row) * result->nCols + cnt].len = 0;
                                }else{
                                        char *val=UdmXmalloc(result->col_size[cnt]+1);
                                        int offset=row*result->col_size[cnt];

					if (result->col_type[cnt] == SQLT_BIN || result->col_type[cnt] == SQLT_CLOB) {
						DecodeHexStr(result->defbuff[cnt] + offset, &result->Items[(result->nRows + row) * result->nCols + cnt], result->col_size[cnt]);
					} else {
	                                        udm_snprintf(val, result->col_size[cnt]+1, "%s",
							result->defbuff[cnt]+offset);
						result->Items[(result->nRows + row) * result->nCols + cnt].val = (char *)UdmStrdup(UdmRTrim(val, " "));
						result->Items[(result->nRows + row) * result->nCols + cnt].len = strlen(result->Items[(result->nRows + row) * result->nCols + cnt].val);
					}
                                        UDM_FREE(val);
                                }
                        }
                }
		result->nRows+=row;
		if(rc==OCI_NO_DATA)
                        break;
                if(!SQL_OK(rc)) {
                        db->errcode=rc;
                        return(UDM_ERROR);
                }

	}
	rc=OCIHandleFree(db->stmthp,OCI_HTYPE_STMT);
	db->res_limit = 0;
	if(!SQL_OK(rc)){
		db->errcode=rc;
		return(UDM_ERROR);
	}
	return(UDM_OK);
}

static int UdmOracle8Query(UDM_DB *db, UDM_SQLRES *res, const char *qbuf) {
	int rc;

	if(!db->connected) {
		UdmOracle8InitDB(db);
		if(db->errcode) {
			UdmOracle8DisplayError(db);
			return(UDM_ERROR);
		} else {
			db->connected=1;
		}
 	}

	rc = sql_oracle_query(db, res, qbuf);

	if(db->errcode) {
		UdmOracle8DisplayError(db);
		return(UDM_ERROR);
	}
	return(rc);
}

int UdmOracle8Prepare (UDM_DB *db, const char *query) {
	int _;

	_ = OCIHandleAlloc(db->envhp, (dvoid *)&(db->stmthp), OCI_HTYPE_STMT, 0, NULL);
	if (! SQL_OK(_)) {
		db->errcode = _;
		return(UDM_ERROR);
	}

        _ = OCIStmtPrepare(db->stmthp, db->errhp, (const unsigned char *)query, strlen(query), OCI_NTV_SYNTAX, OCI_DEFAULT);
	if (! SQL_OK(_)) {
		db->errcode = _;
		return(UDM_ERROR);
	}
	return(UDM_OK);
}

/*
UdmOracle8Define () {
}
*/

int UdmOracle8Bind (UDM_DB *db, int position, void *data, size_t size, int type) {
	int _;

	db->bndhp[position] = NULL;
	_ = OCIBindByPos(db->stmthp, &db->bndhp[position], db->errhp,
	(ub4)position,
	(dvoid *)data,
	(sb4)size,
	(ub2)type,
	(dvoid *)0, (ub2 *)0, (ub2 *)0, (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT);

	if (! SQL_OK(_)) {
		db->errcode = _;
		return(UDM_ERROR);
	}
	return(UDM_OK);
}

int UdmOracle8Exec (UDM_DB *db) {
	int _;
	size_t num_rec = 1;

	_ = OCIStmtExecute(db->svchp, db->stmthp, db->errhp, num_rec, 0, NULL, NULL, OCI_DEFAULT);
	if (! SQL_OK(_)) {
		db->errcode = _;
		return(UDM_ERROR);
	}

	_ = OCIHandleFree(db->stmthp, OCI_HTYPE_STMT);
	if (! SQL_OK(_)) {
		db->errcode = _;
		return(UDM_ERROR);
	}
	return(UDM_OK);
}

#endif






/***********************************************************************/

#if HAVE_CTLIB

#ifndef MAX
#define MAX(X,Y)	(((X) > (Y)) ? (X) : (Y))
#endif

#ifndef MIN
#define MIN(X,Y)	(((X) < (Y)) ? (X) : (Y))
#endif

#define MAX_CHAR_BUF	1024*64

typedef struct _ex_column_data
{
	CS_INT		datatype;
	CS_CHAR		*value;
	CS_INT		valuelen;
	CS_SMALLINT	indicator;
} EX_COLUMN_DATA;

static char sybsrvmsg[2048]="";

static CS_RETCODE CS_PUBLIC
ex_clientmsg_cb(CS_CONTEXT *context,CS_CONNECTION *connection,CS_CLIENTMSG *errmsg)
{
	fprintf(stderr, "\nOpen Client Message:\n");
	fprintf(stderr, "Message number: LAYER = (%ld) ORIGIN = (%ld) ",(long)CS_LAYER(errmsg->msgnumber), (long)CS_ORIGIN(errmsg->msgnumber));
	fprintf(stderr, "SEVERITY = (%ld) NUMBER = (%ld)\n",(long)CS_SEVERITY(errmsg->msgnumber), (long)CS_NUMBER(errmsg->msgnumber));
	fprintf(stderr, "Message String: %s\n", errmsg->msgstring);
	if (errmsg->osstringlen > 0)
	{
		fprintf(stderr, "Operating System Error: %s\n",errmsg->osstring);
	}
	return CS_SUCCEED;
}

static CS_RETCODE CS_PUBLIC
ex_servermsg_cb(CS_CONTEXT *context,CS_CONNECTION *connection,CS_SERVERMSG *srvmsg)
{
#ifdef DEBUG_SQL
	char msg[1024];
	udm_snprintf(msg, sizeof(msg)-1, "Server message: Message number: %ld, Severity %ld, State: %ld, Line: %ld, Msg: '%s'",(long)srvmsg->msgnumber, (long)srvmsg->severity, (long)srvmsg->state, (long)srvmsg->line,srvmsg->text);
	fprintf(stderr, "%s\n",msg);
#endif	

	/*
	if (srvmsg->svrnlen > 0){fprintf(stderr, "Server '%s'\n", srvmsg->svrname);}
	if (srvmsg->proclen > 0){fprintf(stderr, " Procedure '%s'\n", srvmsg->proc);}
	*/

	/*if (srvmsg->msgnumber != 3621)*/ /* for SYBASE*/
	if(srvmsg->severity>1){
		udm_snprintf(sybsrvmsg, sizeof(sybsrvmsg)-1, "Message number: %ld, Severity %ld, State: %ld, Line: %ld, Msg: '%s'",(long)srvmsg->msgnumber, (long)srvmsg->severity, (long)srvmsg->state, (long)srvmsg->line,srvmsg->text);
	}
	
	return CS_SUCCEED;
}

static int
ex_execute_cmd(UDM_DB *db,CS_CHAR *cmdbuf)
{
	CS_RETCODE      retcode;
	CS_INT          restype;
	CS_COMMAND      *cmd;
	CS_RETCODE      query_code=CS_SUCCEED;

	if ((retcode = ct_cmd_alloc(db->conn, &cmd)) != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_cmd_alloc() failed: %s",sybsrvmsg);
		query_code=retcode;
		goto unlock_ret;
	}
	if ((retcode = ct_command(cmd, CS_LANG_CMD, cmdbuf, CS_NULLTERM,CS_UNUSED)) != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_command() failed: %s",sybsrvmsg);
		ct_cmd_drop(cmd);
		query_code=retcode;
		goto unlock_ret;
	}
	if ((retcode = ct_send(cmd)) != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_send() failed: %s",sybsrvmsg);
		ct_cmd_drop(cmd);
		query_code=retcode;
		goto unlock_ret;
	}
	while ((retcode = ct_results(cmd, &restype)) == CS_SUCCEED)
	{
		switch((int)restype)
                {
                    case CS_CMD_SUCCEED:
                    case CS_CMD_DONE:
                        break;

                    case CS_CMD_FAIL:
                        query_code = CS_FAIL;
                        break;

                    case CS_STATUS_RESULT:
                        retcode = ct_cancel(NULL, cmd, CS_CANCEL_CURRENT);
                        if (retcode != CS_SUCCEED)
                        {
                                udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_cancel() failed: %s",sybsrvmsg);
                                query_code = CS_FAIL;
                        }
                        break;

                    default:
                        query_code = CS_FAIL;
                        break;
                }
                if (query_code == CS_FAIL)
                {
                        retcode = ct_cancel(NULL, cmd, CS_CANCEL_ALL);
                        if (retcode != CS_SUCCEED)
                        {
                                udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_cancel() failed: %s",sybsrvmsg);
                        }
                        break;
                }
        }
        if (retcode == CS_END_RESULTS)
        {
                retcode = ct_cmd_drop(cmd);
                if (retcode != CS_SUCCEED)
                {
                        query_code = CS_FAIL;
                }
        }else{
                (void)ct_cmd_drop(cmd);
                query_code = CS_FAIL;
        }
       
unlock_ret:
        return query_code;
}


static int UdmCTLIBInitDB(UDM_DB *db)
{
	CS_RETCODE	retcode;
	CS_INT		netio_type = CS_SYNC_IO;
	CS_INT		len;
	CS_CHAR         *cmdbuf;
	const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
	const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
	const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", NULL);
	
	retcode = cs_ctx_alloc(CS_VERSION_100, &db->ctx);
	if (retcode != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"cs_ctx_alloc() failed: %s",sybsrvmsg);
		db->errcode=1;
		goto unlock_ex;
	}
	retcode = ct_init(db->ctx, CS_VERSION_100);
	if (retcode != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ex_init: ct_init() failed: %s",sybsrvmsg);
		cs_ctx_drop(db->ctx);
		db->ctx = NULL;
		db->errcode=1;
		goto unlock_ex;
	}

#ifdef EX_API_DEBUG
	retcode = ct_debug(db->ctx, NULL, CS_SET_FLAG, CS_DBG_API_STATES,NULL, CS_UNUSED);
	if (retcode != CS_SUCCEED)
	{
		sprintf(db->errstr,"ex_init: ct_debug() failed");
	}
#endif

	if (retcode == CS_SUCCEED)
	{
		retcode = ct_callback(db->ctx, NULL, CS_SET, CS_CLIENTMSG_CB,(CS_VOID *)ex_clientmsg_cb);
		if (retcode != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_callback(clientmsg) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode == CS_SUCCEED)
	{
		retcode = ct_callback(db->ctx, NULL, CS_SET, CS_SERVERMSG_CB,(CS_VOID *)ex_servermsg_cb);
		if (retcode != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_callback(servermsg) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode == CS_SUCCEED)
	{
		retcode = ct_config(db->ctx, CS_SET, CS_NETIO, &netio_type, CS_UNUSED, NULL);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ct_config(netio) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode != CS_SUCCEED)
	{
		ct_exit(db->ctx, CS_FORCE_EXIT);
		cs_ctx_drop(db->ctx);
		db->ctx = NULL;
		db->errcode=1;
		goto unlock_ex;
	}
	if (retcode==CS_SUCCEED){
		retcode = ct_con_alloc(db->ctx, &db->conn);
	}
	if (retcode != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_alloc failed: %s",sybsrvmsg);
		db->errcode=1;
		goto unlock_ex;
	}
	if (retcode == CS_SUCCEED && DBUser != NULL)
	{
		if ((retcode = ct_con_props(db->conn, CS_SET, CS_USERNAME, DBUser, CS_NULLTERM, NULL)) != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(username) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode == CS_SUCCEED && DBPass != NULL)
	{
		if ((retcode = ct_con_props(db->conn, CS_SET, CS_PASSWORD, DBPass, CS_NULLTERM, NULL)) != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(password) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode == CS_SUCCEED)
	{
		if ((retcode = ct_con_props(db->conn, CS_SET, CS_APPNAME, "indexer", CS_NULLTERM, NULL)) != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_con_props(appname) failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode == CS_SUCCEED)
	{
		len = DBHost ? CS_NULLTERM : 0;
		retcode = ct_connect(db->conn, DBHost, len);
		if (retcode != CS_SUCCEED)
		{
			udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ct_connect failed: %s",sybsrvmsg);
			db->errcode=1;
		}
	}
	if (retcode != CS_SUCCEED)
	{
		ct_con_drop(db->conn);
		db->conn = NULL;
		db->errcode=1;
		goto unlock_ex;
	}
	
	db->connected=1;
        cmdbuf = (CS_CHAR *) UdmMalloc(1024);
	sprintf(cmdbuf, "use %s\n", db->DBName);
	
	if ((retcode = ex_execute_cmd(db, cmdbuf)) != CS_SUCCEED)
	{
		udm_snprintf(db->errstr,sizeof(db->errstr)-1,"ex_execute_cmd(use db) failed: %s",sybsrvmsg);
		db->errcode=1;
	}
	UDM_FREE(cmdbuf);
	
unlock_ex:
	return retcode;
}


static void UdmCTLIBCloseDB(UDM_DB *db)
{
	CS_RETCODE	retcode=CS_SUCCEED;
	CS_INT		close_option;
	CS_INT		exit_option;
	CS_RETCODE	status=CS_SUCCEED; /* FIXME: previous retcode was here */

	if(db->conn){
		close_option = (status != CS_SUCCEED) ? CS_FORCE_CLOSE : CS_UNUSED;
		retcode = ct_close(db->conn, close_option);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_con_cleanup: ct_close() failed");
			db->errcode=1;
			goto unlock_ret;
		}
		retcode = ct_con_drop(db->conn);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_con_cleanup: ct_con_drop() failed");
			db->errcode=1;
			goto unlock_ret;
		}
	}	
	if(db->ctx){
		exit_option = (status != CS_SUCCEED) ? CS_FORCE_EXIT : CS_UNUSED;
		retcode = ct_exit(db->ctx, exit_option);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_ctx_cleanup: ct_exit() failed");
			db->errcode=1;
			goto unlock_ret;
		}
		retcode = cs_ctx_drop(db->ctx);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_ctx_cleanup: cs_ctx_drop() failed");
			db->errcode=1;
			goto unlock_ret;
		}
	}
unlock_ret:
	return;
}

static CS_INT CS_PUBLIC
ex_display_dlen(CS_DATAFMT * column)
{
	CS_INT		len;

	switch ((int) column->datatype)
	{
		case CS_CHAR_TYPE:
		case CS_VARCHAR_TYPE:
		case CS_TEXT_TYPE:
		case CS_IMAGE_TYPE:
			len = MIN(column->maxlength, MAX_CHAR_BUF);
			break;

		case CS_BINARY_TYPE:
		case CS_VARBINARY_TYPE:
			len = MIN((2 * column->maxlength) + 2, MAX_CHAR_BUF);
			break;

		case CS_BIT_TYPE:
		case CS_TINYINT_TYPE:
			len = 3;
			break;

		case CS_SMALLINT_TYPE:
			len = 6;
			break;

		case CS_INT_TYPE:
			len = 11;
			break;

		case CS_REAL_TYPE:
		case CS_FLOAT_TYPE:
			len = 20;
			break;

		case CS_MONEY_TYPE:
		case CS_MONEY4_TYPE:
			len = 24;
			break;

		case CS_DATETIME_TYPE:
		case CS_DATETIME4_TYPE:
			len = 30;
			break;

		case CS_NUMERIC_TYPE:
		case CS_DECIMAL_TYPE:
			len = (CS_MAX_PREC + 2);
			break;

		default:
			len = 12;
			break;
	}
	return MAX((CS_INT)(strlen(column->name) + 1), len);
}

static
int ex_fetch_data(UDM_DB *db, UDM_SQLRES *res, CS_COMMAND *cmd)
{
	CS_RETCODE		retcode;
	CS_INT			num_cols;
	CS_INT			i;
	CS_INT			j;
	CS_INT			row_count = 0;
	CS_INT			rows_read;
	CS_DATAFMT		*datafmt;
	EX_COLUMN_DATA		*coldata;

	retcode = ct_res_info(cmd, CS_NUMDATA, &num_cols, CS_UNUSED, NULL);
	if (retcode != CS_SUCCEED)
	{
		sprintf(db->errstr,"ex_fetch_data: ct_res_info() failed");
		return UDM_ERROR;
	}
	if (num_cols <= 0)
	{
		sprintf(db->errstr,"ex_fetch_data: ct_res_info() returned zero columns");
		return UDM_ERROR;
	}

	coldata = (EX_COLUMN_DATA *)UdmMalloc(num_cols * sizeof (EX_COLUMN_DATA));
	datafmt = (CS_DATAFMT *)UdmMalloc(num_cols * sizeof (CS_DATAFMT));

	for (i = 0; i < num_cols; i++)
	{
		retcode = ct_describe(cmd, (i + 1), &datafmt[i]);
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_fetch_data: ct_describe() failed");
			break;
		}

		/* Alloc data and remember the original datatype */
		coldata[i].value = (CS_CHAR *)UdmMalloc((size_t)datafmt[i].maxlength);
		coldata[i].datatype= datafmt[i].datatype;
		
		datafmt[i].maxlength = ex_display_dlen(&datafmt[i]) + 1;
		datafmt[i].datatype = CS_CHAR_TYPE;
		datafmt[i].format   = CS_FMT_NULLTERM;
		
		retcode = ct_bind(cmd, (i + 1), &datafmt[i],coldata[i].value, &coldata[i].valuelen,&coldata[i].indicator);
		
		if (retcode != CS_SUCCEED)
		{
			sprintf(db->errstr,"ex_fetch_data: ct_bind() failed");
			break;
		}
	}
	if (retcode != CS_SUCCEED)
	{
		for (j = 0; j < i; j++)
		{
			UDM_FREE(coldata[j].value);
		}
		UDM_FREE(coldata);
		UDM_FREE(datafmt);
		return UDM_ERROR;
	}

	res->nCols=num_cols;
	if (res->nCols){
		res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
		bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
		for (i=0 ; i < res->nCols; i++){
			res->Fields[i].sqlname = (char*)UdmStrdup(datafmt[i].name);
			res->Fields[i].sqllen= 0;
			res->Fields[i].sqltype= 0;
		}
	}
	
	while (((retcode = ct_fetch(cmd, CS_UNUSED, CS_UNUSED, CS_UNUSED,&rows_read)) == CS_SUCCEED) || (retcode == CS_ROW_FAIL))
	{
		row_count = row_count + rows_read;
		if (retcode == CS_ROW_FAIL)
		{
			sprintf(db->errstr, "Error on row %d.", (int)row_count);
			break;
		}
		res->Items=(UDM_PSTR*)UdmRealloc(res->Items,(res->nRows+1)*res->nCols*sizeof(UDM_PSTR));
		
		for (i = 0; i < num_cols; i++)
		{
			size_t offs= res->nRows*res->nCols+i;
			
			if (coldata[i].datatype == CS_IMAGE_TYPE ||
			    coldata[i].datatype == CS_BINARY_TYPE ||
			    coldata[i].datatype == CS_VARBINARY_TYPE ||
			    coldata[i].datatype == CS_LONGBINARY_TYPE )
			{
				/* 
				  Unfortunately bind to CS_IMAGE_TYPE,
				  CS_BINARY_TYPE, CS_VARBINARY_TYPE,
				  CS_LONGBINARY_TYPE either didn't
				  work or returned zero-padded values
				  with FreeTDS ctlib. So, let's bind 
				  these types as CHAR and convert back
				  into BINARY. This is slower, but at
				  least works as expected.
				*/ 
				DecodeHexStr(coldata[i].value,&res->Items[offs],coldata[i].valuelen-1);
			}
			else
			{
				/* valuelen includes trailing 0 */
				res->Items[offs].val= (char*)UdmMalloc(coldata[i].valuelen);
				memcpy(res->Items[offs].val,coldata[i].value,coldata[i].valuelen);
				res->Items[offs].len= coldata[i].valuelen-1;
			}
		} 
		res->nRows++;
	}

	for (i = 0; i < num_cols; i++)
	{
		UDM_FREE(coldata[i].value);
	}
	UDM_FREE(coldata);
	UDM_FREE(datafmt);

	switch ((int)retcode)
	{
		case CS_END_DATA:
			retcode = CS_SUCCEED;
			db->errcode=0;
			break;

		case CS_FAIL:
			sprintf(db->errstr,"ex_fetch_data: ct_fetch() failed");
			db->errcode=1;
			return UDM_ERROR;
			break;

		default:
			sprintf(db->errstr,"ex_fetch_data: ct_fetch() returned an expected retcode (%d)",(int)retcode);
			db->errcode=1;
			return UDM_ERROR;
			break;
	}
	return UDM_OK;
}


static
int UdmCTLIBQuery(UDM_DB *db, UDM_SQLRES *res, const char * query)
{
	CS_RETCODE	retcode;
	CS_COMMAND	*cmd;
	CS_INT		res_type;

	db->errcode=0;
	db->errstr[0]='\0';

	if(!db->connected){
		UdmCTLIBInitDB(db);
		if(db->errcode)
			goto unlock_ret;
	}

        if ((retcode = ct_cmd_alloc(db->conn, &cmd)) != CS_SUCCEED)
        {
                sprintf(db->errstr,"ct_cmd_alloc() failed");
                db->errcode=1;
                goto unlock_ret;
        }

	retcode = ct_command(cmd, CS_LANG_CMD, query, CS_NULLTERM, CS_UNUSED);
	if (retcode != CS_SUCCEED)
	{
                sprintf(db->errstr,"ct_command() failed");
                db->errcode=1;
                goto unlock_ret;
	}

	if (ct_send(cmd) != CS_SUCCEED)
	{
                sprintf(db->errstr,"ct_send() failed");
                db->errcode=1;
                goto unlock_ret;
	}

	while ((retcode = ct_results(cmd, &res_type)) == CS_SUCCEED)
	{
		switch ((int)res_type)
		{
		case CS_CMD_SUCCEED:
		case CS_CMD_DONE:
   			break;

		case CS_STATUS_RESULT:
			break;

		case CS_CMD_FAIL:
			/*printf("here: %s\n",sybsrvmsg);*/
			if(!strstr(sybsrvmsg,"UNIQUE KEY") && 
			   !strstr(sybsrvmsg,"PRIMARY KEY") &&
			   !strstr(sybsrvmsg,"unique index")){
				udm_snprintf(db->errstr,sizeof(db->errstr)-1,"%s",sybsrvmsg);
				db->errcode=1;
			}
			break;

		case CS_ROW_RESULT:
			if (!res)
			{
				udm_snprintf(db->errstr,sizeof(db->errstr)-1,"CTLibQuery: Query returned result but called without 'res'");
				db->errcode=1;
				goto unlock_ret;
			}
			if(UDM_OK != ex_fetch_data(db,res,cmd))
			{
				db->errcode=1;
				goto unlock_ret;
			}
			break;

		case CS_COMPUTE_RESULT:
			/* FIXME: add checking */
			break; 

		default:
			sprintf(db->errstr,"DoCompute: ct_results() returned unexpected result type (%d)",(int)res_type);
			db->errcode=1;
			goto unlock_ret;
		}
	}

	switch ((int)retcode)
	{
		case CS_END_RESULTS:
			break;

		case CS_FAIL:
			sprintf(db->errstr,"DoCompute: ct_results() failed");
			db->errcode=1;
			goto unlock_ret;

		default:
			sprintf(db->errstr,"DoCompute: ct_results() returned unexpected result code");
			db->errcode=1;
			goto unlock_ret;
	}

	if ((retcode = ct_cmd_drop(cmd)) != CS_SUCCEED)
	{
                sprintf(db->errstr,"DoCompute: ct_cmd_drop() failed");
                db->errcode=1;
                udb_free_result(res);
                goto unlock_ret;
	}

unlock_ret:

	return db->errcode ? UDM_ERROR : UDM_OK;
}

#endif


/*********************************** SQLITE *********************/
#if (HAVE_SQLITE)

static int UdmSQLiteInitDB(UDM_DB *db)
{
  char edbname[1024];
  char dbname[1024], *e, *errmsg;
  strncpy(edbname,db->DBName,sizeof(dbname));
  dbname[sizeof(edbname)-1]='\0';
  
  UdmUnescapeCGIQuery(dbname,edbname);
  /* Remove possible trailing slash */
  e= dbname+strlen(dbname);
  if (e>dbname && e[-1]=='/')
    e[-1]='\0';
  
  if (!(db->sqlt= sqlite_open(dbname, 0, &errmsg)))
  {
    sprintf(db->errstr, "sqlite driver: %s", errmsg ? errmsg : "<NOERROR>");
    UDM_FREE(errmsg);
    db->errcode=1;
    return UDM_ERROR;
  }
  db->connected=1;
  sqlite_busy_timeout(db->sqlt,30*1000);
  return UDM_OK;
}

static int xCallBack(void *pArg, int argc, char **argv, char **name)
{
  UDM_SQLRES *res= (UDM_SQLRES*) pArg;
  int i;

  if (res->nCols == 0)
  {
    res->nCols= argc;
    for (i=0 ; i<argc; i++)
    {
      res->Fields=(UDM_SQLFIELD*)UdmMalloc(res->nCols*sizeof(UDM_SQLFIELD));
      bzero(res->Fields,res->nCols*sizeof(UDM_SQLFIELD));
      for (i=0 ; i < res->nCols; i++){
	res->Fields[i].sqlname = (char*)UdmStrdup(name[i]);
	res->Fields[i].sqllen= 0;
	res->Fields[i].sqltype= 0;
      }
    }
  }
  res->nRows++;
  res->Items=(UDM_PSTR*)UdmRealloc(res->Items,res->nRows*res->nCols*sizeof(UDM_PSTR));
  
  for (i = 0; i < argc; i++)
  {
    size_t offs= (res->nRows-1)*res->nCols+(size_t)i;
    size_t len= strlen(argv[i]?argv[i]:"");
    res->Items[offs].len= len;
    res->Items[offs].val= (char*)UdmMalloc(len+1);
    memcpy(res->Items[offs].val,argv[i]?argv[i]:"",len+1);
  } 
  return 0;
}


static int UdmSQLiteQuery(UDM_DB *db, UDM_SQLRES *res, const char *q)
{
  char *errmsg;
  int rc;
  
#ifdef WIN32
  /** Remove heading "/" from DBName (in case of absolute path) */
  char * tmp_dbname;

  if(db->DBName[2] == ':')
  {
	  tmp_dbname = UdmStrdup(db->DBName + 1);
	  UDM_FREE(db->DBName);
	  db->DBName = tmp_dbname;
  }
#endif /* #ifdef WIN32 */

  db->errcode=0;
  db->errstr[0]='\0';
  if(!db->connected && UDM_OK != UdmSQLiteInitDB(db))
    return UDM_ERROR;
  
  if ((rc= sqlite_exec(db->sqlt, q, xCallBack, (void*)res, &errmsg)))
  {
    sprintf(db->errstr,"sqlite driver: %s",errmsg ? errmsg : "<NOERROR>");
	sqlite_freemem(errmsg);
    if (!strstr(db->errstr,"unique"))
    {
      db->errcode=1;
      return UDM_ERROR;
    }
  }
  return UDM_OK;
}

#endif	/* HAVE_SQLITE */


/***********************************************************************/

/*
 *   Wrappers for different databases
 *
 *   UdmDBEscStr();
 *   UdmSQLQuery();
 *   UdmSQLValue();
 *   UdmSQLLen();
 *   UdmSQLFree();
 *   UdmSQLClose();
 */  


char * UdmDBEscStr(int DBType,char *to,const char *from,size_t len){
	char *s;

	if(!from)return(NULL);
	if(!to)to=(char*)UdmMalloc(len*2+1);

#ifdef HAVE_MYSQL
	if(DBType==UDM_DB_MYSQL){
		mysql_escape_string(to,from,len);
		return(to);
	}
#endif
	s=to;
	if (DBType == UDM_DB_ORACLE8  ||
	    DBType == UDM_DB_MSSQL || DBType == UDM_DB_DB2 ||
	    DBType == UDM_DB_IBASE || DBType == UDM_DB_SAPDB ||
	    DBType == UDM_DB_SQLITE || DBType == UDM_DB_ACCESS ||
	    DBType == UDM_DB_MIMER || DBType == UDM_DB_CACHE)
	 {
	    while(*from){
		switch(*from){
			case '\'':
				/* Note that no break here!*/
				*to=*from;to++;
			default:
				*to=*from;
		}
		to++;from++;
	    }
	} else {
	    while(*from){
		switch(*from){
			case '\'':
			case '\\':
				*to='\\';to++;
			default:*to=*from;
		}
		to++;from++;
	    }
	}
	*to=0;return(s);
}


int UdmSQLBegin(UDM_DB *db)
{
	int rc= UDM_OK;
	switch(db->DBDriver){
		case UDM_DB_PGSQL:
			rc=UdmSQLQuery(db,NULL,"BEGIN WORK");
			break;
		case UDM_DB_SQLITE:
		case UDM_DB_MSSQL:
			rc=UdmSQLQuery(db,NULL,"BEGIN TRANSACTION");
			break;
		case UDM_DB_ORACLE8:
		case UDM_DB_SAPDB:
			rc=UdmSQLQuery(db,NULL,"COMMIT");
			db->commit_fl = 1;
			break;
		case UDM_DB_IBASE:
			rc=UdmSQLQuery(db,NULL,"BEGIN");
			db->commit_fl = 1;
			break;
#if HAVE_ODBC
		case UDM_DB_ODBC:
			db->errcode = SQLSetConnectOption( db->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF);
			if( SQL_SUCCESS != db->errcode )return UDM_ERROR;
			break;
#endif
		default:
			rc=UDM_OK;
			break;
	}
	return rc;
}

int UdmSQLCommit(UDM_DB *db)
{
	int rc= UDM_OK;
	switch(db->DBDriver){
		case UDM_DB_PGSQL:
			rc=UdmSQLQuery(db,NULL,"END WORK");
			break;
		case UDM_DB_SQLITE:
		case UDM_DB_MSSQL:
			rc=UdmSQLQuery(db,NULL,"COMMIT");
			break;
		case UDM_DB_ORACLE8:
		case UDM_DB_SAPDB:
			rc=UdmSQLQuery(db,NULL,"COMMIT");
			db->commit_fl = 0;
			break;
		case UDM_DB_IBASE:
			rc=UdmSQLQuery(db,NULL,"COMMIT");
			db->commit_fl = 1;
			break;
#if HAVE_ODBC
	        case UDM_DB_ODBC:
			if (UDM_OK!=(rc=UdmSQLQuery(db,NULL,"COMMIT")))
				return UDM_ERROR;
			db->errcode = SQLSetConnectOption( db->hDbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_ON);
			if( SQL_SUCCESS != db->errcode )
				return UDM_ERROR;
			break;
#endif
		default:
			db->commit_fl = 0;
			break;
	}
	return rc;
}



int __UDMCALL _UdmSQLQuery(UDM_DB *db, UDM_SQLRES *SQLRes, const char * query, const char *file, const int line) {
	UDM_SQLRES res;
	
#ifdef DEBUG_SQL
	unsigned long ticks;
	ticks=UdmStartTimer();
#endif
	
	/* FIXME: call UdmSQLFree at exit if SQLRes = NULL */
	if (! SQLRes) SQLRes = &res;
	bzero((void*)SQLRes, sizeof(UDM_SQLRES));
	SQLRes->DBDriver = db->DBDriver;
	
#if HAVE_MYSQL
	if(db->DBDriver==UDM_DB_MYSQL){
		UdmMySQLQuery(db,SQLRes,query);
		goto ret;
	}
#endif
	
#if HAVE_SQLITE
	if(db->DBDriver==UDM_DB_SQLITE){
		UdmSQLiteQuery(db,SQLRes,query);
		goto ret;
	}
#endif
	
#if HAVE_PGSQL
	if(db->DBDriver==UDM_DB_PGSQL){
		UdmPgSQLQuery(db,SQLRes,query);
		goto ret;
	}
#endif
	
#if HAVE_IBASE
	if(db->DBDriver==UDM_DB_IBASE){
		UdmIBaseQuery(db,SQLRes,query);
		goto ret;
	}
#endif
	
#if HAVE_ORACLE8
	if(db->DBDriver==UDM_DB_ORACLE8){
		UdmOracle8Query(db, SQLRes, query);
		goto ret;
	}
#endif

#if HAVE_CTLIB
	if(db->DBDriver==UDM_DB_MSSQL){
		UdmCTLIBQuery(db,SQLRes,query);
		goto ret;
	}
#endif

#if (HAVE_ODBC)
	if(db->DBDriver==UDM_DB_ODBC){
		UdmODBCQuery(db,SQLRes,query);
		goto ret;
	}
#endif
	db->errcode=1;
	udm_snprintf(db->errstr,sizeof(db->errstr)-1,"Unsupported SQL database type");
ret:
#ifdef DEBUG_SQL
	ticks=UdmStartTimer()-ticks;
	fprintf(stderr,"[%d] %.2fs SQL {%s:%d}: %s\n", getpid(), (float)ticks/1000, file, line, query);
#endif
	
#ifdef DEBUG_ERR_QUERY
	if (db->errcode == 1)
	  fprintf(stderr, "{%s:%d} Query: %s\n\n", file, line, query);
#endif

	return db->errcode ? UDM_ERROR : UDM_OK;
}




size_t __UDMCALL UdmSQLNumRows(UDM_SQLRES * res)
{
	return(res?res->nRows:0);
}

size_t UdmSQLNumCols(UDM_SQLRES * res)
{
	return(res?res->nCols:0);
}

const char * __UDMCALL UdmSQLValue(UDM_SQLRES * res,size_t i,size_t j){

#if HAVE_PGSQL
	if(res->DBDriver==UDM_DB_PGSQL){
		return(PQgetvalue(res->pgsqlres,(int)(i),(int)(j)));
	}
#endif

	if (i<res->nRows){
		size_t offs=res->nCols*i+j;
		return res->Items[offs].val;
	}else{
		return NULL;
	}
}

int __UDMCALL UdmSQLFetchRow (UDM_SQLRES *res, size_t i, char **buf) {
	size_t j;
	size_t offs = res->nCols * i;

#if HAVE_PGSQL
	if(res->DBDriver==UDM_DB_PGSQL){
		for (j = 0; j < res->nCols; j++) {
			buf[j] = PQgetvalue(res->pgsqlres,(int)(i),(int)(j));
		}
		return(UDM_OK);
	}
#endif

	for (j = 0; j < res->nCols; j++) {
		buf[j] = res->Items[offs + j].val;
	}

	return(UDM_OK);
}

void __UDMCALL UdmSQLFree(UDM_SQLRES * res){

	if (res->Fields){
		size_t i;
		for(i=0;i<res->nCols;i++){
			/*
			printf("%s(%d)\n",res->Fields[i].sqlname,res->Fields[i].sqllen);
			*/
			UDM_FREE(res->Fields[i].sqlname);
		}
		UDM_FREE(res->Fields);
	}
	
#if HAVE_PGSQL
	if(res->DBDriver==UDM_DB_PGSQL){
		PQclear(res->pgsqlres);
		return;
	}
#endif

#if HAVE_ORACLE8
	if(res->DBDriver==UDM_DB_ORACLE8){
		oci_free_result(res);
		udb_free_result(res);
		return;
	}
#endif
	udb_free_result(res);
}


size_t UdmSQLLen(UDM_SQLRES * res,size_t i,size_t j){

#if HAVE_PGSQL
	/* TODO: This should be fixed */
	if(res->DBDriver==UDM_DB_PGSQL){
		return 0;
	}
#endif
	if (i<res->nRows){
		size_t offs=res->nCols*i+j;
		return res->Items[offs].len;
	}else{
		return 0;
	}
	return 0; /* Make compilers happy */
}




void UdmSQLClose(UDM_DB *db){

	if(!db->connected)return;

#if HAVE_MYSQL
	if(db->DBDriver==UDM_DB_MYSQL){
		mysql_close(&db->mysql);
		goto ret;
	}
#endif
	
#if HAVE_PGSQL
	if(db->DBDriver==UDM_DB_PGSQL){
		PQfinish(db->pgsql);
		goto ret;
	}
#endif
	
#if HAVE_IBASE
	if(db->DBDriver==UDM_DB_IBASE){
		UdmIBaseCloseDB(db);
		goto ret;
	}
#endif

#if HAVE_ORACLE8
	if(db->DBDriver==UDM_DB_ORACLE8){
		UdmOracle8CloseDB(db);
		goto ret;
	}
#endif
	
#if HAVE_CTLIB
	if(db->DBDriver==UDM_DB_MSSQL){
		UdmCTLIBCloseDB(db);
		goto ret;
	}
#endif
	
#if HAVE_SQLITE
	if(db->DBDriver==UDM_DB_SQLITE){
		sqlite_close(db->sqlt);
		goto ret;
	}
#endif
	
#if (HAVE_ODBC)
	if(db->DBDriver==UDM_DB_ODBC){
		UdmODBCCloseDB(db);
		goto ret;
	}
#endif
ret:
	db->connected=0;
	return;
}


#include "udm_mutex.h"

/*

  UdmSQLMonitor()
  
  - Executes SQL queries arriving from some virtual input stream
  - Displays query results to some virtual output streams
  
  The incoming strings are checked against:
    
    - SQL comments, i.e. lines starting with "--".
      The comment lines are skipped.
    
    - A set of end-of-query markers, i.e. 
      * MSSQL   style: 'GO' keyword written in the beginning
      * mysql   style: ';'  character written in the end of the string
      * minisql style: '\g' characters written in the end of the string
      
      As soon as a end-of-query marker has found, current collected query
      is executed.
    
    - In other case, the next line is concatenated to the previously
      collected data thus helping to run multi-line queries.
  
  
  The particular ways of handling input and output streams are
  passed in the UDM_SQLMON_PARAM argument. This approach makes 
  it possible to use UdmSQLMonitor() for many purposes:
  
  - executing a bundle of queries written in a text file.
  - running a command-line SQL monitor ("mysql", "pgsql" or "sqlplus" alike),
    i.e. executing queries arriving from the keyboard.
    
  - running a GUI SQL monitor,
    i.e. executing querues arriving from the keyboard in some GUI application.
  
  
  UDM_SQLMON_PARAM has got this structure:
  
  typedef struct udm_sqlmon_param_st
  {
    int flags;
    FILE *infile;
    FILE *outfile;
    char *(*gets)(struct udm_sqlmon_param_st *prm, char *str, size_t size);
    int (*display)(struct udm_sqlmon_param_st *, UDM_SQLRES *sqlres);
    int (*prompt)(struct udm_sqlmon_param_st *, const char *msg);
	void* tag_ptr;
  } UDM_SQLMON_PARAM;
  
  
  gets():     a function to get one string from the incoming stream.
  display():  a function to display an SQL query results
  prompt():   a function to display some prompts:
                - an invitation to enter data, for example: "SQL>".
                - an error message
  
  flags:      sets various aspects of the monitor behaviour:
              the only one so far:
                - whether to display field names in the output

  infile:     if the caller wants to input SQL queries from some FILE stream,
              for example, from STDIN or from a textual file opened via fopen(),
              it can use this field for this purpose
              
  outfile:    if the caller wants to output results to some FILE stream,
              for example, to STDOUT or to a textial file opened via fopen(),
              it can use this field for this purpose

  context_ptr:    extra pointer parameter for various purposes.
  
  To start using UdmSQLMonitor() one have to set
    - gets()
    - diplay()
    - prompt()
  
  If diplay() or prompt() use some FILE streams, the caller should
  also set infile and outfile with appropriative stream pointers.
  
*/


char str[64*1024];

__C_LINK int __UDMCALL UdmSQLMonitor(UDM_AGENT *A, UDM_ENV *Env, UDM_SQLMON_PARAM *prm){
     char *snd=str;
     int  rc=UDM_OK;
     
     str[sizeof(str)-1]='\0';
     while(1){
          int  res;
          int  exec=0;
          char *send0;
          size_t  rbytes= (sizeof(str)-1) - (snd - str);

	  if (!prm->gets(prm,snd,rbytes))
	    break;

          if(snd[0]=='#' || !strncmp(snd, "--", 2)){
               continue;
          }
          send0 = snd;
          snd = snd + strlen(snd);
          while(snd > send0 && strchr(" \r\n\t", snd[-1])){
               *--snd='\0';
          }
		  if(snd == send0)
			  continue;
          
          if(snd[-1]==';'){
               exec=1;
               *--snd='\0';
          }else
          if(snd - 2 >= str && snd[-1]=='g' && snd[-2]=='\\'){
               exec=1;
               snd-=2;
               *snd='\0';
          }else
          if(snd-2 >= str && strchr("oO", snd[-1]) && strchr("gG",snd[-2])){
               exec=1;
               snd-=2;
               *snd='\0';
          }else
          if((size_t)(snd - str + 1) >= sizeof(str)){
               exec=1;
          }
          
          if(exec){
          
               prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "'");
               prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, str);
               prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "'");
               prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
               
               if (!strncasecmp(str,"connection",10)){
                    size_t newnum= atoi(str+10);
                    char msg[255];
                    if (newnum < Env->dbl.nitems){
                         Env->dbl.currdbnum= newnum;
                         sprintf(msg,"Connection changed to #%d",Env->dbl.currdbnum);
                         prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, msg);
                         prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
                    }else{
                         sprintf(msg,"Wrong connection number %d",newnum);
                         prm->prompt(prm, UDM_SQLMON_MSG_ERROR, msg);
                         prm->prompt(prm, UDM_SQLMON_MSG_ERROR, "\n");
                    }
                    
               }
               else if (!strcasecmp(str,"fields=off")){
                    prm->flags=0;
               }
               else if (!strcasecmp(str,"fields=on")){
                    prm->flags=1;
               }
               else if (!strncasecmp(str,"colflags",8))
               {
                    int colnum= atoi(str+8);
                    int colval= atoi(str+10);
                    if (colnum>=0 && colnum<10)
                      prm->colflags[colnum]= colval;
               }
               else
               {
                    UDM_SQLRES sqlres;
                    UDM_DB *db= &Env->dbl.db[Env->dbl.currdbnum];
                    prm->nqueries++;
                    bzero((void*)&sqlres,sizeof(sqlres));
		    UDM_GETLOCK(A, UDM_LOCK_DB);
		    res=UdmSQLQuery(Env->dbl.db,&sqlres,str);
		    UDM_RELEASELOCK(A, UDM_LOCK_DB);
		    if (res!=UDM_OK)
		    {
		      prm->nbad++;
		      rc=UDM_ERROR;
		      prm->prompt(prm, UDM_SQLMON_MSG_ERROR, db->errstr);
		      prm->prompt(prm, UDM_SQLMON_MSG_ERROR, "\n");
		    }
		    else
		    {
		      prm->ngood++;
                      res=prm->display(prm,&sqlres);
                    }
                    UdmSQLFree(&sqlres);
               }
               snd=str;
               str[0]='\0';
          }else{
               if(send0!=snd){
                    *snd=' ';
                    snd++;
                    *snd='\0';
               }
          }
     }
     prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "\n");
     return rc;
}


#endif
