/*
     This code is distributed under the terms and conditions of the
     CCP4 licence agreement as `Part ii)' software.  See the conditions
     in the CCP4 manual for a copyright statement.
*/

/*   cparser.c
     Peter Briggs CCP4 April 2001

     Functions to read in and "parse" (scan, in reality) CCP4-style
     "keyworded" input, plus utility routines.

     cparse_start(maxtokens)
     initialise a PARSERARRAY to be used in subsequent calls to
     cparser routines.

     cparse_end()
     clean up PARSERARRAY after use

     cparse_init_token(parser,itok)
     initialise a single token in PARSERARRAY before use

     cparse_reset(parser)
     initialise PARSERARRAY before use (includes calls to
     cparse_init_token to initialise all tokens)

     cparse_delimiters(parser,delimiters,nulldelimiters)
     set up or restore non-default delimiters

     cparse(line,parser)
     given a string "line", break up into tokens and store in
     PARSERARRAY "parser"

     cparser(line,parser,print)
     read input from stdin or external line, break up into tokens
     and store in PARSERARRAY "parser"

     keymatch(keyin1,keyin2)
     compare input strings to see if they match as CCP4-style keywords

     strtoupper(str1,str2)
     convert string to uppercase

     strmatch(str1,str2)
     check if strings are identical

     charmatch(character,charlist)
     check if character appears in a list of possibilities

*/

/* Header files */

#include <string.h>
#include "cparser.h"

/*------------------------------------------------------------------*/

/* Parser Functions */

/*------------------------------------------------------------------*/

/* cparse_start

   This initialises a PARSERARRAY to be used with the cparse/cparser
   functions.
   Calling function must supply maximum number of tokens.
*/
PARSERARRAY* cparse_start(int maxtokens)
{
  int itok;
  PARSERARRAY *parsePtr;

  /* Initial check for sensible values */
  if (maxtokens < 1) return NULL;

  /* Return a pointer to a PARSERARRAY */
  parsePtr = (PARSERARRAY *) malloc(sizeof(PARSERARRAY));
  if (parsePtr) {
    parsePtr->token = (PARSERTOKEN *) malloc(sizeof(PARSERTOKEN)*maxtokens);
    if (!parsePtr->token) {
      free(parsePtr);
      parsePtr = NULL;
    } else {
      /*parsePtr->ntokens = 0;*/
      parsePtr->maxtokens = maxtokens;
      parsePtr->fp = NULL;

      /* MDW/CCPFYP Memory Fault problem start */
      /* Explicitly ensure that each fullstring is set to NULL
	 before calling cparse_reset
	 There it tries to free memory associated with fullstrings
	 if the pointers are not NULL - but we can't rely on them
	 being initialised with any particular value */
      for (itok = 0; itok < maxtokens; itok++) {
	parsePtr->token[itok].fullstring = NULL;
      }
      /* MDW/CCPFYP Memory Fault problem end */

      cparse_reset(parsePtr);
      /* Initialise the default delimiter and null delimiter characters
	 with a call to cparse_delimiters */
      parsePtr->delim=NULL;
      parsePtr->nulldelim=NULL;
      if (!cparse_delimiters(parsePtr,NULL,NULL)) {
	cparse_end(parsePtr);
	parsePtr = NULL;
      }
    }
  }

  return parsePtr;
}

/*------------------------------------------------------------------*/

/* cparse_end

   This cleans up a PARSEARRAY after being used by cparse/cparser
   functions.
*/
int cparse_end(PARSERARRAY *parsePtr)
{
  int i,maxtokens;

  /* Anything to do? */
  if (parsePtr) {
    /* Free memory for each token */
    maxtokens = parsePtr->maxtokens;
    if (parsePtr->token && parsePtr->maxtokens > 0)
      for (i=0; i<maxtokens; i++)
	if(parsePtr->token[i].fullstring) free(parsePtr->token[i].fullstring); 
    /* Free memory for lists of delimiters */
    if (parsePtr->delim) free(parsePtr->delim);
    if (parsePtr->nulldelim) free(parsePtr->nulldelim);
    /* Free memory for rest of parserarray structure */
    free(parsePtr);
  }
  /* All done */
  return 0;
}

/*------------------------------------------------------------------*/

/* cparse_init_token

   Initialise a token in a parser array
   This sets all string members of the specified token to NULL and
   all numerical values (including flags) to zero
*/

int cparse_init_token(PARSERARRAY *parsePtr, int itok)
{
  if (parsePtr) {
    if (itok < parsePtr->maxtokens) {
      /* Full string is dynamically allocated - free the
	 associated memory, if assigned */
      if (parsePtr->token[itok].fullstring) {
	  free(parsePtr->token[itok].fullstring);
	  parsePtr->token[itok].fullstring = NULL;
      }
      /* Set fixed string tokens to empty string */
      strcpy(parsePtr->token[itok].word,"");
      /* Set numerical value to zero */
      parsePtr->token[itok].value = 0.0;
      /* Set flags to zero */
      parsePtr->token[itok].isstring = 0;
      parsePtr->token[itok].isnumber = 0;
      parsePtr->token[itok].isquoted = 0;
      parsePtr->token[itok].isnull   = 0;
      /* Set start and end positions to zero */
      parsePtr->token[itok].ibeg = 0;
      parsePtr->token[itok].iend = 0;
    }
  }
  return 0;
}

/*------------------------------------------------------------------*/

/* cparse_reset

   Reinitialise a parser array before calling cparse

   An application using cparse (rather than cparser, which also
   calls this routine) can call this function to reset the parser
   array, rather than reinitialising the structure members
   explicitly
*/
int cparse_reset(PARSERARRAY *parsePtr)
{
  int itok;

  if (parsePtr) {
    /* Initialise the tokens to have null values */
    for (itok=0; itok<parsePtr->maxtokens; itok++)
      cparse_init_token(parsePtr,itok);
    /* Initialise number of tokens to zero */
    parsePtr->ntokens = 0;
  }
  return 0;
}

/*------------------------------------------------------------------*/

/* cparse_delimiters

   This allows the application to set its own delimiter characters
   to be used in the cparser routines.

   If a NULL pointer is supplied for either of the two lists then
   then the default delimiters are (re)set.

   Returns 1 on success, 0 if there was an error. In the event of
   an error the delimiter lists will be unchanged.
*/

int cparse_delimiters(PARSERARRAY *parsePtr, char *delim, char *nulldelim)
{
  char defdelim[]=" \t,=",defnulldelim[]=",=";
  char *delimPtr=NULL,*nulldelimPtr=NULL;
  int  ldelim,lnulldelim,istatus=1;

  if (parsePtr) {
    /* If supplied delim is NULL then set to the default */
    if (!delim) delim = defdelim;
    /* If supplied nulldelim is NULL then set to the default */
    if (!nulldelim) nulldelim = defnulldelim;
    /* Set up lists in memory */
    ldelim = strlen(delim) + 1;
    lnulldelim = strlen(nulldelim) + 1;
    /* Standard delimiters */
    delimPtr = (char *) malloc(sizeof(char)*ldelim);
    if (delimPtr) {
      ldelim--;
      strncpy(delimPtr,delim,ldelim+1);
      delimPtr[ldelim] = '\0';
    }
    /* Null delimiters */
    nulldelimPtr = (char *) malloc(sizeof(char)*lnulldelim);
    if (nulldelimPtr) {
      lnulldelim--;
      strncpy(nulldelimPtr,nulldelim,lnulldelim+1);
      nulldelimPtr[lnulldelim] = '\0';
    }
    /* Assign new delimiters in parser array */
    if (delimPtr && nulldelimPtr) {
      if (parsePtr->delim) free(parsePtr->delim);
      parsePtr->delim = delimPtr;
      if (parsePtr->nulldelim) free(parsePtr->nulldelim);
      parsePtr->nulldelim = nulldelimPtr;
    } else {
      /* There is an error - don't reset the parser array */
      if (delimPtr) free(delimPtr);
      if (nulldelimPtr) free(nulldelimPtr);
      istatus = 0;
    };
  } else {
    istatus = 0;
  }
  return istatus;
}

/*------------------------------------------------------------------*/

/* cparse

   This is a scanner based on the old CCP4 Fortranic PARSE routine.
 
   It takes an input string ("line") and returns the number of
   tokens ("ntokens") which are delimited by certain characters
   (defaulted to space, tab, comma, equals - these can be changed by
   the application using a call to cparse_delimiters).
   Information about the tokens themselves is returned as members of
   elements in an array ("tokens") of type PARSERTOKEN (see header
   file for definition and members).

   Substrings can be delimited by single- or double-quotes but must 
   be surrounded by delimiters to be recognised.

   An unquoted ! or # in the input line introduces a trailing comment
   which is ignored.
   Null fields are denoted by two adjacent null delimiters (defaulted
   to comma and equals - these can be changed by the application
   using a call to cparse_delimiters).

   cparse returns the number of tokens found in the line. The tokens
   are returned via the PARSERARRAY parser.
*/

int cparse(char *line, PARSERARRAY *parser)
{
  int quotedstring,starttoken,endtoken;
  char this_char,next_char,matchquote;

  int llen,ich,lword,diag=0;
  int token,nulltoken,isquote,iscommt,isdelim;
  double value;
  char *delim,*nulldelim;
  char comm[]="#!",quot[]="\"\'";
  int  ibeg,iend,start;

  /* Local parser variables */
  int ntok,maxtok;
  PARSERTOKEN *tokenarray;

  /* Begin */

  /* Set diag = 1 and recompile to switch on diagnostic output */
  if (diag) printf("CPARSE: cparse starting\n");

  maxtok = parser->maxtokens;
  ntok = parser->ntokens;
  if (ntok < 0) ntok = 0;

  /* Initialise pointer for local version of the token array */
  tokenarray = parser->token;

  /* Initialise pointers for lists of delimiters */
  delim = parser->delim;
  nulldelim = parser->nulldelim;

  /* Don't process any tokens already dealt with */
  if (ntok > 0)
    start = tokenarray[ntok-1].iend + 1;
  else
    start = 0;

  /* Don't process empty lines */
  llen = strlen(line);
  if (diag) printf("CPARSE: Line is: \"%s\"\nLength of line is %d\n",line,llen);
  if (llen > 0) {

    /* Initialise flags and counters */
    quotedstring = 0;
    token        = 0;
    nulltoken    = 0;

    /* Process the line two characters at a time */

    if (diag) printf("CPARSE: Start position for parsing line is %d\n",start);
    for (ich=start-1; ich<llen; ich++) {

      /* Examine characters in pairs */

      /* Deal with special case, start and end of line
	 This is to avoid accessing non-existant parts of the string */
      if (ich < start) {
	this_char = delim[0];
      } else {
	this_char = line[ich];
      }
      if (ich == llen-1) {
	next_char = delim[0];
      } else {
	next_char = line[ich+1];
      }
      if (diag) printf("CPARSE: %d: Current char = \"%c\" : \n",ich,this_char);

      /* Set flags for this round
	 The pairs of characters are analysed and flags set
	 accordingly to signal actions after the analysis */
      starttoken = 0;
      endtoken   = 0;

      if (!quotedstring) {
	/* Not in a quoted string so look for opening quotes
	   This is delimiter followed by quote character */

	/* Is current character a delimiter character? */
	isdelim = charmatch(this_char,delim);
	
	/* Is next charcter a quote character? */
	isquote = charmatch(next_char,quot);

	/* Start of quoted string? */
	if (isdelim && isquote) {

	  if (diag) printf("CPARSE: start of a quoted string");
	  quotedstring = 1;
	  starttoken = 1;
	  matchquote = next_char;

	} else {
	  /* Not the start of a quoted string */

	  /* Is the current character the start of a comment? */
	  iscommt = charmatch(this_char,comm);
	  if (iscommt) {
	    if (diag) printf("CPARSE: start of a comment\n");
	  }

	  /* Is the current character the start or end of a token? */
	  if (!isdelim) {

	    /* End of the current token?
	       Only if we are in a token now, and the next
	       character is a delimiter or a comment */
	    if (token && (charmatch(next_char,delim) || charmatch(next_char,comm))) {
	      if (diag) printf("CPARSE: end of a token\n");
	      endtoken = 1;
	    }

	  } else {

	    /* Start of a token?
	       Only if we are not in a token now, and the next character
	       is not a delimiter (or a comment) too */
	    if (!token && !(charmatch(next_char,delim) || charmatch(next_char,comm))) {
		if (diag) printf("CPARSE: start of a token\n");
		starttoken = 1;
	    }
	    
	    /* Or - could be a null token?
	       This is a pair of null delimiters together */
	    if (!token && charmatch(this_char,nulldelim)
		&& charmatch(next_char,nulldelim)) {
	      if (diag) printf("CPARSE: null token\n");
	      starttoken = 1;
	      nulltoken  = 1;
	    }

	  }
	  /* End of token identification */
	}
	
      } else {
	/* Inside a quoted string so look for closing quotes
	   This is a matching quote followed by a delimiter */
	
	/* Is current character a matching quote? */
	isquote = (this_char == matchquote);

	/* Is next charcter a delimiter character? */
	isdelim = charmatch(next_char,delim);

	/* End of quoted string */
	if (isdelim && isquote) {
	  if (diag) printf("CPARSE: end of a quoted string");
	  quotedstring = 0;
	  endtoken = 1;
	  matchquote = ' ';
	}
	/* End of character analyses */
      }

      /* Start of new token */
      if (starttoken) {
	/* Check we don't have maximum number of tokens already */
	if (ntok < maxtok) {
	  /* Set flags */
	  token = 1;
	  if (quotedstring) {
	    ibeg = ich + 2;
	  } else if (!nulltoken) {
	    ibeg = ich + 1;
	  } else {
	    ibeg = ich;
	  }
	  if (diag) printf("CPARSE: Start of a new token... ibeg = %d\n",ibeg);
	  /* Initialise values */
	  tokenarray[ntok].fullstring = NULL;
	  tokenarray[ntok].value    = 0.0;
	  tokenarray[ntok].isstring = 0;
	  tokenarray[ntok].isnumber = 0;
	  tokenarray[ntok].isquoted = 0;
	  tokenarray[ntok].isnull   = 0;
	  /* Flag start of quoted string */
	  if (quotedstring) { tokenarray[ntok].isquoted = 1; }
	  /* Flag null token */
	  if (nulltoken) {
	    if (diag) printf("CPARSE: Null token\n");
	    tokenarray[ntok].isnull = 1;
	    tokenarray[ntok].iend   = ich;
	    token = 0;
	    nulltoken = 0;
	  }
	} else {
	  /* Maximum number of tokens already found */
	  ccperror(2,"cparse: maximum number of tokens exceeded");
	  parser->ntokens = ntok;
	  return ntok;
	}
	if (diag) printf("CPARSE: This is the start of token %d\n",ntok);
      }
      /* End of new token */

      /* End of current token */
      if (endtoken) {
	token = 0;
	/* Exclude trailing quote from the token? */
	if (tokenarray[ntok].isquoted) {
	  iend = ich - 1;
	} else {
	  iend = ich;
	}
	if (diag) printf("CPARSE: End of a token... iend = %d\n",iend);
	/* Store the full token in the array */
	lword = iend - ibeg + 1;
	if (diag) printf("CPARSE: lword = %d - start char = %c, end char = %c\n",
			 lword,line[ibeg],line[iend]);
	tokenarray[ntok].fullstring = (char *) malloc(sizeof(char)*(lword+1));
	if (tokenarray[ntok].fullstring) {
	  strncpy(tokenarray[ntok].fullstring,&line[ibeg],lword);
	  tokenarray[ntok].fullstring[lword] = '\0';
	  if (diag) printf("CPARSE: Token is \"%s\"\n",tokenarray[ntok].fullstring);
	} else {
	  ccperror(2,"cparse: couldn't allocate memory to store full token");
	}
	tokenarray[ntok].ibeg = ibeg;
	tokenarray[ntok].iend = iend;
	/* Store the 4 character token in the array */
	if (lword > 4) lword = 4; 
	strncpy(tokenarray[ntok].word,&line[ibeg],lword);
	tokenarray[ntok].word[lword] = '\0';
	/* Determine numerical value (if any) */
	if (sscanf(tokenarray[ntok].fullstring,"%lf",&value)) {
	  if (diag) printf("CPARSE: This has a numerical value of %lf\n",value);
	  tokenarray[ntok].value = value;
	  tokenarray[ntok].isnumber = 1;
	} else {
	  if (diag) printf("CPARSE: There is no numerical value for this token\n");
	  tokenarray[ntok].isstring = 1;
	}
	/* Reset flags etc ready for next token*/
	token  = 0;
	value  = 0.0;
	/* Increment number of tokens */
	ntok++;
	if (diag) printf("CPARSE: This is the end of a token\n");
      }

      /* Don't do any more processing after a comment */
      if (iscommt) {
	parser->ntokens = ntok;
	return ntok;
      }
      /* Check the next pair of characters */
    }
    /* Reset the number of tokens in the parser array */
    parser->ntokens = ntok;
    if (diag) printf("CPARSE: ntokens = %d, and ntok = %d\n",parser->ntokens,ntok);
  }
  return ntok;
}

/*------------------------------------------------------------------*/

/* cparser

   This is based on the old CCP4 Fortranic PARSER routine.

   The normal behaviour is to read "keyworded" data from the input
   stream, and interpret it. Stdin is the default, but a line
   starting with @<name> starts reading from file <name> until eof.

   Each input line may be continued on the next line by the continuation
   characters `&', `-' or `\' at the end of the input line. This
   character is dropped from the list returned to the calling application.

   Pass in a zero length line to force reading from the command line.
   n is the maximum number of characters which will be read into the line.
   (If line is not blank then it will be processed and more input
   read in if it ends in a continuation character, or forces reading from
   an external file.)
   The "print" argument should be supplied as 0 to suppress echoing of the
   input lines to standard output.

   cparser returns the number of tokens parsed in the input line. The
   results of the parsing are stored as members of the PARSEARRAY structure
   "parser" and can be accessed by the application program.

   The function returns the number of tokens, or 0 on reaching end of file.
   On encountering an unrecoverable error cparser returns -1. 
*/

int cparser(char *line, int n, PARSERARRAY *parser, int print)
{
  int fromstdin=0,fromfile=0,fromapp=0,diag=0;
  int nch=200,nold,continuation,first,trunc,llen;
  char linein[200],filename[200];

  /* Local parser variables */
  int  ntok;
  FILE *filein;
  PARSERTOKEN *tokenarray;

  /* Undocumented feature - if print < 0 then also print out
     diagnostic info */
  if (print < 0) {
    print = 1;
    diag  = 1;
  }

  /* Begin */
  if (diag) printf("CPARSER: cparser starting\n");

  /* Abort if parser is a NULL pointer */
  if (!parser) {
    ccperror(4,"cparser: null pointer for PARSERARRAY");
    return -1;
  }

  /* Abort if line is NULL pointer */
  if (!line) {
    ccperror(4,"cparser: null pointer for line");
    return -1;
  }

  /* Reset the parser array for this sweep
     This will prevent phantom values from an earlier call
     to cparser persisting in the parser array */
  cparse_reset(parser);

  /* Blank the keyword */
  strcpy(parser->keyword,"");

  /* Initialise local variables and pointers */
  tokenarray = parser->token;
  ntok       = parser->ntokens;
  filein     = parser->fp;
  if (diag) printf("CPARSER: parser->ntokens = %d, ntok = %d\n",parser->ntokens,ntok);

  /* If line is empty then read from the standard input
     Otherwise process the line from the application first*/
  if (strlen(line)==0) {
    if (!filein) {
      if (diag) printf("CPARSER: Reading from stdin\n");
      fromstdin = 1;
    } else {
      if (diag) printf("CPARSER: Reading from file\n");
      fromfile = 1;
    }
  } else {
    if (diag) printf("CPARSER: Reading line supplied by the application program\n");
    fromapp = 1;
  }

  /* Set flag for first line of input */
  first = 1;
  
  /* Set flag for line continuation */
  continuation = 1;
  
  /* Start the input loop */
  while (continuation) {
    
    /* Read input from stdin a line at a time */
    if (fromstdin) {
      if (!fgets(linein,nch,stdin)) {
	/* Jump out at this point if eof is reached from
	   stdin */
	return 0;
      }
    } else if (fromfile) {
      if (!fgets(linein,nch,filein)) {
	/* Return to input from stdin if eof is read from
	   the external file */
	if (diag) printf("CPARSER: End of external file reached\n");
	fclose(filein);
	filein = NULL;
	fromfile  = 0;
	fromstdin = 1;
	/* Blank the line and reset the first flag to
	   force reading from standard input immediately */
	linein[0] = '\0';
	ntok      = 0;
	parser->ntokens = ntok;
	first     = 1;
      }
    } else if (fromapp) {
      /* If this line contains a continuation mark then
	 read from stdin next time around */
      strncpy(linein,line,n);
    }

    /* Strip any trailing newline e.g. from fgets */
    llen = strlen(linein);
    if (llen > 0)
      if (linein[llen-1] == '\n') {
	linein[llen-1] = '\0';
	llen--;
      }
    
    /* If previous line ended with a continuation character
       then append this one to it
       Check that we don't overflow the number of characters
       specified by the application */
    if (llen > n) {
      ccperror(2,"cparser: input line is too long - truncating");
    }
    if (first) {
      strncpy(line,linein,n);
      first = 0;
    } else {
      strncat(line,linein,n);
    }
    n = n - llen;
    if (diag) printf("CPARSER: line = \"%s\"\n",line);
      
    /* Use cparse to break the input line up into tokens
       Only parse the latest chunk - cparse will append
       new tokens onto the tokenarray */
    nold = ntok;
    ntok = cparse(line,parser);

    /* Blank the line after the last token to remove
       any trailing comments */
    /* 12-06-01
       Don't do this - it zaps everything which is not apparently
       good form */
    /*if (ntok > 0)
      if (tokenarray[ntok-1].isquoted)
      line[tokenarray[ntok-1].iend+2] = '\0';
      else
      line[tokenarray[ntok-1].iend+1] = '\0';*/

    if (diag) printf("CPARSER: ntok = %d, nold = %d\n",ntok,nold);

    /* Have we found more tokens since last time? */
    if (ntok != nold) {
      /* Check first token to see if it is an instruction
	 to read from an external file */
      if (!fromfile && tokenarray[0].word[0] == '@') {
	if (diag) printf("CPARSER: Instruction to read from external file\n");
	/* Get filename and attempt to open file */
	if (tokenarray[0].fullstring) {
	  llen = strlen(tokenarray[0].fullstring);
	  strncpy(filename,&tokenarray[0].fullstring[1],llen);
	  if (diag) printf("CPARSER: External file name is \"%s\"\n",filename);
	  /* Open the external file as read-only */
	  filein = fopen(filename,"r");
	  if (!filein) {
	    ccperror(2,"cparser: failed to open external command file");
	  } else {
	    fromstdin = 0;
	    fromfile  = 1;
	  }
	} else {
	  /* Token with file name is null */
	  ccperror(2,"cparser: failed to get name for external file");
	}
	/* Blank the line and reset the number of tokens
	   to force reading from the external file immediately */
	line[0] = '\0';
	ntok    = 0;
	parser->ntokens = ntok;

      /* Check last token to see if it is continuation
	 character */
      } else if (ntok > 0 &&
	  (strmatch("&",tokenarray[ntok-1].word) ||
	   strmatch("\\",tokenarray[ntok-1].word) ||
	   strmatch("-",tokenarray[ntok-1].word))) {
	if (diag) printf("CPARSER: Detected continuation character\n");
	/* It's a continuation mark 
	   Set flag to indicate this fact in later rounds */
	continuation = 1;
	/* Truncate the line to remove the continuation
	   character */
	if (ntok > 1)
	  trunc = tokenarray[ntok-1].ibeg;
	else
	  trunc = 0;
	if (diag) printf("CPARSER: Continuation character should be at position %d\n\"%c\" is the character at this position\n",trunc,line[trunc]);
	line[trunc] = '\0';
	/* Lose the last token */ 
	ntok--;
	parser->ntokens = ntok;
      } else {
	/* Not a continuation character */
	continuation = 0;
      }
      
    } else {
      /* Didn't get any more tokens from the last pass
	 Check if it is a blank line or comment line */
      if (ntok == 0)
	/* Echo comment line to stdout and blank
	   the line */
	if (strlen(line) > 0) {
	  if (print) printf("Comment line--- %s\n",line);
	  line[0] = '\0';
	}
    }
    if (diag) printf("CPARSER: Continuation = %d\n",continuation);
  }

  /* Fetch and uppercase keyword */
    if (ntok > 0) {
      strtoupper(parser->keyword,tokenarray[0].word);
      if (diag) printf("CPARSER: Keyword is %s\n",parser->keyword);
      /*Echo the line to standard output */ 
      if (print) printf("Data line--- %s\n",line); 
    } else {
      parser->keyword[0] = '\0';
    }

  /* Update the returned variables */
  parser->fp = filein;
  
  if (diag) printf("CPARSER: Returning from cparser\n");
  return ntok;
}

/*------------------------------------------------------------------*/

/* keymatch

   Returns 1 if keywords keyin1 and keyin2 are "identical", 0 otherwise.

   Keywords are identical if they are the same up to the first four
   characters, independent of case.
*/
int keymatch(const char *keyin1, const char *keyin2)
{
  int  len1,len2;
  char key1[5],key2[5],keyup1[5],keyup2[5];

  /* Initial check */
  if (!keyin1 || !keyin2) return 0;

  /* Compare truncated lengths */
  len1 = strlen(keyin1);
  if (len1 > 4) len1 = 4;
 
  len2 = strlen(keyin2);
  if (len2 > 4) len2 = 4;

  /* If lengths don't match then keywords can't be identical */
  if (len1 != len2) return 0;

  /* If supplied words are longer than four characters then
     truncate them after the fourth character */
  strncpy(key1,keyin1,len1);
  key1[len1] = '\0';

  strncpy(key2,keyin2,4);
  key2[len2] = '\0';

  /* Convert strings to uppercase */
  strtoupper(keyup1,key1);
  strtoupper(keyup2,key2);

  /* Compare using strmatch */
  return strmatch(keyup1,keyup2);
}

/*------------------------------------------------------------------*/

/* strtoupper

   Convert string to uppercase.
   
   On exit str1 will contain uppercased copy of str2
*/
int strtoupper (char *str1, char *str2)
{
  int len2,i;

  if (!str2) {
    str1 = NULL;
    return 0;
  }
  
  len2 = strlen(str2);
  if (len2 > 0)
    for (i=0; i<len2 ; i++) str1[i] = toupper(str2[i]);
  str1[len2] = '\0';

  return 0;
}

/*------------------------------------------------------------------*/

/* strmatch

   Compare two strings.

   Returns 1 if strings are identical, 0 if they differ.
*/
int strmatch (char *str1, char *str2)
{
  int len1,len2,i;

  /* Don't process null strings */
  if (!str1 || !str2) return 0;

  len1 = strlen(str1);
  len2 = strlen(str2);

  /* Cannot be identical if lengths differ */
  if (len1 != len2) return 0;

  /* Check character by character
     If any character differs then strings cannot be identical */
  for (i=0; i<len1; i++)
    if (str1[i] != str2[i]) return 0;

  /* Must be a match */
  return 1;
}

/*------------------------------------------------------------------*/

/* charmatch

   Returns 1 if character matches one of the characters in the string,
   0 otherwise.
*/
int charmatch(char character, char *charlist)
{
  int jdo,ismatch=0;

  if (charlist) {
    jdo = 0;
    while (jdo<strlen(charlist) && !ismatch) {
      if (charlist[jdo] == character) ismatch = 1;
      jdo++;
    }
  }

  return ismatch;
}
