//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  sel_exmp4.cpp
//
//  Example:
//     Selection of atoms in a sphere. The parameters are chosen
//  such that the final selection includes the same atoms as
//  those contacting in sel_exmp1.cpp .
//
//  08 Mar 2001
//
//  SGI make:  f77 -o sel_exmp4 sel_exmp4.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Selecting atoms in a sphere.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "sel_exmp4 coor_file [sel_coor_file]\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format, [] means an optional parameter.\n"
    "\n"
    "If 'sel_coor_file' is given, it will receive selected\n"
    "atoms in PDB format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager1,MMDBManager2;
int            RC,lcount;
char           S[500];
int            selHnd,nSelAtoms,i;
PPCAtom        SelAtom;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  MMDBManager1 = NULL;  // all pointers should be either NULL or
  MMDBManager2 = NULL;  // allocated

  //  2.  Create an instance of MMDB
  MMDBManager1 = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager1->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager1->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible reading errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager1->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager1;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager1->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Select atoms.
  //    4.1 Each _new_ selection starts with creation of selection
  //        handle (a handle may be used in several selections)
  selHnd = MMDBManager1->NewSelection();

  //    4.2 Select atoms as needed; first select C_alpha atoms.
  //        Check the selection function and its parameters in file
  //        mmdb_selmngr.h .
  MMDBManager1->SelectAtoms ( selHnd, 0,"A",ANY_RES,"*",ANY_RES,"*",
                              "*","CA","C","*" );

  //        Retrieve the index of 1st selection:
  MMDBManager1->GetSelIndex ( selHnd, SelAtom,nSelAtoms );

  //        Now select only those of already selected which are
  //        found within a 10-angstrom sphere centered at 34th
  //        C_alpha atom. Check the selection function and its
  //        parameters in file mmdb_selmngr.h .
  if (nSelAtoms<34)  {
    printf ( " Chain 'A' is shorter than 34 residues.\n" );
    delete MMDBManager1;
    return 2;
  }
  MMDBManager1->SelectSphere ( selHnd,STYPE_ATOM, SelAtom[33]->x,
                               SelAtom[33]->y, SelAtom[33]->z,
                               10.0, SKEY_AND  );

  //    4.3 Selected atoms may be accessed through the selection
  //        index. Selection index is merely a vector of pointers
  //        on the selected atoms. Check the function and its
  //        parameters in file  mmdb_selmngr.h
  MMDBManager1->GetSelIndex ( selHnd, SelAtom,nSelAtoms );
  printf ( "     %i CA atoms selected in the sphere.\n",nSelAtoms );

  //        print the selection.
  for (i=0;i<nSelAtoms;i++)
    printf ( " %s   %10.4f %10.4f %10.4f\n",SelAtom[i]->GetAtomID(S),
             SelAtom[i]->x,SelAtom[i]->y,SelAtom[i]->z );

  if (argc>2)  {

    //  5. An output file for selected atoms was specified.
    //     Collect selected atoms in a separate instance of MMDB
    //     and then purge them into the file.

    //    5.1  Create one more instance of MMDB
    MMDBManager2 = new CMMDBManager();

    //    5.2  It looks like a good idea to copy the header and
    //         crystallographic information into output file
    //         as well. However this is not needed by MMDB to
    //         function. Check the copy function and its parameters
    //         in file  mmdb_manager.h .
    MMDBManager2->Copy ( MMDBManager1,MMDBFCM_Title | MMDBFCM_Cryst );

    //    5.3  Stuffing an empty MMDB with atoms. Check the
    //         corresponding function and its parameters in file
    //         mmdb_file.h .
    for (i=0;i<nSelAtoms;i++)
      MMDBManager2->PutAtom ( i+1,SelAtom[i] );

    //    5.4  Purge the newly composed MMDB into the file
    printf ( " ...writing PDB file %s\n",argv[2] );
    RC = MMDBManager2->WritePDBASCII ( argv[2] );

  }

  //  6. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection index SelAtom.
  //     It will be disposed by its manager automatically,
  //     therefore an attempt to dispose it from the application
  //     will cause a crash.
  if (MMDBManager1)  delete MMDBManager1;
  if (MMDBManager2)  delete MMDBManager2;

  return 0;

}

