# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------


# load libs
import numpy

# load configuration
import config

# register essential objects
import blocks
import objects


# COMMON MASS RELATED FUNCTIONS
# -----------------------------

def delta(measuredMass, countedMass, units='ppm'):
    """Calculate error between measuredMass and countedMass in specified units.
        measuredMass: (float)
        countedMass: (float)
        units: ('Da' or 'ppm')
    """
    
    if units == 'ppm':
        return (measuredMass - countedMass)/countedMass*1000000
    elif units == 'Da':
        return (measuredMass - countedMass)
    elif units == '%':
        return (measuredMass - countedMass)/countedMass*100
    else:
        raise ValueError, 'Unknown units for delta mass! -->' + units
# ----


def mz(mass, charge, currentCharge=0, agentFormula='H', agentCharge=1, massType='mo'):
    """Calculate m/z value for given mass and charge.
        mass: (tuple of (Mo,Av) or float)
        charge: (int) desired charge of ion
        currentCharge: (int) if mass is charged already
        agentFormula: (str or formula) charging agent formula
        agentCharge: (int) charging agent charge
        massType: ('mo' or 'av') used mass type if mass value is float
    """
    
    electron = 0.00054857990924
    
    # set mass type
    if massType == 'mo':
        massType = 0
    elif massType == 'av':
        massType = 1
    
    # check agent formula
    if agentFormula != 'e' and not isinstance(agentFormula, objects.compound):
        agentFormula = objects.compound(agentFormula)
    
    # get agent mass
    if agentFormula == 'e':
        agentMass = [electron, electron]
    else:
        agentMass = agentFormula.mass()
        agentMass = (agentMass[0]-agentCharge*electron, agentMass[1]-agentCharge*electron)
    
    # recalculate zero charge
    agentCount = currentCharge/agentCharge
    if currentCharge != 0:
        if type(mass) in (tuple, list):
            massMo = mass[0]*abs(currentCharge) - agentMass[0]*agentCount
            massAv = mass[1]*abs(currentCharge) - agentMass[1]*agentCount
            mass = (massMo, massAv)
        else:
            mass = mass*abs(currentCharge) - agentMass[massType]*agentCount
    if charge == 0:
        return mass
    
    # calculate final charge
    agentCount = charge/agentCharge
    if type(mass) in (tuple, list):
        massMo = (mass[0] + agentMass[0]*agentCount)/abs(charge)
        massAv = (mass[1] + agentMass[1]*agentCount)/abs(charge)
        return (massMo, massAv)
    else:
        return (mass + agentMass[massType]*agentCount)/abs(charge)
# ----



# ISOTOPIC PATTERN CALCULATIONS
# -----------------------------

def pattern(compound, fwhm=0.1, relIntThreshold=0.01, charge=1, agentFormula='H', agentCharge=1, rounding=7):
    """Calculate isotopic pattern for given compound.
        fwhm: (float) gaussian peak width
        charge: (int) charge to be calculated
        agentFormula: (str or formula) charging agent formula
        agentCharge: (int) charging agent charge
        massType: ('mo' or 'av') used mass type if mass value is float
    """
    
    electron = 0.00054857990924
    
    finalPattern = []
    
    # set internal threshold
    internalThreshold = relIntThreshold/100.
    
    # check agent formula
    if agentFormula != 'e' and not isinstance(agentFormula, objects.compound):
        agentFormula = objects.compound(agentFormula)
    
    # add charging agent to compound
    if charge and agentFormula != 'e':
        formula = compound.formula()
        for atom, count in agentFormula.composition().items():
            formula += '%s%d' % (atom, count*(charge/agentCharge))
        compound = objects.compound(formula)
    
    # get composition
    composition = compound.composition()
    
    # check composition
    for atom in composition:
        if composition[atom] < 0:
            raise ValueError, 'Pattern cannot be calculated for this formula! --> ' + compound.formula()
    
    # calculate pattern
    for atom in composition:
        atomCount = composition[atom]
        
        # get isotopic profile for current atom or specified isotope only
        atomPattern = []
        match = objects.elementPattern.match(atom)
        symbol, massNumber, tmp = match.groups()
        if massNumber:
            isotope = blocks.elements[symbol].isotopes[massNumber]
            atomPattern.append([isotope[0], 1.])
        else:
            for massNumber, isotope in blocks.elements[atom].isotopes.items():
                if isotope[1] > 0.:
                    atomPattern.append(list(isotope))
        
        # add atoms
        for i in range(atomCount):
            currentPattern = {}
            
            # if pattern is empty (first atom) add current atom pattern
            if not finalPattern:
                finalPattern = atomPattern
                finalPattern.sort()
                continue
            
            # add atom to each peak of final pattern
            for peak in finalPattern:
                
                # skip peak under relevant abundance threshold
                if peak[1] < internalThreshold:
                    continue
                
                # add each isotope of current atom to peak
                for isotope in atomPattern:
                    mass = peak[0] + isotope[0]
                    abundance = peak[1] * isotope[1]
                    
                    # add abundance to stored peak or add new peak
                    mass = round(mass, rounding)
                    if mass in currentPattern:
                        currentPattern[mass] += abundance
                    else:
                        currentPattern[mass] = abundance
            
            # replace final pattern by current
            finalPattern = []
            for mass, abundance in currentPattern.items():
                finalPattern.append([mass, abundance])
            finalPattern.sort()
            
            # noramlize pattern
            finalPattern = _normalize(finalPattern)
    
    # check pattern
    if not finalPattern:
        return None
    
    # correct charge
    if charge:
        for i in range(len(finalPattern)):
            finalPattern[i][0] = (finalPattern[i][0]-electron*charge)/abs(charge)
    
    # collect isotopes according to resolution (FWHM)
    tol = fwhm/2
    if charge:
        tol /= abs(charge)
    collectedPeaks = [finalPattern[0]]
    lastPeak = finalPattern[0]
    for currentPeak in finalPattern[1:]:
        if (lastPeak[0] + tol) > currentPeak[0]:
            abundance = lastPeak[1] + currentPeak[1]
            mass = (lastPeak[0]*lastPeak[1] + currentPeak[0]*currentPeak[1]) / abundance
            collectedPeaks[-1] = [mass, abundance]
            lastPeak = [mass, abundance]
        else:
            collectedPeaks.append(currentPeak)
            lastPeak = currentPeak
    finalPattern = _normalize(collectedPeaks)
    
    # discard peaks below threshold
    filteredPeaks = []
    for peak in finalPattern:
        if peak[1] >= relIntThreshold:
            filteredPeaks.append(peak)
    finalPattern = filteredPeaks
    
    return finalPattern
# ----


def profile(pattern, fwhm=0.1):
    """Make profile spectrum for given isotopic pattern."""
    
    # make raster
    minX = numpy.minimum.reduce(pattern)[0] - 5*fwhm
    maxX = numpy.maximum.reduce(pattern)[0] + 5*fwhm
    raster = numpy.arange(minX,maxX, fwhm/10.)
    points = numpy.zeros(raster.size, float)
    
    # calc gaussian peak for each isotope
    width = fwhm/1.66
    for isotope in pattern:
        lo = isotope[0]-5*fwhm
        hi = isotope[0]+5*fwhm
        for x in range(raster.size):
            if raster[x] < lo or raster[x] > hi:
                continue
            points[x] += isotope[1]*numpy.exp(-1*(pow(raster[x]-isotope[0],2))/pow(width,2))
    
    # make final profile
    finalProfile = numpy.array(zip(raster, points))
    
    return finalProfile
# ----


def gaussian(mz, intensity, fwhm=0.1, steps=500):
    """Make Gaussian peak for given mz, intensity and fwhm."""
    
    data=[]
    
    minX = mz-(5*fwhm)
    maxX = mz+(5*fwhm)
    step = (maxX-minX)/steps
    width = fwhm/1.66
    x = minX
    for i in range(steps):
        y = intensity*numpy.exp(-1*(pow(x-mz,2))/pow(width,2))
        data.append([x,y])
        x += step
    
    data = numpy.array(data)
    
    return data
# ----


def _normalize(data):
    """Normalize data to 100%."""
    
    # get maximum Y
    maximum = data[0][1]
    for item in data:
        if item[1] > maximum:
            maximum = item[1]
    
    # normalize data data
    for x in range(len(data)):
        data[x][1] /= maximum
    
    return data
# ----


