# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import wx
import wx.grid

# load modules
import config
import mwx
import images
import mspy


# FLOATING PANEL WITH COMPARE PEAKLISTS TOOL
# ------------------------------------------

class panelCompare(wx.MiniFrame):
    """Compare peaklists tool."""
    
    def __init__(self, parent):
        wx.MiniFrame.__init__(self, parent, -1, 'Compare Peak Lists', size=(500, 400), style=wx.DEFAULT_FRAME_STYLE)
        
        self.parent = parent
        self.currentDocuments = []
        self.currentPeaklists = []
        self.currentMatches = []
        
        # make gui items
        self.makeGUI()
        wx.EVT_CLOSE(self, self.onClose)
    # ----
    
    
    def makeGUI(self):
        """Make gui notebook."""
        
        # make toolbar
        toolbar = self.makeToolbar()
        
        # make panel
        mainPanel = self.makeMainPanel()
        
        # pack element
        self.mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.mainSizer.Add(toolbar, 0, wx.EXPAND, 0)
        self.mainSizer.Add(mainPanel, 1, wx.EXPAND, 0)
        
        # fit layout
        self.SetMinSize((100,100))
        self.Layout()
        self.mainSizer.Fit(self)
        self.SetSizer(self.mainSizer)
        self.SetMinSize(self.GetSize())
    # ----
    
    
    def makeToolbar(self):
        """Make toolbar."""
        
        # init toolbar
        panel = mwx.bgrPanel(self, -1, images.lib['bgrToolbarNoBorder'], size=(-1, mwx.TOOLBAR_HEIGHT))
        
        # make elements
        compare_label = wx.StaticText(panel, -1, "Compare:")
        compare_label.SetFont(wx.SMALL_FONT)
        
        choices = ['Peak lists', 'Notations (measured)', 'Notations (theoretical)']
        self.compare_combo = wx.ComboBox(panel, -1, choices=choices, size=(180, mwx.SMALL_COMBO_HEIGHT), style=wx.CB_READONLY)
        self.compare_combo.Bind(wx.EVT_COMBOBOX, self.onCompareChanged)
        self.compare_combo.Select(0)
        if config.compare['compare'] == 'measured':
            self.compare_combo.Select(1)
        elif config.compare['compare'] == 'theoretical':
            self.compare_combo.Select(2)
        
        tolerance_label = wx.StaticText(panel, -1, "Tolerance:")
        tolerance_label.SetFont(wx.SMALL_FONT)
        
        self.tolerance_value = wx.TextCtrl(panel, -1, str(config.compare['tolerance']), size=(60, -1), style=wx.TE_PROCESS_ENTER, validator=mwx.validator('floatPos'))
        self.tolerance_value.Bind(wx.EVT_TEXT_ENTER, self.onCompare)
        
        self.unitsDa_radio = wx.RadioButton(panel, -1, "Da", style=wx.RB_GROUP)
        self.unitsDa_radio.SetFont(wx.SMALL_FONT)
        self.unitsDa_radio.SetValue(True)
        
        self.unitsPpm_radio = wx.RadioButton(panel, -1, "ppm")
        self.unitsPpm_radio.SetFont(wx.SMALL_FONT)
        self.unitsPpm_radio.SetValue((config.compare['units'] == 'ppm'))
        
        self.ignoreCharge_check = wx.CheckBox(panel, -1, "Ignore charge")
        self.ignoreCharge_check.SetFont(wx.SMALL_FONT)
        self.ignoreCharge_check.SetValue(config.compare['ignoreCharge'])
        
        self.compare_butt = wx.Button(panel, -1, "Compare", size=(-1, mwx.SMALL_BUTTON_HEIGHT))
        self.compare_butt.SetFont(wx.SMALL_FONT)
        self.compare_butt.Bind(wx.EVT_BUTTON, self.onCompare)
        
        # pack elements
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(mwx.CONTROLBAR_LSPACE)
        sizer.Add(compare_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.compare_combo, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(tolerance_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.tolerance_value, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(10)
        sizer.Add(self.unitsDa_radio, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        sizer.Add(self.unitsPpm_radio, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(20)
        sizer.Add(self.ignoreCharge_check, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddStretchSpacer()
        sizer.AddSpacer(20)
        sizer.Add(self.compare_butt, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(mwx.CONTROLBAR_RSPACE)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(sizer, 1, wx.EXPAND)
        
        panel.SetSizer(mainSizer)
        mainSizer.Fit(panel)
        
        return panel
    # ----
    
    
    def makeMainPanel(self):
        """Make main panel."""
        
        panel = wx.Panel(self, -1)
        
        # make grids
        self.makeDocumentsGrid(panel)
        self.makeMatchesGrid(panel)
        
        # pack main
        mainSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer.Add(self.documentsGrid, 1, wx.EXPAND)
        mainSizer.AddSpacer(mwx.SASH_SIZE)
        mainSizer.Add(self.matchesGrid, 0, wx.EXPAND)
        
        # fit layout
        panel.SetSizer(mainSizer)
        
        return panel
    # ----
    
    
    def makeDocumentsGrid(self, panel):
        """Make documents grid."""
        
        # make table
        self.documentsGrid = wx.grid.Grid(panel, -1, size=(560, 400), style=mwx.GRID_STYLE)
        self.documentsGrid.CreateGrid(0, 0)
        self.documentsGrid.DisableDragColSize()
        self.documentsGrid.DisableDragRowSize()
        self.documentsGrid.SetColLabelSize(19)
        self.documentsGrid.SetRowLabelSize(0)
        self.documentsGrid.SetDefaultRowSize(19)
        self.documentsGrid.AutoSizeColumns(True)
        self.documentsGrid.SetLabelFont(wx.SMALL_FONT)
        self.documentsGrid.SetDefaultCellFont(wx.SMALL_FONT)
        self.documentsGrid.SetDefaultCellAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
        self.documentsGrid.SetDefaultCellBackgroundColour(wx.WHITE)
        
        self.documentsGrid.Bind(wx.grid.EVT_GRID_SELECT_CELL, self.onDocumentsCellSelected)
        self.documentsGrid.Bind(wx.EVT_KEY_DOWN, self.onDocumentsKey)
    # ----
    
    
    def makeMatchesGrid(self, panel):
        """Make matches grid."""
        
        # make table
        self.matchesGrid = wx.grid.Grid(panel, -1, size=(225, 400), style=mwx.GRID_STYLE)
        self.matchesGrid.CreateGrid(0, 0)
        self.matchesGrid.DisableDragColSize()
        self.matchesGrid.DisableDragRowSize()
        self.matchesGrid.SetColLabelSize(19)
        self.matchesGrid.SetRowLabelSize(0)
        self.matchesGrid.SetDefaultRowSize(19)
        self.matchesGrid.AutoSizeColumns(True)
        self.matchesGrid.SetLabelFont(wx.SMALL_FONT)
        self.matchesGrid.SetDefaultCellFont(wx.SMALL_FONT)
        self.matchesGrid.SetDefaultCellAlignment(wx.ALIGN_RIGHT, wx.ALIGN_TOP)
        self.matchesGrid.SetDefaultCellBackgroundColour(wx.WHITE)
        
        self.matchesGrid.Bind(wx.EVT_KEY_DOWN, self.onMatchesKey)
    # ----
    
    
    def onClose(self, evt):
        """Destroy this frame."""
        self.Destroy()
    # ----
    
    
    def onCompare(self, evt):
        """Compare data."""
        
        # check documents
        if not self.currentDocuments:
            wx.Bell()
            return
        
        # get params
        if not self.getParams():
            wx.Bell()
            return
        
        # compare
        self.compareAll()
        
        # update gui
        self.updateDocumentsGrid()
        self.updateMatchesGrid()
    # ----
    
    
    def onCompareChanged(self, evt):
        """Get relevant peaks lists."""
        
        # get data type
        value = self.compare_combo.GetValue()
        if value == 'Notations (measured)':
            config.compare['compare'] = 'measured'
        elif value == 'Notations (theoretical)':
            config.compare['compare'] = 'theoretical'
        else:
            config.compare['compare'] = 'peaklists'
        
        # get peaklists
        self.getPeaklists()
        
        # update gui
        self.updateDocumentsGrid()
        self.updateMatchesGrid()
    # ----
    
    
    def onDocumentsCellSelected(self, evt):
        """Show more info for selected cell."""
        
        # check documents
        if not self.currentDocuments:
            return
        
        # get slection
        col = evt.GetCol()
        row = evt.GetRow()
        docIndex = int(col) / int((len(self.currentDocuments)+1))
        
        # check position
        if col % (len(self.currentDocuments)+1) != 0:
            return
        elif row >= len(self.currentPeaklists[docIndex]):
            return
        else:
            evt.Skip()
        
        # highlight selected cell
        self.documentsGrid.SelectBlock(row, col, row, col)
        
        # highlight mass in plot
        self.parent.updateMassPoints([self.currentPeaklists[docIndex][row][0]])
        
        # compare selected mass
        self.compareSelected(docIndex, row)
        self.updateMatchesGrid()
    # ----
    
    
    def onDocumentsKey(self, evt):
        """Key pressed."""
        
        # get key
        key = evt.GetKeyCode()
        
        # copy
        if key == 67 and evt.CmdDown():
            self.copyDocuments()
        
        # other keys
        else:
            evt.Skip()
    # ----
    
    
    def onMatchesKey(self, evt):
        """Key pressed."""
        
        # get key
        key = evt.GetKeyCode()
        
        # copy
        if key == 67 and evt.CmdDown():
            self.copyMatches()
        
        # other keys
        else:
            evt.Skip()
    # ----
    
    
    def setData(self, documents):
        """Set data."""
        
        self.currentDocuments = []
        self.currentPeaklists = []
        self.currentMatches = []
        
        # get visible documents only
        for document in documents:
            if document.visible:
                self.currentDocuments.append(document)
        
        # get peaklists
        self.getPeaklists()
        
        # update gui
        self.updateDocumentsGrid()
        self.updateMatchesGrid()
    # ----
    
    
    def getParams(self):
        """Get all params from dialog."""
        
        # try to get values
        try:
            
            config.compare['tolerance'] = float(self.tolerance_value.GetValue())
            
            if self.unitsDa_radio.GetValue():
                config.compare['units'] = 'Da'
            else:
                config.compare['units'] = 'ppm'
            
            if self.ignoreCharge_check.GetValue():
                config.compare['ignoreCharge'] = 1
            else:
                config.compare['ignoreCharge'] = 0
            
            return True
            
        except:
            wx.Bell()
            return False
    # ----
    
    
    def getPeaklists(self):
        """Filter peaklists according to specified type."""
        
        # empty current data
        self.currentPeaklists = []
        self.currentMatches = []
        
        # get peaklist
        count = len(self.currentDocuments)
        for x, document in enumerate(self.currentDocuments):
            self.currentPeaklists.append([])
            
            # use measured notations
            if config.compare['compare'] == 'measured':
                items = []
                for item in document.annotations:
                    items.append(item)
                for sequence in document.sequences:
                    for item in sequence.matches:
                        items.append(item)
                for item in items:
                    self.currentPeaklists[-1].append([round(item.mz,6), item.charge, item.baseline-item.intensity, count*[False]])
                    self.currentPeaklists[-1][-1][-1][x] = True
            
            # use theoretical notations
            elif config.compare['compare'] == 'theoretical':
                items = []
                for item in document.annotations:
                    if item.theoretical != None:
                        items.append(item)
                for sequence in document.sequences:
                    for item in sequence.matches:
                        if item.theoretical != None:
                            items.append(item)
                for item in items:
                    self.currentPeaklists[-1].append([round(item.theoretical,6), item.charge, item.baseline-item.intensity, count*[False]])
                    self.currentPeaklists[-1][-1][-1][x] = True
            
            # use peaklists
            else:
                for item in document.spectrum.peaklist:
                    self.currentPeaklists[-1].append([round(item.mz,6), item.charge, item.baseline-item.intensity, count*[False]])
                    self.currentPeaklists[-1][-1][-1][x] = True
    # ----
    
    
    def updateDocumentsGrid(self):
        """Update current documents grid."""
        
        # erase grid
        if self.documentsGrid.GetNumberCols():
            self.documentsGrid.DeleteCols(0, self.documentsGrid.GetNumberCols())
        if self.documentsGrid.GetNumberRows():
            self.documentsGrid.DeleteRows(0, self.documentsGrid.GetNumberRows())
        
        # check peaklist
        if not self.currentPeaklists:
            return
        
        # make grid
        count = len(self.currentDocuments)
        self.documentsGrid.AppendCols(count**2+count)
        self.documentsGrid.AppendRows(max(len(d) for d in self.currentPeaklists))
        
        # set formats
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        cellAttr = wx.grid.GridCellAttr()
        cellAttr.SetReadOnly(True)
        
        # make grid
        for i in range(count):
            col = i*(count+1)
            
            # make labels
            self.documentsGrid.SetColLabelValue(col, 'm/z')
            self.documentsGrid.SetColAttr(col, cellAttr)
            for x in range(count):
                self.documentsGrid.SetColLabelValue(col+1+x, '*')
                self.documentsGrid.SetColAttr(col+1+x, cellAttr)
                self.documentsGrid.SetColSize(col+1+x, 20)
            
            # add data
            for j, item in enumerate(self.currentPeaklists[i]):
                mz = mzFormat % item[0]
                self.documentsGrid.SetCellValue(j, col, mz)
                self.documentsGrid.SetCellValue(j, col+i+1, '*')
                self.documentsGrid.SetCellAlignment(j, col+i+1, wx.ALIGN_CENTER, wx.ALIGN_CENTER)
                for x in range(count):
                    if item[-1][x]:
                        self.documentsGrid.SetCellBackgroundColour(j, col+x+1, self.currentDocuments[x].colour)
        
        self.documentsGrid.AutoSizeColumns(True)
    # ----
    
    
    def updateMatchesGrid(self):
        """Update current matches."""
        
        # erase grid
        if self.matchesGrid.GetNumberCols():
            self.matchesGrid.DeleteCols(0, self.matchesGrid.GetNumberCols())
        if self.matchesGrid.GetNumberRows():
            self.matchesGrid.DeleteRows(0, self.matchesGrid.GetNumberRows())
        
        # check matches
        if not self.currentMatches:
            return
        
        # make grid
        self.matchesGrid.AppendCols(5)
        self.matchesGrid.AppendRows(len(self.currentMatches))
        self.matchesGrid.SetColLabelValue(0, '*')
        self.matchesGrid.SetColLabelValue(1, 'm/z')
        self.matchesGrid.SetColLabelValue(2, 'error')
        self.matchesGrid.SetColLabelValue(3, 'a/b')
        self.matchesGrid.SetColLabelValue(4, 'b/a')
        cellAttr = wx.grid.GridCellAttr()
        cellAttr.SetReadOnly(True)
        for x in range(5):
            self.matchesGrid.SetColAttr(x, cellAttr)
        self.matchesGrid.SetColSize(0, 20)
        
        # set formats
        mzFormat = '%0.' + `config.main['mzDigits']` + 'f'
        errFormat = '%0.' + `config.main['mzDigits']` + 'f'
        if config.compare['units'] == 'ppm':
            errFormat = '%0.' + `config.main['ppmDigits']` + 'f'
        
        # add data
        for i, match in enumerate(self.currentMatches):
            mz = mzFormat % match[1]
            error = errFormat % match[2]
            ratio1 = '%0.2f' % match[3]
            ratio2 = '%0.2f' % match[4]
            
            self.matchesGrid.SetCellValue(i, 1, mz)
            self.matchesGrid.SetCellValue(i, 2, error)
            self.matchesGrid.SetCellValue(i, 3, ratio1)
            self.matchesGrid.SetCellValue(i, 4, ratio2)
            self.matchesGrid.SetCellBackgroundColour(i, 0, self.currentDocuments[match[0]].colour)
            if match[5] == True:
                self.matchesGrid.SetCellAlignment(i, 0, wx.ALIGN_CENTER, wx.ALIGN_CENTER)
                self.matchesGrid.SetCellValue(i, 0, '*')
        
        self.matchesGrid.AutoSizeColumns(True)
    # ----
    
    
    def compareAll(self):
        """Compare all peaklists."""
        
        self.currentMatches = []
        
        # get total peaklist and erase previous matches
        peaklist = []
        count = len(self.currentPeaklists)
        for i, pkl in enumerate(self.currentPeaklists):
            for j, pk in enumerate(pkl):
                peaklist.append([pk[0], pk[1], i, j])
                self.currentPeaklists[i][j][-1] = count*[False]
                self.currentPeaklists[i][j][-1][i] = True
        peaklist.sort()
        
        # compare peaklists
        count = len(peaklist)
        for i in range(count):
            for j in range(i, count):
                p1 = peaklist[i]
                p2 = peaklist[j]
                
                # check charge
                if not config.compare['ignoreCharge'] and (p1[1] != p2[1]) and (p1[1] != None and p2[1] != None):
                    continue
                
                # check error
                error = mspy.delta(p1[0], p2[0], config.compare['units'])
                if abs(error) <= config.compare['tolerance']:
                    self.currentPeaklists[p1[2]][p1[3]][-1][p2[2]] = True
                    self.currentPeaklists[p2[2]][p2[3]][-1][p1[2]] = True
                elif error < 0:
                    break
    # ----
    
    
    def compareSelected(self, docIndex, pkIndex):
        """Compare selected mass only."""
        
        self.currentMatches = []
        
        # get current peak
        peak = self.currentPeaklists[docIndex][pkIndex]
        p1 = [peak[0], peak[1], peak[2], docIndex, pkIndex]
        
        # get total peaklist
        peaklist = []
        for i, pkl in enumerate(self.currentPeaklists):
            if i != docIndex:
                for j, pk in enumerate(pkl):
                    peaklist.append([pk[0], pk[1], pk[2], i, j])
        peaklist.sort()
        
        # compare mass
        for p2 in peaklist:
            
            # check charge
            if not config.compare['ignoreCharge'] and (p1[1] != p2[1]) and (p1[1] != None and p2[1] != None):
                continue
            
            # check error
            error = mspy.delta(p1[0], p2[0], config.compare['units'])
            if abs(error) <= config.compare['tolerance']:
                ratio1 = p1[2]/p2[2]
                ratio2 = 1/ratio1
                self.currentMatches.append([p2[3], p2[0], error, ratio1, ratio2, p2[3]==docIndex])
            elif error < 0:
                break
        
        # sort matches by document
        self.currentMatches.sort()
    # ----
    
    
    def copyDocuments(self):
        """Copy documents table into clipboard."""
        
        # get default bgr colour
        defaultColour = self.documentsGrid.GetDefaultCellBackgroundColour()
        
        # get data
        buff = ''
        for row in range(self.documentsGrid.GetNumberRows()):
            line = ''
            for col in range(self.documentsGrid.GetNumberCols()):
                value = self.documentsGrid.GetCellValue(row, col)
                if value == '' and defaultColour != self.documentsGrid.GetCellBackgroundColour(row, col):
                    value = "x"
                line += value + '\t'
            buff += '%s\n' % (line.rstrip())
        
        # make text object for data
        obj = wx.TextDataObject()
        obj.SetText(buff.rstrip())
        
        # paste to clipboard
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(obj)
            wx.TheClipboard.Close()
    # ----
    
    
    def copyMatches(self):
        """Copy matches table into clipboard."""
        
        # get data
        buff = ''
        for row in range(self.matchesGrid.GetNumberRows()):
            line = ''
            for col in range(1, self.matchesGrid.GetNumberCols()):
                line += self.matchesGrid.GetCellValue(row, col) + '\t'
            buff += '%s\n' % (line.rstrip())
        
        # make text object for data
        obj = wx.TextDataObject()
        obj.SetText(buff.rstrip())
        
        # paste to clipboard
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(obj)
            wx.TheClipboard.Close()
    # ----
    
    

