# -------------------------------------------------------------------------
#     Copyright (C) 2005-2011 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

#load libs
import re
import numpy
import copy

# load stopper
from stopper import CHECK_FORCE_QUIT

# register essential objects
import blocks

# register essential modules
import processing


# compile basic patterns
formulaPattern = re.compile(r'''
    ^(
        ([\(])* # start parenthesis
        (
            ([A-Z][a-z]{0,2}) # atom symbol
            (\{[\d]+\})? # isotope
            (([\-][\d]+)|[\d]*) # atom count
        )+
        ([\)][\d]*)* # end parenthesis
    )*$
''', re.X)

elementPattern = re.compile(r'''
            ([A-Z][a-z]{0,2}) # atom symbol
            (?:\{([\d]+)\})? # isotope
            ([\-]?[\d]*) # atom count
''', re.X)



# BASIC OBJECTS DEFINITION
# ------------------------

class compound:
    """Compound object definition."""
    
    def __init__(self, rawFormula):
        
        # user defined formula
        self.rawFormula = rawFormula
        
        # buffers
        self._formula = None
        self._composition = None
        self._mass = None
        
        # user defined params
        self.userParams = {}
        
        # check formula
        if not formulaPattern.match(self.rawFormula):
            raise ValueError, 'Wrong formula! --> ' + self.rawFormula
        
        # check elements and isotopes
        for atom in elementPattern.findall(self.rawFormula):
            if not atom[0] in blocks.elements:
                raise ValueError, 'Unknown element in formula! --> ' + atom[0] + ' in ' + self.rawFormula
            elif atom[1] and not int(atom[1]) in blocks.elements[atom[0]].isotopes:
                raise ValueError, 'Unknown isotope in formula! --> ' + atom[0] + atom[1] + ' in ' + self.rawFormula
        
        # check brackets
        if self.rawFormula.count(')') != self.rawFormula.count('('):
            raise ValueError, 'Wrong number of brackets in formula! --> ' + self.rawFormula
    # ----
    
    
    def reset(self):
        """Clear formula buffers."""
        
        self._formula = None
        self._mass = None
        self._composition = None
    # ----
    
    
    # GETTERS
    
    def formula(self):
        """Get formula."""
        
        # check formula buffer
        if self._formula != None:
            return self._formula
        
        # get composition
        comp = self.composition()
        
        # format composition
        self._formula = ''
        for el in sorted(comp.keys()):
            if comp[el] == 1:
                self._formula += el
            else:
                self._formula += '%s%d' % (el, comp[el])
        
        return self._formula
    # ----
    
    
    def composition(self):
        """Get elemental composition as dict."""
        
        # check composition buffer
        if self._composition != None:
            return self._composition
        
        # unfold brackets
        unfoldedFormula = self._unfoldBrackets(self.rawFormula)
        
        # group same elements
        self._composition = {}
        for symbol, isotop, count in elementPattern.findall(unfoldedFormula):
            
            # make atom
            if isotop:
                atom = '%s{%s}' % (symbol, isotop)
            else:
                atom = symbol
            
            # convert counting
            if count:
                count = int(count)
            else:
                count = 1
            
            # add atom
            if atom in self._composition:
                self._composition[atom] += count
            else:
                self._composition[atom] = count
        
        # remove zeros
        for atom in self._composition.keys():
            if self._composition[atom] == 0:
                del self._composition[atom]
        
        return self._composition
    # ----
    
    
    def mass(self):
        """Get mass."""
        
        # get mass
        if self._mass == None:
            massMo = 0
            massAv = 0
            
            # get composition
            comp = self.composition()
            
            # get mass for each atom
            for atom in comp:
                count = comp[atom]
                
                # check specified isotope and mass
                match = elementPattern.match(atom)
                symbol, massNumber, tmp = match.groups()
                if massNumber:
                    isotope = blocks.elements[symbol].isotopes[int(massNumber)]
                    atomMass = (isotope[0], isotope[0])
                else:
                    atomMass = blocks.elements[symbol].mass
                
                # multiply atom
                massMo += atomMass[0]*count
                massAv += atomMass[1]*count
            
            # store mass in buffer
            self._mass = (massMo, massAv)
        
        return self._mass
    # ----
    
    
    def mz(self, charge, agentFormula='H', agentCharge=1):
        """Get ion m/z."""
        
        # get current mass and calculate mz
        return processing.mz(self.mass(), charge, agentFormula=agentFormula, agentCharge=agentCharge)
    # ----
    
    
    def pattern(self, fwhm=0.1, threshold=0.01, charge=0, agentFormula='H', agentCharge=1, rounding=7):
        """Get isotopic pattern."""
        
        return processing.pattern(self, \
            fwhm=fwhm, \
            threshold=threshold, \
            charge=charge, \
            agentFormula=agentFormula, \
            agentCharge=agentCharge, \
            rounding=rounding)
    # ----
    
    
    def isvalid(self, charge=0, agentFormula='H', agentCharge=1):
        """Utility to check ion composition."""
        
        # check agent formula
        if agentFormula != 'e' and not isinstance(agentFormula, compound):
            agentFormula = compound(agentFormula)
        
        # make ion compound
        if charge and agentFormula != 'e':
            ionFormula = self.rawFormula
            for atom, count in agentFormula.composition().items():
                ionFormula += '%s%d' % (atom, count*(charge/agentCharge))
            ion = compound(ionFormula)
        else:
            ion = compound(self.rawFormula)
        
        # get composition
        for atom, count in ion.composition().items():
            if count < 0:
                return False
        
        return True
    # ----
    
    
    # HELPERS
    
    def _unfoldBrackets(self, string):
        """Unfold formula and count each atom."""
        
        unfoldedFormula = ''
        brackets = [0,0]
        enclosedFormula = ''
        
        i = 0
        while i < len(string):
            
            # handle brackets
            if string[i] == '(':
                brackets[0] += 1
            elif string[i] == ')':
                brackets[1] += 1
            
            # part outside brackets
            if brackets == [0,0]:
                unfoldedFormula += string[i]
            
            # part within brackets
            else:
                enclosedFormula += string[i]
                
                # unfold part within brackets
                if brackets[0] == brackets[1]:
                    enclosedFormula = self._unfoldBrackets(enclosedFormula[1:-1])
                    
                    # multiply part within brackets
                    count = ''
                    while len(string)>(i+1) and string[i+1].isdigit():
                        count += string[i+1]
                        i += 1
                    if count:
                        enclosedFormula = enclosedFormula * int(count)
                    
                    # add and clear
                    unfoldedFormula += enclosedFormula
                    enclosedFormula = ''
                    brackets = [0,0]
            
            i += 1
        return unfoldedFormula
    # ----
    


class sequence:
    """Sequence object definition."""
    
    def __init__(self, chain, title='', accession='', nTermFormula='H', cTermFormula='OH', lossFormula=''):
        
        # get chain
        chain = chain.upper()
        for char in ('\t','\n','\r','\f','\v', ' ', '-', '*', '.'):
                chain = chain.replace(char, '')
        self.chain = chain.upper()
        
        self.nTermFormula = nTermFormula
        self.cTermFormula = cTermFormula
        self.modifications = [] # [[name, position=[#|symbol], state[f|v]], ] (f-fixed, v-variable)
        self.labels = [] # [[name, position=[#|symbol], state[f|v]], ] (f-fixed, v-variable)
        
        # for proteins
        self.title = title
        self.accession = accession
        self.orgName = ''
        self.pi = None
        self.score = None
        
        # for peptides
        self.userRange = []
        self.aaBefore = ''
        self.aaAfter = ''
        self.miscleavages = 0
        
        # for fragments
        self.fragmentSerie = None
        self.fragmentIndex = None
        self.fragmentFiltered = False
        self.lossFormula = lossFormula
        
        # buffers
        self._formula = None
        self._composition = None
        self._mass = None # (monoisotopic, average)
        
        # user defined params
        self.userParams = {}
        
        # check amino acids
        for amino in self.chain:
            if not amino in blocks.aminoacids:
                raise ValueError, 'Unknown amino acid in sequence! --> ' + amino
    # ----
    
    
    def __len__(self):
        """Get sequence length."""
        return len(self.chain)
    # ----
    
    
    def __getitem__(self, i):
        return self.chain[i]
    # ----
    
    
    def __getslice__(self, start, stop):
        """Get slice of the sequence."""
        
        # check slice
        if stop < start:
            raise ValueError, 'Invalid sequence slice definition!'
        
        # break the links
        parent = copy.deepcopy(self)
        
        # check slice
        start = max(start, 0)
        stop = min(stop, len(parent.chain))
        
        # make new sequence object
        seq = parent.chain[start:stop]
        peptide = sequence(seq)
        
        # add modifications
        for mod in parent.modifications:
            if type(mod[1]) == int and mod[1] >= start and mod[1] < stop:
                mod[1] -= start
                peptide.modifications.append(mod)
            elif type(mod[1]) in (str, unicode) and mod[1] in peptide.chain:
                peptide.modifications.append(mod)
        
        # add labels
        for mod in parent.labels:
            if type(mod[1]) == int and mod[1] >= start and mod[1] < stop:
                mod[1] -= start
                peptide.labels.append(mod)
            elif type(mod[1]) in (str, unicode) and mod[1] in peptide.chain:
                peptide.labels.append(mod)
        
        # set range in user coordinates
        peptide.userRange = [start+1, stop]
        
        # set adjacent amino acids
        if start > 0:
            peptide.aaBefore = parent.chain[start-1]
        if stop < len(parent.chain):
            peptide.aaAfter = parent.chain[stop]
        
        # add terminal modifications
        if start == 0:
            peptide.nTermFormula = parent.nTermFormula
        if stop >= len(parent.chain):
            peptide.cTermFormula = parent.cTermFormula
        
        # ensure buffers are cleaned
        peptide.reset()
        
        return peptide
    # ----
    
    
    def __setslice__(self, start, stop, value):
        """Insert sequence object (essential for sequence editor)."""
        
        # check slice
        if stop < start:
            raise ValueError, 'Invalid slice!'
        
        # check value
        if not isinstance(value, sequence):
            raise TypeError, 'Invalid object to instert!'
        
        # break the links
        value = copy.deepcopy(value)
        
        # delete slice
        if stop != start:
            del(self[start:stop])
        
        # insert sequence
        self.chain = self.chain[:start] + value.chain + self.chain[start:]
        
        # shift modifications
        for x, mod in enumerate(self.modifications):
            if type(mod[1]) == int and mod[1] >= start:
                self.modifications[x][1] += (len(value))
        
        # shift labels
        for x, mod in enumerate(self.labels):
            if type(mod[1]) == int and mod[1] >= start:
                self.labels[x][1] += (len(value))
        
        # insert modifications
        for mod in value.modifications:
            if type(mod[1]) == int:
                mod[1] += start
            self.modifications.append(mod)
        
        # insert labels
        for mod in value.labels:
            if type(mod[1]) == int:
                mod[1] += start
            self.labels.append(mod)
        
        # clear some values
        self.userRange = []
        self.aaBefore = ''
        self.aaAfter = ''
        self.miscleavages = 0
        
        # clear buffers
        self.reset()
    # ----
    
    
    def __delslice__(self, start, stop):
        """Delete slice of sequence (essential for sequence editor)."""
        
        # check slice
        if stop < start:
            raise ValueError, 'Invalid slice!'
        
        # remove sequence
        self.chain = self.chain[:start] + self.chain[stop:]
        
        # remove modifications
        keep = []
        for mod in self.modifications:
            if type(mod[1]) == int and (mod[1] < start or mod[1] >= stop):
                if mod[1] >= stop:
                    mod[1] -= (stop - start)
                keep.append(mod)
            elif type(mod[1]) in (str, unicode) and mod[1] in self.chain:
                keep.append(mod)
        self.modifications = keep
        
        # remove labels
        keep = []
        for mod in self.labels:
            if type(mod[1]) == int and (mod[1] < start or mod[1] >= stop):
                if mod[1] >= stop:
                    mod[1] -= (stop - start)
                keep.append(mod)
            elif type(mod[1]) in (str, unicode) and mod[1] in self.chain:
                keep.append(mod)
        self.labels = keep
        
        # clear some values
        self.userRange = []
        self.aaBefore = ''
        self.aaAfter = ''
        self.miscleavages = 0
        
        # clear buffers
        self.reset()
    #----
    
    
    def __add__(self, value):
        """Join sequences and return result (essential for sequence editor)."""
        
        # check value
        if not isinstance(value, sequence):
            raise TypeError, 'Invalid object to join with sequence!'
        
        # join sequences
        result = self[:]
        result[len(result):] = value
        
        # set C terminus
        result.cTermFormula = value.cTermFormula
        
        # set neutral loss
        result.lossFormula = value.lossFormula
        
        # clear some values
        result.userRange = []
        result.aaBefore = ''
        result.aaAfter = ''
        result.miscleavages = 0
        
        # clear buffers
        result.reset()
        
        return result
    # ----
    
    
    def __iter__(self):
        self._index = 0
        return self
    # ----
    
    
    def next(self):
        if self._index < len(self.chain):
            self._index += 1
            return self.chain[self._index-1]
        else:
            raise StopIteration
    # ----
    
    
    def reset(self):
        """Clear sequence buffers."""
        
        self._formula = None
        self._mass = None
        self._composition = None
    # ----
    
    
    # GETTERS
    
    def duplicate(self):
        """Return copy of current sequence."""
        return copy.deepcopy(self)
    # ----
    
    
    def count(self, item):
        """Count item in the chain."""
        return self.chain.count(item)
    # ----
    
    
    def formula(self):
        """Get formula."""
        
        # check formula buffer
        if self._formula != None:
            return self._formula
        
        # get composition
        comp = self.composition()
        
        # format composition
        self._formula = ''
        for el in sorted(comp.keys()):
            if comp[el] == 1:
                self._formula += el
            else:
                self._formula += '%s%d' % (el, comp[el])
        
        return self._formula
    # ----
    
    
    def composition(self):
        """Get elemental composition as dict."""
        
        # check composition buffer
        if self._composition != None:
            return self._composition
        
        self._composition = {}
        
        # add amino acids to formula
        for amino in self.chain:
            for el, count in blocks.aminoacids[amino].composition.items():
                if el in self._composition:
                    self._composition[el] += count
                else:
                    self._composition[el] = count
        
        # add modifications and labels
        mods = self.modifications + self.labels
        for name, position, state in mods:
            multi = 1
            if type(position) in (str, unicode) and position !='':
                multi = self.chain.count(position)
            for el, count in blocks.modifications[name].composition.items():
                if el in self._composition:
                    self._composition[el] += multi*count
                else:
                    self._composition[el] = multi*count
        
        # add terminal modifications
        termCmpd = compound(self.nTermFormula + self.cTermFormula)
        for el, count in termCmpd.composition().items():
            if el in self._composition:
                self._composition[el] += count
            else:
                self._composition[el] = count
        
        # subtract neutral loss for fragments
        lossCmpd = compound(self.lossFormula)
        for el, count in lossCmpd.composition().items():
            if el in self._composition:
                self._composition[el] -= count
            else:
                self._composition[el] = -1*count
        
        # remove zeros
        for atom in self._composition.keys():
            if self._composition[atom] == 0:
                del self._composition[atom]
        
        return self._composition
    # ----
    
    
    def mass(self):
        """Get mass."""
        
        # check mass buffer
        if self._mass != None:
            return self._mass
        
        # get mass
        self._mass = compound(self.formula()).mass()
        
        return self._mass
    # ----
    
    
    def mz(self, charge, agentFormula='H', agentCharge=1):
        """Get ion m/z"""
        
        # get current mass and calculate mz
        return processing.mz(self.mass(), charge, agentFormula=agentFormula, agentCharge=agentCharge)
    # ----
    
    
    def pattern(self, fwhm=0.1, threshold=0.01, charge=0, agentFormula='H', agentCharge=1, rounding=7):
        """Get isotopic pattern."""
        
        return processing.pattern(self, \
            fwhm=fwhm, \
            threshold=threshold, \
            charge=charge, \
            agentFormula=agentFormula, \
            agentCharge=agentCharge, \
            rounding=rounding)
    # ----
    
    
    def format(self, template='S [m]'):
        """Get formated sequence."""
        
        # make keys
        keys = {}
        keys['s'] = self.chain.lower()
        keys['S'] = self.chain
        keys['N'] = self.nTermFormula
        keys['C'] = self.cTermFormula
        keys['b'] = self.aaBefore.lower()
        keys['B'] = self.aaBefore
        keys['a'] = self.aaAfter.lower()
        keys['A'] = self.aaAfter
        keys['m'] = self._formatModifications(self.modifications)
        keys['M'] = self._formatModifications(self.modifications + self.labels)
        keys['l'] = self._formatModifications(self.labels)
        keys['p'] = self.miscleavages
        
        if self.userRange:
            keys['r'] = '%s-%s' % tuple(self.userRange)
        
        if self.fragmentSerie != None and self.fragmentIndex != None:
            keys['f'] = '%s %s' % (self.fragmentSerie, self.fragmentIndex)
        elif self.fragmentSerie != None:
            keys['f'] =  self.fragmentSerie
        
        # format
        buff = ''
        for char in template:
            if char in keys:
                buff += keys[char]
            else:
                buff += char
        
        # clear format
        buff = buff.replace('[]', '')
        buff = buff.replace('()', '')
        buff = buff.strip()
        
        return buff
    # ----
    
    
    def digest(self, enzyme, miscleavage=0, allowMods=False, strict=True):
        """Digest seuence by specified enzyme.
            enzyme: (str) enzyme name - must be defined in mspy.enzymes
            miscleavage: (int) number of allowed misscleavages
            allowMods: (bool) do not care about modifications in cleavage site
            strict: (bool) do not cleave even if variable modification is in cleavage site
        """
        
        return processing.digest(self,
            enzyme=enzyme, \
            miscleavage=miscleavage, \
            allowMods=allowMods, \
            strict=strict
        )
    # ----
    
    
    def fragment(self, serie, index=None):
        """Generate list of neutral peptide fragments from given peptide.
            serie: (str) fragment serie name - must be defined in mspy.fragments
            index: (int) fragment index
        """
        
        return processing.fragment(self, \
            serie=serie, \
            index=index \
        )
    # ----
    
    
    def search(self, mass, charge, tolerance, enzyme=None, semiSpecific=True, tolUnits='Da', massType='mo', maxMods=1, position=False):
        """Search sequence for specified ion.
            mass: (float) m/z value to search for
            charge: (int) charge of the m/z value
            tolerance: (float) mass tolerance
            tolUnits: ('Da', 'ppm') tolerance units
            enzyme: (str) enzyme used for peptides endings, if None H/OH is used
            semiSpecific: (bool) semispecific cleavage is checked (enzyme must be set)
            massType: ('mo' or 'av') mass type of the mass value
            maxMods: (int) maximum number of modifications at one residue
            position: (bool) retain position for variable modifications (much slower)
        """
        
        matches = []
        
        # set mass type
        if massType == 'mo':
            massType = 0
        elif massType == 'av':
            massType = 1
        
        # set terminal modifications
        if enzyme:
            enzyme = blocks.enzymes[enzyme]
            expression = re.compile(enzyme.expression+'$')
            nTerm = enzyme.nTermFormula
            cTerm = enzyme.cTermFormula
        else:
            semiSpecific = False
            nTerm = 'H'
            cTerm = 'OH'
        
        # set mass limits
        if tolUnits == 'ppm':
            lowMass = mass - (tolerance * mass/1000000)
            highMass = mass + (tolerance * mass/1000000)
        else:
            lowMass = mass - tolerance
            highMass = mass + tolerance
        
        # search sequence
        length = len(self)
        for i in range(length):
            for j in range(i+1, length+1):
                
                CHECK_FORCE_QUIT()
                
                # get peptide
                peptide = self[i:j]
                if i != 0:
                    peptide.nTerminalFormula = nTerm
                if j != length:
                    peptide.cTerminalFormula = cTerm
                
                # check enzyme specifity
                if semiSpecific and peptide.aaBefore and peptide.aaAfter:
                    if not expression.search(peptide.aaBefore+peptide.chain[0]) and not expression.search(peptide.chain[-1]+peptide.aaAfter):
                        continue
                
                # variate modifications
                variants = peptide.variations(maxMods=maxMods, position=position)
                
                # check mass limits
                peptides = []
                masses = []
                for pep in variants:
                    pepMZ = pep.mz(charge)[massType]
                    peptides.append((pepMZ, pep))
                    masses.append(pepMZ)
                if max(masses) < lowMass:
                    continue
                elif min(masses) > highMass:
                    break
                
                # search for matches
                for pep in peptides:
                    if lowMass <= pep[0] <= highMass:
                        matches.append(pep[1])
        
        return matches
    # ----
    
    
    def variations(self, maxMods=1, position=True, enzyme=None):
        """Calculate all possible combinations of variable modifications.
            maxMods: (int) maximum modifications allowed per one residue
            position: (bool) retain modifications positions (much slower)
            enzyme: (str) enzyme name to ensure that modifications are not presented in cleavage site
        """
        
        variablePeptides = []
        
        # get modifications
        fixedMods = []
        variableMods = []
        for mod in self.modifications:
            if mod[2] == 'f':
                fixedMods.append(mod)
            elif type(mod[1]) == int:
                variableMods.append(mod)
            else:
                if not position:
                    variableMods += [mod] * self.chain.count(mod[1])
                else:
                    for x, amino in enumerate(self.chain):
                        if amino == mod[1]:
                            variableMods.append([mod[0], x, 'v'])
        
        # make combinations of variable modifications
        variableMods = self._countUniqueModifications(variableMods)
        combinations = []
        for x in self._uniqueCombinations(variableMods):
            combinations.append(x)
        
        # disable positions occupied by fixed modifications
        occupied = []
        for mod in fixedMods:
            count = max(1, self.chain.count(str(mod[1])))
            occupied += [mod[1]]*count
        
        # disable modifications at cleavage sites
        if enzyme:
            enz = blocks.enzymes[enzyme]
            if not enz.modsBefore and self.aaAfter:
                occupied += [len(self.chain)-1]*maxMods
            if not enz.modsAfter and self.aaBefore:
                occupied += [0]*maxMods
        
        CHECK_FORCE_QUIT()
        
        # filter modifications
        buff = []
        for combination in combinations:
            positions = occupied[:]
            for mod in combination:
                positions += [mod[0][1]]*mod[1]
            if self._checkModifications(positions, self.chain, maxMods):
                buff.append(combination)
        combinations = buff
        
        CHECK_FORCE_QUIT()
        
        # format modifications and filter same
        buff = []
        for combination in combinations:
            mods = []
            for mod in combination:
                if position:
                    mods += [[mod[0][0], mod[0][1], 'f']]*mod[1]
                else:
                    mods += [[mod[0][0],'','f']]*mod[1]
            mods.sort()
            if not mods in buff:
                buff.append(mods)
        combinations = buff
        
        # make new peptides
        for combination in combinations:
            
            CHECK_FORCE_QUIT()
            
            variablePeptide = copy.deepcopy(self)
            variablePeptide.modifications[:] = fixedMods+combination
            variablePeptide.reset()
            
            if variablePeptide.isvalid():
                variablePeptides.append(variablePeptide)
        
        return variablePeptides
    # ----
    
    
    def ismodified(self, position=None, strict=False):
        """Check if selected amino acid or whole sequence has any modification.
            position: (int) amino acid index
            strict: (bool) check variable modifications
        """
        
        # check specified position only
        if position != None:
            for mod in self.modifications:
                if (strict or mod[2]=='f') and (mod[1] == position or mod[1] == self.chain[position]):
                    return True
        
        # check whole sequence
        else:
            for mod in self.modifications:
                if strict or mod[2]=='f':
                    return True
        
        # not modified
        return False
    # ----
    
    
    def isvalid(self, charge=0, agentFormula='H', agentCharge=1):
        """Utility to check ion composition."""
        
        formula = compound(self.formula())
        return formula.isvalid(charge=charge, agentFormula=agentFormula, agentCharge=agentCharge)
    # ----
    
    
    # MODIFIERS
    
    def modify(self, name, position, state='f'):
        """Apply modification to sequence."""
        
        # check modification
        if not name in blocks.modifications:
            raise KeyError, 'Unknown modification! --> ' + name
        
        # check position
        try: position = int(position)
        except: position = str(position)
        if type(position) == str and self.chain.count(position) == 0:
            return False
        
        # add modification
        self.modifications.append([name, position, str(state)])
        
        # clear buffers
        self.reset()
        
        return True
    # ----
    
    
    def unmodify(self, name=None, position=None, state='f'):
        """Remove modification from sequence."""
        
        # remove all modifications
        if name == None:
            del self.modifications[:]
        
        # remove modification
        else:
            try: mod = [name, int(position), str(state)]
            except: mod = [name, str(position), str(state)]
            while mod in self.modifications:
                i = self.modifications.index(mod)
                del self.modifications[i]
        
        # clear buffers
        self.reset()
    # ----
    
    
    def label(self, name, position, state='f'):
        """Apply label modification to sequence."""
        
        # check modification
        if not name in blocks.modifications:
            raise KeyError, 'Unknown modification! --> ' + name
        
        # check position
        try: position = int(position)
        except: position = str(position)
        if type(position) == str and self.chain.count(position) == 0:
            return False
        
        # add label
        self.labels.append([name, position, state])
        
        # clear buffers
        self.reset()
        
        return True
    # ----
    
    
    # HELPERS
    
    def _formatModifications(self, modifications):
        """Format modifications."""
        
        # get modifications
        modifs = {}
        for mod in modifications:
            
            # count modification
            if mod[1] == '' or type(mod[1]) == int:
                count = 1
            elif type(mod[1]) in (str, unicode):
                count = self.chain.count(mod[1])
            
            # add modification to dic
            if count and mod[0] in modifs:
                modifs[mod[0]] += count
            elif count:
                modifs[mod[0]] = count
        
        # format modifications
        if modifs:
            mods = ''
            for mod in sorted(modifs.keys()):
                mods += '%sx%s; ' % (modifs[mod], mod)
            return '%s' % mods[:-2]
        else:
            return ''
    # ----
    
    
    def _uniqueCombinations(self, items):
        """Generate unique combinations of items."""
        
        for i in range(len(items)):
            for cc in self._uniqueCombinations(items[i+1:]):
                for j in range(items[i][1]):
                    yield [[items[i][0],items[i][1]-j]] + cc
        yield []
    # ----
    
    
    def _countUniqueModifications(self, modifications):
        """Get list of unique modifications with counter."""
        
        uniqueMods = []
        modsCount = []
        for mod in modifications:
            if mod in uniqueMods:
                modsCount[uniqueMods.index(mod)] +=1
            else:
                uniqueMods.append(mod)
                modsCount.append(1)
        
        modsList = []
        for x, mod in enumerate(uniqueMods):
            modsList.append([mod, modsCount[x]])
        
        return modsList
    # ----
    
    
    def _checkModifications(self, positions, chain, maxMods):
        """Check if current modification set is applicable."""
        
        for x in positions:
            count = positions.count(x)
            if type(x) == int:
                if count > maxMods:
                    return False
            elif type(x) in (str, unicode):
                available = chain.count(x)
                for y in positions:
                    if type(y) == int and chain[y] == x:
                        available -= 1
                if count > (available * maxMods):
                    return False
        
        return True
    # ----
    
    


class peak:
    """Peak object definition"""
    
    def __init__(self, mz, ai=0., base=0., sn=None, charge=None, isotope=None, fwhm=None):
        
        self.mz = float(mz)
        self.ai = float(ai)
        self.base = float(base)
        self.sn = sn
        self.charge = charge
        self.isotope = isotope
        self.fwhm = fwhm
        self.relIntensity = 1.
        self.childScanNumber = None
        
        # buffers
        self._mass = None # neutral mass
        
        # user defined params
        self.userParams = {}
        
        # clear buffers and set intensity and resolutio
        self.reset()
    # ----
    
    
    def reset(self):
        """Clear peak buffers and set intensity and resolution."""
        
        # clear mass buffer
        self._mass = None
        
        # set intensity
        self.intensity = self.ai - self.base
        
        # set resolution
        self.resolution = None
        if self.fwhm:
            self.resolution = self.mz/self.fwhm
    # ----
    
    
    # GETTERS
    
    def mass(self):
        """Get neutral peak mass."""
        
        # check charge
        if self.charge == None:
            return None
        
        # check mass buffer
        if self._mass != None:
            return self._mass
            
        # calculate neutral mass
        self._mass = processing.mz(self.mz, 0, self.charge, agentFormula='H', agentCharge=1)
        
        return self._mass
    # ----
    
    
    # SETTERS
    
    def setMz(self, mz):
        """Set new m/z value."""
        self.mz = mz
        self.reset()
    # ----
    
    
    def setAi(self, ai):
        """Set new a.i. value."""
        self.ai = ai
        self.reset()
    # ----
    
    
    def setBase(self, base):
        """Set new baseline value."""
        self.base = base
        self.reset()
    # ----
    
    
    def setSN(self, sn):
        """Set new s/n value."""
        self.sn = sn
    # ----
    
    
    def setCharge(self, charge):
        """Set new charge value."""
        self.charge = charge
        self.reset()
    # ----
    
    
    def setIsotope(self, isotope):
        """Set new isotope value."""
        self.isotope = isotope
    # ----
    
    
    def setFwhm(self, fwhm):
        """Set new fwhm value."""
        self.fwhm = fwhm
        self.reset()
    # ----
    
    


class peaklist:
    """Peaklist object definition."""
    
    def __init__(self, peaks=[]):
        
        # check data
        self._peaks = []
        for item in peaks:
            item = self._checkPeak(item)
            self._peaks.append(item)
        
        # buffers
        self._basepeak = None
        
        # sort peaklist by m/z
        self._sort()
        
        # set basepeak
        self._setBasepeak()
        
        # set relative intensities
        self._setRelativeIntensities()
    # ----
    
    
    def __len__(self):
        return len(self._peaks)
    # ----
    
    
    def __setitem__(self, i, item):
        
        # check item
        item = self._checkPeak(item)
        
        # basepeak is edited - set new
        if self._peaks[i] is self._basepeak:
            self._peaks[i] = item
            self._setBasepeak()
            self._setRelativeIntensities()
        
        # new basepeak set
        elif self._basepeak and item.intensity > self._basepeak.intensity:
            self._peaks[i] = item
            self._basepeak = item
            self._setRelativeIntensities()
        
        # lower than basepeak
        elif self._basepeak:
            item.relIntensity = item.intensity / self._basepeak.intensity
            self._peaks[i] = item
        
        # no basepeak set
        else:
            self._peaks[i] = item
            self._setBasepeak()
            self._setRelativeIntensities()
        
        # sort peaklist
        self._sort()
    # ----
    
    
    def __getitem__(self, i):
        return self._peaks[i]
    # ----
    
    
    def __delitem__(self, i):
        
        # delete basepeak
        if self._peaks[i] is self._basepeak:
            del self._peaks[i]
            self._setBasepeak()
            self._setRelativeIntensities()
        
        # delete others
        else:
            del self._peaks[i]
    # ----
    
    
    def __iter__(self):
        self._index = 0
        return self
    # ----
    
    
    def __add__(self, peaksB):
        """Return A+B."""
        
        new = self.duplicate()
        new.concatenate(peaksB)
        return new
    # ----
    
    
    def __mul__(self, x):
        """Return A*X."""
        
        new = self.duplicate()
        new.multiply(x)
        return new
    # ----
    
    
    def next(self):
        if self._index < len(self._peaks):
            self._index += 1
            return self._peaks[self._index-1]
        else:
            raise StopIteration
    # ----
    
    
    def append(self, item):
        """Append new peak.
            item: (peak or [#, #] or (#,#)) peak to be added
        """
        
        # check peak
        item = self._checkPeak(item)
        
        # add peak and sort peaklist
        if self._peaks and self._peaks[-1].mz > item.mz:
            self._peaks.append(item)
            self._sort()
        else:
            self._peaks.append(item)
        
        # new basepeak set
        if self._basepeak and item.intensity > self._basepeak.intensity:
            self._basepeak = item
            self._setRelativeIntensities()
        
        # lower than basepeak
        elif self._basepeak and self._basepeak.intensity != 0:
            item.relIntensity = item.intensity / self._basepeak.intensity
        
        # no basepeak set
        else:
            item.relIntensity = 1.
            self._setBasepeak()
    # ----
    
    
    # GETTERS
    
    def duplicate(self):
        """Return copy of current peaklist."""
        return copy.deepcopy(self)
    # ----
    
    
    def basepeak(self):
        """Return basepeak."""
        return self._basepeak
    # ----
    
    
    # MODIFIERS
    
    def refresh(self):
        """Sort peaklist and recalculate basepeak and relative intensities."""
        
        self._sort()
        self._setBasepeak()
        self._setRelativeIntensities()
    # ----
    
    
    def delete(self, indexes=[]):
        """Delete selected peaks.
            indexes: (list or tuple) indexes of peaks to be deleted
        """
        
        # check peaklist
        if not self._peaks:
            return
        
        # delete peaks
        indexes.sort()
        indexes.reverse()
        relint = False
        for i in indexes:
            if self._peaks[i] is self._basepeak:
                relint = True
            del self._peaks[i]
        
        # recalculate basepeak and relative intensities
        if relint:
            self._setBasepeak()
            self._setRelativeIntensities()
    # ----
    
    
    def empty(self):
        """Remove all peaks."""
        
        del self._peaks[:]
        self._basepeak = None
    # ----
    
    
    def crop(self, minX, maxX):
        """Delete peaks outside given range.
            minX: (float) lower m/z limit
            maxX: (float) upper m/z limit
        """
        
        # check peaklist
        if not self._peaks:
            return
        
        # get indexes to delete
        indexes = []
        for x, peak in enumerate(self._peaks):
            if peak.mz < minX or peak.mz > maxX:
                indexes.append(x)
        
        # delete peaks
        self.delete(indexes)
    # ----
    
    
    def multiply(self, x):
        """Multiply each peak intensity by X.
            x: (int or float) multiplier factor
        """
        
        for peak in self._peaks:
            peak.ai *= x
            peak.base *= x
    # ----
    
    
    def concatenate(self, peaksB):
        """Add data from given peaklist.
            peaksB: (peaklist or list or tuple of peaks) peaklist to be added
        """
        
        # add new peaks
        for peak in copy.deepcopy(peaksB):
            peak = self._checkPeak(peak)
            self._peaks.appen(peak)
        
        # update final peaklist
        self._sort()
        self._setBasepeak()
        self._setRelativeIntensities()
    # ----
    
    
    def applyCalibration(self, fce, params):
        """Apply calibration to peaks.
            fce: (function) calibration model
            params: (list or tuple) params for calibration model
        """
        
        # check peaklist
        if not self._peaks:
            return
        
        # apply calibration
        for peak in self._peaks:
            peak.setMz(fce(params, peak.mz))
    # ----
    
    
    def findIsotopes(self, maxCharge=1, mzTolerance=0.15, intTolerance=0.5, isotopeShift=0.0):
        """Calculate peak charges and find isotopes.
            maxCharge: (float) max charge to be searched
            mzTolerance: (float) absolute mass tolerance for isotopes
            intTolerance: (float) relative intensity tolerance for isotopes in %/100
            isotopeShift: (float) isotope distance correction (neutral mass)
        """
        
        processing.findIsotopes(
            peaklist=self, \
            maxCharge=maxCharge, \
            mzTolerance=mzTolerance, \
            intTolerance=intTolerance, \
            isotopeShift=isotopeShift \
            )
    # ----
    
    
    def removeBelow(self, absThreshold=0., relThreshold=0., snThreshold=0.):
        """Remove peaks below threshold.
            absThreshold: (float) absolute intensity threshold
            relThreshold: (float) relative intensity threshold
            snThreshold: (float) signal to noise threshold
        """
        
        # check peaklist
        if not self._peaks:
            return
        
        # get absolute threshold
        threshold = self._basepeak.intensity * relThreshold
        threshold = max(threshold, absThreshold)
        
        # get indexes to delete
        indexes = []
        for x, peak in enumerate(self._peaks):
            if peak.intensity < threshold or (peak.sn != None and peak.sn < snThreshold):
                indexes.append(x)
        
        # delete peaks
        self.delete(indexes)
    # ----
    
    
    def removeShoulders(self, window=2.5, relThreshold=0.05, fwhm=0.01):
        """Remove shoulder peaks from FTMS data.
            window: (float) peak width multiplier to make search window
            relThreshold: (float) max relative intensity of shoulder/parent peak in %/100
            fwhm: (float) default peak width if not set in peak
        """
        
        # check peaklist
        if not self._peaks:
            return
        
        # get possible parent peaks
        candidates = []
        for peak in self._peaks:
            if not peak.sn or peak.sn*relThreshold > 3:
                candidates.append(peak)
            
        # filter shoulder peaks
        indexes = []
        for parent in candidates:
            
            # get shoulder window
            if parent.fwhm:
                lowMZ = parent.mz - parent.fwhm * window
                highMZ = parent.mz + parent.fwhm * window
            elif fwhm:
                lowMZ = parent.mz - fwhm * window
                highMZ = parent.mz + fwhm * window
            else:
                continue
            
            # get intensity threshold
            intThreshold = parent.intensity * relThreshold
            
            # get indexes to delete
            for x, peak in enumerate(self._peaks):
                if (lowMZ < peak.mz < highMZ) and (peak.intensity < intThreshold) and (not x in indexes):
                    indexes.append(x)
                if peak.mz > highMZ:
                    break
        
        # delete peaks
        self.delete(indexes)
    # ----
    
    
    def removeIsotopes(self):
        """Remove isotopes."""
        
        # check peaklist
        if not self._peaks:
            return
        
        # get indexes to delete
        indexes = []
        for x, peak in enumerate(self._peaks):
            if peak.isotope != 0 and peak.charge != None:
                indexes.append(x)
        
        # delete peaks
        self.delete(indexes)
    # ----
    
    
    def removeUnknown(self):
        """Remove unknown peaks - no charge set."""
        
        # check peaklist
        if not self._peaks:
            return
        
        # get indexes to delete
        indexes = []
        for x, peak in enumerate(self._peaks):
            if peak.charge == None:
                indexes.append(x)
        
        # delete peaks
        self.delete(indexes)
    # ----
    
    
    # HELPERS
    
    def _sort(self):
        """Sort data according to mass."""
        
        buff = []
        for item in self._peaks:
            buff.append((item.mz, item))
        buff.sort()
        
        self._peaks = []
        for item in buff:
            self._peaks.append(item[1])
    # ----
    
    
    def _checkPeak(self, item):
        """Check item to be a valid peak."""
        
        # peak instance
        if isinstance(item, peak):
            return item
        
        # make peak from list or tuple
        elif type(item) in (list, tuple) and len(item)==2:
            return peak(item[0], item[1])
        
        # not valid peak data
        raise TypeError, 'Item must be a peak object or list/tuple of two floats!'
    # ----
    
    
    def _setBasepeak(self):
        """Get most intens peak."""
        
        # check peaklist
        if not self._peaks:
            self._basepeak = None
            return
        
        # set new basepeak
        self._basepeak = self._peaks[0]
        maxInt = self._basepeak.intensity
        for item in self._peaks[1:]:
            if item.intensity > maxInt:
                self._basepeak = item
                maxInt = item.intensity
    # ----
    
    
    def _setRelativeIntensities(self):
        """Set relative intensities for all peaks."""
        
        # check peaklist
        if not self._peaks:
            return
        
        # set relative intensities
        maxInt = self._basepeak.intensity
        for item in self._peaks:
            if maxInt:
                item.relIntensity = item.intensity / maxInt
            else:
                item.relIntensity = 1.
    # ----
    
    


class scan:
    """Scan object definition."""
    
    def __init__(self, points=[], peaks=[]):
        
        self.title = ''
        self.scanNumber = None
        self.parentScanNumber = None
        self.polarity = None
        self.msLevel = None
        self.retentionTime = None
        self.totIonCurrent = None
        self.basePeakMZ = None
        self.basePeakIntensity = None
        self.precursorMZ = None
        self.precursorIntensity = None
        self.precursorCharge = None
        
        # buffers
        self._baseline = None
        self._baselineParams = {'window': None, 'smooth': None, 'offset': None}
        
        # user defined params
        self.userParams = {}
        
        # convert points to numPy array
        self.points = numpy.array(points)
        
        # convert peaks to peaklist
        if isinstance(peaks, peaklist):
            self.peaklist = peaks
        else:
            self.peaklist = peaklist(peaks)
    # ----
    
    
    def __len__(self):
        return len(self.points)
    # ----
    
    
    def __add__(self, scanB):
        """Return A+B."""
        
        new = self.duplicate()
        new.concatenate(peaksB)
        return new
    # ----
    
    
    def __sub__(self, scanB):
        """Return A-B."""
        
        new = self.duplicate()
        new.subtract(peaksB)
        return new
    # ----
    
    
    def __mul__(self, x):
        """Return A*X."""
        
        new = self.duplicate()
        new.multiply(x)
        return new
    # ----
    
    
    def reset(self):
        """Clear scan buffers."""
        
        self._baseline = None
        self._baselineParams = {'window': None, 'smooth': None, 'offset': None}
    # ----
    
    
    # GETTERS
    
    def duplicate(self):
        """Return copy of current scan."""
        return copy.deepcopy(self)
    # ----
    
    
    def noise(self, minX=None, maxX=None, mz=None, window=0.1):
        """Return noise level and width for specified m/z range or m/z value.
            minX: (float) lower m/z limit
            maxX: (float) upper m/z limit
            mz: (float) m/z value
            window: (float) percentage around specified m/z value to use for noise calculation
        """
        
        # calculate noise
        return processing.noise(self.points, minX=minX, maxX=maxX, mz=mz, window=window)
    # ----
    
    
    def baseline(self, window=0.1, smooth=True, offset=0.):
        """Return spectrum baseline data.
            window: (float) noise calculation window in %/100
            smooth: (bool) smooth final baseline
            offset: (float) global intensity offset in %/100
        """
        
        # calculate baseline
        if self._baseline == None \
            or self._baselineParams['window'] != window \
            or self._baselineParams['smooth'] != smooth \
            or self._baselineParams['offset'] != offset:
            
            self._baseline = processing.baseline(self.points, window=window, smooth=smooth, offset=offset)
            self._baselineParams['window'] = window
            self._baselineParams['smooth'] = smooth
            self._baselineParams['offset'] = offset
        
        return self._baseline
    # ----
    
    
    def normalization(self):
        """Return normalization params."""
        
        # calculate range for spectrum and peaklist
        if len(self.points) > 0 and len(self.peaklist) > 0:
            spectrumMax = numpy.maximum.reduce(self.points)[1]
            spectrumMin = numpy.minimum.reduce(self.points)[1]
            peaklistMax = max([peak.ai for peak in self.peaklist])
            peaklistMin = min([peak.base for peak in self.peaklist])
            shift = min(spectrumMin, peaklistMin)
            scale = (max(spectrumMax, peaklistMax)-shift)/100
        
        # calculate range for spectrum only
        elif len(self.points) > 0:
            spectrumMax = numpy.maximum.reduce(self.points)[1]
            shift = numpy.minimum.reduce(self.points)[1]
            scale = (spectrumMax-shift)/100
        
        # calculate range for peaklist only
        elif len(self.peaklist) > 0:
            peaklistMax = max([peak.ai for peak in self.peaklist])
            shift = min([peak.base for peak in self.peaklist])
            scale = (peaklistMax-shift)/100
        
        # no data
        else:
            return 1., 0.
        
        return scale, shift
    # ----
    
    
    def pointsSelection(self, minX, maxX):
        """Return data points for selected m/z range.
            minX: (float) lower m/z limit
            maxX: (float) upper m/z limit
        """
        
        # check slice
        if maxX < minX:
            raise ValueError, 'Invalid m/z slice definition!'
        
        # crop points
        i1 = self._getIndex(self.points, minX)
        i2 = self._getIndex(self.points, maxX)
        
        return self.points[i1:i2]
    # ----
    
    
    def peakIntensity(self, mz):
        """Return interpolated intensity for given m/z.
            mz: (float) m/z value
        """
        
        # calculate peak intensity
        return processing.peakIntensity(self.points, mz)
    # ----
    
    
    def peakWidth(self, mz, intensity):
        """Return peak width for given m/z and height.
            mz: (float) peak m/z value
            intensity: (float) intensity of width measurement
        """
        
        # calculate peak width
        return processing.peakWidth(self.points, mz, intensity)
    # ----
    
    
    def hasPoints(self):
        """Return number of data points."""
        return len(self.points)
    # ----
    
    
    def hasPeaks(self):
        """Return number of peaks in peaklist."""
        return len(self.peaklist)
    # ----
    
    
    # SETTERS
    
    def setPoints(self, points):
        """Set new point."""
        self.points = points
        self.reset()
    # ----
    
    
    def setPeaklist(self, peaks):
        """Set new point."""
        
        # convert peaks to peaklist
        if isinstance(peaks, peaklist):
            self.peaklist = peaks
        else:
            self.peaklist = peaklist(peaks)
    # ----
    
    
    # MODIFIERS
    
    def swap(self):
        """Swap data between spectrum and peaklist."""
        
        # make new spectrum
        points = [[i.mz, i.ai] for i in self.peaklist]
        points = numpy.array(points)
        
        # make new peaklist
        peaks = [peak(i[0],i[1]) for i in self.points]
        peaks = peaklist(peaks)
        
        # update document
        self.points = points
        self.peaklist = peaks
        
        # clear buffers
        self.reset()
    # ----
    
    
    def crop(self, minX, maxX):
        """Crop data points and peaklist.
            minX: (float) lower m/z limit
            maxX: (float) upper m/z limit
        """
        
        # crop spectrum data
        i1 = self._getIndex(self.points, minX)
        i2 = self._getIndex(self.points, maxX)
        self.points = self.points[i1:i2]
        
        # crop peaklist data
        self.peaklist.crop(minX, maxX)
        
        # clear buffers
        self.reset()
    # ----
    
    
    def normalize(self):
        """Normalize data points and peaklist."""
        
        # get normalization params
        normalization = self.normalization()
        
        # normalize spectrum points
        if len(self.points) > 0:
            self.points -= numpy.array((0, normalization[1]))
            self.points /= numpy.array((1, normalization[0]))
        
        # normalize peaklist
        if len(self.peaklist) > 0:
            for peak in self.peaklist:
                peak.setAi((peak.ai - normalization[1]) / normalization[0])
                peak.setBase((peak.base - normalization[1]) / normalization[0])
            self.peaklist.refresh()
        
        # clear buffers
        self.reset()
    # ----
    
    
    def multiply(self, x):
        """Multiply data points and peaklist by X.
            x: (int or float) multiplier factor
        """
        
        # multiply spectrum
        if len(self.points):
            self.points *= numpy.array((1.0, x))
        
        # multiply peakslist
        self.peaklist.multiply(x)
        
        # clear buffers
        self.reset()
    # ----
    
    
    def concatenate(self, scanB):
        """Add data from given scan.
            scanB: (scan) scan to be added
        """
        
        # check scan
        if not isinstance(scanB, scan):
            raise TypeError, "Cannot concatenate with non-scan object!"
        
        # use spectra only
        if len(self.points) or len(scanB.points):
            
            # unify raster
            data = processing.unifyRaster(self.points, scanB.points)
            
            # convert back to arrays
            pointsA = numpy.array(data[0])
            pointsB = numpy.array(data[1])
            
            # remove X axis from points B
            pointsB[:,0] = 0
            
            # concatenate points
            self.points = pointsA + pointsB
            
            # empty peaklist
            self.peaklist.empty()
        
        # use peaklists only
        elif len(self.peaklist) or len(scanB.peaklist):
            self.peaklist.concatenate(scanB.peaklist)
        
        # clear buffers
        self.reset()
    # ----
    
    
    def subtract(self, scanB):
        """Subtract given data points from current scan."""
        
        # check scan
        if not isinstance(scanB, scan):
            raise TypeError, "Cannot subtract non-scan object!"
        
        # use spectra only
        if len(self.points) and len(scanB.points):
            
            # unify raster
            data = processing.unifyRaster(self.points, scanB.points)
            
            # convert back to arrays
            pointsA = numpy.array(data[0])
            pointsB = numpy.array(data[1])
            
            # remove X axis from points B
            pointsB[:,0] = 0
            
            # subtract points
            self.points = pointsA - pointsB
            
            # empty peaklist
            self.peaklist.empty()
            
            # clear buffers
            self.reset()
    # ----
    
    
    def smooth(self, method, window, cycles=1):
        """Smooth data points.
            method: ('MA' or 'SG') smoothing method, MA - moving average, SG - Savitzky-Golay
            window: (float) m/z window size for smoothing
            cycles: (int) number of repeating cycles
        """
        
        # apply moving average filter
        if method.upper() == 'MA':
            points = processing.smoothMA(self.points, window, cycles)
        
        # apply Savitzky-Golay filter
        elif method.upper() == 'SG':
            points = processing.smoothSG(self.points, window, cycles)
        
        # unknown method
        else:
            raise KeyError, "Unknown smoothing method"
        
        # store data
        self.points = points
        self.peaklist.empty()
        
        # clear buffers
        self.reset()
    # ----
    
    
    def applyCalibration(self, fce, params):
        """Apply calibration to data points and peaklist.
            fce: (function) calibration model
            params: (list or tuple) params for calibration model
        """
        
        # calibrate data points
        for x, point in enumerate(self.points):
            self.points[x][0] = fce(params, point[0])
        
        # calibrate peaklist
        self.peaklist.applyCalibration(fce, params)
        
        # clear buffers
        self.reset()
    # ----
    
    
    def correctBaseline(self, window=0.1, smooth=True, offset=0.):
        """Subtract baseline from data points.
            window: (float) noise calculation window in %/100
            smooth: (bool) smooth final baseline
            offset: (float) global intensity offset in %/100
        """
        
        # correct baseline
        points = processing.correctBaseline(\
            points=self.points, \
            window=window, \
            smooth=smooth, \
            offset=offset \
        )
        
        # store data
        self.points = points
        self.peaklist.empty()
        
        # clear buffers
        self.reset()
    # ----
    
    
    def labelScan(self, pickingHeight=0.75, absThreshold=0., relThreshold=0., snThreshold=0., baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., smoothMethod=None, smoothWindow=0.2, smoothCycles=1):
        """Label centroides in current scan.
            pickingHeight: (float) peak picking height for centroiding
            absThreshold: (float) absolute intensity threshold
            relThreshold: (float) relative intensity threshold
            snThreshold: (float) signal to noise threshold
            baselineWindow: (float) noise calculation window in %/100
            baselineSmooth: (bool) smooth baseline
            baselineOffset: (float) baseline intensity offset in %/100
            smoothMethod: (None or MA or SG) smoothing method, MA - moving average, SG - Savitzky-Golay
            smoothWindows: (float) m/z window size for smoothing
            smoothCycles: (int) number of repeating cycles
        """
        
        # label scan
        newPeaklist = processing.labelScan(self.points, \
            pickingHeight=pickingHeight, \
            absThreshold=absThreshold, \
            relThreshold=relThreshold, \
            snThreshold=snThreshold, \
            baselineWindow=baselineWindow, \
            baselineSmooth=baselineSmooth, \
            baselineOffset=baselineOffset, \
            smoothMethod=smoothMethod, \
            smoothWindow=smoothWindow, \
            smoothCycles=smoothCycles)
        
        # update peaklist
        if newPeaklist != None:
            self.peaklist = newPeaklist
            return True
        else:
            return False
    # ----
    
    
    def labelPeak(self, mz=None, minX=None, maxX=None, pickingHeight=0.75, baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., baselineData=None):
        """Return labeled peak in given m/z range.
            mz: (float) single m/z value
            minX: (float) starting m/z value
            maxX: (float) ending m/z value
            pickingHeight: (float) peak picking height for centroiding
            baselineWindow: (float) noise calculation window in %/100
            baselineSmooth: (bool) smooth baseline
            baselineOffset: (float) baseline intensity offset in %/100
            baselineData: (list of [x, noiseLevel, noiseWidth]) precalculated baseline
        """
        
        # label peak
        newPeak = processing.labelPeak(self.points, \
            mz=mz, \
            minX=minX, \
            maxX=maxX, \
            pickingHeight=pickingHeight, \
            baselineWindow=baselineWindow, \
            baselineSmooth=baselineSmooth, \
            baselineOffset=baselineOffset, \
            baselineData=baselineData \
        )
        
        # append peak
        if newPeak:
            self.peaklist.append(newPeak)
            return True
        else:
            return False
    # ----
    
    
    def labelPoint(self, mz, baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., baselineData=None):
        """Label peak at given m/z value.
            mz: (float) m/z value to label
            baselineWindow: (float) noise calculation window in %/100
            baselineSmooth: (bool) smooth baseline
            baselineOffset: (float) baseline intensity offset in %/100
            baselineData: (list of [x, noiseLevel, noiseWidth]) precalculated baseline
        """
        
        # label peak
        newPeak = processing.labelPoint(self.points, \
            mz=mz, \
            baselineWindow=baselineWindow, \
            baselineSmooth=baselineSmooth, \
            baselineOffset=baselineOffset, \
            baselineData=baselineData \
        )
        
        # append peak
        if newPeak:
            self.peaklist.append(newPeak)
            return True
        else:
            return False
    # ----
    
    
    def findIsotopes(self, maxCharge=1, mzTolerance=0.15, intTolerance=0.5, isotopeShift=0.0):
        """Calculate peak charges and find isotopes.
            maxCharge: (float) max charge to be searched
            mzTolerance: (float) absolute mass tolerance for isotopes
            intTolerance: (float) relative intensity tolerance for isotopes in %/100
            isotopeShift: (float) isotope distance correction (neutral mass)
        """
        
        # find istopes
        self.peaklist.findIsotopes(
            maxCharge=maxCharge, \
            mzTolerance=mzTolerance, \
            intTolerance=intTolerance, \
            isotopeShift=isotopeShift \
            )
    # ----
    
    
    def removeBelow(self, absThreshold=0., relThreshold=0., snThreshold=0.):
        """Remove peaks below threshold.
            absThreshold: (float) absolute intensity threshold
            relThreshold: (float) relative intensity threshold
            snThreshold: (float) signal to noise threshold
        """
        
        # remove peaks below threshold
        self.peaklist.removeBelow( \
            absThreshold=absThreshold, \
            relThreshold=relThreshold, \
            snThreshold=snThreshold \
            )
    # ----
    
    
    def removeShoulders(self, window=2.5, relThreshold=0.05, fwhm=0.01):
        """Remove shoulder peaks from current peaklist.
            window: (float) peak width multiplier to make search window
            relThreshold: (float) max relative intensity of shoulder/parent peak in %/100
            fwhm: (float) default peak width if not set in peak
        """
        
        # remove shoulder peaks
        self.peaklist.removeShoulders(\
            window=window, \
            relThreshold=relThreshold, \
            fwhm=fwhm \
            )
    # ----
    
    
    def removeIsotopes(self):
        """Remove isotopes from current peaklist."""
        
        # remove isotopes
        self.peaklist.removeIsotopes()
    # ----
    
    
    def removeUnknown(self):
        """Remove unknown peaks (no charge set) from current peaklist."""
        
        # remove unknowns
        self.peaklist.removeUnknown()
    # ----
    
    
    # HELPERS
    
    def _getIndex(self, points, x):
        """Get nearest higher index for selected point."""
        
        lo = 0
        hi = len(points)
        while lo < hi:
            mid = (lo + hi) / 2
            if x < points[mid][0]:
                hi = mid
            else:
                lo = mid + 1
        
        return lo
    # ----
    
    
    def _interpolateLine(self, p1, p2, x=None, y=None):
        """Get line interpolated X or Y value."""
        
        # check points
        if p1[0] == p2[0] and x!=None:
            return max(p1[1], p2[1])
        elif p1[0] == p2[0] and y!=None:
            return p1[0]
        
        # get equation
        m = (p2[1] - p1[1])/(p2[0] - p1[0])
        b = p1[1] - m * p1[0]
        
        # get point
        if x != None:
            return m * x + b
        elif y != None:
            return (y - b) / m
    # ----
    
    

