####################################################
# This is where the music is created.

import copy, random

import MMAglobals;  gbl = MMAglobals
from MMAcommon import *
import MMAmidi


defaultChannelVolume = 100

class struct:
	pass
	
class Pats:
	""" Pattern class.
	
		Define classes for processing drum, chord, arp, and chord track.
		These are mostly the same, so we create a base class and derive
		the others from it.
	
	"""
	
	def __init__(self, nm):

		self.inited = 0		
		self.name = nm	
		self.channel = 0
		self.pats = {}
		self.grooves = {}
		self.saveVols = {}
		self.transpose = 0	
		self.ssvoice = -1	# Track the voice set for the track
		self.artic = (90,)	# Note length clip adjustment

		# Midi commands like Pan, Glis, etc. are stacked until musical
		# data is actually written to the track. Each item in
		# the midiPending list is a name (PAN, GLIS, etc), timeoffset, value.

		self.midiPending = []		

		self.riff = ''
				
		self.disable = 0
		
		if self.vtype == 'DRUM':
			self.ssvoice = 0

		self.setChannelVolume(defaultChannelVolume)

		self.clearSequence()		
					
		self.inited = 1
		
	##########################################
	## These are called from process() to set options

	def setCompress(self, ln):
		""" set/unset the compress flag. """
	
		ln = self.lnExpand(ln, 'Compress')

		vwarn = 0
		
		for i in range(len(ln)):
			
			n = stoi(ln[i], "Argument for %s Compress must be a value." \
					% self.name)
					
			if n < 0 or n > 5:
				error("Compress %s out-of-range; must be 0 to 5." % n)
			
				
			if n and self.voicingMode:
				vwarn = 1
						
			ln[i] = n

		if vwarn:
			warning("Setting both VoicingMode and Compress is not a good idea")			
		
		self.compress = tuple( ln )
		
		if gbl.debug:
			print "Set %s Compress to " % self.name,
			printList(ln)
		
		
	def setVOmode(self, ln):
		""" set the VoicingMode. """
	
		n=ln.upper()
		
		valid= ("-", "OPTIMAL", "NONE", "ROOT", "COMPRESSED", "INVERT")
		
		if not n in  valid:
			error("Valid VoicingModes are: %s" % " ".join(valid))
				
		if n in ('-', 'NONE',"ROOT"):
			n = None
		

		if n and (max(self.invert) + max(self.compress)):
			warning("Setting both VoicingMode and Invert/Compress is not a good idea")	
			
		# When we set voicing mode we always reset this. This forces
		# the voicingmode code to restart its rotations.

		self.lastChord = []	  

		self.voicingMode = n
		
		if self.vtype != "CHORD":
			warning("VoicingMode is not supported for %s tracks." % self.vtype)
			
			
		if gbl.debug:
			print "Set %s VoicingMode to %s" % (self.name, n)


	def setVOrange(self, ln):
		""" set the VoicingRange. """
	
		n = stoi(ln, "Argument for %s VoicingRange must be a value." \
					% self.name)
					
		if n < 1 or n > 30:
			error("VoiceRange %s out-of-range; must be 1 to 30." % n)
						
		self.voicingRange = n
		
		if self.vtype != "CHORD":
			warning("VoicingRange is not supported for %s tracks." % self.vtype)
			
		if gbl.debug:
			print "Set %s VoicingRange to %s" % (self.name,n)


	def setVOcenter(self, ln):
		""" set the VoicingCenter. """
	
		n = stoi(ln, "Argument for %s VoicingCenter must be a value." \
				% self.name)
					
		if n < 1 or n > 12:
			error("VoiceCenter %s out-of-range; must be 1 to 12." % n)
						
			
		self.voicingCenter = n
		
		if self.vtype != "CHORD":
			warning("VoicingCenter is not supported for %s tracks." % self.vtype)
			
		if gbl.debug:
			print "Set %s VoicingCenter to %s." % (self.name, self.voicingCenter)


	def setVOmove(self, ln):
		""" Set the VoicingMove. 
		
			Voice VoicingMove  BarCount Direction
			Voice VoicingMove  Random  Percentage
		
		"""

		if len(ln) == 1 and ln[0] == '0':
			ln.append('0')
			
		if len(ln)!=2:
			error("VoicingMove must be in form <BarCount Dir> or <Random Percent>.")
			
		n=ln[0].upper()

		if n == 'RANDOM':
			self.voicingMove[0] = n
		else:
			n=stoi(ln[0], "Argument 1 for %s VoicingMove must be a value or RANDOM." \
				% self.name)
				
			if n<0:
				error("VoicingMove Barcount must >= 0, not %s" % n)
			if n>20:
				warning("VoicingMove Barcound %s quite large" % n)
			
			self.voicingMove[0] = n
			
		d=stoi(ln[1], "Argument 2 for %s VoicingMove must be a value")
		
		if n=='RANDOM':
			if d<=0 or d>100:
				error("VoicingMove RANDOM value must be 1 to 100 not %s" % d)
			
		else:
			if d<-1 or d>1:
				error("VoicingMove Barcount value must be -1, 0 or 1, not %s" % d)
				
		self.voicingMove[1]=d
			
		if self.vtype != "CHORD":
			warning("VoicingMove is not supported for %s tracks." % self.vtype)
			
		if gbl.debug:
			print "Set %s VoicingMove to %s %s." % (self.name, self.voicingMove[0],
				self.voicingMove[1]) 


	def setDuplicate(self, ln):
		""" set/unset octave duplication. """

		ln = self.lnExpand(ln, 'Duplicate')
			
		for i in range(len(ln)):
			n = stoi(ln[i], "Argument for %s Duplicate must be a value." \
					% self.name)
					
			if n < -9 or n > 9:
				error("Duplicate %s out-of-range; must be -9 to 9." % n)
						
			ln[i] = n * 12
			
		self.duplicate = tuple( ln )
		
		if gbl.debug:
			print "Set %s Duplicate to " % self.name,
			printList(ln)

	def setDupRoot(self, ln):
		""" set/unset root duplication. """

		ln = self.lnExpand(ln, 'DupRoot')
			
		if self.vtype != 'CHORD':
			error("RootDup can only be applied to CHORD tracks.")
			
		for i in range(len(ln)):
			n = stoi(ln[i], "Argument for %s DupRoot must be a value." \
					% self.name)
					
			if n < -9 or n > 9:
				error("DupRoot %s out-of-range; must be -9 to 9." % n)
						
			ln[i] = n * 12
			
		self.dupRoot = tuple( ln )
		
		if gbl.debug:
			print "Set %s DupRoot to " % self.name,
			printList(ln)

	
	def setChordLimit(self, ln):
		""" set/unset the chordLimit flag. """
	
		n = stoi(ln, "Argument for %s ChordLimit must be a value." \
					% self.name)
					
		if n < 0 or n > 8:
			error("ChordLimit %s out-of-range; must be 0 to 8." % n)
						
		self.chordLimit = n
		
		if gbl.debug:
			print "Set %s ChordLimit to %s" % (self.name, n)

		
	def setChannel(self, ln):
		""" Set the midi-channel number for a track. 
		
			Checks for channel duplication
			Sets default channel volume.
		"""
	
		c = stoi(ln, "%s Channel assignment expecting Value, not %s" %
			(self.name, ln))

		if c<0 or c>16:
			error("%s Channel must be 0..16, not %s" % (self.name, ln))
			
		if self.vtype == 'DRUM' and c != 10:
			error("MIDI Channel for DRUM tracks cannot be reassigned.")
			
		if c == 10 and self.vtype != 'DRUM':
			error("Channel 10 is reserved for DRUM, not %s." % self.name)
		
		old = self.channel
		
		# Disable the channel.

		if c == 0:
			self.channel = 0
			if gbl.midiAssigns[c].count(self.name):
				gbl.midiAssings[c].remove(self.name)
			warning("%s channel disabled, %s available." %
				(self.name, old ))
			return
			
			
		for a in gbl.tnames.keys():
			if a == self.name:	# okay to reassign same number
				continue
				
			if gbl.tnames[a].channel == c:
				error("%s is assigned to %s." % (c, gbl.tnames[a].name ) )
				
		self.channel = c
		gbl.midiAssigns[c].append(self.name)

		if not gbl.mtrks.has_key(c):
			gbl.mtrks[c]=MMAmidi.Mtrk(c)
			self.midiPending.append(('TNAME', 0, self.name.title() ))
			if gbl.debug:
				print "MIDI channel %s buffer created." % c



	
		if gbl.debug:
			print "MIDI Channel %s assigned to %s." % (self.channel, self.name)
			

	def setChShare(self, ln):
		""" Share midi-channel setting. """
		
		sc = ln.upper()
		
		if not gbl.tnames.has_key(sc):
			error("Channel '%s' does not exist. No such name." % sc)
		
		if sc == self.name:
			error("%s can't share MIDI channel with itself." % sc)

		
		schannel = gbl.tnames[sc].channel
		
		if not schannel:
			for c in range(16, 0, -1):
				if not gbl.midiAssigns[c]:
					gbl.tnames[sc].setChannel('%s' % c)
					break

		schannel = gbl.tnames[sc].channel
		if not schannel:
				error("CHShare attempted to assign MIDI channel for %s, but "
					"none avaiable." %  self.name)


		self.channel = schannel
		
			
		if self.channel == 0:
			self.setChannel()
			return
			
		warning("%s and %s now share MIDI channel %s" %
			(sc, self.name, self.channel))
		
		if gbl.midiAssigns[self.channel].count(self.name):
			error("Channel sharing between %s and %s previously enabled." % \
				(self.name, ln))
				
		gbl.midiAssigns[self.channel].append(self.name)
			
					
	def setChannelVolume(self, v):
		""" LowLevel MIDI command. Set Channel Voice. """
		
		self.midiPending.append(( "CVOLUME", gbl.tickOffset, v) )

				
	def setPan(self, ln):
		""" Set MIDI Pan for this track. """
		
		v = stoi(ln[0], "Expecting integer value 0..127") 
	
		if v<0 or v>127:
			error("PAN value must be 0..127")

		self.midiPending.append( ("PAN", gbl.tickOffset, v))

		
	def setGlis(self, ln):
		""" Set MIDI Glis for this track. """
		
		v = stoi(ln, "Expecting integer for Portamento")	
		
		if v<0 or v>127:
			error("Value for Portamento must be 0..127")
			
		self.midiPending.append( ("GLIS", gbl.tickOffset, v))


	def setStrum(self, ln):
		""" Set Strum time. """

		# Strum is only valid for CHORD tracks.
		
		if self.vtype != "CHORD":
			error( "Strum is only valid in Chord tracks, you tried to " 
				"set it in a %s track." % self.name)

		ln=self.lnExpand(ln, 'Strum')
		
		for i in range(len(ln)):
			n = stoi(ln[i], "Argument for %s Strum must be an integer" \
					% self.name)
		
			if n < 0 or n > 100:
				error("Strum %s out-of-range; must be 0..100." % n)
			
			ln[i] = n
			
		self.strum = ln
		
		if gbl.debug:
			print "Set %s Strum to %s" % (self.name, self.strum)


	def setTone(self, ln):
		""" Set Tone. Error trap, only drum tracks have tone. """
		
		error("Tone command not supported for %s track." % self.name)

		
	def setOn(self):
		""" Turn ON track. """
		
		self.disable = 0

		if gbl.debug:
			print "%s Enabled" % self.name


		
	def setOff(self):
		""" Turn OFF track. """
		
		self.disable = 1

		if gbl.debug:
			print "%s Disabled" % self.name

	

	def setRVolume(self, ln):
		""" Set the volume randomizer for a track. """

		ln = self.lnExpand(ln, 'RVolume')

		for i in range(len(ln)):
			
			n = stoi(ln[i], "Argument for %s RVolume must be a value." \
					% self.name)
					
			if n < 0 or n > 100:
				error("RVolume %s out-of-range; must be 0..100." % n)
						
			if n > 30:
				warning("%s is a large RVolume value!" % n)
					
			ln[i] = n

			
		self.rVolume = tuple( ln )
			
		if gbl.debug:
			print "Set %s Rvolume to " % self.name,
			printList(ln)


	def setRSkip(self, ln):
		""" Set the note random skip factor for a track. """

		ln = self.lnExpand(ln, 'RSkip')

		for i in range(len(ln)):

			n = stoi(ln[i], "Expecting integer after in RSkip")
			
			if n < 0 or n > 99:
				error("RSkip arg must be 0..99")
			
			ln[i] = n 

		self.rSkip = tuple( ln )
			
		if gbl.debug:
			print "Set %s RSkip to " % self.name,
			printList(ln)



	def setRTime(self, ln):
		""" Set the timing randomizer for a track. """
		
		ln=self.lnExpand(ln,  'RTime')
		
		for i in range(len(ln)):
			n=stoi(ln[i], "Expecting an integer for Rtime")
			if n < 0 or n > 100:
				error("RTime %s out-of-range; must be 0..100." % n)
			
			ln[i]=n
				
		self.rTime = tuple( ln )

		if gbl.debug:
			print "Set %s RTime to " % self.name,
			printList(ln)

		
	def setRnd(self):
		""" Enable random pattern selection from sequence."""
		
		self.seqRnd = 1
		if gbl.debug:
			print "Set %s SeqRnd." % self.name

		
	def setNoRnd(self):
		""" Disable random pattern selection from sequence."""

		self.seqRnd = 0
		if gbl.debug:
			print "Cleared %s SeqRnd." % self.name


	def setDirection(self, ln):
		""" Set scale direction. """
		
		ln = self.lnExpand(ln, "Direction")
		
		vald = ('UP', 'DOWN', 'BOTH', 'RANDOM')
		for i in range(len(ln)):
			n = ln[i].upper()
			if not n in vald:
				error("Unknown %s Direction. Only '%s' are valid." \
					% (self.name, ' '.join(vald) ))
			ln[i]=n
			
		self.direction = tuple( ln )

		if gbl.debug:
			print "Set %s Direction to " % self.name,
			printList(ln)


	def setScaletype(self, ln):
		""" Set scale type. """
		
		ln = self.lnExpand(ln, "ScaleType")
		
		vald = ('MAJOR', 'MINOR', 'CHROMATIC', 'AUTO')
		for i in range(len(ln)):
			n = ln[i].upper()
			if not n in vald:
				error("Unknown %s ScaleType. Only '%s' are valid." \
					% (self.name, ' '.join(vald) ))
			ln[i]=n
			
		if self.vtype == 'SCALE':
			self.scaleType = tuple( ln )
	
			self.notes = None		
	
		if gbl.debug:
			if self.vtype=='SCALE':
				print "Set %s ScaleType to " % self.name,
				printList(ln)
			else:
				print "Set %s ScaleType INGORED" % self.name


	
	def setInvert(self, ln):
		""" Set inversion for track.
			
			This can be applied to any track,
			but has no effect in drum tracks. It inverts the chord
			by one rotation for each value.
		"""
		
		ln=self.lnExpand(ln, "Invert")
		
		vwarn = 0
		
		for i in range(len(ln)):
			ln[i] = stoi(ln[i], "Argument for %s Invert must be an integer" \
					% self.name)
			
			if ln[i] and self.voicingMode:
				vwarn = 1
				
		self.invert = tuple( ln )

		if vwarn:
			warning("Setting both VoicingMode and Invert is not a good idea")
		
		if gbl.debug:
			print "Set %s Invert to " % self.name,
			printList(ln)

							
	def setOctave(self, ln):
		""" Set the octave for a track. """
		
		ln=self.lnExpand(ln, 'Octave')
		
		for i in range(len(ln)):
			n = stoi(ln[i], "Argument for %s Octave must be an integer" \
					% self.name)
			if n < 0 or n > 10:
				error("Octave %s out-of-range; must be 0..10." % n)

			ln[i] = n * 12
		
		self.octave = tuple( ln )
		
		if gbl.debug:
			print "Set %s Octave to" % self.name,
			for i in self.octave:
				print i/12,
			print

	def setHarmony(self, ln):
		""" Set the harmony (only valid in solo track). """
		
		
		ln=self.lnExpand(ln, 'Harmony')
		
		for i in range(len(ln)):
			n = stoi(ln[i], "Argument for %s Harmony must be an integer" \
					% self.name)
			if n not in (0,2,3):
				error("Harmony values must be 0, 2 or 3; not '%s'." % n)

			ln[i] = n 
		
		self.harmony = tuple( ln )

		if self.vtype != 'SOLO':
			warning("Harmony setting for %s track ignored" % self.vtype)
			
		if gbl.debug:
			print "Set %s Harmony to" % self.name,
			for i in self.harmony:
				print i,
			print


	def setTranspose(self, t):
		""" Set the transpostione for a track.
		
			This will be called for all tracks. Process() doesn't
			have a point for transpose to be applied to only one track.
			No effect on drum tracks.
		"""
		
		if t < -12 or t > 12:
			error("Tranpose %s out-of-range; must be -12..12." % t)
			
		self.transpose=t
		
	
	def setSeqSize(self, ssize):
		""" Expand existing pattern list. """
		
		def bump(l):
			while len(l) < ssize:
				l += l
			return l
			
		if len(self.sequence):
			self.sequence = bump(self.sequence)
		
		self.invert     = bump(self.invert)
		self.artic      = bump(self.artic)
		self.volume     = bump(self.volume)
		self.voice      = bump(self.voice)
		self.rVolume    = bump(self.rVolume)
		self.rSkip      = bump(self.rSkip)
		self.rTime      = bump(self.rTime)
		self.strum      = bump(self.strum)
		self.octave     = bump(self.octave)
		self.harmony    = bump(self.harmony)
		self.direction  = bump(self.direction)
		self.scaleType  = bump(self.scaleType)
		self.compress   = bump(self.compress)
		self.duplicate  = bump(self.duplicate)
		self.dupRoot  = bump(self.dupRoot)


		if self.vtype == "DRUM":
			self.toneList = bump(self.toneList)


	def setVoice(self, ln):
		""" Set the voice for a track.
		
			Note, this just sets flags, the voice is set in bar().
			ln[] is not nesc. set to the correct length.
		"""
		
		ln=self.lnExpand(ln, 'Voice')
				
		for i in range(len(ln)):
			a = ln[i]
			n=MMAmidi.instToValue(a)

			if n < 0:
				n=stoi(a, "Expecting a valid voice name or value, "
					"not '%s'" % a)
				if n <0 or n > 127:
					error("Voice must be 0..127.")
			ln[i]=n

		self.voice = ln

		if gbl.debug:
			print "Set %s Voice to: " % self.name,
			for a in ln:
				print MMAmidi.valueToInst(a),
			print


	def setVolume(self, ln):
		""" Set the volume for a pattern.
			ln - list of volume names (pp, mf, etc)
			ln[] not nesc. correct length
		"""

		ln=self.lnExpand(ln, 'Volume')
		
		d=gbl.vols
		for i in range(len(ln)):
			a=ln[i].upper()	

			if not d.has_key( a ):
				error("Expecting a valid volume, not '%s'" % ln[i])
			ln[i] = d[a]
			
		self.volume = tuple(ln)
			
		if gbl.debug:
			print "Set %s Volume to " % self.name,
			for a in ln:
				print keyLookup(d,a),
			print


	def setArtic(self, ln):
		""" Set the note articuation value. """
	
		ln=self.lnExpand(ln, 'Articulate')
				
		for i in range(len(ln)):
			a = stoi(ln[i], "Expecting value in articulation setting.")
			if a < 1 or a > 100:
				error("Articulation setting must be 1..100, "
					"not %s." % a)
			
			if a == 100:
				warning("Articulation of 100 for %s might " 
					"cause problems." % self.name )

			ln[i] = a
			
		self.artic = tuple(ln)
			
		if gbl.debug:
			print "Set %s Articulation to " % self.name,
			printList(ln)


	
	def lnExpand(self, ln, cmd):
		""" Validate and expand a list passed to a set command. """
		
		if len(ln) > gbl.seqSize:
			warning("%s list truncated to %s patterns." % 
				(self.name, gbl.seqSize) )
			
		while len(ln) < gbl.seqSize:
			ln += ln
		
		if ln[0] == '/':
			error ("You cannot use a '/' as the first item in a %s list." 
				% cmd)
	
		for i in range(len(ln)):
			if ln[i]=='/':
				ln[i]=ln[i-1]
				
		return ln[0:gbl.seqSize]



	def copySettings(self, cp):
		""" Copy the voicing from a 2nd voice to the current one. """
		
		
		cp=gbl.tnames[cp]
		
		if cp.vtype != self.vtype:
			error("Tracks must be of same type for copy ... %s and %s aren't." % \
				(self.name, cp.name))
				
		self.volume		= cp.volume[:]
		self.rVolume	= cp.rVolume[:]
		self.rSkip		= cp.rSkip[:]
		self.rTime		= cp.rTime[:]
		self.strum		= cp.strum[:]
		self.octave		= cp.octave[:]
		self.harmony	= cp.harmony[:]
		self.direction	= cp.direction[:]
		self.scaleType	= cp.scaleType[:]
		self.voice		= cp.voice[:]
		self.invert		= cp.invert[:]
		self.artic		= cp.artic[:]
		self.compress	= cp.compress[:]
			
		if self.vtype == 'DRUM':
			self.toneList = cp.toneList[:]
		

		if gbl.debug:
			print "Settings from %s copied to %s." % (cp.name, self.name)
	


	##################################################
	## Save/restore grooves
						
	def saveGroove(self, gname):
		""" Define a groove.
		
			Called by the 'DefGroove Name'. This is called for
			each track. 
		
			If 'gname' is already defined it is overwritten.	
		
			Note aux. function which may be defined for each track type.
		"""
		
		self.grooves[gname] = {
			'PATS':    self.pats.copy(),	
			'SEQ':     self.sequence[:],
			'VOLUME':  self.volume[:],
			'RVOLUME': self.rVolume[:],
			'RSKIP':   self.rSkip[:],
			'RTIME':   self.rTime[:],
			'STRUM':   self.strum[:],
			'OCTAVE':  self.octave[:],
			'HARMONY': self.harmony[:],
			'DIR':     self.direction[:],
			'SCALE':   self.scaleType[:],
			'VOICE':   self.voice[:],
			'INVERT':  self.invert[:],
			'ARTIC':   self.artic[:],
			'SEQRND':  self.seqRnd,
			'COMPRESS':self.compress[:],
			'VMODE':   self.voicingMode,
			'VRANGE':   self.voicingRange,
			'VCENTER':   self.voicingCenter,
			'VMOVE':   self.voicingMove[:],
			'DUPLICATE':self.duplicate[:],
			'DUPROOT':self.dupRoot[:],
			'LIMIT':   self.chordLimit}
	
		if self.vtype == 'DRUM':
			self.grooves[gname]['TONES'] = self.toneList[:]
		
	def isGrooveDefined(self, gname):
		""" See if groove name has been defined. """
		
		return self.grooves.has_key(gname)
		
				
	def restoreGroove(self, gname):
		""" Restore a defined groove. """
		

		g = self.grooves[gname]
		
		self.sequence   =  g['SEQ']
		self.volume     =  g['VOLUME']
		self.rTime      =  g['RTIME']
		self.rVolume    =  g['RVOLUME']
		self.rSkip      =  g['RSKIP']
		self.strum      =  g['STRUM']
		self.octave     =  g['OCTAVE']
		self.voice      =  g['VOICE']
		self.harmony    =  g['HARMONY']
		self.direction  =  g['DIR']
		self.scaleType  =  g['SCALE']
		self.invert     =  g['INVERT']
		self.artic      =  g['ARTIC']
		self.seqRnd     =  g['SEQRND' ]
		self.compress   =  g['COMPRESS']
		self.voicingMode = g['VMODE']
		self.voicingRange = g['VRANGE']
		self.voicingCenter = g['VCENTER']
		self.voicingMove = g['VMOVE']
		self.duplicate  =  g['DUPLICATE']
		self.dupRoot  =  g['DUPROOT']
		self.chordLimit =  g['LIMIT']

		if self.vtype == 'DRUM':
			self.toneList = g['TONES']	
				
		# This loop  is necessary. If we just restore self.pats{} 
		# from the saved version, new defs will be lost.
		# Don't ask how I know this.

		for p in g['PATS'].keys():
			self.pats[p]=g['PATS'][p]
		
		
	
	def defVolume(self, name):
		""" Save current volume setting to a name.
		
			Called by the 'SetVolume Name'. This is called for
			each track. 
			
			If 'name' is already defined it is overwritten.	
		"""
			
		self.saveVols[name] = self.volume[:]
		

	def restoreVolume(self, name):
		""" Restore a saved volume. """
		
		if not self.saveVols.has_key(name):
			error("Volume name '%s' not defined" % name)
			
		self.volume=self.saveVols[name]
		
			
	####################################
	## Sequence functions
	
	def setSequence(self, ln):
		""" Set the sequence for a track.
		
		names - a list of pattern names.
			The names do not have to be defined (yet).
			
			Existing sequences are silently overwritten.
		"""

		ln=self.lnExpand(ln, 'Sequence')
		
		if ln[0] == '-':
			ln=[]
			
		else:
			for i in range(len(ln)):	
				ln[i]=ln[i].upper()
		
		self.sequence = tuple(ln)
		
		if gbl.seqshow:
			print "%s sequence set:" % self.name,
			for a in self.sequence:
				print a,
			print

		
	def clearSequence(self):
		""" Clear sequence for track.
			This is also called from __init__() to set the 
			initial defaults for each track.
		"""

		if self.vtype != 'SOLO' or not self.inited:
			self.sequence      =  ()
			self.seqRnd        =  0
			self.scaleType     =  ('AUTO',)
			self.rVolume       =  (0,)
			self.rSkip         =  (0,)
			self.rTime         =  (0,)
			self.octave        =  (4 * 12,)
			self.voice         =  (1,)
			self.harmony       =  (0,)
			self.strum         =  (0,)		
			self.volume        =  (gbl.vols['MF'],)
			self.compress      =  (0,)
			self.voicingMode   =  None
			self.voicingRange  =  12
			self.voicingCenter =  4
			self.voicingMove   =  [0,0]
			self.duplicate     =  (0,)
			self.dupRoot       =  (0,)
			self.chordLimit    =  0
			self.invert = (0,)			
			self.lastChord = []	
	
		if self.riff:
			warning("%s sequence clear deleting unused riff" % self.name)
		self.riff = None
		
		if self.vtype == 'CHORD':
			self.direction = ('UP',)
		else:
			self.direction  =  ('BOTH',)
				
		self.setSeqSize(gbl.seqSize)


	############################
	### Pattern functions
	############################
	
	

	def definePattern(self, name, ln):
		""" Define a Pattern. """

		""" Patterns are shared for all tracks of the same type. 
			We store and retrieve from the base track, so use
			the track type which should be same. Ie: track DRUM3
			has a type name of DRUM.
		"""
		
		name = name.upper()
		pattern, pats, slot = self.defPatRiff(ln)

		if pats.has_key(name):
			redef = 'REDEFINED'
		else:
			redef = 'created'

		pats[name] = pattern
		
		if gbl.pshow:
			print "%s pattern %s %s:" % (slot.title(), name, redef )
			self.printPattern(pattern)


	def setRiff(self, ln):
		""" Define and set a Riff. """

		
		if self.riff:
			warning("%s overwriting unused riff." % self.name)
			
		if len(ln) == 1 and ln[0].upper() == 'Z':
			self.riff = 'Z'
			if gbl.pshow:
				print "%s Riff 'Z'" % self.name
				
		else: 
			pattern, pats, slot = self.defPatRiff(ln)
			self.riff = pattern
				
			if gbl.pshow:
				print "%s Riff:" % self.name
				self.printPattern(pattern)

		

	def defPatRiff(self, ln):
		""" Worker function to define pattern. Shared by definePattern()
			and setRiff().
		"""
			
		def patsort(c1, c2):
			""" Sort a pattern tuple. """
		
			if c1.offset < c2.offset: return -1
			if c1.offset == c2.offset: return 0
			else: return 1
	
				
		# Convert "1 2 3; 4 5 6" to  [ [1,2,3], [4,5,6] ]

		p = []
		ln = ln.split(';')
		for l in ln:
			p.append(l.split())
	
		plist=[]

		slot = self.vtype
		pats=gbl.tnames[slot].pats		
		
		
		for ev in p:
			if len(ev) == 1:
				nm = ev[0].upper()
				if pats.has_key(nm):
					plist.extend(pats[nm])
					continue
				else:
					error("%s is not an existing %s pattern." \
						% (nm, self.vtype.title()) )
					
			plist.append( self.getPgroup(ev))
		

		
		plist.sort(patsort)
		return ( tuple(plist), pats, slot)


	def mulPattern(self, newpat, oldpat, fact):
		"""Create a new pattern by  multiplying an existing pattern.
		
		This can be used to take a pattern of 4 1/4 notes and turn
		it into a pattern of 8 1/8 notes.
		"""

		newpat = newpat.upper()
		pattern, pats, slot = self.mulPatRiff(oldpat, fact)
		
		if pats.has_key(newpat):
			redef = 'REDEFINED'
		else:
			redef = 'created'

		pats[newpat] = pattern

		if gbl.pshow:
			print "%s pattern %s %s as %s * %s:" \
				% (slot, newpat, redef, fact, oldpat)
			self.printPattern(pattern)
		
		

	def mulRiff(self, oldpat, fact):
		""" Define a multiplied riff. """
		
		if self.riff:
			warning("%s overwriting unused riff." % self.name)
			
		pattern, pats, slot = self.mulPatRiff(oldpat, fact)
		
		self.riff = pattern
		
		if gbl.pshow:
			print "%s Riff as %s * %s:" % (slot, fact, oldpat)
			self.printPattern(pattern)
		
		

	def mulPatRiff(self, oldpat, fact):
		""" Shared function to define a pattern or riff using OLD * Factor. """
			
		slot = self.vtype
		pats = gbl.tnames[slot].pats
		
		fact = stoi(fact, "The multiplier arg must be an integer")
		
		if fact<1 or fact >100:
			error("The multiplier arg must be in the range 2 to 100.")

		oldpat = oldpat.upper()
		
		if not pats.has_key(oldpat):
			error("%s pattern '%s' must be defined before we can multipy it"
				% (slot, oldpat))
		
		# Make N copies of pattern, adjusted so that the new copy has all
		# note lengths and start times  adjusted.
		# eg: [[1, 2, 66], [3, 2, 88]]  * 2
		# becomes [[1,4,66], [2,4,88], [3,4,66], [4,4,88]].
				
		new = []
		add = 0
		step = (gbl.BperQ * gbl.QperBar)/fact


		for n in range(fact):
			orig = copy.deepcopy(pats[oldpat])
			for z in orig:
				z.offset = (z.offset / fact) + add
				z.duration /= fact
				if z.duration < 1:
					z.duration = 1
	
				new.append(z)
			add += step
		
		return (tuple(new), pats, slot)
		
	
	def shiftPattern(self, newpat, oldpat, fact):
		"""Create a new pattern by  shifting an existing pattern. """


		newpat = newpat.upper()
		pattern, pats, slot = self.shiftPatRiff(oldpat, fact)

		if pats.has_key(newpat):
			redef = 'REDEFINED'
		else:
			redef = 'created'

		pats[newpat] = pattern

		if gbl.pshow:
			print "%s pattern %s %s as %s Shift %s:" \
				% (slot, newpat, redef, oldpat, fact)
			self.printPattern(pattern)
		
		
	def shiftRiff(self, oldpat, fact):
		""" Create riff by shifting existing pattern. """

		if self.riff:
			warning("%s overwriting unused riff." % self.name)
		
		pattern, pats, slot = self.shiftPatRiff(oldpat, fact)

		self.riff = pattern

		if gbl.pshow:
			print "%s Riff as %s Shift %s:" % (slot, oldpat, fact)
			self.printPattern(pattern)

	def shiftPatRiff(self, oldpat, fact):
			
		slot = self.vtype
		pats = gbl.tnames[slot].pats
		
		fact = stof(fact, "The multiplier arg must be an value")
		
		oldpat = oldpat.upper()

		if not pats.has_key(oldpat):
			error("%s pattern '%s' must be defined before we can shift it"
				% (slot, oldpat))
		
		# Adjust all the beat offsets
		
		new = copy.deepcopy(pats[oldpat])				
		max = gbl.BperQ * (gbl.QperBar)
		for n in new:
			n.offset += fact * gbl.BperQ
			if n.offset < 0 or n.offset > max:
				error("Pattern shift with factor %f has resulted in an "
					"illegal offset." % fact )
		
	
		return (tuple(new), pats, slot)
		
					
	def printPattern(self, pat):
		""" Print a pattern. Used by debugging code."""
		
		s=[]
		for p in pat:
			s.append(" %2.2f %2.0f" % (1+(p.offset/float(gbl.BperQ)),
				p.duration))
				
			if self.vtype == 'CHORD':
				for a in p.vol:
					s.append( " %2.0f" % a)

			elif self.vtype == 'BASS':
				s.append( " %2.0f %2.0f" % (p.noteoffset, p.vol ) )
			
			elif self.vtype == 'ARPEGGIO':
				s.append( " %2.0f " % p.vol )
				
			elif self.vtype == 'DRUM':
				s.append(" %2.0f" %  p.vol)
			
			elif self.vtype == 'WALK':
				s.append(" %2.0f" % p.vol )
				
			s.append(' ;')
			s.append('\n')
		s[-2]=' @'
		print "".join(s)

	
	#########################
	## Music processing
	#########################
		
	def bar(self, ctable):
		""" Process a bar of music. """

		
		# Not having a sequence defined for a track is fine.
		# Just don't do anything for it...
		
		if (not self.sequence and not self.riff) or self.disable:
			return

		# We only permit patterns to be defined for tracks NOT ending
		# in a digit (ie DRUM can have a pattern, DRUM1 can't).
		# So, all the Drum patterns are shared for all the DRUM tracks.
		# We use the 'vtype' as the name for this track.

		pats=gbl.tnames[self.vtype].pats			
		
		# Decide which seq to use. This is either the current
		# seqCount, or if SeqRnd has been set for the track
		# it is a random pattern in the sequence. 

		# The class variable self.seq is set to the sequence to
		# use. 
		
		
		if self.seqRnd:
			self.seq = random.randrange(gbl.seqSize)
		else:
			self.seq = gbl.seqCount
			
		
		sc = self.seq

		# The special pattern 'Z' is reserved. It means do nothing...		
		
		if not self.riff and (self.sequence[sc] == 'Z' or self.sequence[sc]==''):
			return
			
		# MIDI Channel assignment. If no channel is assigned try to find
		# an unused number and assign that.
		
		if not self.channel:
			for c in range(16, 0, -1):
				if not gbl.midiAssigns[c]:
					self.setChannel('%s' % c)
					break

		if not self.channel:
				error("No MIDI channel is avaiable for %s. Try CHShare?" % self.name)


		# We are ready to create musical data. 1st do pending midi commands.
		
		while self.midiPending:
			c, off, v = self.midiPending.pop(0)
			if c == 'TNAME':
				gbl.mtrks[self.channel].addTrkName(off, v)
				if gbl.debug:
					print "%s Track name inserted at offset %s." % \
						(self.name, off)		

			
			elif c == 'GLIS':
				gbl.mtrks[self.channel].addGlis(off, v)
				if gbl.debug:
					print "%s Glis at offset %s set to %s." % \
						(self.name, off, ord(chr(v)))		
				
			elif c == 'PAN':
				gbl.mtrks[self.channel].addPan(off, v)
				if gbl.debug:
					print "%s Pan at offset %s set to %s." % \
						(self.name, off, v)		
				
			elif c == 'CVOLUME':
				gbl.mtrks[self.channel].addChannelVol(off, v)
				if gbl.debug:
					print "%s ChannelVolume at offset %s set to %s" % \
						(self.name, off, v)

			else:
				error("Unknown midi command pending. Call Bob.")

	
		# Set the voice in the midi track if not previously done.

		v = self.voice[sc]		
		if v != self.ssvoice:
			gbl.mtrks[self.channel].addProgChange( gbl.tickOffset,  v)
			self.ssvoice = v
			
			if gbl.debug:
				print "Track %s Voice %s inserted." \
					% (self.name, MMAmidi.valueToInst(v) )

		if self.riff:
			pattern = self.riff
			if pattern == 'Z':
				self.riff = None
				return
		else:
			try:
				pattern = pats[self.sequence[sc]]

			except:
				error("%s Pattern %s is not defined" %
					(self.name, self.sequence[sc]))

		self.trackBar(pattern, ctable)
		
		if self.riff:
			self.riff = None

	
		
	def getChordInPos( self, offset, ctable):
		""" Compare an offset to a list of ctables and return
			the table entry active for the given beat. 
			
			We assume that the first offset in 'ctable' is 0!
			We assme that 'offset' is >= 0!
	
			Returns a ctable.
		"""
		
		for i in range(len(ctable)-1, -1, -1):	# reverse order
			if offset >= ctable[i].offset:
				break
		return ctable[i]



	def adjustVolume(self, v):
		""" Adjust a note volume based on the track and global volume setting. """

		if not v:
			return 0
		
		sc = self.seq
		
		if self.rSkip[sc] and random.random() * 100 < self.rSkip[sc]:
			return 0

		v =  (v * self.volume[sc] * gbl.volume)	/ 10000

		if self.rVolume[sc]:
			a = (v * self.rVolume[sc])/100
			if a:
				v += random.randrange(-a, a)
				
		if v > 127:
			v = 127
		elif  v < 1:
			v = 1
			
		return v
		
			
	def adjustNote(self, n):
		""" Adjust a note for a given octave/transposition.
			Ensure that the note is in range.
		"""

		n += self.octave[self.seq] + self.transpose
		
		while n < 0:
			n += 12
		while n > 127:
			n -= 12
		
		return n

	
	def setBarOffset(self, v):
		""" Convert a string into a valid bar offset in midi ticks. """

		v=stof(v, "Value for %s bar offset must be integer/float" % self.name)	
		
		if v < 1:
			error("Defining %s Pattern, bar offset must be 1 or greater." %
				 self.name)
				 
		if v >= gbl.QperBar+1:
			error("Defining %s Pattern, bar offset must be less than %s." %
				(self.name, gbl.QperBar + 1))
				
				
		return int((v-1) * gbl.BperQ)
		
		
	def getDur(self, d):
		""" Return the adjusted duration for a note. 
		
			The adjustment makes notes more staccato. Valid
			adjustments are 1 to 100. 100 is not recommended.
		"""

		d = (d * self.artic[self.seq]) / 100
		if not d:
			d = 1
			
		return d
	
	
##############################################################
# We have a class for each track type. They are all derived
# from the base 'Pats' class. These classes do special processing
# like parsing a pattern tuple and creating a chord.
##############################################################
	
	
class Bass(Pats):
	""" Pattern class for a bass track. """

	vtype = 'BASS'	
			
	def getPgroup(self, ev):
		""" Get group for bass pattern.
				
			Tuples:  [start, length, note, volume]
		
		"""
		
		a = struct()
		if len(ev) != 4:	
			error("There must be n groups of 4 in a pattern definition.")
			
		a.noteoffset = stoi(ev[2], "Note offset in Bass definition not int.")
		a.vol = stoi(ev[3], "Note volume in Bass definition not int.")
		
		a.duration = getNoteLen( ev[1] )

		a.offset = self.setBarOffset(ev[0])
					
		if a.noteoffset<0 or a.noteoffset>3:
			error("Chord offsets must be 0..3.")
		
		return a


	def trackBar(self, pattern, ctable):
		""" Do the bass bar.
		
		Called from self.bar()
		
		"""
		
		sc = self.seq
		
		for p in pattern:
			ct = self.getChordInPos(p.offset, ctable)

			if ct.bassZ:
				continue
						
			if self.chordLimit:
				ct.chord.limit(self.limit)
				
			if self.compress[sc]:
				ct.chord.compress()
				if p.noteoffset >= ct.chord.len():
					continue
									
			if self.invert[sc]:
				ct.chord.invert(self.invert[sc])

			n = self.adjustNote(ct.chord.offsetNote(p.noteoffset))


			v=self.adjustVolume(p.vol)
			
			if not v:
				continue
			
			gbl.mtrks[self.channel].addPairToTrack(p.offset,
				self.rTime[sc],
				self.getDur(p.duration), n, v )
			
			if self.duplicate[sc]:
				n += self.duplicate[sc]
				if n>=0 and n<128:
					gbl.mtrks[self.channel].addPairToTrack(p.offset,
						self.rTime[sc],
						self.getDur(p.duration), n, v )
						
			ct.chord.reset()	# important, other tracks chord object
	

class Chord(Pats):
	""" Pattern class for a chord track. """

	vtype = 'CHORD'
	
	
	
	
	
	def getPgroup(self, ev):
		""" Get group for chord pattern.
		
			Tuples: [start, length, volume (,volume ...) ]
		"""
			
		a = struct()
		if len(ev) < 3:	
			error("There must be at least 3 items in each group "
				"of a chord pattern definition.")
		
		a.vol = []
		for i in ev[2:]:
			a.vol.append(stoi(i,
				"Expecting integer in vols for Chord definition."))

		while(len(a.vol)<8):	# force 8 volumes
			a.vol.append(a.vol[-1])
		
		a.duration = getNoteLen(ev[1])	
							
		a.offset = self.setBarOffset(ev[0])
			
		return a
		

	def chordVoicing(self, chord):
		""" Voicing algorithm by Alain Brenzikofer. """
		

		sc = self.seq
		vmode=self.voicingMode

		if vmode == "OPTIMAL":
			
			# initialize with a voicing around centerNote
			
			chord.center1(self.lastChord)
			
			# adjust range and center
			
			if not self.vmove:
				chord.center2(self.voicingCenter, self.voicingRange/2)
				
			# move voicing

			elif self.lastChord:
				if (self.lastChord != chord.notes() ) and self.vmove:
					chord.invert(self.vmove)
					self.vmove = 0
					
					# now update voicingCenter
					
					sum=0
					for n in chord.notes():
						sum += n
					c=sum/chord.len()
					
					# If using random voicing move it it's possible to
					# get way off the selected octave. This check ensures
					# that the centerpoint stays in a 1 octave range.
					# Note that if using voicingMove manually (not random)
					# it is quite possible to move the chord centers to very
					# low or high keyboard positions!

					if self.voicingMove[0] == 'RANDOM':
						if c < -6:
							c = -6
						if  c > 9:
							c = 9
						
					self.voicingCenter=c

		elif vmode == "COMPRESSED":
			chord.compress()
		
		elif vmode == "INVERT":
			if chord.root() < -2:
				chord.invert(1)

			elif chord.root() > 2:
				chord.invert(-1)
			chord.compress()
				
		self.lastChord = chord.notes()
		

	def trackBar(self, pattern, ctable):
		""" Do a chord bar.
		
		Called from self.bar()
		
		"""
			
		sc = self.seq
		
		# Set voicing move ONCE at the top of each bar.
		# The voicing code resets vmove to 0 the first
		# time it's used. That way only one movement is
		# done in a bar.
		
		self.vmove = 0
		
		if self.voicingMove[0]:
			if self.voicingMove[0] == 'RANDOM':
				if random.randrange(100) <= self.voicingMove[1]:
					self.vmove = random.choice((-1,1))
			else:
				self.vmove = self.voicingMove[1]

		for p in pattern:
			tb = self.getChordInPos(p.offset, ctable)

			if tb.chordZ:
				continue

			self.crDupRoot = self.dupRoot[sc]
			
			vmode = self.voicingMode
			vols = p.vol[0:tb.chord.len()]
			
			# Limit the chord notes. This works even if THERE IS A VOICINGMODE
			
			if self.chordLimit:
				tb.chord.limit(self.chordLimit)
				
			# Compress chord into single octave if 'compress' is set
			# We do it here, before octave, transpose and invert!
			# Ignored if we have a VOICINGMODE
			
			if self.compress[sc] and not vmode:
				tb.chord.compress()

			# Do the voicing stuff.
			
			if vmode:
				self.chordVoicing(tb.chord)

			# Invert. 

			if self.invert[sc]:
				tb.chord.invert(self.invert[sc])

			# Duplicate the root. This can be set from a DupRoot command
			# or by chordVoicing(). Notes:
			#   root was set earlier and is guaranteed to be the root
			#     note of the current chord.
			#   The volume for the added root will be loudest of the
			#     notes in the chord.
			#   If the new note (after transpose and octave adjustments
			#     is out of MIDI range it will be ignored.
			#   The new note is added at the bottom of the chord so strum
			#     will work properly.
				
			dupn = None
			if self.crDupRoot:
				root = tb.chord.root() + self.crDupRoot
				t = root + self.octave[sc] + self.transpose
				if t >=0 and t < 128:
					dupn = tb.chord.root() + self.crDupRoot
		
			strumAdjust = self.strum[sc] 
			strumOffset = 0
			if strumAdjust and self.direction[sc]=='DOWN':
				strumOffset += strumAdjust * len(chord)
				strumAdjust = -strumAdjust
				
			
			loo =zip(tb.chord.notes(), vols)	# do notes in order
			if dupn:
				loo.insert(0, [dupn, max(vols)] )
				
			loo.sort()				# mainly for strum
			for n,v in loo:
				v = self.adjustVolume( v )
				if not v:
					continue

				n = self.adjustNote(n)
				gbl.mtrks[self.channel].addPairToTrack(p.offset+strumOffset,
					 self.rTime[sc],
					 self.getDur(p.duration), n, v )


				if self.duplicate[sc]:
					n += self.duplicate[sc]
					if n>=0 and n<128:
						gbl.mtrks[self.channel].addPairToTrack(p.offset,
							self.rTime[sc],
							self.getDur(p.duration), n, v )

				
				strumOffset += strumAdjust
				
			tb.chord.reset()	# important, other tracks chord object
	
		# Adjust the voicingMove counter at the end of the bar
		
		if self.voicingMove[0] and self.voicingMove[0] != 'RANDOM':
				self.voicingMove[0] -= 1
				


class Arpeggio(Pats):
	""" Pattern class for an arpeggio track. """

	vtype = 'ARPEGGIO'
	arpOffset=0
	arpDirection=1
	lastDirection=1

	def getPgroup(self, ev):
		""" Get group for apreggio pattern.
		
			Tuples: [start, length, volume]
		"""
	
		a = struct()
		if len(ev) != 3:	
			error("There must be at exactly 3 items in "
				"each group for an apreggio define.")
				
		a.vol = stoi(ev[2], "Type error in Arpeggio definition.")
		
		a.duration = getNoteLen(ev[1])
				
		a.offset = self.setBarOffset(ev[0])
		
		return a
	 		

	
	def trackBar(self, pattern, ctable):
		""" Do a arpeggio bar.
		
		Called from self.bar()
		
		"""
		
		sc = self.seq
		direct = self.direction[sc]
		
		if self.lastDirection != direct:
			self.lastDirection = direct
			self.arpOffset=0
		
		for p in pattern:
			
			tb = self.getChordInPos(p.offset, ctable)

			if tb.arpeggioZ:
				continue

			if direct == 'DOWN':
				self.arpDirection = -1

			if self.chordLimit:
				tb.chord.limit(self.chordLimit)

			if self.compress[sc]:
				tb.chord.compress(chord)
				
			if self.invert[sc]:
				tb.chord.invert(self.invert[sc])
			
			if direct == 'BOTH':
				if self.arpOffset < 0:
					self.arpOffset = 1
					self.arpDirection = 1
				elif self.arpOffset >= tb.chord.len():
					self.arpOffset = tb.chord.len()-2
					self.arpDirection = -1
			
			elif direct == 'UP':
				if self.arpOffset >= tb.chord.len() or self.arpOffset < 0:
					self.arpOffset = 0
					self.arpDirection = 1
					
			elif direct == 'DOWN':
				if self.arpOffset < 0 or self.arpOffset >= tb.chord.len():
					self.arpOffset = tb.chord.len()-1
					self.arpDirection = -1

			if direct == 'RANDOM':
				n = self.adjustNote( random.choice(tb.chord.notes() ))
			else:
				n = self.adjustNote( tb.chord.offsetNote(self.arpOffset) )

			self.arpOffset += self.arpDirection
			
			v=self.adjustVolume(p.vol)
			if not v:
				continue

			
			gbl.mtrks[self.channel].addPairToTrack(p.offset,
				self.rTime[sc],
				self.getDur(p.duration), n, v )

			if self.duplicate[sc]:
				n += self.duplicate[sc]
				if n>=0 and n<128:
					gbl.mtrks[self.channel].addPairToTrack(p.offset,
						self.rTime[sc],
						self.getDur(p.duration), n, v )

			
			tb.chord.reset()	# important, other tracks chord object


class Drum(Pats):
	""" Pattern class for a drum track. """

	vtype = 'DRUM'
	toneList = [38]

	def setTone(self, ln):
		""" Set a tone list. Only valid for DRUMs.
		    ln[] is not nesc. the right length.
		"""
		
		ln=self.lnExpand(ln, 'Tone')
		
		for i in range(len(ln)):
			a = ln[i]
			n=MMAmidi.drumToValue(a)
			if n < 0:
				n=stoi(a, "Expecting a valid drum name or value for "
					"drum, not '%s'" % a)
			if n <0 or n > 127:
				error("Note in Drum Tone list must be 0..127.")
			ln[i] = n

		self.toneList = ln

	
	def getPgroup(self, ev):
		""" Get group for a drum pattern.
		
			Tuples: [start, length, volume]
		"""

		a = struct()

		if len(ev) != 3:	
			error("There must be at exactly 3 items in "
				"each group of a drum define.")
		
		# Volume
					
		a.vol = stoi(ev[2], "Type error in Drum volume.")

		# Duration
		
		a.duration = getNoteLen(ev[1])
		
		a.offset = self.setBarOffset(ev[0])

		return a
		

	
	def trackBar(self, pattern, ctable):
		""" Do a drum bar.
		
		Called from self.bar()
		
		"""
		
		sc = self.seq
		
		for p in pattern:
			tb = self.getChordInPos(p.offset, ctable)
			
			if tb.drumZ:
				continue

			n = self.toneList[sc]
			
			v = self.adjustVolume(p.vol)
			if not v:
				continue

			gbl.mtrks[self.channel].addPairToTrack( p.offset,
				self.rTime[sc],
				self.getDur(p.duration), n, v )
			

	
class Walk(Pats):
	""" Pattern class for a walking bass track. """

	vtype = 'WALK'
	
	lastScale = ''
	lastType = ''
	bassOffset = 0
	notes = None
	

	def getPgroup(self, ev):
		""" Get group for walking bass pattern.
		
			Tuples: [start, length, volume]
		"""

		a = struct()
		if len(ev) != 3:	
			error("There must be at exactly 3 items in each group " +\
					"in a Walking Bass definition.")
			
		a.vol = stoi(ev[2], "Type error in Walking Bass definition")

		a.duration = getNoteLen(ev[1])				
					
		a.offset = self.setBarOffset(ev[0])	
					
		return a
		

	
	def trackBar(self, pattern, ctable):
		""" Do a waling  bass bar.
		
		Called from self.bar()
		
		"""

		# Careful if you change this. The up/down direction setting relies
		# on the up/down pattern with the 5 position being the highest.
		
		majNotes=[0,2,4,5,7,5,4,2]	# first 5 notes of a major scale
		minNotes=[0,2,3,5,7,5,3,2]	# minor
		dimNotes=[0,2,3,6,6,3,2,0]	# dim ???

		sc=self.seq		
		dir = self.direction[sc]

		for p in pattern:
		
			tb = self.getChordInPos(p.offset, ctable)
			
			if tb.walkZ:
				continue
				
			root = tb.chord.root() 		# root note of chord
			type = tb.chord.type()		# M, m, 7, etc.
			scale = tb.chord.scale()	# C, Eb, etc		
			
			if scale != self.lastScale or type != self.lastType or \
					not self.notes:
				self.lastScale = scale
				self.lastType = type
				self.bassOffset = 0
				if type[0] == 'm':
					self.notes=minNotes
				elif type=='dim':
					self.notes=dimNotes
				else:
					self.notes=majNotes
					
				if dir == 'UP' or dir == 'DOWN':
					self.notes = self.notes[:5]
					if dir == 'DOWN':
						self.notes.reverse()				
			
			
			if self.bassOffset >= len(self.notes):
				self.bassOffset = 0


			# Note is determined by adding root note to note list
		

			n = self.adjustNote( root + self.notes[self.bassOffset] )

			if dir == 'UP' or dir =='DOWN':
				self.bassOffset += 1
			else:
				self.bassOffset += random.choice((0,1,1,1,1,1,2,2,3,3))
	
			v=self.adjustVolume(p.vol)

			if not v:
				continue

			gbl.mtrks[self.channel].addPairToTrack( p.offset,
				self.rTime[sc],
				self.getDur(p.duration), n, v )
	
			if self.duplicate[sc]:
				n += self.duplicate[sc]
				if n>=0 and n<128:
					gbl.mtrks[self.channel].addPairToTrack(p.offset,
						self.rTime[sc],
						self.getDur(p.duration), n, v )

		

class Scale(Pats):
	""" Pattern class for a Scale track. """

	vtype = 'SCALE'
	
	lastScale = None
	lastType = 'x'
	lastDirect = 1
	lastCompress = 0
	sOffset = 0
	notes = None
	dirfact = 1

	def getPgroup(self, ev):
		""" Get group for scale patterns.
		
			Tuples: [start, length, volume]
		"""

		a = struct()
		if len(ev) != 3:	
			error("There must be at exactly 3 items in each group " +\
					"in a Scale definition.")
			
		a.vol = stoi(ev[2],"Type error in Scale definition")

		a.duration = getNoteLen(ev[1])				
					
		a.offset = self.setBarOffset(ev[0])	
					
		return a
	
	def getScale(self, c, root, len):
		if c == 'm':
			s = [0, 2, 3, 5, 7, 8, 10, 12]	# natural minor
		elif c == 'M':
			s = [0, 2, 4, 5, 7, 9, 11, 12]	# major
		elif c == 'c':
			s = range(0,12)
		
		scale=[]
		for t in range(len):
			for n in s:
				scale.append(n + t*12)
			

		return [root + x for x in scale]
	

	def trackBar(self, pattern, ctable):
		""" Do a scale bar.
		
			Called from self.bar()
		"""
	
		sc = self.seq
		direct = self.direction[sc]

		# Set/check any difference in the compress setting between
		# this and the previous bar.

		t = self.compress[sc]
		if t != self.lastCompress:
			self.lastCompress = t
			self.notes = None
			
		if self.lastDirect != direct:
			self.lastDirect = direct
			self.notes = None

		for p in pattern:
		
			tb = self.getChordInPos(p.offset, ctable)
			
			if tb.scaleZ:
				continue
			
			root = tb.chord.root()	 		# root note of chord
			ctype = tb.chord.type()			# M, m, 7, m6,  etc.
			scale = tb.chord.scale()		# C, Eb, etc		

			
			# We set a new run of notes if the chordtype changes between
			# major and minor or if there isn't a note run.
			# This also resets the sOffset, the current position
			# in the run.
			
			if (ctype[0] == 'm' and self.lastType[0] != 'm') or \
					(ctype[0] != 'm' and self.lastType[0] == 'm') or \
					(not self.notes):
				
				self.lastScale = scale
				self.lastType = ctype
	
				ocount = 3
				
				if self.compress[sc]:
					ocount = self.compress[sc]

				stype = self.scaleType[sc]
				
				if stype == "CHROMATIC":
					self.notes = self.getScale('c', root, ocount)
				elif stype == 'MAJOR':
					self.notes = self.getScale('M', root, ocount)
				elif stype == 'MINOR':
					self.notes = self.getScale('m', root, ocount)
				else:
					if ctype[0] == 'm':
						self.notes=self.getScale('m', root, ocount)
					else:
						self.notes=self.getScale('M', root, ocount)

				if direct == 'DOWN':
					self.dirfact = -1
					self.sOffset = len(self.notes)-1
				else:
					self.dirfact = 1
					self.sOffset = 0


			# Keep offset into note list in range
			
			if self.sOffset >= len(self.notes):
				if direct == 'BOTH':
					self.dirfact = -1
					self.sOffset = len(self.notes)-2
				else:
					self.sOffset = 0
					
			elif self.sOffset < 0:
				if direct == 'BOTH':
					self.dirfact = 1
					self.sOffset = 1
				else:
					self.sOffset = len(self.notes)-1
					
			if direct == 'RANDOM':
				n=self.adjustNote(random.choice(self.notes))
			else:
				n = self.adjustNote( self.notes[self.sOffset] )
				self.sOffset += self.dirfact
			
			v=self.adjustVolume(p.vol)
			if not v:
				continue


			gbl.mtrks[self.channel].addPairToTrack( p.offset,
				self.rTime[sc],
				self.getDur(p.duration), n, v )
	
			if self.duplicate[sc]:
				n += self.duplicate[sc]
				if n>=0 and n<128:
					gbl.mtrks[self.channel].addPairToTrack(p.offset,
						self.rTime[sc],
						self.getDur(p.duration), n, v )



class Melody(Pats):
	""" The melody and solo tracks are identical, expect that
		the solo tracks DO NOT get saved in grooves and are only
		initialized once.
	"""
	
	vtype = 'MELODY'
	
	midiNotes = {'c':0, 'd':2, 'e':4, 'f':5, 'g':7, 'a':9, 'b':11}
	
	sigFlat=['b','e','a','d','g','c','f']
	sigSharp=['f','c','g','d','a','e','b']

	""" These are replace funcs from the parent class. They overide
		the normal pattern/solo setting routines. Note that the only
		valid routine is setSolo(). 
	"""

	def setRiff(self, ln):
		if self.riff and not self.disable:
			warning("%s: Overwriting unused melody/solo" % self.name)
		self.riff = ln
		
	def definePattern(self, name, ln):
		self.paterror()
		
	def mulPattern(self, newpat, oldpat, fact):
		self.paterror()

	def shiftPattern(self, newpat, oldpat, fact):		
		self.paterror()
		
	def paterror():
		error("Melody/solo patterns cannot be defined.")
	
	def mulRiff(self, oldpat, fact):
		error("Melody/solo track patterns cannot be multiplied.")
		
	def shiftRiff(self, oldpat, fact):
		error("Melody/solo track patterns cannot be shifted.")

	
	# Add a harmony note based on the chord.

	def harmonize(self, intr, note, chord):

		# Copy the chord and adjust the octave of the chord
		# to match the note

		intr -= 1
		chord = chord[:]
		while note <= chord[0]:
			for i in range(len(chord)):
				chord[i]-=12
		while note > chord[0]+12:
			for i in range(len(chord)):
				chord[i]+=12
				
		# find the highest note in the chord less than the note
		
		ret=-999
		for n in chord:
			if n<note:
				ret=n
		
		# Failsave, should never occur. But if we don't find a harmony note
		# just duplicate the note.
		
		if ret==-999:
			ret = note
			
		ret = [ret]

		# If called with intr==1 (harmony==2) then grab a 2nd harmony note
		
		if intr:
			ret.extend(self.harmonize(intr, ret[len(ret)-1], chord))
			
		return ret



	def trackBar(self, notes, ctable):
		""" Do the solo/melody line. Called from self.bar() """

		length = '4'
		lastc = ''
		sc = self.seq
		barEnd = gbl.BperQ*gbl.QperBar

		velocity=self.adjustVolume( self.volume[sc] ) 
		
		harmony = self.harmony[sc]		

		""" We maintain a keysig table. There is an entry for each note,
			either -1,0,1 corresponding to flat,natural,sharp. We populate
			the table for each bar from the keysig value. As we process
			the bar data we update the table. There is one flaw here---in
			real music an accidental on a note in a give octave does not
			effect the following same-named notes in different octaves.
			In this routine IT DOES. 
		"""
		
		acc = {'a':0, 'b':0, 'c':0, 'd':0, 'e':0, 'f':0, 'g':0}

		if gbl.keySig[0]=='b':
			for a in range(gbl.keySig[1]):
				acc[self.sigFlat[a]]=-1
				
		if gbl.keySig[0]=='#':
			for a in range(gbl.keySig[1]):
				acc[self.sigSharp[a]]=1


		""" The initial string is in the format "1ab;4c;;4r;". The trailing
			';' is important and needed. If we don't have this requirement
			we can tell if the last note is a repeat of previous. For example,
			if we have coded "2a;2a;" as "2a;;" and we didn't have the 'must end
			with ;' rule, we end up with "2a;" and then we make this into
			2 notes...or do we. Easier to make a rule.
		"""

		if not notes.endswith(';'):
			error("All Solo strings must end with a ';'")
		
		n = notes.split(';')
		n.pop()	
		
		""" Take our list of note/value pairs and decode into
			a list of midi values. Quite ugly.
		"""
		
		notes={}
		offset = 0
		
		for a in n:
		
			if offset >= barEnd:
				error("Attempt to start a Solo note '%s' after end of bar." % a)
	
			# Strip out spaces, tabs were deleted with various split()s
			
			a=a.replace(' ', '')
				
			# strip out all '<volume>' setting and adjust velocity
			
			a, vls = pextract(a, "<", ">")
			if vls:
				if len(vls) > 1:
					error("Only 1 volume string is permitted per note-set")
		
				vls = vls[0].upper()
				if not gbl.vols.has_key( vls ):
					error("%s string Expecting a valid volume, not '%s'" % \
						(self.name, vls))
				velocity=self.adjustVolume(gbl.vols[vls])
				
		
			p=len(a)
			for i in range(p):
				if a[i].isalpha():
					p=i
					break

			l = a[:p]
			c = a[p:]
	
			if not l:
				l=length
			else:
				l=getNoteLen(l)
				
			if not c:
				c=lastc
				if c == '':
					error("You must specify the first note in a solo line")

			length = l	# defaults for next loop
			lastc = c
				
			""" Convert the note part into a series of midi values
				Notes can be a single note, or a series of notes. And
				each note can be a letter a-g (or r), a '#,&,n' plus
				a series of '+'s or '-'s
			"""
				
			c=list(c)

			while c:
			
				# Parse off note name or 'r' for a rest
				
				name = c.pop(0)
				
				if name == 'r':
					if notes.has_key(offset) or c:
						error("You cannot combine a rest with a note in "
							"a chord for solos.")
					break;

				if not self.midiNotes.has_key(name):
					error("%s encountered illegal note name '%s'." % (self.name, name))
					
				v=self.midiNotes[ name ] 
				
				
				# Parse out a "#', '&' or 'n' accidental.
				
				if c and c[0]=='#':
					c.pop(0)
					acc[name] = 1
				elif c and c[0]=='&':
					c.pop(0)
					acc[name] = -1
				elif c and c[0]=='n':
					c.pop(0)
					acc[name] = 0
						
				v+=acc[name]
				
				# Parse out +/- (or series) for octave
				
				if c and c[0] == '+':
					while c and c[0] == '+':
						c.pop(0)
						v+=12
				elif c and c[0] == '-':
					while c and c[0] == '-':
						c.pop(0)
						v-=12
				
				if not notes.has_key(offset):
					nry=notes[offset]=struct()
					nry.dur=l
					nry.velocity=velocity
					nry.nl=[]
				notes[offset].nl.append(v)
			 
			if harmony >= 2 and notes.has_key(offset):
				ary=notes[offset]
				if len(ary.nl)==1:
					tb = self.getChordInPos(offset, ctable)

					if not tb.chordZ:
						ary.nl.extend(self.harmonize(harmony-1, ary.nl[0],
							tb.chord.bnotes() ) )
					
			offset += length

		if offset > barEnd:
			warning("%s, end of last note overlaps end of bar by %s beat(s)." % \
				(self.name, (offset-barEnd)/gbl.BperQ))
	
		keys=notes.keys()
		keys.sort()

		for offset in keys:
			ary=notes[offset]
			l=ary.dur
			v=ary.velocity
			for n in ary.nl:
				gbl.mtrks[self.channel].addPairToTrack( offset,
					self.rTime[sc],
					self.getDur(l),
					self.adjustNote(n),
					v )
			

			


class Solo(Melody):
	""" Pattern class for a solo track. """

	vtype = 'SOLO'	
	
	
	# Grooves are not saved/restored for solo tracks.
	
	def restoreGroove(self, gname):
		self.setSeqSize(gbl.seqSize)
		
	def saveGroove(self, gname):
		pass
	
	
