
# MMApatWalk.py

"""
	This module is an integeral part of the program 
	MMA - Musical Midi Accompaniment.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	Bob van der Poel <bvdp@uniserve.com>
	
"""
	



import random
from MMAharmony import harmonize
import MMAglobals;  gbl = MMAglobals
from MMAcommon import *
from MMApat import PC


class Walk(PC):
	""" Pattern class for a walking bass track. """

	vtype = 'WALK'
	
	def getPgroup(self, ev):
		""" Get group for walking bass pattern.
		
			Tuples: [start, length, volume]
		"""

		a = struct()
		if len(ev) != 3:	
			error("There must be at exactly 3 items in each group in " 
					"a Walking Bass definition, not <%s>." % ' '.join(ev))
			
		a.vol = stoi(ev[2], "Type error in Walking Bass definition")

		a.duration = getNoteLen(ev[1])				
					
		a.offset = self.setBarOffset(ev[0])	
					
		return a
		

	walkChoice = 0	
	
	def trackBar(self, pattern, ctable):
		""" Do a waling  bass bar.
		
		Called from self.bar()
		
		"""

		sc=self.seq		
		dir = self.direction[sc]
		unify = self.unify[sc]
		
		for p in pattern:
		
			tb = self.getChordInPos(p.offset, ctable)
			
			if tb.walkZ:
				continue
				
			root = tb.chord.root() 		# root note of chord

			""" Create a scale based on the chord. We do this for each
				beat. A bit of overwork if the chord between beats
				is the same, but it's pretty fast as is. Note that
				the offset is kept between calls/beats/bars and
				with changing chords a certain amount of variety is
				introduced.
				
				The scale produced starts off being the 1st 6 notes
				of a major scale (up and down): (c,d,e,f,g,a,g,f,e,d,c).
				This is transformed so that no notes conflict with the
				scale notes. A few examples based on the chords:
					
					C      C,D,E,F,G,A,B 
					C9     C,D,E,F,G,Bb,
					C7b5   C,D,E,F,Gb,Bb
					Cdim   C,Eb,Eb,F,Gb,A,B
					
				The complete range of notes is NOT present in the
				resulting scale, but I think this is better than having
				wrong notes. With the randomizing of the steps, changing
				chords, etc. the walk sounds pretty good, if un-inspired.					"""
			
			cd=tb.chord.bnotes() 
			wNotes = []
			r= tb.chord.root()
			scale = [ r + a  for a in (0,2,4,5,7,9,11,9,7,5,4,2) ]
			for n in scale:
				if n in cd:
					pass
				elif n-1 in cd:
					n -= 1
				elif n+1 in cd:
					n += 1 	
				else:
					continue
				wNotes.append(n)
	
			if dir in ('UP', 'DOWN'):
				wNotes =  wNotes[:wNotes.index(max(wNotes))+1]
				if dir == 'DOWN':
					wNotes.reverse()				
		

			# Actual note is determined by adding root note to note list

			if self.walkChoice >= len(wNotes) or self.walkChoice < 0:
				self.walkChoice = 0

			note = wNotes[self.walkChoice] 
			
			# Adjust offset for next time 
			
			if dir in ('UP', 'DOWN'):
				self.walkChoice += 1
			else:
				self.walkChoice += random.choice( (-1,0,0,2,2,1,1,1,1,1,1,1))

	
			v=self.adjustVolume(p.vol, p.offset)

			if not v:
				continue

			if not self.harmonyOnly[sc]:
				gbl.mtrks[self.channel].addPairToTrack(
					p.offset,
					self.rTime[sc],
					self.getDur(p.duration),
					self.adjustNote(note),
					v, unify )
	
			if self.duplicate[sc]:
				n = self.adjustNote(note) + self.duplicate[sc]
				if n>=0 and n<128:
					gbl.mtrks[self.channel].addPairToTrack(
						p.offset,
						self.rTime[sc],
						self.getDur(p.duration),
						n,	v, unify)

			if self.harmony[sc]:
				ch = self.getChordInPos(p.offset, ctable).chord.notes()
				h = harmonize(self.harmony[sc], note, ch)
				for n in h:
					gbl.mtrks[self.channel].addPairToTrack(
						p.offset,
						self.rTime[sc],
						self.getDur(p.duration),
						self.adjustNote(n), 
						self.adjustVolume(p.vol, -1),
						unify )



