<?php

# Copyright 2006 Soeren Boll Overgaard <boll@fork.dk>
#
# This file is part of mlmmjadmd.
#
# mlmmjadmd is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# mlmmjadmd is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with mlmmjadmd; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

class Mlmmjadmd {

	var $debug = 0;
	var $host = "localhost";
	var $port = "10000";
	var $username = "None";
	var $password = "None";
	var $client;
	var $message = ""; /* The message part of the reply */
	var $ok = ""; /* The status code from the reply */
	var $reply = ""; /* The complete reply */
	var $err = false;
	var $err_str = "";
	var $data_lines = array();
	var $booleanValue;

	/* Constructor */
	function Mlmmjadmd($host, $port) {
		if ($this->debug) print "Attempting to connect to $host at port $port\n";
		$this->client = stream_socket_client("tcp://".$host.":".$port);
		if (!$this->client) {
			if ($this->debug) print "Unable to connect. Setting err=true\n";
			$this->err = true;
			$this->err_str = "Unable to connect to $host at port $port";
		}
	}

	/* Getters */

	function getDebug() {
		return $this->debug;
	}

	function setDebug($debug) {
		$this->debug = $debug;
	}

	function getBooleanResult() {
		return $this->booleanValue;
	}

	/* Return true if status of last command was OK, false otherwise */
	function getStatusCode() {
		return $this->ok;
	}

	/* Return the complete verbatim reply received from 
	 * the server after sending previous command */
	function getReply() {
		return $this->reply;
	}

	/* Return true if we were unable to parse the previous command */
	function getError() {
		return $this->err;
	}

	/* Return a string describing the error that occurred last. 
	 * Only meaningful if getError() returns true */
	function getErrorString() {
		return $this->err_str;
	}

	/* Get the array containing the data lines (if any) returned from 
	 * the last command */
	function getData() {
		return $this->data_lines;
	}

	function getDataItem() {
		if (!is_array($this->data_lines) || sizeof($this->data_lines)<1) {
			return null;
		} else {
			return $this->data_lines[0];
		}
	}

	/* Internal utility functions */
	function parseReply($reply) {

		$this->err = false;
		$this->err_str = "";
		$this->ok = true;

		$parts = explode(" ", $reply, 2);
		if (sizeof($parts)==2) {
			$this->message = $parts[1];
			$this->reply = $reply;
			if ($parts[0]=="OK") {
				if ($this->debug) print "Reply contained OK\n";
				$this->ok = true;
			} else {
				if ($this->debug) print "Reply did not contain OK\n";
				$this->ok = false;
				$this->err_str = $parts[1];
			}
		} else {
			$this->err = true;
			$this->err_str = "Unable to parse reply";
		}
	}

	function sendCommand($command) {

		global $debug;

		if ($this->debug) print "-> $command\n";
		fwrite($this->client, $command."\n");
		$result = fgets($this->client);
		if ($this->debug) print "<- $result\n";
		return $result;
	}

	function sendBooleanCommand($command) {
		$this->sendDataCommand($command);
		if ($this->getStatusCode()) {
			$data = $this->getData();
			if (sizeof($data)==1) {
				if (trim($data[0])=="True") {
					$this->booleanValue = true;
				} else {
					$this->booleanValue = false;
				}
			} else {
				if ($this->debug) print "size of data not 1 (it is ".sizeof($data).") in call to sendBooleanCommand.";
			}
		} else {
			if ($this->debug) print "sendBooleanCommand call to sendDataCommand failed\n";
		}
	}

	function sendDataCommand($command) {

		if ($this->debug) print "Sending data command\n-> $command\n";
		fwrite($this->client, $command."\n");
		/* If the first line received starts with the "LI" keyword, we will 
		 * receive data, otherwise, something strange happened */
		$result = trim(fgets($this->client));
		if ($this->debug) print "<- $result\n";
		$parts = explode(" ", $result, 2);
		if ($this->debug) print_r($parts);

		if (sizeof($parts)==2) {
			if ($this->debug) print "Parsing LI line\n";
			/* Ok, we have the status code in $parts[0] */
			if ($parts[0]=="LI") {
				/* We will receive data. The number of lines is in $parts[1] */
				if ($this->debug) print "Detected LI token\n";
				if (is_numeric($parts[1])) {
					$lineCount = (int)$parts[1];
					if ($this->debug) print "Expecting $lineCount data lines\n";
					$this->data_lines = array();
					for ($i=0; $i<$lineCount; $i++) {
						if ($this->debug) print "Reading data line\n";
						$line = trim(fgets($this->client));
						$lineParts = explode(" ", $line, 2);
						if (sizeof($lineParts)==2) {
							array_push($this->data_lines, $lineParts[1]);
						} else {
							if ($this->debug) print "Unable to parse line '".$line."'\n";
						}
					}
				} else {
					$this->err = true;
					$this->err_str = "LI parameter is not numeric. Unable to parse data reply";
					return;
				}
			} else {
				$this->err = true;
				$this->err_str = "First line did not contain LI. Unable to parse data reply";
				return;
			}
		} else {
			$this->err = true;
			$this->err_str = "Unable to parse first data line, so unable to determine number of data lines";
			return;
		}
	
		if ($this->debug) print_r($this->data_lines);
		$result = trim(fgets($this->client));
		$this->parseReply($result);
	}

	function authenticate($username, $password) {
		$result = $this->sendCommand("authenticate $username $password");
		$this->parseReply($result);
	}

	function listlists() {
		$result = $this->sendDataCommand("listlists");
	}

	function addlist($list, $domain, $owner) {
		$result = $this->sendCommand("addlist $list $domain $owner");
		$this->parseReply($result);
	}

	function removelist($list) {
		$result = $this->sendCommand("removelist $list");
		$this->parseReply($result);
	}

	function listmoderators($list) {
		$result = $this->sendDataCommand("listmoderators $list");
	}

	function addsubscriber($list, $address) {
		$result = $this->sendCommand("addsubscriber $list $address");
		$this->parseReply($result);
	}

	function removesubscriber($list, $address) {
		$result = $this->sendCommand("removesubscriber $list $address");
		$this->parseReply($result);
	}

	function listdigestsubscribers($list) {
		$result = $this->sendDataCommand("listdigestsubscribers $list");
	}

	function adddigestsubscriber($list, $address) {
		$result = $this->sendCommand("adddigestsubscriber $list $address");
		$this->parseReply($result);
	}

	function removedigestsubscriber($list, $address) {
		$result = $this->sendCommand("removedigestsubscriber $list $address");
		$this->parseReply($result);
	}

	function addmoderator($list, $address) {
		$result = $this->sendCommand("addmoderator $list $address");
		$this->parseReply($result);
	}

	function removemoderator($list, $address) {
		$result = $this->sendCommand("removemoderator $list $address");
		$this->parseReply($result);
	}

	function setmoderated($list) {
		$result = $this->sendCommand("setmoderated $list");
		$this->parseReply($result);
	}

	function unsetmoderated($list) {
		$result = $this->sendCommand("unsetmoderated $list");
		$this->parseReply($result);
	}

	function getmoderated($list) {	
		$this->sendBooleanCommand("getmoderated $list");
	}

	function setclosedlist($list) {
		$result = $this->sendCommand("setclosedlist $list");
		$this->parseReply($result);
	}

	function unsetclosedlist($list) {
		$result = $this->sendCommand("unsetclosedlist $list");
		$this->parseReply($result);
	}

	function getclosedlist($list) {
		$result = $this->sendBooleanCommand("getclosedlist $list");
	}

	function setclosedlistsub($list) {
		$result = $this->sendCommand("setclosedlistsub $list");
		$this->parseReply($result);
	}

	function unsetclosedlistsub($list) {
		$result = $this->sendCommand("unsetclosedlistsub $list");
		$this->parseReply($result);
	}

	function getclosedlistsub($list) {
		$result = $this->sendBooleanCommand("getclosedlistsub $list");
	}

	function settocc($list) {
		$result = $this->sendCommand("settocc $list");
		$this->parseReply($result);
	}

	function unsettocc($list) {
		$result = $this->sendCommand("unsettocc $list");
		$this->parseReply($result);
	}

	function gettocc($list) {
		$result = $this->sendBooleanCommand("gettocc $list");
	}

	function setsubonlypost($list) {
		$result = $this->sendCommand("setsubonlypost $list");
		$this->parseReply($result);
	}

	function unsetsubonlypost($list) {
		$result = $this->sendCommand("unsetsubonlypost $list");
		$this->parseReply($result);
	}

	function setprefix($list, $prefix) {
		$result = $this->sendCommand("setprefix $list $prefix");
		$this->parseReply($result);
	}

	function unsetprefix($list) {
		$result = $this->sendCommand("unsetprefix $list");
		$this->parseReply($result);
	}

	function getprefix($list) {
		$result = $this->sendDataCommand("getprefix $list");
	}

	function addowner($list, $owner) {
		$result = $this->sendCommand("addowner $list $owner");
		$this->parseReply($result);
	}

	function removeowner($list, $owner) {
		$result = $this->sendCommand("removeowner $list $owner");
		$this->parseReply($result);
	}

	function listowners($list) {
		$result = $this->sendDataCommand("listowners $list");
	}

	function addcustomheader($list, $header) {
		$result = $this->sendCommand("addcustomheader $list $header");
		$this->parseReply($result);
	}

	function removecustomheader($list, $header) {
		$result = $this->sendCommand("removecustomheader $list $header");
		$this->parseReply($result);
	}

	function listcustomheaders($list) {
		$result = $this->sendDataCommand("listcustomheaders $list");
	}

	function adddelheader($list, $header) {
		$result = $this->sendCommand("adddelheader $list $header");
		$this->parseReply($result);
	}

	function removedelheader($list, $header) {
		$result = $this->sendCommand("removedelheader $list $header");
		$this->parseReply($result);
	}

	function listdelheaders($list) {
		$result = $this->sendDataCommand("listdelheaders $list");
	}

	function setmemorymailsize($list, $bytes) {
		$result = $this->sendCommand("setmemorymailsize $list $bytes");
		$this->parseReply($result);
	}

	function unsetmemorymailsize($list) {
		$result = $this->sendCommand("unsetmemorymailsize $list");
		$this->parseReply($list);
	}

	function getmemorymailsize($list) {
		$result = $this->sendDataCommand("getmemorymailsize $list");
	}

	function setaddtohdr($list) {
		$result = $this->sendCommand("setaddtohdr $list");
		$this->parseReply($result);
	}

	function unsetaddtohdr($list) {
		$result = $this->sendCommand("unsetaddtohdr $list");
		$this->parseReply($result);
	}

	function getaddtohdr($list) {
		$result = $this->sendBooleanCommand("getaddtohdr $list");
	} 

	function setrelayhost($list, $host) {
		$result = $this->sendCommand("setrelayhost $list $host");
		$this->parseReply($result);
	}

	function unsetrelayhost($list) {
		$result = $this->sendCommand("unsetrelayhost $list");
		$this->parseReply($result);
	}

	function getrelayhost($list) {
		$result = $this->sendDataCommand("getrelayhost $list");
	}

	function setnotifysub($list) {
		$result = $this->sendCommand("setnotifysub $list");
		$this->parseReply($result);
	}

	function unsetnotifysub($list) {
		$result = $this->sendCommand("unsetnotifysub $list");
		$this->parseReply($result);
	}

	function getnotifysub($list) {
		$result = $this->sendBooleanCommand("getnotifysub $list");
	}

	function setdigestinterval($list, $seconds) {
		$result = $this->sendCommand("setdigestinterval $list $seconds");
		$this->parseReply($result);
	}

	function unsetdigestinterval($list) {
		$result = $this->sendCommand("unsetdigestinterval $list");
		$this->parseReply($result);
	}

	function getdigestinterval($list) {
		$result = $this->sendDataCommand("getdigestinterval $list");
	}

	function setdigestmaxmails($list, $count) {
		$result = $this->sendCommand("setdigestmaxmails $list $count");
		$this->parseReply($result);
	}

	function unsetdigestmaxmails($list) {
		$result = $this->sendCommand("unsetdigestmaxmails $list");
		$this->parseReply($result);
	}

	function getdigestmaxmails($list) {
		$result = $this->sendDataCommand("getdigestmaxmails $list");
	}

	function setbouncelife($list, $lifetime) {
		$result = $this->sendCommand("setbouncelife $list $lifetime");
		$this->parseReply($result);
	}

	function unsetbouncelife($list) {
		$result = $this->sendCommand("unsetbouncelife $list");
		$this->parseReply($result);
	}

	function getbouncelife($list) {
		$result = $this->sendDataCommand("getbouncelife $list");
	}

	function setnoarchive($list) {
		$result = $this->sendCommand("setnoarchive $list");
		$this->parseReply($result);
	}

	function unsetnoarchive($list) {
		$result = $this->sendCommand("unsetnoarchive $list");
		$this->parseReply($result);
	}

	function getnoarchive($list) {
		$result = $this->sendBooleanCommand("getnoarchive $list");
	}

	function setnosubconfirm($list) {
		$result = $this->sendCommand("setnosubconfirm $list");
		$this->parseReply($result);
	}

	function unsetnosubconfirm($list) {
		$result = $this->sendCommand("unsetnosubconfirm $list");
		$this->parseReply($result);
	}

	function getnosubconfirm($list) {
		$result = $this->sendBooleanCommand("getnosubconfirm $list");
	}

	function setnoget($list) {
		$result = $this->sendCommand("setnoget $list");
		$this->parseReply($result);
	}

	function unsetnoget($list) {
		$result = $this->sendCommand("unsetnoget $list");
		$this->parseReply($result);
	}

	function getnoget($list) {
		$result = $this->sendBooleanCommand("getnoget $list");
	}
	
	function setsubonlyget($list) {
		$result = $this->sendCommand("setsubonlyget $list");
		$this->parseReply($result);
	}

	function unsetsubonlyget($list) {
		$result = $this->sendCommand("unsetsubonlyget $list");
		$this->parseReply($result);
	}

	function getsubonlyget($list) {
		$result = $this->sendBooleanCommand("getsubonlyget $list");
	}

	function setsmtpport($list, $port) {
		$result = $this->sendCommand("setsmtpport $list $port");
		$this->parseReply($result);
	}

	function unsetsmtpport($list) {
		$result = $this->sendCommand("unsetsmtpport $list");
		$this->parseReply($result);
	}

	function getsmtpport($list) {
		$result = $this->sendDataCommand("getsmtpport $list");
	}
	
	function listsubscribers($list) {
		$this->sendDataCommand("listsubscribers $list");
	}

	function quit() {
		fclose($this->client);
	}
}

?>
