/***********************************************************************\
		Editor mined
		menu handling part
\***********************************************************************/

#include "syntax.h"

#include "mined.h"
#define gotoxy(x, y)	move_cursor (x, y - 1)

#include "io.h"


/***********************************************************************\
	Menu types
\***********************************************************************/

typedef struct {
	char * menuname; int menulen;
	menuitemtype * menuitems;
	} menutype;

typedef struct {
	char * (* dispflag) ();
	void (* toggle) ();
	char * menutitle;
	menuitemtype * menu;
	int menulen;
	} flagitemtype;


/***********************************************************************\
	Global variables
\***********************************************************************/

/* mode tuning */
local FLAG use_graphic_borders = True;
local int menumargin = 1;
local int popupmenumargin = 1;

local int flags_pos = 62;
local int flags_displayed = 9;

/* layout tuning */
local int menu_width = 12;
local int popupmenu_width = 16;
local int horizontal_bar_width = 1;

/* state */
local int first_dirty_line = 0;
local int last_dirty_line = 0;
local menuitemtype * last_menu;
local int last_menulen;
local int last_menuwidth;
local char * last_menutitle;

local FLAG pulldownmenu_displayed = False;
local FLAG popupmenu_displayed = False;

local FLAG menuline_dirty = False;

#ifdef full_Keymap_menu
local char keymap_menuname [22];
#endif


/***********************************************************************\
	Local functions and tables
\***********************************************************************/
local void
	action_menu ();
local void
	MINMEN ();
local menuitemtype Quotemenu [];


/***********************************************************************\
	Flag operations
\***********************************************************************/

local void
	togglenothing ()
{
}

local char *
	dispnothing ()
{
	return " ";
}


local void
	toggleHOP ()
{
	If hop_flag > 0
	Then	hop_flag = 0;
	Else	hop_flag = 1;
	Fi
	displayflags ();
}

local char *
	dispHOP ()
{
	return hop_flag > 0 ? "#H" : "h";
}


#ifdef debug_ring_buffer
local char *
	disp_buffer_open ()
{
	static char f [3];
	f [0] = '#';
	f [1] = buffer_open_flag + '0';
	f [2] = '\0';
	return f;
}
#endif


local void
	toggleVIEW ()
{
	If viewonly == True
	Then	EDITmode ();
	Else	VIEWmode ();
	Fi
}

local void
	select_editmode (menu, i)
	menuitemtype * menu;
	int i;
{
	If i == 0
	Then	EDITmode ();
	Else	VIEWmode ();
	Fi
}

local char *
	dispVIEW ()
{
	return viewonly == True ? "#V" : "E";
}


global void
	toggleappend ()
{
	If append_flag == True
	Then	append_flag = False;
	Else	append_flag = True;
	Fi
	displayflags ();
}

local void
	select_buffermode (menu, i)
	menuitemtype * menu;
	int i;
{
	If i == 0
	Then	append_flag = False;
	Else	append_flag = True;
	Fi
}

local char *
	dispappend ()
{
	If append_flag == True
	Then	return "#+";
	Else	return "=";
	Fi
}


local void
	toggle_autoindent ()
{
	If autoindent == True
	Then	autoindent = False;
	Else	autoindent = True;
	Fi
}

local void
	select_autoindent (menu, i)
	menuitemtype * menu;
	int i;
{
	If i == 0
	Then	autoindent = True;
	Else	autoindent = False;
	Fi
}

local char *
	disp_autoindent ()
{
	If autoindent == True
	Then	return "#»";
	Else	return "#¦";
	Fi
}


local void
	toggleJUSlevel ()
{
	JUSlevel ++;
	If JUSlevel > 2
	Then	JUSlevel = 0;
	Fi
}

local void
	select_justify (menu, i)
	menuitemtype * menu;
	int i;
{
	JUSlevel = i;
}

local char *
	dispJUSlevel ()
{
	If JUSlevel == 0
	Then	return "j";
	Elsif JUSlevel == 1
	Then	return "#j";
	Else	return "#J";
	Fi
}


local void
	toggleJUSmode ()
{
	JUSmode = 1 - JUSmode;
}

local void
	select_paragraph (menu, i)
	menuitemtype * menu;
	int i;
{
	JUSmode = i;
}

local char *
	dispJUSmode ()
{
	return JUSmode == 1 ? "#«" : "# ";
}


local char last_encoding_type = ' ';

local void
	toggle_encoding ()
{
	char * curpos = cur_text;
	If cjk_term == False
	Then
		If utf8_text == True
		Then	utf8_text = False;
			last_encoding_type = 'u';
		Elsif	cjk_text == True
		Then	cjk_text = False;
			last_encoding_type = 'c';
		Elsif	last_encoding_type == 'c'
		Then	cjk_text = True;
			last_encoding_type = ' ';
		Else	utf8_text = True;
			last_encoding_type = ' ';
		Fi
		mapped_text = utf8_text == False && cjk_text == False
				&& cjk_encoding == 'V';
		RD ();
		menuline_dirty = True;
		/* move cursor to or behind actual previous character position */
		move_address (curpos, y);
		/* adjust not to stay amidst a character */
		move_to (x, y);
	Fi
}

local void
	select_encoding (menu, i)
	menuitemtype * menu;
	int i;
{
	char * curpos = cur_text;
	If utf8_text == True
	Then	last_encoding_type = 'u';
	Elsif	cjk_text == True
	Then	last_encoding_type = 'c';
	Fi
	switch (menu [i].tag) {
	case 'L':	/* Latin-1 */
		If cjk_term == True
		Then	ring_bell ();
			return;
		Fi
		utf8_text = False;
		cjk_text = False;
		mapped_text = False;
		break;
	case 'U':	/* Unicode */
		If cjk_term == True
		Then	ring_bell ();
			return;
		Fi
		utf8_text = True;
		cjk_text = False;
		mapped_text = False;
		break;
	case 'B':	/* Big5 */
	case 'G':	/* GB */
	case 'C':	/* CNS */
	case 'J':	/* EUC-JP */
	case 'S':	/* Shift-JIS */
	case 'K':	/* UHC */
	case 'H':	/* Johab */
		utf8_text = False;
		cjk_text = True;
		set_cjk_table (menu [i].tag);
		break;
	case 'V':	/* VISCII */
		If cjk_term == True
		Then	ring_bell ();
			return;
		Fi
		set_mapped_text (menu [i].tag);
		break;
	}
	RD ();
	menuline_dirty = True;
	/* move cursor to or behind actual previous character position */
	move_address (curpos, y);
	/* adjust not to stay amidst a character */
	move_to (x, y);
}

local char cjk_flag [3];

local char *
	disp_encoding_1 ()
{
	If utf8_text == True
	Then	return "#U";
	Elsif	cjk_text == True || visciimode ()
	Then	cjk_flag [0] = '#';
		cjk_flag [1] = cjk_encoding_flag [0];
		cjk_flag [2] = '\0';
		return cjk_flag;
	Else	return "#L";
	Fi
}

local char *
	disp_encoding_2 ()
{
	If utf8_text == True
	Then	return "#8";
	Elsif	cjk_text == True || visciimode ()
	Then	cjk_flag [0] = '#';
		cjk_flag [1] = cjk_encoding_flag [1];
		cjk_flag [2] = '\0';
		return cjk_flag;
	Else	return "#1";
	Fi
}


local void
	toggle_combining ()
{
	char * curpos = cur_text;
	If combining_mode == True
	Then	combining_mode = False;
	Else	combining_mode = True;
	Fi
	RD ();
	menuline_dirty = True;
	/* move cursor to actual previous character position */
	move_address (curpos, y);
	/* adjust not to stay amidst a combined character */
/*	move_to (x, y);	*/
}

local void
	select_combining (menu, i)
	menuitemtype * menu;
	int i;
{
	char * curpos = cur_text;
	FLAG old_combining_mode = combining_mode;
	If i == 0
	Then	combining_mode = True;
	Else	combining_mode = False;
	Fi
	If combining_mode != old_combining_mode
	Then	RD ();
		menuline_dirty = True;
		/* move cursor to actual previous character position */
		move_address (curpos, y);
	Fi
}

local char *
	disp_combining ()
{
	If utf8_text == False || combining_screen == False
	Then	return " ";
	Elsif combining_mode == True
	Then	return "ç";
	Else	return "#´";
	Fi
}


local char *
	disp_leftquote ()
{
	If utf8_text == True
	Then	return quote_mark (quote_type, 0);
	Else	return " ";
	Fi
}

local char *
	disp_rightquote ()
{
	int utfcount;
	unsigned long unichar;
	char * rightquotation;

	If utf8_text == True
	Then	rightquotation = quote_mark (quote_type, 0);
		utf8_info (rightquotation, & utfcount, & unichar);
		If iswide (unichar)
		Then	return "";
		Else	advance_utf8 (& rightquotation);
			utf8_info (rightquotation, & utfcount, & unichar);
			If iswide (unichar)
			Then	return " ";
			Else	return rightquotation;
			Fi
		Fi
	Else	return " ";
	Fi
}


local void
	toggle_HTML ()
{
	If dim_HTML == True
	Then	dim_HTML = False;
	Else	dim_HTML = True;
	Fi
	RD ();
}


local void
	select_keymap_entry (menu, i)
	menuitemtype * menu;
	int i;
{
	setKEYMAP (menu [i].hopitemname);
}


local void
	select_quote_type (menu, i)
	menuitemtype * menu;
	int i;
{
	set_quote_type (i);
}


local char km [3];

local char *
	dispKEYMAP0 ()
{
	If allow_keymap
	Then	km [0] = '#';
		km [1] = keyboard_mapping [0];
		km [2] = '\0';
		return km;
	Else	return " ";
	Fi
}

local char *
	dispKEYMAP1 ()
{
	If allow_keymap
	Then	km [0] = '#';
		km [1] = keyboard_mapping [1];
		km [2] = '\0';
		return km;
	Else	return " ";
	Fi
}


/***********************************************************************\
	Pulldown menu tables
\***********************************************************************/

local void
	separator ()
{
}


local menuitemtype Filemenu [] =
{
	{"Open", EDIT, ""},
	{"View", VIEW, ""},
	{"Save", WT, ""},
	{"New Name", NN, ""},
	{"Save As ...", SAVEAS, ""},
	{"Save & Exit", EXED, ""},
	{"", separator, ""},
	{"paste file", INSFILE, ""},
	{"copy to file", WB, "append to file"},
	{"print buffer", PBUF, ""},
	{"", separator, ""},
	{"check out", checkout, ""},
	{"check in", checkin, ""},
	{"", separator, ""},
	{"next file", NXTFILE, "first file"},
	{"prev file", PRVFILE, "last file"},
	{"n-th file", NTHFILE, ""},
	{"save position", SAVPOS, ""},
	{"Discard & Quit", QUED, ""},
};

local menuitemtype Editmenu [] =
{
	{"mark", MARK, "Go to Mark"},
	{"cut", CUT, "cut & append"},
	{"copy", COPY, "append"},
	{"paste", PASTE, "paste external buf"},
	{"paste previous", YANKRING, ""},
	{"", separator, ""},
	{"Go to Mark", GOMA, ""},
	{"set marker N", MARKER, ""},
	{"go marker N", GOMARKER, ""},
};

local menuitemtype Searchmenu [] =
{
	{"find", SFW, "find Identifier"},
	{"find backw.", SRV, "find Idf. backw."},
	{"find again", RS, "previous search"},
	{"", separator, ""},
	{"find Identifier", SIDFW, ""},
	{"find Idf. backw.", SIDRV, ""},
	{"find matching ()", SCORR, "find wrong enc."},
	{"go to Idf. def.", Stag, "go to def. ..."},
	{"", separator, ""},
	{"substitute", GR, ""},
	{"replace (?)", REPL, ""},
	{"replace on line", LR, ""},
	{"", separator, ""},
	{"Go to Mark", GOMA, ""},
	{"Go to ...", GOTO, ""},
};

local menuitemtype Paragraphmenu [] =
{
	{"justify clever", JUSclever, "clever justify other"},
	{"justify simple", JUS, "justify other mode"},
	{"", separator, ""},
	{"set left margin", ADJLM, ""},
	{"set first line left m.", ADJFLM, ""},
	{"set other lines l. m.", ADJNLM, ""},
	{"set right margin", ADJRM, ""},
	{"toggle auto-justif.", toggleJUSlevel, ""},
	{"toggle line-end mode", toggleJUSmode, ""},
};

local menuitemtype Specialeditmenu [] =
{
	{"insert control char", CTRLINS, ""},
	{"insert HTML tag", HTML, "embed in HTML tags"},
	{"case switch", LOWCAP, "case switch word"},
	{"", separator, ""},
	{"encoding", handleEncodingmenu, ""},
	{"combined display", toggle_combining, ""},
	{"file info", FS, "toggle file info"},
	{"char info", display_code, "toggle char info"},
	{"toggle auto indent", toggle_autoindent, ""},
	{"toggle HTML disp", toggle_HTML, ""},
	{"toggle TAB width", toggletab, ""},
	{"", separator, ""},
	{"combine 2 chars", (voidfunc) UML, ""},
	{"ins. hex char bytes", changehex, "char (hex)"},
	{" unicode char value", changeuni, "char (unicode)"},
	{" decimal char value", changedec, "char (dec)"},
	{" octal char value", changeoct, "char (octal)"},
};

#ifdef msdos
local menuitemtype Screensizemenu [] =
{
#ifdef msdos
	{"bigger", screenbigger, "video mode"},
	{"smaller", screensmaller, "graphic mode"},
	{"more lines", screenmorelines, "font bank"},
	{"less lines", screenlesslines, "char height"},
#else
	{"bigger", screenbigger, ""},
	{"smaller", screensmaller, ""},
	{"more lines", screenmorelines, ""},
	{"less lines", screenlesslines, ""},
#endif
	{"switch line #", LNSW, ""},
	{"circul line #", LNCI, ""},
};
#endif


local menutype menus [] =
{
	{"File", arrlen (Filemenu), Filemenu},
	{"Edit", arrlen (Editmenu), Editmenu},
	{"Search", arrlen (Searchmenu), Searchmenu},
	{"eXtra", arrlen (Specialeditmenu), Specialeditmenu},
	{"Paragraph", arrlen (Paragraphmenu), Paragraphmenu},
#ifdef msdos
	{"Screen size", arrlen (Screensizemenu), Screensizemenu},
#endif
/* this is replaced by the flags keymap menu: */
#ifdef full_Keymap_menu
	{keymap_menuname, arrlen (Keymapmenu), Keymapmenu},
#endif
};


/***********************************************************************\
	Flag menu tables
\***********************************************************************/

/* Unicode quotation marks and their usage in some languages
   The UTF-8 codes of all quotation marks are either C2AB or C2BB 
   or start with either E280 or E380. This may help for efficient 
   detection during file loading (e.g. by checking if 
   current_byte & 0xDE == 0xC2).

	“ 201C; LEFT DOUBLE QUOTATION MARK; DOUBLE TURNED COMMA QUOTATION MARK
		left English, Spanish, Turkish
		right German, Danish, Polish, Russian, Romanian, Slovak, Slovenian, Czech, Hungarian
	” 201D; RIGHT DOUBLE QUOTATION MARK; DOUBLE COMMA QUOTATION MARK
		right English, Spanish, Turkish
		right Dutch, Hungarian
		left/right Swedish, Finnish
		right nested traditional Greek
	„ 201E; DOUBLE LOW-9 QUOTATION MARK; LOW DOUBLE COMMA QUOTATION MARK
		left German, Danish, Polish, Russian, Romanian, Slovak, Sloven, Czech, Hungarian
		left Dutch, Hungarian
	‟ 201F; DOUBLE HIGH-REVERSED-9 QUOTATION MARK; DOUBLE REVERSED COMMA QUOTATION MARK
		left nested traditional Greek
	‘ 2018; LEFT SINGLE QUOTATION MARK; SINGLE TURNED COMMA QUOTATION MARK
	’ 2019; RIGHT SINGLE QUOTATION MARK; SINGLE COMMA QUOTATION MARK
	‚ 201A; SINGLE LOW-9 QUOTATION MARK; LOW SINGLE COMMA QUOTATION MARK
?	‛ 201B; SINGLE HIGH-REVERSED-9 QUOTATION MARK; SINGLE REVERSED COMMA QUOTATION MARK
	« 00AB; LEFT-POINTING DOUBLE ANGLE QUOTATION MARK; LEFT POINTING GUILLEMET
		left French, Italian, Norwegian, Portuguese, Russian, Slovenian, Turkish
		right German, Polish, Slovak, Czech, Serbian, Croatian
	» 00BB; RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK; RIGHT POINTING GUILLEMET
		right French, Italian, Norwegian, Portuguese, Russian, Slovenian, Turkish
		left German, Polish, Slovak, Czech, Serbian, Croatian
		left/right Swedish, Finnish
	‹ 2039; SINGLE LEFT-POINTING ANGLE QUOTATION MARK; LEFT POINTING SINGLE GUILLEMET
	› 203A; SINGLE RIGHT-POINTING ANGLE QUOTATION MARK; RIGHT POINTING SINGLE GUILLEMET
C?	〈3008; LEFT ANGLE BRACKET; OPENING ANGLE BRACKET
C?	〉3009; RIGHT ANGLE BRACKET; CLOSING ANGLE BRACKET
C?	《300A; LEFT DOUBLE ANGLE BRACKET; OPENING DOUBLE ANGLE BRACKET
		left Chinese
C?	》300B; RIGHT DOUBLE ANGLE BRACKET; CLOSING DOUBLE ANGLE BRACKET
		right Chinese
J	「300C; LEFT CORNER BRACKET; OPENING CORNER BRACKET
		left Japanese
J	」300D; RIGHT CORNER BRACKET; CLOSING CORNER BRACKET
		right Japanese
J?	『300E; LEFT WHITE CORNER BRACKET; OPENING WHITE CORNER BRACKET
J?	』300F; RIGHT WHITE CORNER BRACKET; CLOSING WHITE CORNER BRACKET
*/

#define Chinese_book_marks

local menuitemtype Quotemenu [] =
{
	   {"\"plain\"", select_quote_type, "  \"\" ''"},
	/* English, Spanish, Turkish */
	   {"“English”", select_quote_type, "  “” ‘’"},
	/* German, Danish, Polish, Russian, Romanian, Slovak, Slovenian, Czech, Hungarian */
	   {"„German“", select_quote_type, "  „“ ‚‘"},
	/* French, Italian, Norwegian, Portuguese, Russian, Slovenian, Turkish */
	   {"«French»", select_quote_type, "  «» ‹›"},
	/* German, Polish, Slovak, Czech, Serbian, Croatian */
	   {"»inwards«", select_quote_type, "  »« ›‹"},
	/* Dutch, Hungarian */
	   {"„Dutch”", select_quote_type, "  „” ‚’"},
	/* Swedish, Finnish */
	   {"”Swedish”", select_quote_type, "  ”” ’’"},
	/* Swedish, Finnish */
	   {"»Swedish»", select_quote_type, "  »» ››"},
	/* traditional Greek */
	   {"«Greek» ‟.”", select_quote_type, "  «» ‟”"},
#ifdef Chinese_book_marks
	/* Chinese */
	   {"《Chinese》", select_quote_type, "《》 〈〉"},
#endif
	/* Japanese */
	   {"「Japan」", select_quote_type, "『』 「」"},
};

local menuitemtype Keymapmenu [] =
{
	{"none", select_keymap_entry, "--"},
#ifdef use_keymap_tables
#include "keymapsm.h"
#endif
};

local menuitemtype encodingmenu [] =
{
	{"8 bit", select_encoding, "Latin-1/ISO 8859-1", 'L'},
	{"Unicode", select_encoding, "UTF-8", 'U'},
#ifdef use_cjk_tables
	{"", separator, ""},
	{"Hongkong (Big 5 HKSCS)", select_encoding, "Big 5 HKSCS", 'B'},
	{"China (simplified, GB)", select_encoding, "GB18030>GBK>GB2312/EUC-CN", 'G'},
	{"Taiwan (CNS)", select_encoding, "CNS/EUC-TW", 'C'},
	{"Japanese (EUC)", select_encoding, "JIS/EUC-JP", 'J'},
	{"Japanese (Shift-Jis)", select_encoding, "JIS/Shift-JIS", 'S'},
	{"Korean (UHC, KS)", select_encoding, "UHC>KS C/X/EUC-KR", 'K'},
	{"Korean (Johab)", select_encoding, "Johab", 'H'},
	{"", separator, ""},
	{"Vietnamese (VISCII)", select_encoding, "VISCII", 'V'},
#endif
};

local menuitemtype combiningmenu [] =
{
	{"combined", select_combining, ""},
	{"separated", select_combining, ""},
};

local menuitemtype textmenu [] =
{
	{"edit", select_editmode, "modify"},
	{"view", select_editmode, "readonly"},
};

local menuitemtype buffermenu [] =
{
	{"overwrite", select_buffermode, ""},
	{"append", select_buffermode, ""},
};

local menuitemtype autoindentmenu [] =
{
	{"auto indent", select_autoindent, ""},
	{"off", select_autoindent, ""},
};

local menuitemtype justifymenu [] =
{
	{"on command", select_justify, ""},
	{"at line end", select_justify, ""},
	{"always", select_justify, ""},
};

local menuitemtype paragraphmenu [] =
{
	{"non-blank line-end", select_paragraph, ""},
	{"empty line", select_paragraph, ""},
};


/***********************************************************************\
	Popup menu table
\***********************************************************************/

local menuitemtype Popupmenu [] =
{
	{"Mark", MARK, "Go to Mark"},
	{"Cut", CUT, "Cut-Append"},
	{"Copy", COPY, "Append"},
	{"Paste", PASTE, "Paste external"},
	{"HTML command", HTML, ""},
	{"", separator, ""},
	{"Find", SFW, "FindIdf"},
	{"FindReverse", SRV, "FindIdfRev"},
	{"FindNext", RS, "Previous Find"},
	{"FindIdf", SIDFW, ""},
	{"FindIdfRev", SIDRV, ""},
	{"go to Idf. def.", Stag, "go to def. ..."},
	{"Go to Mark", GOMA, ""},
	{"matching paren", SCORR, ""},
};


/***********************************************************************\
	menu structure auxiliary functions
\***********************************************************************/
global int
	count_quote_types ()
{
	return arrlen (Quotemenu);
}


global char *
	quote_mark (i, n)
	int i;
	int n;
{
	char * q = Quotemenu [i].hopitemname;
	while (* q == ' ') {
		q ++;
	}
	while (n > 0) {
		advance_utf8 (& q);
		while (* q == ' ') {
			q ++;
		}
		n --;
	}
	return q;
}


global int
	lookup_quotes (q)
	char * q;
{
	int i;
	For i = 0 While i < arrlen (Quotemenu) Step i ++
	Do	If strisprefix (q, quote_mark (i, 0))
		Then	return i;
		Fi
	Done
	For i = 0 While i < arrlen (Quotemenu) Step i ++
	Do	If strisprefix (q, quote_mark (i, 2))
		Then	return i;
		Fi
	Done
	return -1;
}




/***********************************************************************\
	Flag table and next/previous function
\***********************************************************************/

local char * dispKEYMAP0 ();
local char * dispKEYMAP1 ();


local flagitemtype Flagmenu [] =
{
	{dispKEYMAP0, cycleKEYMAP, "Keyboard Map", Keymapmenu, arrlen (Keymapmenu)},
	{dispKEYMAP1, cycleKEYMAP, "Keyboard Map", Keymapmenu, arrlen (Keymapmenu)},
	{dispnothing, togglenothing, NIL_PTR},
	{disp_leftquote, quote_type_down, "Quotes", Quotemenu, arrlen (Quotemenu)},
	{disp_rightquote, quote_type_up, "Quotes", Quotemenu, arrlen (Quotemenu)},
	{dispnothing, togglenothing, NIL_PTR},
	{disp_encoding_1, toggle_encoding, "Encoding", encodingmenu, arrlen (encodingmenu)},
	{disp_encoding_2, toggle_encoding, "Encoding", encodingmenu, arrlen (encodingmenu)},
	{disp_combining, toggle_combining, "Combining", combiningmenu, arrlen (combiningmenu)},
	{dispnothing, togglenothing, NIL_PTR},
	{dispHOP, toggleHOP, NIL_PTR},
#ifdef debug_ring_buffer
	{disp_buffer_open, togglenothing, NIL_PTR},
#endif
	{dispnothing, togglenothing, NIL_PTR},
	{dispVIEW, toggleVIEW, "Text", textmenu, arrlen (textmenu)},
	{dispappend, toggleappend, "Paste buf", buffermenu, arrlen (buffermenu)},
	{dispnothing, togglenothing, NIL_PTR},
	{disp_autoindent, toggle_autoindent, "Auto indent", autoindentmenu, arrlen (autoindentmenu)},
	{dispJUSlevel, toggleJUSlevel, "Word wrap", justifymenu, arrlen (justifymenu)},
	{dispJUSmode, toggleJUSmode, "Paragraph ends at", paragraphmenu, arrlen (paragraphmenu)},
};


local void
	flag_menu (i)
	int i;
{
	action_menu (Flagmenu [i].menu, 
			Flagmenu [i].menulen, 
			- (flags_pos + i), 0, 
			Flagmenu [i].menutitle);
}


local void
	next_flag_menu (current_menu)
	menuitemtype * current_menu;
{
	int i;
	FLAG passed = False;
	If current_menu == (menuitemtype *) 0
	Then	passed = True;
	Fi

	For i = 0 While i < arrlen (Flagmenu) Step i ++
	Do
		If Flagmenu [i].menutitle != NIL_PTR
		&& Flagmenu [i].menu == current_menu
		Then	passed = True;
		Elsif passed == True && Flagmenu [i].menutitle != NIL_PTR
		      && * ((* Flagmenu [i].dispflag) ()) != ' '
		Then
			/* on a double-column flag menu, move right */
			If i + 1 < arrlen (Flagmenu)
			&& Flagmenu [i + 1].menutitle != NIL_PTR
			&& streq (Flagmenu [i].menutitle, Flagmenu [i + 1].menutitle)
			Then	i ++;
			Fi
			flag_menu (i);
			return;
		Fi
	Done
	openmenu (0);
}


local void
	prev_flag_menu (current_menu)
	menuitemtype * current_menu;
{
	int i;
	FLAG passed = False;
	If current_menu == (menuitemtype *) 0
	Then	passed = True;
	Fi

	For i = arrlen (Flagmenu) - 1 While i >= 0 Step i --
	Do	If Flagmenu [i].menutitle != NIL_PTR
		&& Flagmenu [i].menu == current_menu
		Then	passed = True;
		Elsif	passed == True && Flagmenu [i].menutitle != NIL_PTR
			&& * ((* Flagmenu [i].dispflag) ()) != ' '
		Then	flag_menu (i);
			return;
		Fi
	Done
	openmenu (arrlen (menus) - 1);
}




/***********************************************************************\
	putnstr () writes a string to screen at adjusted length
	putnstr_mark () does the same and marks given item (word)
\***********************************************************************/

#define dont_debug_item_marking

local void
	putnstr_mark (s, l, mark_item)
	char * s;
	int l;
	int mark_item;
{
	int i = 0;
	char * spoi = s;
	int utfcount;
	unsigned long unichar;
	unsigned long cjkchar;
	int index = 0;
	char marked = ' ';
#ifdef debug_item_marking
	printf ("putnstr <%s>\n", s);
#endif

	If mark_item == 0
	Then
#ifdef debug_item_marking
	printf ("border on @ %d\n", i);
#endif
		menuitem_on ();
		marked = 'b';
	Elsif mark_item > 0
	Then
#ifdef debug_item_marking
	printf ("header on @ %d\n", i);
#endif
		menuheader_on ();
		marked = 'h';
	Fi
	Dowhile i < l
	Do	If * spoi == '\0'
		Then	If index == mark_item
			Then	/* end marked item */
#ifdef debug_item_marking
	printf ("end string @ %d\n", i);
#endif
				menuitem_off ();
				menuheader_on ();
				marked = 'h';
			Fi
			index ++;
			putchar (' ');
			i ++;
		Elsif (* spoi & 0x80) != 0	/* UTF-8 character */
		Then	If cjk_term == True
			Then	utf8_info (spoi, & utfcount, & unichar);
				cjkchar = cjk (unichar, False);
				If cjkchar == quit_char
				Then	/* unencoded mapping, no output */
				Else
					i += put_cjkchar (cjkchar);
				Fi
				advance_utf8 (& spoi);
			Else	put_utfchar (spoi);
				advance_utf8_scr (& spoi, & i, s);
			Fi
		Else	/* ASCII character */
			If * spoi == ' '
			Then	If index == mark_item
				Then	/* end marked item */
#ifdef debug_item_marking
	printf ("end marked item @ %d\n", i);
#endif
					menuitem_off ();
					menuheader_on ();
					marked = 'h';
				Fi
				index ++;
				putchar (* spoi ++);
				If index == mark_item
				Then	/* begin marked item */
#ifdef debug_item_marking
	printf ("begin marked item @ %d\n", i);
#endif
					menuheader_off ();
					menuitem_on ();
					marked = 'b';
				Fi
			Else
				putchar (* spoi ++);
			Fi
			i ++;
		Fi
	Done
	If marked == 'h'
	Then
#ifdef debug_item_marking
	printf ("header off @ %d\n", i);
#endif
		menuheader_off ();
	Elsif marked == 'b'
	Then
#ifdef debug_item_marking
	printf ("border off @ %d\n", i);
#endif
		menuitem_off ();
		menuheader_off ();
	Fi
}


local void
	putnstr (s, l)
	char * s;
	int l;
{
	putnstr_mark (s, l, -2);
}




/***********************************************************************\
	putblock () writes a block graphics character
	menu*_on/off () switch menu border display mode
	putborder_top/middle/bottom () writes menu borders
\***********************************************************************/
FLAG in_menu_border = False;

local void
	putblock (c)
	char c;
{
	If in_menu_border == True
	Then	putblockchar (c);
	Else	menuborder_on ();
		putblockchar (c);
		menuborder_off ();
	Fi
}

local void
	putborder_top (x, y, width, title, hook)
	int x, y, width;
	char * title;
	FLAG hook;
{
	int i;

	gotoxy (x, y);
	If title != NIL_PTR
	Then	If use_graphic_borders == True
		Then	menuborder_on ();
			putblock ('l');
			menuheader_on ();
			putnstr ("", menumargin);
		Else	menudisp_on ();
			putnstr ("", 1 + menumargin);
		Fi
		putnstr (title, width - 2 - 2 * menumargin);
		If use_graphic_borders == True
		Then	putnstr ("", menumargin);
			menuheader_off ();
			menuborder_on ();
			If hook
			Then	putblock ('k');	/* 'u' ? */
			Else	putblock ('k');
			Fi
		Else	putnstr ("", 1 + menumargin);
		Fi
	Else
		If use_graphic_borders == True
		Then	menuborder_on ();
			putblock ('l');
			For i = 2 While i < width Step i += horizontal_bar_width
			Do	putblock ('q');
			Done
			putblock ('k');
		Else	menudisp_on ();
			putnstr ("", width);
		Fi
	Fi
}

local void
	putborder_bottom (x, y, width)
	int x, y, width;
{
	int i;

	gotoxy (x, y);
	If use_graphic_borders == True
	Then
		putblock ('m');
		For i = 2 While i < width Step i += horizontal_bar_width
		Do	putblock ('q');
		Done
		putblock ('j');
		menuborder_off ();
	Else	putnstr ("", width);
		menudisp_off ();
	Fi
}

local void
	putborder_middle (x, y, width)
	int x, y, width;
{
	int i;

	gotoxy (x, y);
	If use_graphic_borders == True
	Then
		putblock ('t');
		For i = 2 While i < width Step i += horizontal_bar_width
		Do	putblock ('q');
		Done
		putblock ('u');
		menuborder_off ();
	Else	putnstr ("", width);
	Fi
}


/***********************************************************************\
	displaymenuheader () displays a menu header
\***********************************************************************/
local void
	displaymenuheader (meni)
	int meni;
{
	gotoxy (meni * menu_width, 0);
	putchar (' ');
	menudisp_on ();
	putnstr ("", menumargin);
	putnstr (menus [meni].menuname, menu_width - 2 - 2 * menumargin);
	putnstr ("", menumargin);
	menudisp_off ();
	putchar (' ');
}


/***********************************************************************\
	number_menus () determines the number of pull-down menus
	set_scripttag () sets script tag into name of Keymap menu
\***********************************************************************/
local int
	number_menus ()
{
#ifdef full_Keymap_menu
	If allow_keymap
	Then	return arrlen (menus);
	Else	return arrlen (menus) - 1;
	Fi
#else
	return arrlen (menus);
#endif
}

#ifdef full_Keymap_menu
global void
	set_scripttag ()
{
	strcpy (keymap_menuname, keyboard_mapping);
	strcat (keymap_menuname, " Keymap");
}
#endif


/***********************************************************************\
	calcmenuvalues () determines menu screen positions
	width_of_menu () determines the display width of the menu
\***********************************************************************/
local int
	width_of_menu (menu, menulen, startcol)
	menuitemtype * menu; int menulen; int startcol;
{
	int i;
	menuitemtype * item;
	int itemlen;
	int width = 0;
	For i = 0 While i < menulen Step i ++
	Do
		item = & (menu [i]);
		If hop_flag > 0 && item->hopitemname != NIL_PTR && item->hopitemname [0] != '\0'
		Then	itemlen = utf8_col_count (item->hopitemname);
		Else	itemlen = utf8_col_count (item->itemname);
		Fi
		itemlen += 2 + 2 * menumargin;
		If itemlen > width
		Then	width = itemlen;
		Fi
	Done
	If width > XMAX
	Then	width = XMAX;
	Fi
	If startcol > 0 && startcol + width > XMAX - 1
	Then	width = XMAX + 1 - startcol;
	Fi
	If width_data_version == 3 && width & 1
	Then	width ++;
	Fi
	return width;
}


local void
	calcmenuvalues ()
{
	int flags_gap;

	/* adjust terminal feature usage */
	If menu_border_style == '@'
	Then	use_graphic_borders = False;
	Fi

#ifdef pc_dont_use_graphic_borders
	If use_curses == False
	Then	use_ascii_graphics = True;
	Fi
#endif
	If cjk_term == True && use_vt100_block_graphics == False
	Then	use_ascii_graphics = True;
	Fi

	If width_data_version == 3
	Then	horizontal_bar_width = 2;
	Fi

	/* determine positions for menus and flags display */
	menu_width = (XMAX - arrlen (Flagmenu) - 4) / number_menus ();

	If menu_width < 10 || use_graphic_borders
	Then	menumargin = 0;
	Else	menumargin = 1;
	Fi

	/* determine popup menu width: */
	popupmenu_width = width_of_menu (Popupmenu, arrlen (Popupmenu), 0);

	If popupmenu_width < 15 || use_graphic_borders
	Then	popupmenumargin = 0;
	Else	popupmenumargin = 1;
	Fi

	/* determine position and length of displayed flags */
	flags_pos = XMAX - 2 - arrlen (Flagmenu);
	flags_gap = flags_pos - number_menus () * menu_width;
	flags_displayed = arrlen (Flagmenu);
	If flags_gap < 2
	Then	flags_pos += 2 - flags_gap;
		flags_displayed -= 2 - flags_gap;
	Fi
}


/***********************************************************************\
	displaymenuheaders () displays the menu headers
\***********************************************************************/
local void
	displaymenuheaders_from (firsti)
	int firsti;
{
	int i;
	int starti;

	If menuline_dirty == True
	Then	starti = 0;
		menuline_dirty = False;
	Else	starti = firsti;
	Fi

	calcmenuvalues ();

	If menu_width < 3
	Then	return;
	Fi

#ifdef full_Keymap_menu
	If number_menus () == arrlen (menus)
	Then	set_scripttag ();
	Fi
#endif
	For i = starti While i < number_menus () Step i ++
	Do
		displaymenuheader (i);
	Done
}

local void
	displaymenuheaders ()
{
	int i;

	calcmenuvalues ();

	If menu_width < 3
	Then	return;
	Fi

#ifdef full_Keymap_menu
	If number_menus () == arrlen (menus)
	Then	set_scripttag ();
	Fi
#endif
	For i = 0 While i < number_menus () Step i ++
	Do
		displaymenuheader (i);
	Done
}


/***********************************************************************\
	cleargap () clears the gap between menu headers and flags
\***********************************************************************/
local void
	cleargap ()
{
	int i;
	int gappos = number_menus () * menu_width;

	gotoxy (gappos, 0);
	For i = gappos While i < flags_pos Step i ++
	Do
		putchar (' ');
	Done
}


/***********************************************************************\
	displayflags () displays the menu flags
\***********************************************************************/
global void
	displayflags ()
{
	int i;
	char * flags;
	FLAG is_reverse;
	unsigned long unichar;
	int utflen;

	calcmenuvalues ();

	If menu_width < 3
	Then	return;
	Fi

#ifdef full_Keymap_menu
	If number_menus () == arrlen (menus)
	Then	set_scripttag ();
		displaymenuheader (number_menus () - 1);
	Fi
#endif

	gotoxy (flags_pos, 0);
	For i = 0 While i < flags_displayed Step i ++
	Do
		flags = (* Flagmenu [i].dispflag) ();

		If * flags == '#'
		Then	is_reverse = True;
			flags ++;
			menudisp_on ();
		Else	is_reverse = False;
		Fi
		If (* flags & 0x80) != 0
		Then
			If cjk_term == True
			Then	utf8_info (flags, & utflen, & unichar);
				If unichar < 0x80
				Then	put_unichar (unichar);
				Else	If unichar == 0xAB
					Then	putchar ('<');
					Elsif unichar == 0xBB
					Then	putchar ('>');
					Elsif unichar == 0xA6
					Then	putchar ('|');
					Else	putchar ('?');
					Fi
				Fi
			Else	put_utfchar (flags);
			Fi
		Else	putchar (* flags);
		Fi
		If	is_reverse == True
		Then	menudisp_off ();
		Fi
	Done
}


/***********************************************************************\
	displaymenuline () displays the menu line
\***********************************************************************/
local void
	displaymenuline_from (firsti)
	int firsti;
{
	displaymenuheaders_from (firsti);
	cleargap ();
	displayflags ();
	clear_eol ();
	top_line_scrolled = False;
}


global void
	displaymenuline ()
{
	displaymenuheaders ();
	cleargap ();
	displayflags ();
	clear_eol ();
	top_line_scrolled = False;
}


/***********************************************************************\
	prepare_menuline prepares menu line to prevent 
	right-to-left menu obstruction
\***********************************************************************/
local void
	prepare_menuline (line, menu_line_poi)
		int line; LINE * * menu_line_poi;
{
	char * cp;
	int utfcount;
	unsigned long unichar;

	If * menu_line_poi == tail || * menu_line_poi == NIL_LINE
	Then	return;
	Fi

	If bidi_screen == True && utf8_text == True
	Then	For cp = (* menu_line_poi)->text
		While * cp != '\0'
		Step advance_utf8 (& cp)
		Do	utf8_info (cp, & utfcount, & unichar);
			If is_right_to_left (unichar)
			Then	break;
			Fi
		Done
		If * cp != '\0'
		Then	gotoxy (0, line);
			clear_eol ();
		Fi
	Fi
	If * menu_line_poi == bot_line
	Then	* menu_line_poi = NIL_LINE;
	Else	* menu_line_poi = (* menu_line_poi)->next;
	Fi
}




/***********************************************************************\
	menu state and saved menu status for redrawmenu ()
\***********************************************************************/

local FLAG pulldownmenu_active = False;
local FLAG popupmenu_active = False;

local int last_pulldown_menu;
local int last_pulldown_item;

local int last_popup_column;
local int last_popup_line;
local int last_popup_item;
local int last_popup_index;


/***********************************************************************\
	pulldown_menu () displays a pulldown menu
\***********************************************************************/
local int
	pulldown_menu (meni)
	int meni;
{
	int i;
	int menulen = menus [meni].menulen;
	int meniwidth;
	int menuxpos = meni * menu_width;
	menuitemtype * item;
	LINE * menu_line;

	menu_mouse_mode (True);

	/* save redraw values: */
	last_pulldown_menu = meni;
	last_pulldown_item = -1;

	If menulen > YMAX - 1
	Then menulen = YMAX - 1;
	Fi

	/* determine menu width: */
	meniwidth = width_of_menu (menus [meni].menuitems, menulen, menuxpos);

	/* top menu border, with menu name: */
	putborder_top (menuxpos, 0, meniwidth, menus [meni].menuname, False);

	/* text lines may have to be cleared: */
	menu_line = top_line;

	/* menu items: */
	For i = 0 While i < menulen Step i ++
	Do
	    prepare_menuline (1 + i, & menu_line);
	    item = & (menus [meni].menuitems [i]);
	    If item->itemfu == separator
	    Then
		putborder_middle (menuxpos, 1 + i, meniwidth);
	    Else
		gotoxy (menuxpos, 1 + i);
		If use_graphic_borders == True
		Then	putblock ('x');
			menuborder_off ();
		Else	putchar (' ');
			menudisp_off ();
		Fi
		putnstr ("", menumargin);

		If hop_flag > 0 && item->hopitemname != NIL_PTR && item->hopitemname [0] != '\0'
		Then	putnstr (item->hopitemname, meniwidth - 2 - 2 * menumargin);
		Else	putnstr (item->itemname, meniwidth - 2 - 2 * menumargin);
		Fi

		putnstr ("", menumargin);
		If use_graphic_borders == True
		Then	menuborder_on ();
			putblock ('x');
		Else	menudisp_on ();
			putchar (' ');
		Fi
	    Fi
	Done

	/* bottom menu border: */
	prepare_menuline (1 + menulen, & menu_line);
	putborder_bottom (menuxpos, 1 + menulen, meniwidth);

	gotoxy (menuxpos, 0);
	If menulen > last_dirty_line
	Then	last_dirty_line = menulen;
	Fi
	pulldownmenu_displayed = True;
	pulldownmenu_active = True;

	return meniwidth;
}


/***********************************************************************\
	display_menu () displays the given menu, returns menu width
\***********************************************************************/
local int
	display_menu (menu, menulen, column, line, title)
		menuitemtype * menu; int menulen;
		int column; int line;
		char * title;
{
	int i;
	menuitemtype * item;
	int menu_width = width_of_menu (menu, menulen, 0);
	LINE * menu_line;

	menu_mouse_mode (True);

	/* save redraw values: */
	last_popup_column = column;
	last_popup_line = line;
	last_popup_item = -1;

	/* text lines may have to be cleared: */
	menu_line = proceed (top_line, line - 1);

	/* top menu border: */
	prepare_menuline (line, & menu_line);
	putborder_top (column, line, menu_width, title, True);

	/* menu items: */
	For i = 0 While i < menulen Step i ++
	Do
	    prepare_menuline (line + 1 + i, & menu_line);
	    item = & (menu [i]);
	    If item->itemfu == separator
	    Then
		putborder_middle (column, line + 1 + i, menu_width);
	    Else
		gotoxy (column, line + 1 + i);
		If use_graphic_borders == True
		Then	putblock ('x');
			menuborder_off ();
		Else	putchar (' ');
			menudisp_off ();
		Fi
		putnstr ("", popupmenumargin);

		If hop_flag > 0 && item->hopitemname != NIL_PTR && item->hopitemname [0] != '\0'
		Then	putnstr (item->hopitemname, menu_width - 2 - 2 * popupmenumargin);
		Else	putnstr (item->itemname, menu_width - 2 - 2 * popupmenumargin);
		Fi

		putnstr ("", popupmenumargin);
		If use_graphic_borders == True
		Then	menuborder_on ();
			putblock ('x');
		Else	menudisp_on ();
			putchar (' ');
		Fi
	    Fi
	Done

	/* bottom menu border: */
	prepare_menuline (line + 1 + menulen, & menu_line);
	putborder_bottom (column, line + 1 + menulen, menu_width);

	If line > 0 && title != NIL_PTR
	Then	/* flag menu */
		gotoxy (column + menu_width - 2 + horizontal_bar_width, line);
	Else	gotoxy (column, line);
	Fi
	first_dirty_line = line - 1;
	last_dirty_line = first_dirty_line + menulen + 2;
	last_menu = menu;
	last_menulen = menulen;
	last_menuwidth = menu_width;
	last_menutitle = title;
	popupmenu_displayed = True;
	popupmenu_active = True;

	return menu_width;
}


/***********************************************************************\
	menselected () highlights a selected pulldown menu item
\***********************************************************************/
local void
	menselected (selected, meni, itemno)
		FLAG selected; int meni; int itemno;
{
	menuitemtype * item;
	int meniwidth;
	int menulen = menus [meni].menulen;
	int menuxpos = meni * menu_width;

	/* save redraw values: */
	If selected == True
	Then	last_pulldown_item = itemno;
	Else	last_pulldown_item = -1;
	Fi

	/* determine menu width: */
	meniwidth = width_of_menu (menus [meni].menuitems, menulen, menuxpos);

	gotoxy (menuxpos, 1 + itemno);
	If use_graphic_borders == True
	Then	putblock ('x');
	Else	menudisp_on ();
		putchar (' ');
		menudisp_off ();
	Fi
	If selected == True
	Then	menuheader_on ();
	Fi
	putnstr ("", menumargin);

	item = & (menus [meni].menuitems [itemno]);
	If hop_flag > 0 && item->hopitemname != NIL_PTR && item->hopitemname [0] != '\0'
	Then	putnstr (item->hopitemname, meniwidth - 2 - 2 * menumargin);
	Else	putnstr (item->itemname, meniwidth - 2 - 2 * menumargin);
	Fi

	putnstr ("", menumargin);
	If selected == True
	Then	menuheader_off ();
	Fi
	If use_graphic_borders == True
	Then	putblock ('x');
	Else	menudisp_on ();
		putchar (' ');
		menudisp_off ();
	Fi
}


/***********************************************************************\
	popupmenselected () highlights a selected popup menu item
\***********************************************************************/
local void
	popupmenselected (menu, menu_width, selected, column, line, i, index)
		menuitemtype * menu; FLAG selected;
		int column; int line; int i; int index;
{
	menuitemtype * item;

	/* save redraw values: */
	If selected == True
	Then	last_popup_item = i;
		last_popup_index = index;
	Else	last_popup_item = -1;
	Fi

	gotoxy (column, line + 1 + i);
	If use_graphic_borders == True
	Then	putblock ('x');
	Else	menudisp_on ();
		putchar (' ');
		menudisp_off ();
	Fi
	If selected == True && index < 0
	Then	menuheader_on ();
	Fi
	putnstr ("", popupmenumargin);

	item = & (menu [i]);
	If selected == True && index >= 0
	Then	putnstr_mark (item->itemname, menu_width - 2 - 2 * popupmenumargin, index);
	Elsif hop_flag > 0 && item->hopitemname != NIL_PTR && item->hopitemname [0] != '\0'
	Then	putnstr (item->hopitemname, menu_width - 2 - 2 * popupmenumargin);
	Else	putnstr (item->itemname, menu_width - 2 - 2 * popupmenumargin);
	Fi

	putnstr ("", popupmenumargin);
	If selected == True && index < 0
	Then	menuheader_off ();
	Fi
	If use_graphic_borders == True
	Then	putblock ('x');
	Else	menudisp_on ();
		putchar (' ');
		menudisp_off ();
	Fi
}


/***********************************************************************\
	clean_menus () wipes menus which are still displayed
	clear_menus () clears menus which are still displayed
\***********************************************************************/
global void
	clean_menus ()
{
	LINE * start_line;

	menu_mouse_mode (False);
	If pulldownmenu_displayed == True
	Then
		pulldownmenu_displayed = False;
		display (0, top_line, last_dirty_line, last_dirty_line);
		last_dirty_line = 0;
		set_cursor_xy ();
	Fi
	If popupmenu_displayed == True
	Then
		start_line = proceed (top_line, first_dirty_line);
		popupmenu_displayed = False;
		display (first_dirty_line, start_line, 1 + last_menulen, first_dirty_line + 1 + last_menulen);
		last_dirty_line = 0;
		set_cursor_xy ();
	Fi
}

local void
	clear_menus ()
{
	clean_menus ();
	pulldownmenu_active = False;
	popupmenu_active = False;
}


/***********************************************************************\
	redrawmenu () redraws an open menu after screen refresh
\***********************************************************************/
global void
	redrawmenu ()
{
	If pulldownmenu_active == True
	Then
		int item = last_pulldown_item;
		pulldown_menu (last_pulldown_menu);
		If item >= 0
		Then	menselected (True, last_pulldown_menu, item);
		Fi
	Fi
	If popupmenu_active == True
	Then
		int item = last_popup_item;
		display_menu (last_menu, last_menulen, last_popup_column, last_popup_line, last_menutitle);
		If item >= 0
		Then	popupmenselected (last_menu, last_menuwidth, True, 
					last_popup_column, last_popup_line, 
					item, last_popup_index);
		Fi
	Fi
}


/***********************************************************************\
	is_menu_open () checks if a menu is open
\***********************************************************************/
global int
	is_menu_open ()
{
	return pulldownmenu_active == True || popupmenu_active == True;
}


/***********************************************************************\
	QUICKMEN () handles right mouse button menu
	action_menu () handles given action menu
	popup_menu () handles given action or selection menu
\***********************************************************************/
global int
	popup_menu (menu, menulen, column, line, title, select_keys)
		menuitemtype * menu; int menulen;
		int column; int line;
		char * title;
		char * select_keys;
{
	unsigned long c;
	int startline;
	int startcolumn;
	int itemno;
	int ret = -1;
	char * cpoi;
	char * select;
	int scol;
	int select_index = -2;

	int menu_width = width_of_menu (menu, menulen, 0);

	startline = line + 1;
	If startline > YMAX - menulen - 1
	Then	startline = YMAX - menulen - 1;
	Fi
	If column >= 0
	Then	startcolumn = column;
	Else	startcolumn = 4 - column - menu_width - 2 * horizontal_bar_width;
	Fi
	If startcolumn > XMAX - menu_width - 2 * horizontal_bar_width + 2
	Then	startcolumn = XMAX - menu_width - 2 * horizontal_bar_width + 2;
	Fi
	If startline < 1 || startcolumn < 0 || menu_width < 3
	Then	return ret;
	Fi

	/* initial display */
	menu_width = display_menu (menu, menulen, startcolumn, startline, title);

	/* initial selection */
	If select_keys == NIL_PTR
	Then	itemno = 0;
	Else	itemno = 1;
		select_index = 0;
		popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
	Fi

	flush ();

	c = readcharacter_allbuttons ();
	Dowhile command (c) == MDN || command (c) == MUP
		|| (c == ' '
			&& (select_keys == NIL_PTR
			    || selection_space != SPACE_SELECT))
		|| command (c) == HOP || command (c) == GOTO
		|| (command (c) == MOUSEescape
			&& mouse_button != leftbutton
			&& (mouse_button != releasebutton
			    || ! (mouse_prevbutton == leftbutton 
			          || mouse_prevbutton == rightbutton
			         )
			    || ! (
				  mouse_xpos > startcolumn
				   && mouse_xpos < startcolumn + menu_width
				  && mouse_ypos >= startline
				   && mouse_ypos < startline + menulen
			         )
			   )
		   )
		|| (select_keys != NIL_PTR
			&& (command (c) == MLF || command (c) == MRT
			   || c == '<' || c == '>'
			   )
		   )
	Do
	    If (c == ' ' && select_keys == NIL_PTR)
		|| command (c) == HOP
		|| command (c) == GOTO
		|| (command (c) == MOUSEescape && mouse_button == middlebutton)
	    Then
		toggleHOP ();
		clear_menus ();
		/* recalculate x pos to prevent menu display wrap-around */
		menu_width = width_of_menu (menu, menulen, 0);
		If column >= 0
		Then	startcolumn = column;
		Else	startcolumn = 4 - column - menu_width - 2 * horizontal_bar_width;
		Fi
		If startcolumn > XMAX - menu_width - 2 * horizontal_bar_width + 2
		Then	startcolumn = XMAX - menu_width - 2 * horizontal_bar_width + 2;
		Fi
		If startline < 1 || startcolumn < 0 || menu_width < 3
		Then	return ret;
		Fi
		/* display the menu */
		menu_width = display_menu (menu, menulen, startcolumn, startline, title);
		If itemno > 0
		Then	popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
		Fi
	    Elsif command (c) == MOUSEescape
	    Then
		If itemno > 0
		Then	popupmenselected (menu, menu_width, False, startcolumn, startline, itemno - 1, select_index);
		Fi
		itemno = mouse_ypos - startline + 1;
		If itemno <= 0 || itemno > menulen
		|| menu [itemno - 1].itemfu == separator
		Then	itemno = 0;
		Else
			popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
		Fi
	    Elsif command (c) == MLF || c == '<'
	    Then
		If select_index > 0
		Then	select_index --;
		Else	select_index = strlen (select_keys) - 1;
		Fi
		popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
	    Elsif command (c) == MRT || c == '>'
	    Then
		If select_index < strlen (select_keys) - 1
		Then	select_index ++;
		Else	select_index = 0;
		Fi
		popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
	    Elsif c == ' ' && selection_space == SPACE_NEXT
	    Then
		If select_index < strlen (select_keys) - 1
		Then	select_index ++;
		Else	popupmenselected (menu, menu_width, False, startcolumn, startline, itemno - 1, select_index);
			select_index = 0;
			itemno ++;
			If menu [itemno - 1].itemfu == separator
			Then	itemno ++;
			Fi
			If itemno > menulen
			Then	itemno = 1;
			Fi
		Fi
		popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
	    Else
		If itemno > 0
		Then	popupmenselected (menu, menu_width, False, startcolumn, startline, itemno - 1, select_index);
		Fi
		If command (c) == MDN ||
		   (c == ' ' && selection_space == SPACE_NEXTROW)
		Then	itemno ++;
			If menu [itemno - 1].itemfu == separator
			Then	itemno ++;
			Fi
			If itemno > menulen
			Then	itemno = 1;
			Fi
		Elsif command (c) == MUP
		Then	itemno --;
			If menu [itemno - 1].itemfu == separator
			Then	itemno --;
			Fi
			If itemno <= 0
			Then	itemno = menulen;
			Fi
		Fi
		popupmenselected (menu, menu_width, True, startcolumn, startline, itemno - 1, select_index);
	    Fi
	    flush ();
	    c = readcharacter_allbuttons ();
	Done
	If command (c) == MOUSEescape
	Then	/* mouse click */
		mouse_xpos -= startcolumn;
		mouse_ypos -= startline;
		If mouse_xpos > 0 && mouse_xpos < menu_width
		&& mouse_ypos >= 0 && mouse_ypos < menulen
		Then
		    If select_keys != NIL_PTR
		    Then
			/* determine mouse position in menu line */
			/* used for keyboard mapping selection */
			select = menu [mouse_ypos].itemname;
			cpoi = select;
			ret = 0;
			scol = 1;
			Dowhile scol < mouse_xpos && * cpoi != '\0'
			Do	If * cpoi == ' '
				Then	ret ++;
				Fi
				advance_utf8_scr (& cpoi, & scol, select);
			Done
			If ret < strlen (select_keys)
			Then	ret = mouse_ypos * strlen (select_keys) + ret;
			Else	ret = -1;
			Fi
		    Else
			popupmenselected (menu, menu_width, True, startcolumn, startline, mouse_ypos, select_index);
			menu_mouse_mode (False);
			(* menu [mouse_ypos].itemfu) (menu, mouse_ypos);
			displayflags ();
		    Fi
		Elsif mouse_ypos + startline == -1
		Then	clear_menus ();
			displaymenuheaders ();
			MINMEN (mouse_xpos + startcolumn, 0);
		Fi
	Elsif (c == '\n' || command (c) == SNL 
		|| (c == ' ' && select_keys != NIL_PTR
			     && selection_space == SPACE_SELECT)
		) && itemno > 0
	Then
		menu_mouse_mode (False);
		If select_keys != NIL_PTR
		Then	ret = (itemno - 1) * strlen (select_keys)
				+ select_index;
		Else	(* menu [itemno - 1].itemfu) (menu, itemno - 1);
			displayflags ();
		Fi
	Elsif select_keys != NIL_PTR
	Then
		cpoi = strchr (select_keys, c);
		If cpoi != NIL_PTR && * cpoi != '\0'
		Then	ret = (itemno - 1) * strlen (select_keys)
				+ (cpoi - select_keys);
		Fi
	Elsif command (c) == MLF && input_active == False
	Then	clear_menus ();
		prev_flag_menu (menu);
	Elsif command (c) == MRT && input_active == False
	Then	clear_menus ();
		next_flag_menu (menu);
	Fi
	clear_menus ();
	return ret;
}


local void
	action_menu (menu, menulen, column, line, title)
		menuitemtype * menu; int menulen;
		int column; int line;
		char * title;
{
	clear_menus ();
	(void) popup_menu (menu, menulen, column, line, title, NIL_PTR);
}


global void
	QUICKMEN (column, line)
	int column;
	int line;
{
	action_menu (Popupmenu, arrlen (Popupmenu), column, line, NIL_PTR);
}


/***********************************************************************\
	MINMEN (screen_column, itemno) handles menu dialog or flag toggling
		if itemno > 0, the item is pre-selected
	openmenu (i) opens the i-th menu
	openmenuat (c) opens the menu at screen column c
\***********************************************************************/
global void
	openmenu (i)
	int i;
{
	MINMEN (i * menu_width, 0);
}

global void
	openmenuat (c)
	int c;
{
	If mouse_button == middlebutton && hop_flag == 0
	Then	toggleHOP ();
	Fi

	MINMEN (c, 0);
}

local void
	MINMEN (column, itemno)
	int column;
	int itemno;
{
	int meni;
	unsigned long c;
	int popmenwidth;	/* width of the menu, determined dynamically */

	calcmenuvalues ();

	If menu_width < 3
	Then	return;
	Fi
	meni = column / menu_width;
	If meni < 0 || meni >= number_menus ()
	Then	/* check if it's a flag toggle position */
		meni = column - flags_pos;
		If meni >= 0 && meni < flags_displayed
		Then
			If mouse_button == rightbutton
			&& Flagmenu [meni].menutitle != NIL_PTR
			Then	If * ((* Flagmenu [meni].dispflag) ()) != ' '
				Then	action_menu (
						Flagmenu [meni].menu, 
						Flagmenu [meni].menulen, 
						- (column - 2), 
						0, 
						Flagmenu [meni].menutitle);
				Fi
			Else	(* Flagmenu [meni].toggle) (column);
			Fi
			displayflags ();
			set_cursor_xy ();
		Fi
		return;
	Fi

	popmenwidth = pulldown_menu (meni);
	If itemno > 0
	Then	menselected (True, meni, itemno - 1);
	Fi

	flush ();
	c = readcharacter_allbuttons ();
	Dowhile command (c) == MDN || command (c) == MUP
		|| (command (c) == MOUSEescape
			&& mouse_button != leftbutton
			&& (mouse_button != releasebutton ||
			    mouse_ypos < 0 ||
			    ! (mouse_prevbutton == leftbutton 
			       || mouse_prevbutton == rightbutton
			      )
			   )
			&& mouse_button != middlebutton
			&& (mouse_button != rightbutton ||
			    mouse_ypos >= 0 || mouse_xpos / menu_width == meni)
			&& (mouse_button != movebutton ||
			    mouse_ypos >= 0 || mouse_xpos / menu_width == meni)
		   )
	Do
	    If command (c) == MOUSEescape
	    Then
		If itemno > 0
		Then	menselected (False, meni, itemno - 1);
		Fi
		itemno = mouse_ypos + 1;
		If itemno <= 0 || itemno > menus [meni].menulen
		|| menus [meni].menuitems [itemno - 1].itemfu == separator
		Then	itemno = 0;
		Else
			menselected (True, meni, itemno - 1);
		Fi
	    Else
		If itemno > 0
		Then	menselected (False, meni, itemno - 1);
		Fi
		If command (c) == MDN
		Then	itemno ++;
			If menus [meni].menuitems [itemno - 1].itemfu == separator
			Then	itemno ++;
			Fi
			If itemno > menus [meni].menulen
			Then	itemno = 1;
			Fi
		Else	itemno --;
			If menus [meni].menuitems [itemno - 1].itemfu == separator
			Then	itemno --;
			Fi
			If itemno <= 0
			Then	itemno = menus [meni].menulen;
			Fi
		Fi
		menselected (True, meni, itemno - 1);
	    Fi
	    flush ();
	    c = readcharacter_allbuttons ();
	Done
	If c == ' ' || command (c) == HOP || command (c) == GOTO
		|| (command (c) == MOUSEescape && mouse_button == middlebutton)
	Then	toggleHOP ();
		clear_menus ();
		displaymenuline_from (meni);
		MINMEN (column, itemno);
	Elsif command (c) == MOUSEescape
	Then	/* mouse click */
		If mouse_xpos >= menu_width * meni
		&& mouse_xpos < menu_width * meni + popmenwidth
		&& mouse_ypos >= 0 && mouse_ypos < menus [meni].menulen
		&& (mouse_button == leftbutton || mouse_button == releasebutton)
		Then
			/*menselected (True, meni, mouse_ypos);*/
			clear_menus ();
			menu_mouse_mode (False);
			(* menus [meni].menuitems [mouse_ypos].itemfu)
				(menus [meni].menuitems [mouse_ypos].hopitemname);
		Elsif mouse_ypos == -1 && mouse_xpos / menu_width != meni
		Then	clear_menus ();
		/*	displaymenuheaders ();	*/
			displaymenuline_from (meni);
			MINMEN (mouse_xpos, 0);
		Fi
	Elsif (c == '\n' || command (c) == SNL) && itemno > 0
	Then
		clear_menus ();
		menu_mouse_mode (False);
		(* menus [meni].menuitems [itemno - 1].itemfu)
			(menus [meni].menuitems [itemno - 1].hopitemname);
	Elsif command (c) == MLF || command (c) == MRT
	Then	clear_menus ();
		displaymenuline_from (meni);
		If command (c) == MLF
		Then	If meni > 0
			Then	meni --;
			Else	/*meni = number_menus () - 1;*/
				/*flag_menu (arrlen (Flagmenu) - 1);*/
				prev_flag_menu ((menuitemtype *) 0);
				return;
			Fi
		Else	If meni < number_menus () - 1
			Then	meni ++;
			Else	/*meni = 0;*/
				/*flag_menu (0);*/
				next_flag_menu ((menuitemtype *) 0);
				return;
			Fi
		Fi
		openmenu (meni);
	Fi
	clear_menus ();
/*	displaymenuheaders ();	*/
/*	displaymenuline_from (meni);	*/
	displaymenuline ();
}


/***********************************************************************\
	Flag-triggered menus
\***********************************************************************/

global void
	handleQuotemenu (col)
	int col;
{
	If col == 0
	Then	col = flags_pos + 3;
	Fi
	action_menu (Quotemenu, arrlen (Quotemenu), - col, 0, "Quotes");
}


global void
	handleKeymapmenu (col)
	int col;
{
	If col == 0
	Then	col = flags_pos;
	Fi
	action_menu (Keymapmenu, arrlen (Keymapmenu), - col, 0, "Keyboard Map");
}


global void
	handleEncodingmenu ()
{
	int col = flags_pos + 6;
	action_menu (encodingmenu, arrlen (encodingmenu), - col, 0, "Encoding");
}


/***********************************************************************\
	End
\***********************************************************************/
