/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include "proto.h"

extern GLOBAL *global;

void Stats::Display(Filebuf *filebuf) {
	unsigned int tvalue;
	char *statvalue;
	StatGroupList::iterator groupitem;
	StatCounterList::iterator countitem;

	for (groupitem = group_list.begin(); groupitem != group_list.end(); groupitem++) {
		filebuf->Addf("<tr><td align=\"center\"><table class=\"smalldialog\">\n");
		filebuf->Addf("<tr><td class=\"listhead\" align=\"center\" colspan=\"2\">%s</td></tr>\n", groupitem->name.c_str());

		for (countitem = groupitem->counter_list.begin(); countitem != groupitem->counter_list.end(); countitem++) {
			if (groupitem->type == StatGroup::RATIO) {
				if (countitem->type & StatCounter::UINT)
					tvalue = countitem->value.count;
				else {
					/* can't use callbacks for ratio groups */
					ASSERT(0);
				}

				
				filebuf->Addf("<tr><td width=\"40%%\">%s</td><td>%u (%d%%)</td></tr>\n", countitem->name.c_str(), tvalue, (groupitem->total != 0) ? (int)(((float)tvalue / (float)groupitem->total) * 100.0) : 0); 
			} else {
				statvalue = countitem->Get();
				filebuf->Addf("<tr><td width=\"40%%\">%s</td><td>%s</td></tr>\n", countitem->name.c_str(), statvalue);
				xfree(statvalue);
			}
		}

		filebuf->Addf("</table></td></tr> <tr><td><br></td></tr>\n");
	}
}

bool Stats::AddGroup(string groupname, int type) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname) return FALSE;

	StatGroup group(groupname, type);
	group_list.push_back(group);

	return FALSE;
}

bool Stats::AddCounter(string groupname, string countername, int type) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname)
			return item->Add(countername, type);

	return FALSE;
}

bool Stats::AddCounter(string groupname, string countername, unsigned int (*cb)(), int type) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname)
			return item->Add(countername, cb, type);

	return FALSE;
}

bool Stats::AddCounter(string groupname, string countername, char *(*cb)(), int type) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname)
			return item->Add(countername, cb, type);

	return FALSE;
}

unsigned int Stats::Increment(string groupname, string stat) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname) break;

	if (item != group_list.end())
		return item->Increment(stat);

	return 0;
}

unsigned int Stats::Decrement(string groupname, string stat) {
	StatGroupList::iterator item;


	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname) break;

	if (item != group_list.end())
		return item->Decrement(stat);

	return 0;
}

unsigned int Stats::Increment(string groupname, string stat, unsigned int amount) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname) break;

	if (item != group_list.end())
		return item->Increment(stat, amount);

	return 0;
}

unsigned int Stats::Decrement(string groupname, string stat, unsigned int amount) {
	StatGroupList::iterator item;

	for (item = group_list.begin(); item != group_list.end(); item++)
		if (item->name == groupname) break;

	if (item != group_list.end())
		return item->Decrement(stat, amount);

	return 0;
}

bool StatGroup::Add(string stat, int type) {
	StatCounter sc;
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++)
		if (item->name == stat) return FALSE;

	sc.name = stat;
	sc.type = type;
	sc.value.count = 0;

	counter_list.push_back(sc);

	return TRUE;
}

bool StatGroup::Add(string stat, unsigned int (*cb)(), int type) {
	StatCounter sc;
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++)
		if (item->name == stat) return FALSE;

	if (!(type & (StatCounter::UINT | StatCounter::FILESIZE))) return FALSE;

	sc.name = stat;
	sc.type = type | StatCounter::CALLBACK;
	(void *)sc.value.callback = (void *)cb;

	counter_list.push_back(sc);

	return TRUE;
}

bool StatGroup::Add(string stat, char *(*cb)(), int type) {
	StatCounter sc;
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++)
		if (item->name == stat) return FALSE;

	if (!(type & StatCounter::STRING)) return FALSE;

	sc.name = stat;
	sc.type = type | StatCounter::CALLBACK;
	(void *)sc.value.callback = (void *)cb;

	counter_list.push_back(sc);

	return TRUE;
}

StatGroup::StatGroup(string name, int type) {
	this->name = name;
	this->type = type;

	total = 0;
}

unsigned int StatGroup::Increment(string stat) {
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++) {
		if (item->name == stat) {
			total++;
			return ++item->value.count;
		}
	}

	return 0;
}

unsigned int StatGroup::Decrement(string stat) {
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++) {
		if (item->name == stat) {
			total--;
			return --item->value.count;
		}
	}

	return 0;
}

unsigned int StatGroup::Increment(string stat, unsigned int amount) {
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++) {
		if (item->name == stat) {
			total += amount;
			return item->value.count += amount;
		}
	}

	return 0;
}

unsigned int StatGroup::Decrement(string stat, unsigned int amount) {
	StatCounterList::iterator item;

	for (item = counter_list.begin(); item != counter_list.end(); item++) {
		if (item->name == stat) {
			total -= amount;
			return item->value.count -= amount;
		}
	}

	return 0;
}

char *StatCounter::Get() {
	char buf[1024];
	unsigned int uint_val = 0;
	char *string_val = NULL;

	if (type & CALLBACK) {
		if (type & STRING)
			string_val = (char *)value.callback();
		else
			uint_val = (unsigned int)value.callback();
	} else
		uint_val = value.count;

	switch(type & ~StatCounter::CALLBACK) {
		case UINT:
			snprintf(buf, sizeof(buf), "%u", uint_val);
			return xstrdup(buf);
		case FILESIZE:
			return filesize(uint_val);
		case STRING:
			return string_val;
		default:
			ASSERT(0);
	}
}



char *stat_usertime() {
	struct timeval tv;

	pthread_mutex_lock(&global->rusage_lock);
	tv = global->rusage.ru_utime;
	pthread_mutex_unlock(&global->rusage_lock);

	return timeval_to_string(&tv);
}

char *stat_systime() {
	struct timeval tv;

	pthread_mutex_lock(&global->rusage_lock);
	tv = global->rusage.ru_stime;
	pthread_mutex_unlock(&global->rusage_lock);

	return timeval_to_string(&tv);
}

unsigned int stat_majflt() {
	return atomic_read_mutex(&global->rusage_lock, &global->rusage.ru_majflt);
}

unsigned int stat_minflt() {
	return atomic_read_mutex(&global->rusage_lock, &global->rusage.ru_minflt);
}

unsigned int stat_size() {
	struct statm_t sm;

	if (!statm_load(&sm)) return 0;

	return sm.size;
}

unsigned int stat_resident() {
	struct statm_t sm;

	if (!statm_load(&sm)) return 0;

	return sm.resident;
}

unsigned int stat_share() {
	struct statm_t sm;

	if (!statm_load(&sm)) return 0;

	return sm.share;
}

/* very unportable code, but getrusage doesn't report memory usage in most unix's */
bool statm_load(struct statm_t *statm) {
	int x, pagesize = sysconf(_SC_PAGESIZE);
	char buf[1024];
	FILE *fptr = fopen("/proc/self/statm", "r");

	if (fptr == NULL) return FALSE;

	if (!fgets(buf, sizeof(buf), fptr)) {
		fclose(fptr);
		return FALSE;
	}
	fclose(fptr);

	x = sscanf(buf, "%lu %lu %lu %lu %lu %lu %lu", &statm->size, &statm->resident, &statm->share, &statm->trs, &statm->drs, &statm->lrs, &statm->dt);
	if (x != 7)
		return FALSE;

	statm->size *= pagesize;
	statm->resident *= pagesize;
	statm->share *= pagesize;
	statm->trs *= pagesize;
	statm->drs *= pagesize;
	statm->lrs *= pagesize;
	statm->dt *= pagesize;

	return TRUE;
}
