/*
    Middleman Proxy Server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include "proto.h"

extern char sectionfile[];
extern time_t documentsynctime;
extern pthread_mutex_t documentlock;

InterfaceError::InterfaceError(const char *fmt, ...)
{
	int ret;
	char buf[8096];
	va_list valist;

	va_start(valist, fmt);
	ret = vsnprintf(buf, sizeof(buf), fmt, valist);
	va_end(valist);

	msg = buf;
}

int Item::count = 0;

/*
 * Construct a new item in a Subsection based on the Subsection's template (templ)
 * if and XmlElement is given, load the fields from it
 */
Item::Item(const Subsection& sub, const XmlElement* element)
{
	id = count++;
	section = sub.section;
	subsection = sub.name;

	field_vec = sub.templ;       

	if (element) {
		FieldVector::iterator field;
		for (field = field_vec.begin(); field != field_vec.end(); field++) {
			try {
				field->load(*element);
			} catch (ConfigError) {
				field->error_log(*element);
			}
		}
	}
}

StringList Item::read(CGIMap &args)
{
	return field_vec.read(args);
}

int Item::id_get() const
{
	return id;
}

inline int Item::is(int id) const
{
	return (this->id == id);
}

const Item* ItemList::find(const string& name)
{
	ItemList::const_iterator item;

	for (item = begin(); item != end(); item++) {

		FieldVector::const_iterator field;
		for (field = item->field_vec.begin(); field != item->field_vec.end(); field++) {
			
			if (field->type == Field::UNIQUE_ID && field->string_value == name)
				return &*item;
		}

	}

	return NULL;
}

void Item::display(Filebuf* filebuf, int direction) const
{
	field_vec.display(filebuf);

	filebuf->Addf("<tr><td colspan=\"2\" align=\"center\"><table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\">\n");
	filebuf->Addf("<tr class=\"dialogfoot\"><td align=\"center\"><a href=\"config?section=%s&amp;subsection=%s&amp;dialog=show&amp;id=%d\">Edit</a> <a href=\"config?section=%s&amp;subsection=%s&amp;action=delete&amp;id=%d\">Delete</a></td>\n", section.c_str(), subsection.c_str(), id, section.c_str(), subsection.c_str(), id);

	if (direction) {
		filebuf->Addf("<td align=\"center\"><a href=\"config?section=%s&amp;subsection=%s&amp;action=shift&amp;id=%d&amp;direction=up\">Up</a> <a href=\"config?section=%s&amp;subsection=%s&amp;action=shift&amp;id=%d&amp;direction=down\">Down</a></td>\n", section.c_str(), subsection.c_str(), id, section.c_str(), subsection.c_str(), id);
		filebuf->Addf("<td align=\"center\"><a href=\"config?section=%s&amp;subsection=%s&amp;action=shift&amp;id=%d&amp;direction=top\">Top</a> <a href=\"config?section=%s&amp;subsection=%s&amp;action=shift&amp;id=%d&amp;direction=bottom\">Bottom</a></td></tr>\n", section.c_str(), subsection.c_str(), id, section.c_str(), subsection.c_str(), id);
	}


	filebuf->Addf("</table></td></tr>\n");
}

StringSelect::StringSelect(string name, string comment, string desc, bool selected)
{
	this->name = name;
	this->comment = comment;
	this->selected = selected;
	this->desc = desc;
}

StringSelect::StringSelect()
{
}


int string_to_int(string str)
{
	char* endptr;

	int res = strtol(str.c_str(), &endptr, 10);
	if (*endptr != '\0') {
		throw ConfigError();
	}

	return res;
}

unsigned int string_to_uint(string str)
{
	char* endptr;

	unsigned int res = strtoul(str.c_str(), &endptr, 10);
	if (*endptr != '\0') {
		throw ConfigError();
	}

	return res;
}
		
unsigned int string_to_filesize(string str)
{
	char unit[11];
	unsigned int res;
	int x = sscanf(str.c_str(), "%u%10s", &res, unit);

	if (x == 1)
		return res;
	else if (x == 2) {

		if (!strcasecmp(unit, ""))
			return res;
		else if (!strcasecmp(unit, "k") || !strcasecmp(unit, "kb") || !strcasecmp(unit, "kbit"))
			return 1024 * res;
		else if (!strcasecmp(unit, "m") || !strcasecmp(unit, "mb") || !strcasecmp(unit, "mbit"))
			return 1024 * 1024 * res;
		else if (!strcasecmp(unit, "g") || !strcasecmp(unit, "gb") || !strcasecmp(unit, "gbit"))
			return 1024 * 1024 * 1024 * res;
		else
			throw ConfigError();
	} else
		throw ConfigError();
}

Field::Field(string n, string c, string d, string value, int t, bool op)
{
	name = n;
	type = t;
	comment = c;
	desc = d;
	space = 20;
	min = 0;
	max = 0;
	optional = op;

	active = false;
	int_value = int_value2 = uint_value = 0;
	string_value = value;

	switch (type) {
	case INT:
		int_value = string_to_int(value);
		break;
	case BOOL:
		int_value = !strcasecmp(value.c_str(), "true");
		break;
	case UINT:
		uint_value = string_to_uint(value);
		break;
	case STRING:
	case STRING_LIST:
	case IP_RANGE_LIST:
	case PORT_RANGE_LIST:
		space = 80;
		break;
	case FILE_SIZE:
		uint_value = string_to_filesize(value);
		break;
	case INT_RANGE: 
		{
			Couple couple(value);
			min = string_to_int(couple.first());
			max = string_to_int(couple.second());
		}
		break;
	default:
		break;
	}
}

Field::Field(string n, string c, string d, StringSelectVector& v, int t, bool op)
{
	name = n;
	type = t;
	comment = c;
	desc = d;
	space = 0;  /* meaningless */
	optional = op;

	active = false;
	int_value = int_value2 = uint_value = 0;
	string_select_value = string_select_value2 = v;
}

string Field::possible_values() const
{
	if (string_select_value.size() == 1)
		return string_select_value.begin()->name;

	string res;
	StringSelectVector::const_iterator v, last;

	last = string_select_value.end();
	last--;

	for (v = string_select_value.begin(); v != last; v++)
		res += v->name + ",";
	res += last->name;
	return res;	
}

int Field::type_get(const string& str)
{
	bool found = true;

	if (str == "INT") return INT;

	else if (str == "UINT") return UINT;

	else if (str == "BOOL") return BOOL;

	else if (str == "STRING") return STRING;

	else if (str == "MULTILINE_STRING") return MULTILINE_STRING;

	else if (str == "STRING_SELECT_ONE") return STRING_SELECT_ONE;

	else if (str == "STRING_SELECT_MANY") return STRING_SELECT_MANY;

	else if (str == "STRING_LIST") return STRING_LIST;

	else if (str == "IP_RANGE_LIST") return IP_RANGE_LIST;

	else if (str == "PORT_RANGE_LIST") return PORT_RANGE_LIST;

	else if (str == "STRING_RANGE") return STRING_RANGE;

	else if (str == "INT_RANGE") return INT_RANGE;

	else if (str == "FILE_SIZE") return FILE_SIZE;

	else found = false;

	ASSERT(found);

	return 0;
}

string Field::nature() const
{
	switch (type) {
	case INT:
		return "an integer";
	case UINT:
		return "a positive integer";
	case BOOL:
		return "true or false";
	case STRING:
		return "a string";
	case MULTILINE_STRING:
		return "a multi-line string";
	case STRING_SELECT_ONE:
		return "a string among these values : " + possible_values();
	case STRING_SELECT_MANY:
		return "a list of strings among these values : " + possible_values();
	case STRING_LIST:
		return "a string list";
	case IP_RANGE_LIST:
		return "an IP range list (see manual)";
	case PORT_RANGE_LIST:
		return "a port range list (see manual)";
	case STRING_RANGE:
		return "a couple of strings among these values : " + possible_values();
	case INT_RANGE:
		char str[100];
		snprintf(str, 100, "a couple of integers between %d and %d inclusive", min, max);
		return str;
	case FILE_SIZE:
		return "a file size (for example 120, 10K, 1M)";
	default:
		return "";
	}
}

void Field::error_log(const XmlElement& element) const
{	
	switch (type) {
	case UNIQUE_ID:
		putlog(MMLOG_ERROR, "Parse error in XML file, section %s, element %s: %s must be a unique identifier, but its value %s is already used before", element.section_get().c_str(), element.element_get().c_str(), name.c_str(), string_value.c_str());
		break;
	default:
		putlog(MMLOG_ERROR, "Parse error in XML file, section %s, element %s: %s must be %s, but it is given value %s", element.section_get().c_str(), element.element_get().c_str(), name.c_str(), nature().c_str(), string_value.c_str());
		break;
	}

	ASSERT(0);
}

void Field::load(const XmlElement& element)
{
	StringSelectVector::iterator ssv;
	StringList::iterator sl;
	int i = 0;

	string_value = element.string_read(name);
	if (optional)
		active = element.check_active(name);

	if (optional && !active)
		return;

	switch (type) {
	case BOOL:
		if ((string_value != "true") && (string_value != "TRUE") && (string_value != "false") && (string_value != "FALSE")) {
			error_log(element);	
		}
			
		int_value = (string_value == "true" || string_value == "TRUE") ? 1 : 0;
		break;
	case INT:
		try {
			int_value = string_to_int(string_value);
		} catch (ConfigError) {
			error_log(element);
		}
		break;
	case UINT:
		try {
			uint_value = string_to_uint(string_value);
		} catch (ConfigError) {
			error_log(element);
		}
		break;
	case STRING:
	case MULTILINE_STRING:
		break;
	case STRING_SELECT_ONE:
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) {
			ssv->selected = (ssv->name == string_value);
			if (ssv->selected)
				int_value = i;
			i++;
		}
		break;
	case STRING_SELECT_MANY:
		string_list_value = StringList(string_value);
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) {
			ssv->selected = false;
			
			for (sl = string_list_value.begin(); sl != string_list_value.end(); sl++) {
				if (*sl == ssv->name) {
					ssv->selected = true;
					int_value |= (1<<i);
					break;
				}
			}
			i++;
		}
		break;
	case STRING_LIST:
		string_list_value = StringList(string_value);
		break;
	case IP_RANGE_LIST:		
		try {
			ip_range_list_value = IpRangeList(string_value);
		} catch (ConfigError e) {
			error_log(element);	
		}
		break;
	case PORT_RANGE_LIST:
		try {
			port_range_list_value = PortRangeList(string_value);
		} catch (ConfigError e) {
			error_log(element);
		}
		break;
	case UNIQUE_ID:
		/* unique id is not yet used */
		break;
	case STRING_RANGE:
		try {
			Couple couple(string_value);
			int_value = string_select_value.case_indexof(couple.first());
			int_value2 = string_select_value2.case_indexof(couple.second());
		} catch (ConfigError e) {
			error_log(element);
		}
		break;
	case INT_RANGE:
		try {
			Couple couple(string_value);
			int_value = string_to_int(couple.first());
			int_value2 = string_to_int(couple.second());

			if (!(min <= int_value && int_value <= max && min <= int_value2 && int_value2 <= max))
				error_log(element);

		} catch (ConfigError e) {
			error_log(element);
		}
		break;
	case FILE_SIZE:
		try {
			uint_value = string_to_filesize(string_value);
		} catch (ConfigError e) {
			error_log(element);
		}
		break;
	default:
		break;
	}
}

void Field::save(TiXmlElement* element) const
{
	TiXmlElement *field = xnew TiXmlElement(this->name.c_str());
	ASSERT(field);

	if (optional)
		field->SetAttribute("active", (active) ? "true" : "false");
	
	TiXmlText *text = xnew TiXmlText(this->string_value.c_str());
	ASSERT(text);
	
	TiXmlNode *ret = field->LinkEndChild(text);
	ASSERT(ret);

	ret = element->LinkEndChild(field);
	ASSERT(ret);
}

void Field::zero()
{
	active = false;

	string_list_value.clear();
	ip_range_list_value.clear();
	port_range_list_value.clear();
	

	if (type == STRING_SELECT_MANY) {
		string_value = "";
		int_value = 0;
		StringSelectVector::iterator ssv;
	
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++)
			ssv->selected = false;
	}
}

void Field::read(CGIMap &args, int id)
{
	string value;
	int i = 0;

	StringSelectVector::iterator ssv;

	char sid [5];
	snprintf(sid, 5, "F%d", id);

	value = args[sid];

	if (optional) {
		string s = args[string(sid) + "_active"];
		active = (s == "on");
	}

	if (optional && !active)
		return;

	pthread_mutex_lock(&documentlock);
	documentsynctime = time(NULL);
	pthread_mutex_unlock(&documentlock);

	switch (type) {
	case INT:
		string_value = value;
		int_value = string_to_int(value);
		break;
	case UINT:
		string_value = value;
		uint_value = string_to_uint(value);
		break;
	case BOOL:
		string_value = value;
		int_value = (value == "true") ? 1 : 0;
		break;

	case MULTILINE_STRING:
		string_list_value.clear();
	case STRING:
		string_value = value;	      	
		break;

	case STRING_SELECT_ONE:
		string_value = value;
		string_select_value.reset();
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) {
			if (ssv->name == string_value) {
				ssv->selected = true;
				int_value = i;
				break;
			}
			i++;
		}
		break;

	case STRING_SELECT_MANY:
		string_value = "";
		int_value = 0;
		string_select_value.reset();
		string_list_value.clear();
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) {
			if (args[string(sid) + "_" + ssv->name] == "on") {
				ssv->selected = true;
				int_value |= (1<<i);
				if (string_value == "")
					string_value = ssv->name;
				else
					string_value += ',' + ssv->name;

				string_list_value.push_back(ssv->name);
			}
			i++;
		}
		break;

	case STRING_LIST:
		string_value = value;
		string_list_value.clear();
		string_list_value.parse(string_value);
		break;
	case IP_RANGE_LIST:
		string_value = value;
		ip_range_list_value.clear();
		ip_range_list_value.parse(string_value);
		break;
	case PORT_RANGE_LIST:
		string_value = value;
		port_range_list_value.clear();
		port_range_list_value.parse(string_value);
		break;
	case UNIQUE_ID:
		/*
		if (item_list->find(value) && value != string_value)
			return comment + " must be a unique identifier, but its value is already used";
		else {
			string_value = value;
			return "";
		}
		*/
		break;
	case STRING_RANGE:
		string_select_value.reset();
		string_select_value2.reset();

		string_value = value;		
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) {
			if (ssv->name == value) {
				ssv->selected = true;
				int_value = i;
				break;
			}
			i++;
		}

		
		value = args[string(sid) + "_2"];
		string_value += "," + value;

		for (ssv = string_select_value2.begin(); ssv != string_select_value2.end(); ssv++) {
			if (ssv->name == value) {
				ssv->selected = true;
				int_value2 = i;
				break;
			}
			i++;
		}

		break;

	case INT_RANGE:		
		string_value = value;
		
		int_value = string_to_int(value);
		if (!(min <= int_value && int_value <= max))
			throw ConfigError();

		value = args[string(sid) + "_2"];
		string_value += "," + value;
			
		int_value2 = string_to_int(value);
		if (!(min <= int_value2 && int_value2 <= max))
			throw ConfigError();
		
		break;
	case FILE_SIZE:
		string_value = value;
		uint_value = string_to_filesize(string_value);
		break;
	default:
		break;
	}
	
}

void Field::display(Filebuf * filebuf) const
{
	if ((optional && active) || (!optional && string_value != "")) {
		char* ptr = string_to_html(string_value.c_str(), HTML_NEWLINES);
		filebuf->Addf("<tr><td width=\"20%%\">%s</td><td>%s</td></tr>\n", comment.c_str(), ptr);
		xfree(ptr);
	}
}


void Field::dialog_display(Filebuf * filebuf, CGIMap &args, int id) const
{
	string value = string_value;
	bool active = this->active;
	char* ptr;
	StringSelectVector::const_iterator ssv;

	/* a value in the arguments overwrites the field value, when this value is submitted */
	/*
	this doesn't work right... if we add/edit an entry this will cause the global options to reflect
	the new/edited entries because the same naming scheme is used (F0, F1, F2, etc.) for fields.

	char sid [5];
	snprintf(sid, 5, "F%d", id);

	string s = args[sid];
	if (s != "") value = s;
	*/

	ptr = string_to_html(desc.c_str(), HTML_NEWLINES | HTML_JAVASCRIPT);
	filebuf->Addf("<tr><td width=\"20%\" onMouseOver=\"EnterContent('%s', '%s'); Activate();\" onMouseOut=\"deActivate();\">%s</td><td>", comment.c_str(), ptr, comment.c_str());
	xfree(ptr);

	if (optional) {
		filebuf->Addf("<input type=\"checkbox\" name=\"F%d_active\" %s>active &nbsp;&nbsp;&nbsp;", id, (active) ? "checked" : "");
	}

	switch (type) {
	case BOOL:
		filebuf->Addf("Yes: <input type=\"radio\" name=\"F%d\" value=\"true\" %s> ", id, (value == "true") ? "checked" : "");
		filebuf->Addf("No: <input type=\"radio\" name=\"F%d\" value=\"false\" %s> ", id, (value == "true") ? "" : "checked");
		filebuf->Addf("</td></tr>\n");
		break;
	case MULTILINE_STRING:
		ptr = string_to_html(value.c_str(), 0);
		filebuf->Addf("<textarea name=\"F%d\" cols=\"80\" rows=\"8\" wrap=\"virtual\">%s</textarea></td></tr>\n", id, (ptr != NULL) ? ptr : "", space);
		FREE_AND_NULL(ptr);
		break;
	case INT:
	case UINT:
	case STRING:
	case STRING_LIST:
	case IP_RANGE_LIST:
	case PORT_RANGE_LIST:
	case UNIQUE_ID:
	case FILE_SIZE:
		ptr = string_to_html(value.c_str(), FALSE);
		filebuf->Addf("<input type=\"text\" name=\"F%d\" value=\"%s\" size=\"%d\">", id, (ptr != NULL) ? ptr : "", space);
		FREE_AND_NULL(ptr);
		break;
	case STRING_SELECT_ONE:
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) 
			filebuf->Addf("%s: <input type=\"radio\" name=\"F%d\" value=\"%s\" %s> ", ssv->comment.c_str(), id, ssv->name.c_str(), (ssv->selected) ? "checked" : "");
		break;
	case STRING_SELECT_MANY:
		filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\">\n");
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) 
			filebuf->Addf("<tr><td>%s</td><td><input type=\"checkbox\" name=\"F%d_%s\" %s></td></tr>\n", ssv->comment.c_str(), id, ssv->name.c_str(), (ssv->selected) ? "checked" : "");		       
		filebuf->Addf("</table>");
		break;
	case STRING_RANGE:
		filebuf->Addf("<select name=\"F%d\">", id);
		for (ssv = string_select_value.begin(); ssv != string_select_value.end(); ssv++) 
			filebuf->Addf("<option value=\"%s\" %s>%s\n", ssv->name.c_str(), (ssv->selected) ? "selected" : "", ssv->comment.c_str());
	
		filebuf->Addf("</select>");
		filebuf->Addf("&nbsp;to ");
		filebuf->Addf("<select name=\"F%d_2\">", id);
		for (ssv = string_select_value2.begin(); ssv != string_select_value2.end(); ssv++) 
			filebuf->Addf("<option value=\"%s\" %s>%s\n", ssv->name.c_str(), (ssv->selected) ? "selected" : "", ssv->comment.c_str());
		
		filebuf->Addf("</select>");
		break;
	case INT_RANGE:
		filebuf->Addf("<input type=\"text\" name=\"F%d\" size=\"%d\" value=\"%d\"> to <input type=\"input\" name=\"F%d_2\" size=\"%d\" value=\"%d\">", id, space, int_value, id, space, int_value2);
		break;
	default:
		break;
	}

	filebuf->Addf("</td></tr>\n");
}

FieldVector::FieldVector(const TiXmlElement& option_element, SSVMap &ssv_map)
{
	parse(option_element, ssv_map);
}

void FieldVector::parse(const TiXmlElement& option_element, SSVMap &ssv_map)
{
	TiXmlElement *element;
	TiXmlNode *node;
	int num, count = 0;

	for (TiXmlElement* option = option_element.FirstChildElement(); option; option = option->NextSiblingElement()) {
		ASSERT(option->Attribute("n", &num));
		ASSERT((num == count));
		
		string name = option->Value();
		
		ASSERT(element = option->FirstChildElement("comment"));
		ASSERT(node = element->FirstChild());
		string comment = node->Value();

		ASSERT(element = option->FirstChildElement("value"));
		string value = (node = element->FirstChild()) ? node->Value() : "";

		ASSERT(element = option->FirstChildElement("type"));
		ASSERT(node = element->FirstChild());
		int type = Field::type_get(node->Value());

		bool optional = false;
		element = option->FirstChildElement("optional");
		if (element) {
			ASSERT(node = element->FirstChild());
			optional = !strcasecmp(node->Value(), "true");
		}

		ASSERT(element = option->FirstChildElement("desc"));
		node = element->FirstChild();
		string desc = (node) ? node->Value() : "";

		if (type == Field::STRING_SELECT_ONE || type == Field::STRING_SELECT_MANY || type == Field::STRING_RANGE)
			push_back(Field(name, comment, desc, ssv_map[value], type, optional));
		else
			push_back(Field(name, comment, desc, value, type, optional));
			
		count++;
	}
}

StringList FieldVector::read(CGIMap &args)
{
	StringList error_list;
	int id = 0;

	FieldVector::iterator field;
	for (field = begin(); field != end(); field ++) {
		
		try {
			field->read(args, id);
		} catch (ConfigError) {
			error_list.push_back(field->comment + " must be " + field->nature());
		}

		id++;
	}

	return error_list;
}

void FieldVector::save(TiXmlElement* element) const
{
	FieldVector::const_iterator field;
	for (field = begin(); field != end(); field++)
		field->save(element);		
}

void FieldVector::display(Filebuf* filebuf) const
{
	for (FieldVector::const_iterator field = begin(); field != end(); field++) 
		field->display(filebuf);
}

void FieldVector::dialog_display(Filebuf* filebuf, CGIMap &args) const
{
	int id = 0;

	for (FieldVector::const_iterator field = begin(); field != end(); field++) {
		field->dialog_display(filebuf, args, id);
		id++;
	}
}

Subsection::Subsection(const string s, const string n, const string c, const FieldVector& t):
     section(s),
     name(n),
     comment(c),
     templ(t)
{
}

typedef map<string, FieldVector> FieldVectorMap;

/* 
 * Section constructor based on the XML file section.xml
 */
Section::Section(string n, int lt, bool r):
	lock_type(lt),
	name(n),
	restart(r)
{
	static bool loaded = false;
	static TiXmlDocument doc(sectionfile);
	TiXmlElement *element;
	TiXmlNode *node;
	SSVMap  ssv_map;
	FieldVectorMap templ_map;

	map[name] = this;

	if (lock_type == RWLOCK)
		pthread_rwlock_init(&this->rwlock, NULL);
	else
		pthread_mutex_init(&this->mutex, NULL);



	if (!loaded) {
		bool loadOkay = doc.LoadFile();
		if (!loadOkay) {
			putlog(MMLOG_ERROR, "Error when parsing section.xml: %s", doc.ErrorDesc());
			exit(1);
		}
			       
		loaded = true;
	}

	
	ASSERT(element = doc.FirstChildElement("section"));
	ASSERT(element = element->FirstChildElement(name.c_str()));
	TiXmlElement& section = *element;

	ASSERT(element = section.FirstChildElement("comment"));
	ASSERT(node = element->FirstChild());
	this->comment = node->Value();
	
	if ((element = section.FirstChildElement("select"))) {
		for (TiXmlElement* select = element->FirstChildElement(); select; select = select->NextSiblingElement()) {
			
			string select_name = select->Value();
			
			StringSelectVector ssv;

			for (TiXmlElement* item = select->FirstChildElement(); item; item = item->NextSiblingElement()) {
				string item_name = item->Value();

				string item_comment;
				element = item->FirstChildElement("comment");
				if (element) {					
					ASSERT(node = element->FirstChild());
					item_comment = node->Value();
				} else {
					item_comment = item_name;
				}
				
				bool item_value = false;
				element = item->FirstChildElement("value");
				if (element) {
					ASSERT(node = element->FirstChild());
					item_value = !strcasecmp(node->Value(), "true");
				}

				string item_desc = "";
				element = item->FirstChildElement("desc");
				if (element) {
					node = element->FirstChild();
					item_desc = (node) ? node->Value() : "";
				}
				
				ssv.push_back(StringSelect(item_name, item_comment, item_desc, item_value));
			}

			ssv_map[select_name] = ssv;
		}
	}

	ASSERT(element = section.FirstChildElement("global"));
	TiXmlElement& global_options = *element;

	field_vec.parse(global_options, ssv_map);

	element = section.FirstChildElement("template");
	
	if (element) {
		/* We read entry templates that will be used to build 
		   subsection */
		
		for (TiXmlElement* templ = element->FirstChildElement(); templ; templ = templ->NextSiblingElement()) {
			FieldVector templ_fileds(*templ, ssv_map);
			
			templ_map[templ->Value()] = templ_fileds;
		}
	}

	element = section.FirstChildElement("subsection");

	if (element) {
		/* Build the subsections with the given templates */
		
		for (TiXmlElement* subsec = element->FirstChildElement(); subsec; subsec = subsec->NextSiblingElement()) {
			string subsec_name = subsec->Value();
			
			ASSERT(element = subsec->FirstChildElement("comment"));
			ASSERT(node = element->FirstChild());
			string subsec_comment = node->Value();
			
			ASSERT(element = subsec->FirstChildElement("template"));
			ASSERT(node = element->FirstChild());
			string subsec_templ = node->Value();
			
			this->sub_vec.push_back(Subsection(this->name, subsec_name, subsec_comment, templ_map[subsec_templ]));
		}
	}
}		    

Section::~Section()
{
	pthread_rwlock_destroy(&this->rwlock);
}

void Section::read_lock() const
{
	if (lock_type == RWLOCK)
		pthread_rwlock_rdlock(&this->rwlock);
	else
		pthread_mutex_lock(&this->mutex);
}

void Section::write_lock() const
{
	if (lock_type == RWLOCK)
		pthread_rwlock_wrlock(&this->rwlock);
	else
		pthread_mutex_lock(&this->mutex);
}

void Section::mutex_lock() const
{
	ASSERT(lock_type == MUTEX);
	pthread_mutex_lock(&this->mutex);
}

void Section::unlock() const
{
	if (lock_type == RWLOCK)
		pthread_rwlock_unlock(&this->rwlock);
	else
		pthread_mutex_unlock(&this->mutex);
}

int Section::atomic_read(int* x) const
{
	if (lock_type == RWLOCK)
		return atomic_read_rwlock(&this->rwlock, x);
	else
		return atomic_read_mutex(&this->mutex, x);
}

int Section::atomic_read(unsigned int* x) const
{
	if (lock_type == RWLOCK)
		return atomic_read_rwlock(&this->rwlock, x);
	else
		return atomic_read_mutex(&this->mutex, x);
}

int Section::atomic_read(long * x) const
{
	if (lock_type == RWLOCK)
		return atomic_read_rwlock(&this->rwlock, x);
	else
		return atomic_read_mutex(&this->mutex, x);
}

int Section::atomic_read(unsigned long * x) const
{
	if (lock_type == RWLOCK)
		return atomic_read_rwlock(&this->rwlock, x);
	else
		return atomic_read_mutex(&this->mutex, x);
}

int Section::cond_wait(pthread_cond_t* cond)
{
	ASSERT(lock_type == MUTEX);
	return pthread_cond_wait(cond, &this->mutex);
}


void Section::load(TiXmlDocument* doc, int overwrite)
{
	write_lock();

	/* overwrite is 3 at startup. For sections that needs restart before that their 
	   changes take effect (like network), we load the section only at startup time */
	if (restart && overwrite != 3) {
		unlock();
		return;
	}
	

	XmlElement element = XmlElement(doc, this->name);
	
	FieldVector::iterator field;
	for (field = field_vec.begin(); field != field_vec.end(); field++)
		field->load(element);

	SubsectionVector::iterator sub;
	for (sub = sub_vec.begin(); sub != sub_vec.end(); sub++) {
		
		XmlElement subelement = XmlElement(doc, this->name, sub->name);
		
		if (overwrite)
			sub->item_list.clear();

		while (subelement.full()) {

			sub->item_list.push_back(Item(*sub, &subelement));
			subelement.next();
				
		}
	}
	
	update();
	
	unlock();
}

void Section::save(TiXmlDocument* doc)
{
	read_lock();

	TiXmlElement* tixml_section = section_select(doc, this->name.c_str())->ToElement();

	/* clear section */
	tixml_section->Clear();

	field_vec.save(tixml_section);
	
	SubsectionVector::const_iterator sub;
	ItemList::const_iterator item;
	for (sub = sub_vec.begin(); sub != sub_vec.end(); sub++) {

		for (item = sub->item_list.begin(); item != sub->item_list.end(); item++) {
			
			TiXmlElement *tixml_item = xnew TiXmlElement(sub->name.c_str());
			ASSERT(tixml_item);

			item->field_vec.save(tixml_item);

			TiXmlNode *ret = tixml_section->LinkEndChild(tixml_item);
			ASSERT(ret);
		}
	}
		
	unlock();
}

void Section::interface(Filebuf * filebuf, CGIMap &args, CONNECTION * connection, const StringList& error_list)
{
	read_lock();
	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\">\n");

	if (field_vec.size() != 0) { 

		filebuf->Addf("<tr><td align=\"center\"><table class=\"bigdialog\">");

		if (error_list.size() != 0) {
			filebuf->Addf("<tr><td colspan=\"2\"><span style=\"color: rgb(102, 0, 0);\">Submission failed, please correct these errors:<br><ul>\n");
			
			StringList::const_iterator error;
			for (error = error_list.begin(); error != error_list.end(); error++)
				filebuf->Addf("<li>%s</li>\n", error->c_str());
			
			filebuf->Addf("</ul></span></td></tr>\n");
		}

		filebuf->Addf("<tr></td><form action=\"config\" method=\"POST\">\n");
		filebuf->Addf("<input type=\"hidden\" name=\"section\" value=\"%s\">\n", this->name.c_str());
		filebuf->Addf("<input type=\"hidden\" name=\"action\" value=\"global\">\n");
		
		field_vec.dialog_display(filebuf, args);

		filebuf->Addf("<tr><td colspan=\"2\"><br></td></tr> <tr class=\"dialogfoot\"><td colspan=\"2\" align=\"center\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");
		filebuf->Addf("</form></td></tr></table></td></tr> <tr><td><br></td></tr>\n");
	}

	SubsectionVector::iterator sub;
	for (sub = sub_vec.begin(); sub != sub_vec.end(); sub++) {

		filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%%\">\n");
		filebuf->Addf("<tr><td align=\"center\"><table class=\"centerdialog\">\n");
		filebuf->Addf("<tr class=\"listhead\"><td colspan=\"2\" align=\"center\"><b>%s%s</b></td></tr>\n", sub->comment.c_str(), (this->restart) ? " (need restart)" : "");
		filebuf->Addf("<tr><td colspan=\"2\" align=\"center\"><a href=\"config?section=%s&amp;subsection=%s&amp;dialog=show\">Add</a></td></tr>\n", this->name.c_str(), sub->name.c_str());
		filebuf->Addf("</table></td></tr> <tr><td><br></td></tr>\n");

		ItemList::iterator item;
		for (item = sub->item_list.begin(); item != sub->item_list.end(); item++) {
			
			filebuf->Addf("<tr><td align=\"center\">\n");

			filebuf->Addf("<table class=\"dialog\">\n");
			item->display(filebuf, TRUE);

			filebuf->Addf("</table></td></tr> <tr><td><br></td></tr>\n");
		}
		
		filebuf->Addf("</table>\n");
	}

	filebuf->Addf("</table>\n");
	
	unlock();
}

void Section::interface_dialog(Filebuf * filebuf, CGIMap &args, CONNECTION * connection, const StringList& error_list)
{
	read_lock();

	int id = -1;
	Item* item = 0;
	Subsection* sub = 0;       

	string s = args["id"];
	id = (s != "") ? atoi(s.c_str()) : -1;

	sub = subsection_get(args["subsection"]);

	ASSERT(sub);

	if (id != -1)
		item = item_get(id);
	

	filebuf->Addf("<tr><td align=\"center\"><table class=\"bigdialog\">\n");

	/* LIST ERRORS */ 
	if (error_list.size() != 0) {
		filebuf->Addf("<tr><td colspan=\"2\"><span style=\"color: rgb(102, 0, 0);\">Submission failed, please correct these errors:<br><ul>\n");
		
		StringList::const_iterator error;
		for (error = error_list.begin(); error != error_list.end(); error++)
			filebuf->Addf("<li>%s</li>\n", error->c_str());
		
		filebuf->Addf("</ul></span></td></tr>\n");
	}

	filebuf->Addf("<form action=\"config\" method=\"POST\">\n");

	filebuf->Addf("<input type=\"hidden\" name=\"section\" value=\"%s\">\n", this->name.c_str());
	filebuf->Addf("<input type=\"hidden\" name=\"subsection\" value=\"%s\">\n", sub->name.c_str());

	if (id != -1) {
		filebuf->Addf("<input type=\"hidden\" name=\"action\" value=\"edit\">\n");
		filebuf->Addf("<input type=\"hidden\" name=\"id\" value=\"%d\">\n", id);
	} else
		filebuf->Addf("<input type=\"hidden\" name=\"action\" value=\"add\">\n");

	if (item) {
		item->field_vec.dialog_display(filebuf, args);
	} else {
		sub->templ.dialog_display(filebuf, args);
	}

	filebuf->Addf("<tr><td colspan=\"2\"><br></td></tr>\n");
	filebuf->Addf("<tr class=\"dialogfoot\"><td colspan=\"2\" align=\"center\"><input type=\"submit\" value=\"Submit\"></td></tr>\n");

	filebuf->Addf("</form></table></td></tr>\n");

	unlock();
}

StringList Section::interface_action(Filebuf * filebuf, CGIMap &args, CONNECTION * connection)
{
	write_lock();

	int id = -1;
	string action;
	Subsection* sub = 0;       
	StringList error_list;

	string s = args["id"];
	id = (s != "") ? atoi(s.c_str()) : -1;

	sub = subsection_get(args["subsection"]);
	
	action = args["action"];

	if (action == "")
		goto interface_action_fail;

	if (id == -1 && action != "add") {
		/* We modify the fields of this->field_vec */

		/* Chek first the syntax and place the result in new_field_vec */
		FieldVector new_field_vec = field_vec;
		error_list = new_field_vec.read(args);

		/* If no error, update field_vec with new values */
		if (error_list.size() == 0) {
			field_vec = new_field_vec;
		}

	} else if (action == "add") {
		/* Add a new item in sub->item_list */
	
		Item new_item(*sub);
		
		error_list = new_item.read(args);
		
		if (error_list.size() == 0) {
			sub->item_list.push_back(new_item);
		}
		
	} else {
		/* Modify or remove an item in sub_bec */
		
		ItemList::iterator item = item_iterator_get(id);

		ItemList& item_list = sub->item_list;

		if (action == "delete") {
			item_list.erase(item);
		} else if (action == "edit") {
			/* Chek first the syntax and place the result in new_field_vec */

			FieldVector new_field_vec = item->field_vec;
			error_list = new_field_vec.read(args);

			/* If no error, update filed_vec with new values */
			if (error_list.size() == 0) {
				item->field_vec = new_field_vec;
			}

		} else if (action == "shift") {
			const string dir = args["direction"];

			if (dir == "up") {
				if (item != item_list.begin()) iter_swap(--item, item);
			} else if (dir == "down") {
				if (item != --item_list.end()) iter_swap(++item, item);
			} else if (dir == "top") {
				if (item != item_list.begin()) {
					item_list.push_front(*item);
					item_list.erase(item);
				}
			} else if (dir == "bottom") {
				if (item != --item_list.end()) {
					item_list.push_back(*item);
					item_list.erase(item);
				}
			}
		}
	}

	update();

 interface_action_fail:
	unlock();

	return error_list;
}

Item* Section::item_get(int id)
{
	return &*item_iterator_get(id);
}

ItemList::iterator Section::item_iterator_get(int id)
{
	SubsectionVector::iterator sub;
	ItemList::iterator item;

	for (sub = sub_vec.begin(); sub != sub_vec.end(); sub++) {
		for (item = sub->item_list.begin(); item != sub->item_list.end(); item++) {
			if (item->is(id)) {
				return item;
			}
		}
	}

	return sub->item_list.end();	
}


Subsection* Section::subsection_get(const string &name)
{
	SubsectionVector::iterator sub;
	for (sub = sub_vec.begin(); sub != sub_vec.end(); sub++) {
		if (sub->name == name)
			return &*sub;
	}

	return 0;
}

void Section::display_enabled(Filebuf* filebuf)
{
	read_lock();
	if (field_vec.size() != 0 && field_vec[0].name == "enabled") {
		int checked = field_vec[0].int_value == TRUE;
		filebuf->Addf("<tr><td>%s</td><td><input type=\"radio\" name=\"%s\" value=\"on\" %s></td><td><input type=\"radio\" name=\"%s\" value=\"off\" %s></td></tr>\n", comment.c_str(), name.c_str(), (checked) ? "checked" : "", name.c_str(), (!checked) ? "checked" : "");
	}
	unlock();
}

string Section::name_get() const
{
	string temp;
	read_lock();
	temp = name;
	unlock();
	return temp;	
}

unsigned int address_to_int(const string& ip)
{
	StringList string_list(ip, '.');
	StringList::const_iterator s;
	unsigned int res = 0;
	char* endptr;

	if (string_list.size() != 4)
		throw ConfigError();

	for (s = string_list.begin(); s != string_list.end(); s++) {
		int a = strtol(s->c_str(), &endptr, 10);
		if ((*endptr != '\0') || (a > 255) || (a < 0))
			throw ConfigError();

		res = (res << 8) + a;
	}
		
	return res;
}

/*
 * Constructor of IpRange : parse "a-b"
 */
IpRange::IpRange(const string& str)
{
	if (str.find('-') == string::npos) {
		if (str == "") {
			ip_start = "0.0.0.0";
			ip_end = "255.255.255.255";
		} else {
			ip_start = ip_end = str;
		}
	} else {

		ip_start = str.substr(0, str.find('-'));
		ip_end   = str.substr(str.find('-') + 1);
		
		if (ip_start == "") ip_start = "0.0.0.0";
		if (ip_end == "") ip_end = "255.255.255.255";

	}
	
	ip_start_int = address_to_int(ip_start);
	ip_end_int   = address_to_int(ip_end);
}


/*
 * Is the IP in the range?
 */
int IpRange::has(char* ip) const
{
	unsigned int ip_int = address_to_int(ip);

	return (ip_start_int <= ip_int && ip_int <= ip_end_int);
}


IpRangeList::IpRangeList() 
{
}

IpRangeList::IpRangeList(const string& s) 
{
	parse(s);
}

void IpRangeList::parse(const string& s)
{
	StringList string_list(s);

	StringList::const_iterator i;
	for (i = string_list.begin(); i != string_list.end(); i++)
		this->push_back(IpRange(*i));
	
}

int IpRangeList::has(char* ip) const
{
	if (size() == 0)
		return TRUE;

	IpRangeList::const_iterator i;

	for (i = begin(); i != end(); i++) {
		if (i->has(ip))
			return TRUE;
	}

	return FALSE;
}

PortRange::PortRange(const string& str)
{
	char* endptr;

	if (str.find('-') == string::npos) {
		start = end = strtol(str.c_str(), &endptr, 10);
		if ((*endptr != '\0') || (start > MAX_PORT) || (start <= 0))
			throw ConfigError();
	} else {

		string p1, p2;
		
		p1 = str.substr(0, str.find('-'));
		p2 = str.substr(str.find('-') + 1);
		
		start = strtol(p1.c_str(), &endptr, 10);
		if ((*endptr != '\0') || (start > MAX_PORT) || (start <= 0))
			throw ConfigError();

		end = strtol(p2.c_str(), &endptr, 10);
		if ((*endptr != '\0') || (end > MAX_PORT) || (end <= 0))
			throw ConfigError();
	}
}

PortRangeList::PortRangeList()
{
}

PortRangeList::PortRangeList(const string& str)
{
	parse(str);
}

void PortRangeList::parse(const string& str)
{
	StringList string_list(str);

	StringList::const_iterator i;
	for (i = string_list.begin(); i != string_list.end(); i++)
		this->push_back(PortRange(*i));
}

int PortRangeList::has(int port) const
{
	if (size() == 0)
		return TRUE;

	list<PortRange>::const_iterator i;

	for (i = begin(); i != end(); i++)
		if (i->start <= port && port <= i->end)
			return TRUE;

	return FALSE;
}

