/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <unistd.h>
#include "proto.h"

extern GeneralSection *general_section;
extern GLOBAL *global;

POOL *pool_init()
{
	POOL *pool;

	pool = (POOL*)xmalloc(sizeof(POOL));

	pool->entries = 0;
	pool->pool_list = NULL;

	pthread_mutex_init(&pool->lock, NULL);

	return pool;
}

void pool_add(POOL * pool, Socket *sock, char *proto, char *host, int port, char *username, char *password)
{
	struct pool_t *pitem;

	pthread_mutex_lock(&pool->lock);

	putlog(MMLOG_DEBUG, "pool_add: entries: %d max: %d", pool->entries, general_section->poolsize_get());

	if (pool->entries >= general_section->poolsize_get() && pool->pool_list != NULL) {
		pitem = pool->pool_list;
		pool->pool_list = pool->pool_list->next;
		pool->pool_list->prev = NULL;
		pool->entries--;

		xdelete pitem->sock;
		xfree(pitem->proto);
		xfree(pitem->host);
		FREE_AND_NULL(pitem->username);
		FREE_AND_NULL(pitem->password);
		xfree(pitem);
	}

	if (pool->pool_list == NULL) {
		pitem = (pool_t*)xmalloc(sizeof(struct pool_t));
		pitem->prev = NULL;
		pool->pool_list = pitem;
	} else {
		for (pitem = pool->pool_list; pitem->next; pitem = pitem->next);
		pitem->next = (pool_t*)xmalloc(sizeof(struct pool_t));
		pitem->next->prev = pitem;
		pitem = pitem->next;
	}

	pitem->next = NULL;
	pitem->age = time(NULL);
	pitem->sock = sock;
	pitem->proto = xstrdup(proto);
	pitem->host = xstrdup(host);
	pitem->username = (username != NULL) ? xstrdup(username) : NULL;
	pitem->password = (password != NULL) ? xstrdup(password) : NULL;
	pitem->port = port;

	pool->entries++;

	pthread_mutex_unlock(&pool->lock);
}

Socket *pool_find(POOL * pool, char *proto, char *host, int port, char *username, char *password)
{
	Socket *sock = NULL;
	struct pool_t *pitem;

	pthread_mutex_lock(&pool->lock);

	for (pitem = pool->pool_list; pitem; pitem = pitem->next) {
		if (username != NULL && pitem->username != NULL) {
			if (strcasecmp(pitem->username, username))
				continue;
		} else if (username != NULL || pitem->username != NULL)
			continue;

		if (password != NULL && pitem->password != NULL) {
			if (strcasecmp(pitem->password, password))
				continue;
		} else if (username != NULL || pitem->username != NULL)
			continue;

		if (!strcasecmp(proto, pitem->proto) && !strcasecmp(pitem->host, host) && pitem->port == port)
			break;
	}

	if (pitem != NULL) {
		if (pitem->next != NULL)
			pitem->next->prev = pitem->prev;
		if (pitem->prev != NULL)
			pitem->prev->next = pitem->next;
		else
			pool->pool_list = pitem->next;

		sock = pitem->sock;
		xfree(pitem->proto);
		xfree(pitem->host);
		FREE_AND_NULL(pitem->username);
		FREE_AND_NULL(pitem->password);
		xfree(pitem);

		pool->entries--;
	}

	pthread_mutex_unlock(&pool->lock);

	putlog(MMLOG_DEBUG, "pool_find %s %s %d = %s", proto, host, port, (sock != NULL) ? "TRUE" : "FALSE");

	if (sock)
		global->stats.Increment("connection pool", "hit");
	else
		global->stats.Increment("connection pool", "miss");

	return sock;
}

void pool_clean(POOL * pool, int force)
{
	pthread_mutex_lock(&pool->lock);

	pool_clean_unlocked(pool, force);

	pthread_mutex_unlock(&pool->lock);
}

void pool_clean_unlocked(POOL * pool, int force)
{
	int count = 0, psize, ptimeout;
	time_t curtime;
	struct pool_t *pitem, *tmp;

	curtime = time(NULL);
	psize = general_section->poolsize_get();
	ptimeout = general_section->pooltimeout_get();

	for (count = 0, pitem = pool->pool_list; pitem; count++) {
		if (pitem->age + ptimeout <= curtime || count > psize || force) {
			putlog(MMLOG_DEBUG, "pool_clean: removed %s from pooled connections", pitem->host);

			pool->entries--;

			tmp = pitem->next;

			if (pitem->next != NULL)
				pitem->next->prev = pitem->prev;
			if (pitem->prev != NULL)
				pitem->prev->next = pitem->next;
			else
				pool->pool_list = tmp;

			xdelete pitem->sock;
			xfree(pitem->proto);
			xfree(pitem->host);
			FREE_AND_NULL(pitem->username);
			FREE_AND_NULL(pitem->password);
			xfree(pitem);

			pitem = tmp;
		} else
			pitem = pitem->next;
	}
}
