/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin
    Copyright (C) 2003  Riadh Elloumi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "proto.h"

extern GeneralSection *general_section;
extern long gmtoffset;


StringList::StringList()
{
}

StringList::StringList(const string& str, const char separator)
{
	parse(str, separator);
}

void StringList::parse(const string& str, const char separator)
{
	if (str != "") {
		string p1, p2 = str;

		while (p2 != "") {

			if (p2.find(separator) == string::npos) {
				this->push_back(p2);
				p2 = "";
			}
			else {
				p1 = p2.substr(0, p2.find(separator));
				p2 = p2.substr(p2.find(separator) + 1);
				
				this->push_back(p1);
			}
		}
	}
}

int StringList::has(const string s) const
{
	for (StringList::const_iterator str = begin(); str != end(); str++) {
		if (*str == s)
			return TRUE;
	}
	return FALSE;
}

Couple::Couple(const string& str):
	StringList(str)
{
	if (size() != 2)
		throw ConfigError();
	
}
	
string Couple::first() const
{
	return *begin();
}

string Couple::second() const
{
	return *(++begin());
}

int StringSelectVector::case_indexof(const string& str) const
{
	int i = 0;
	for (StringSelectVector::const_iterator ss = begin(); ss != end(); ss++) {
		if (!strcasecmp(str.c_str(), ss->name.c_str()))
			return i;
		i++;
	}

	throw ConfigError();

}

void StringSelectVector::reset()
{
	for (StringSelectVector::iterator ssv = begin(); ssv != end(); ssv++)
		ssv->selected = false;
}

/*
strncpy which always NULL terminates
*/
char *s_strncpy(char *d, const char *s, size_t len)
{
	char *dest = d;

	for (; len > 1 && (*dest = *s); s++, dest++, len--);
	*dest = '\0';

	return d;
}

/*
strncat which always NULL terminates
*/
char *s_strncat(char *d, char *s, size_t len)
{
	char *dest = d;

	for (; len > 1 && *dest; dest++, len--);
	for (; len > 1 && (*dest = *s); s++, dest++, len--);
	*dest = '\0';

	return d;
}

/*
case-insensitive strstr
*/
char *s_strcasestr(char *haystack, char *needle)
{
	char *n, *tmp;

	for (; *haystack; haystack++) {
		for (tmp = haystack, n = needle; *tmp && *n && tolower(*tmp) == tolower(*n); tmp++, n++);
		if (!*n)
			return haystack;
	}

	return NULL;
}

/*
return descriptive error for a code
*/
char *code_to_error(int code)
{
	switch (code) {
	case 200:
		return "OK";
	case 301:
		return "Document Moved";
	case 302:
		return "Found";
	case 304:
		return "Not Modified";
	case 400:
		return "Bad Request";
	case 403:
		return "Forbidden";
	case 404:
		return "Not Found";
	case 401:
	case 407:
		return "Authenticaton required";
	case 500:
		return "Internal Error";
	case 501:
		return "Not Implemented";
	case 502:
		return "Service Temporarily Unavailable";
	case 503:
		return "Connection Failed";
	default:
		return "Unknown";
	}

	return NULL;
}

size_t s_strnlen(const char *s, size_t len)
{
	size_t ret;

	for (ret = 0; len && *s; s++, len--, ret++);

	return ret;
}

/*
convert an error flag into a template name
*/
char *error_to_template(int error)
{
	switch (error) {
	case ERROR_CONNECT:
		return "noconnect";
	case ERROR_DNS:
		return "nodns";
	case ERROR_AUTH:
		return "badauth";
	case ERROR_UNKNOWN:
	default:
		return "unknown";
	}

	return NULL;
}

/*
free HOSTENT structure
*/
void hostent_free(HOSTENT * hostent)
{
	xfree(hostent->addr);
	xfree(hostent);
}

/*
return TRUE if a string appears to be an IP address
*/
int isip(const char *string)
{
	int dots = 0;

	for (; *string; string++) {
		if (*string == '.')
			dots++;
		else if (!isdigit((int) *string))
			return FALSE;
	}

	return (dots == 3) ? TRUE : FALSE;
}

void string_tolower(char *string)
{
	for (; *string; string++)
		*string = tolower(*string);
}

/*
replace characters that have a special meaning in html with their respective entity
*/
char *string_to_html(const char *s, int flags)
{
	int maxlen, len = 0;
	char *ret;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	maxlen = general_section->linelength_get();

	for (; *s; s++) {
		if ((flags & HTML_JAVASCRIPT) && strchr("';.?/\\[]{}()+*|$", *s))
			filebuf->Addf("\\%c", *s);
		else if (*s == '"')
			filebuf->Add("&quot;", 6);
		else if (*s == '<')
			filebuf->Add("&lt;", 4);
		else if (*s == '>')
			filebuf->Add("&gt;", 4);
		else if (*s == '&')
			filebuf->Addf("&amp;", 5);
		else if ((flags & HTML_NEWLINES) && *s == '\n')
			filebuf->Add("<br>", 4);
		else if ((flags & HTML_SPACES) && *s == ' ')
			filebuf->Addf("&nbsp;", 6);
		else
			filebuf->Add(s, 1);

		/* the browser will wrap at spaces */
		if (*s == ' ')
			len = 0;

		if (flags == HTML_NEWLINES && ++len % maxlen == 0)
			filebuf->Add("<br>", 4);
	}

	filebuf->Add("", 1);
	filebuf->Shorten();

	ret = filebuf->data;

	filebuf->data = NULL;
	xdelete filebuf;

	return ret;
}

/*
decode all hex characters in a url
*/
char *url_decode(char *url, size_t len)
{
	char *ret, *u = url, tmp[3];
	Filebuf *filebuf;

	tmp[2] = '\0';

	filebuf = xnew Filebuf();

	while (*u && u - url < len) {
		if (*u == '%' && u[1] != '\0') {
			tmp[0] = u[1];
			tmp[1] = u[2];

			filebuf->Addf("%c", strtol(tmp, NULL, 16));

			u += 3;
		} else if (*u == '+') {
			filebuf->Add(" ", 1);

			u++;
		} else {
			filebuf->Add(u, 1);

			u++;
		}
	}

	filebuf->Add("", 1);
	filebuf->Shorten();

	ret = filebuf->data;

	filebuf->data = NULL;
	xdelete filebuf;

	return ret;
}

/*
encode special characters in a url
*/
char *url_encode(char *url, size_t len)
{
	char *ret, *u = url;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	for (; *u && u - url < len; u++) {
		if (*u == ' ')
			filebuf->Add("+", 1);
		else if ((*u >= 0 && *u <= 42) || (*u >= 58 && *u <= 64) || (*u >= 91 && *u <= 96) || *u >= 123)
			filebuf->Addf("%%%.2x", *u);
		else
			filebuf->Add(u, 1);
	}

	filebuf->Add("", 1);
	filebuf->Shorten();

	ret = filebuf->data;

	filebuf->data = NULL;
	xdelete filebuf;

	return ret;
}

/*
escape all <, >, and \'s in a string destined for the xml file
*/
char *string_to_xml(char *s)
{
	char *ret;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	for (; *s; s++) {
		if (*s == '>')
			filebuf->Add("&gt;", 4);
		else if (*s == '<')
			filebuf->Add("&lt;", 4);
		else if (*s == '&')
			filebuf->Add("&amp;", 5);
		else if (*s == '\"')
			filebuf->Add("&quot;", 6);
		else if (*s == '\'')
			filebuf->Add("&apos;", 6);
		else
			filebuf->Add(s, 1);
	}

	filebuf->Add("", 1);
	filebuf->Shorten();

	ret = filebuf->data;

	filebuf->data = NULL;
	xdelete filebuf;

	return ret;
}

/*
tokenize a string and return a NULL-terminated char **
*/
char **string_break(const char *s, const char d)
{
	int count = 0;
	char **ret = NULL;
	const char *start = s;

	for (; *s; s++) {
		if (*s == d) {
			if (s != start) {
				ret = (char**)xrealloc(ret, sizeof(char *) * (count + 2));
				ret[count] = xstrndup(start, s - start);

				count++;
			}

			while (*s == d)
				s++;
			start = s;
		}

		if (*s == '\0')
			break;
	}

	if ((*start && *start != d) || ret == NULL) {
		ret = (char**)xrealloc(ret, sizeof(char *) * (count + 2));
		ret[count] = xstrndup(start, s - start);

		count++;
	}

	ret[count] = NULL;

	return ret;
}

char *array_merge(char **a, int sep)
{
	char *ret;
	Filebuf *filebuf;

	filebuf = xnew Filebuf();

	for (; a && *a; a++) {
		filebuf->Addf("%s", *a);
		if (a[1] != NULL)
			filebuf->Addf("%c", sep);
	}

	filebuf->Add("", 1);
	filebuf->Shorten();

	ret = filebuf->data;

	filebuf->data = NULL;
	xdelete filebuf;

	return ret;
}

int array_find(char **a, char *s)
{
	if (!a) return FALSE;

	for (; *a; a++)
		if (!strcmp(*a, s))
			return TRUE;

	return FALSE;
}

int array_length(char **a)
{
	int i;

	if (!a) return 0;

	for (i = 0; a[i]; i++);

	return i;
}

void array_free(char **a)
{
	int i;

	if (!a) return;

	for (i = 0; a[i]; i++)
		xfree(a[i]);

	xfree(a);
}

char **array_dup(char **a)
{
	int len;
	char **ret;

	if (!a) return NULL;

	len = array_length(a);

	ret = (char**)xmalloc((sizeof(char **) * len) + 1);

	ret[len] = '\0';
	for (--len; len >= 0; len--)
		ret[len] = xstrdup(a[len]);

	return ret;
}

char **array_add(char **a, char *s) {
	int len = array_length(a);

	a = (char**)xrealloc(a, sizeof(char *) * (len + 2));
	a[len] = xstrdup(s);
	a[len + 1] = NULL;

	return a;
}

char **array_delete(char **a, char *s) {
	int i = 0, x;
	char **arg = a;

	if (!a) return NULL;
	
	while (*arg) {
		if (!strcmp(*arg, s)) {
			xfree(*arg);
			x = (array_length(arg + 1) + 1) * sizeof(char *);
			memmove(arg, arg + 1, x);

			a = (char**)xrealloc(a,  (i * sizeof(char *)) + x);
			arg = a + i;
		} else { 
			i++;
			arg++;
		}
	}

	return a;
}

int range_check(char *range, unsigned int number)
{
	int i, found = FALSE;
	unsigned int high;
	char **items, *ptr;

	items = string_break(range, ',');

	for (i = 0; items[i]; i++) {
		if (found == FALSE) {
			ptr = strchr(items[i], '-');
			if (ptr != NULL) {
				high = strtoul(&ptr[1], NULL, 10);

				if (number >= ((items[i][0] != '-') ? strtoul(items[i], NULL, 10) : 0) && number <= ((high == 0) ? ~0 : high))
					found = TRUE;
			} else if (number == strtoul(items[i], NULL, 10))
				found = TRUE;
		}

		xfree(items[i]);
	}

	xfree(items);

	return found;
}

void stack_free(struct STACK_T * stack)
{
	struct STACK_T *tmp;

	if (stack == NULL)
		return;

	while (stack->prev != NULL)
		stack = stack->prev;

	while (stack != NULL) {
		tmp = stack->next;

		xfree(stack->data);
		xfree(stack);

		stack = tmp;
	}

}

mode_t mode_parse(char *mode)
{
	mode_t ret = 0;

	if (strlen(mode) < 10)
		return 0;

	if (mode[1] == 'r')
		ret |= 0400;
	if (mode[2] == 'w')
		ret |= 0200;
	if (mode[3] == 'x')
		ret |= 0100;
	else if (mode[3] == 's')
		ret |= 04000 | 0100;

	if (mode[4] == 'r')
		ret |= 0040;
	if (mode[5] == 'w')
		ret |= 0020;
	if (mode[6] == 'x')
		ret |= 0010;
	else if (mode[6] == 'S')
		ret |= 02000 | 0010;

	if (mode[7] == 'r')
		ret |= 0004;
	if (mode[8] == 'w')
		ret |= 0002;
	if (mode[9] == 'x')
		ret |= 0001;
	else if (mode[9] == 't')
		ret |= 01000 | 0001;

	return ret;
}

char *mode_create(mode_t mode)
{
	char *ret;

	ret = (char*)xmalloc(11);

	ret[0] = '-';
	if (mode & 0400)
		ret[1] = 'r';
	else
		ret[1] = '-';
	if (mode & 0200)
		ret[2] = 'w';
	else
		ret[2] = '-';
	if (mode & 04000)
		ret[3] = 's';
	else if (mode & 0100)
		ret[3] = 'x';
	else
		ret[3] = '-';

	if (mode & 0040)
		ret[4] = 'r';
	else
		ret[4] = '-';
	if (mode & 0020)
		ret[5] = 'w';
	else
		ret[5] = '-';
	if (mode & 02000)
		ret[6] = 'S';
	else if (mode & 0010)
		ret[6] = 'x';
	else
		ret[6] = '-';

	if (mode & 0004)
		ret[7] = 'r';
	else
		ret[7] = '-';
	if (mode & 0002)
		ret[8] = 'w';
	else
		ret[8] = '-';
	if (mode & 01000)
		ret[9] = 't';
	else if (mode & 0001)
		ret[9] = 'x';
	else
		ret[9] = '-';

	ret[10] = '\0';

	return ret;
}

/*
parse an ftp PORT in the form "h1,h2,h3,h4,p1,p2" where h1 is the 8 most significant bits of the IP address
and p1 is the 8 most significant bits of the port, both in network byte order 
*/
struct sockaddr_in *ftpport_parse(char *ftpport)
{
	char **args;
	struct sockaddr_in *ret;

	args = string_break(ftpport, ',');

	if (array_length(args) != 6) {
		array_free(args);

		return NULL;
	}

	ret = (sockaddr_in*)xmalloc(sizeof(struct sockaddr_in));
	memset(ret, 0, sizeof(struct sockaddr_in));

	ret->sin_family = AF_INET;
	ret->sin_port = atoi(args[4]) | atoi(args[5]) << 8;
	ret->sin_addr.s_addr = atoi(args[0]) | atoi(args[1]) << 8 | atoi(args[2]) << 16 | atoi(args[3]) << 24;

	array_free(args);

	return ret;
}

char *ftpport_create(struct sockaddr_in *saddr)
{
	char buf[128];

	snprintf(buf, sizeof(buf), "%d,%d,%d,%d,%d,%d", saddr->sin_addr.s_addr & 0xff, (saddr->sin_addr.s_addr >> 8) & 0xff, (saddr->sin_addr.s_addr >> 16) & 0xff, (saddr->sin_addr.s_addr >> 24) & 0xff, saddr->sin_port & 0xff, (saddr->sin_port >> 8) & 0xff);

	putlog(MMLOG_DEBUG, "ftpport_create: %s", buf);

	return xstrdup(buf);
}

char *filesize(size_t size)
{
	double sz;
	char *s, buf[16];

	if (size >= 1024 * 1024 * 1024) {
		s = "GB";
		sz = (float) size / 1024 / 1024 / 1024;
	} else if (size >= 1024 * 1024) {
		s = "MB";
		sz = (float) size / 1024 / 1024;
	} else if (size >= 1024) {
		s = "KB";
		sz = (float) size / 1024;
	} else {
		s = "B";
		sz = (float) size;
	}

	snprintf(buf, sizeof(buf), "%.1f %s", sz, s);

	return xstrdup(buf);
}

time_t local_mktime(struct tm *t) {
	time_t ret;

	ret = mktime(t);

	return ret - gmtoffset;
}

time_t utc_time(time_t *t) {
	time_t curtime = time(NULL);

	curtime += gmtoffset;
	if (t != NULL) *t = curtime;

	return curtime;
}

char **cmdline_parse(const char *cmdline)
{
	int i = 0, bufpos = 0, new_ = TRUE;  /* new is a key word in C++ */ 
	int insquote = FALSE, indquote = FALSE, escape = FALSE;
	char **ret = NULL, buf[256];

	ret = (char**)xmalloc(sizeof(char *));
	*ret = NULL;

	for (; *cmdline; cmdline++) {
		if (*cmdline == '\\' && indquote == FALSE && insquote == FALSE && escape == FALSE) {
			escape = TRUE;
			continue;
		}

		if (escape == FALSE) {
			if (*cmdline == '\"' && insquote == FALSE) {
				indquote = !indquote;

				continue;
			} else if (*cmdline == '\'' && indquote == FALSE) {
				insquote = !indquote;

				continue;
			} else if (*cmdline == ' ' && indquote == FALSE && insquote == FALSE) {
				new_ = TRUE;

				continue;
			}
		}

		escape = FALSE;

		if (new_ == TRUE) {
			if (bufpos != 0) {
				ret = (char**)xrealloc(ret, (++i + 1) * sizeof(char *));

				buf[bufpos] = '\0';

				ret[i - 1] = xstrdup(buf);
				ret[i] = NULL;

				bufpos = 0;
			}


			new_ = FALSE;
		}

		if (bufpos != sizeof(buf) - 1)
			buf[bufpos++] = *cmdline;
	}

	if (bufpos != 0) {
		ret = (char**)xrealloc(ret, (++i + 1) * sizeof(char *));

		buf[bufpos] = '\0';

		ret[i - 1] = xstrdup(buf);
		ret[i] = NULL;
	}

	return ret;
}

char *string_append(char *s1, char *s2) {
	int slen1 = 0, slen2;

	if (s1 != NULL) slen1 = strlen(s1);
	slen2 = strlen(s2);

	s1 = (char*)xrealloc(s1, slen1 + slen2 + 1);
	memcpy(s1 + slen1, s2, slen2 + 1);

	return s1;
}

char *path_to_url(char *path) {
	int i;
	char buf[512], p[1024], *pptr = p, *ret = NULL;
	char **dirs;

	dirs = string_break(path, '/');

	if (strcmp(dirs[0], "")) {
		pptr += snprintf(pptr, sizeof(p) - (pptr - p), "/");		
		snprintf(buf, sizeof(buf), "<a href=\"/\">/</a>");
		ret = xstrdup(buf);
	}

	for (i = 0; dirs[i]; i++) {
		pptr += snprintf(pptr, sizeof(p) - (pptr - p), "%s/", dirs[i]);		
		snprintf(buf, sizeof(buf), "<a href=\"%s\">%s/</a>", p, dirs[i]);
		ret = string_append(ret, buf);

		xfree(dirs[i]);
	}

	xfree(dirs);

	return ret;
}

int in_absolutetimerange(const struct tm &subject, const struct tm_partial &from, const struct tm_partial &to) {
	if (from.m_year) {
		if (subject.tm_year < from.tm_year)
			return FALSE;
		else if (subject.tm_year != from.tm_year)
			goto checkto;
	}

	if (from.m_mon) {
		if (subject.tm_mon < from.tm_mon)
			return FALSE;
		else if (subject.tm_mon != from.tm_mon)
			goto checkto;
	}

	if (from.m_mday) {
		if (subject.tm_mday < from.tm_mday)
			return FALSE;
		else if (subject.tm_mday != from.tm_mday)
			goto checkto;
	}

	if (from.m_wday) {
		if (subject.tm_wday < from.tm_wday)
			return FALSE;
		else if (subject.tm_wday != from.tm_wday)
			goto checkto;
	}

	if (from.m_hour) {
		if (subject.tm_hour < from.tm_hour)
			return FALSE;
		else if (subject.tm_hour != from.tm_hour)
			goto checkto;
	}

	if (from.m_min) {
		if (subject.tm_min < from.tm_min)
			return FALSE;
		else if (subject.tm_min != from.tm_min)
			goto checkto;
	}

	checkto:

	if (to.m_year) {
		if (subject.tm_year > to.tm_year)
			return FALSE;
		else if (subject.tm_year != to.tm_year)
			goto out;
	}

	if (to.m_mon) {
		if (subject.tm_mon > to.tm_mon)
			return FALSE;
		else if (subject.tm_mon != to.tm_mon)
			goto out;
	}

	if (to.m_mday) {
		if (subject.tm_mday > to.tm_mday)
			return FALSE;
		else if (subject.tm_mday != to.tm_mday)
			goto out;
	}

	if (to.m_wday) {
		if (subject.tm_wday > to.tm_wday)
			return FALSE;
		else if (subject.tm_wday != to.tm_wday)
			goto out;
	}

	if (to.m_hour) {
		if (subject.tm_hour > to.tm_hour)
			return FALSE;
		else if (subject.tm_hour != to.tm_hour)
			goto out;
	}

	if (to.m_min) {
		if (subject.tm_min > to.tm_min)
			return FALSE;
		else if (subject.tm_min != from.tm_min)
			goto out;
	}

	out:

	return TRUE;
}

int in_alltimerange(const struct tm &subject, const struct tm_partial &from, const struct tm_partial &to) {
	if (from.m_year && from.tm_year > subject.tm_year) return FALSE;
	if (from.m_mon && from.tm_mon > subject.tm_mon) return FALSE;
	if (from.m_mday && from.tm_mday > subject.tm_mday) return FALSE;
	if (from.m_wday && from.tm_wday > subject.tm_wday) return FALSE;
	if (from.m_hour && from.tm_hour > subject.tm_hour) return FALSE;
	if (from.m_min && from.tm_min > subject.tm_min) return FALSE;

	if (to.m_year && to.tm_year < subject.tm_year) return FALSE;
	if (to.m_mon && to.tm_mon < subject.tm_mon) return FALSE;
	if (to.m_mday && to.tm_mday < subject.tm_mday) return FALSE;
	if (to.m_wday && to.tm_wday < subject.tm_wday) return FALSE;
	if (to.m_hour && to.tm_hour < subject.tm_hour) return FALSE;
	if (to.m_min && to.tm_min < subject.tm_min) return FALSE;

	return TRUE;
}

char *month_name(int mnum) {
	switch(mnum) {
		case 1:
			return "January";
		case 2:
			return "February";
		case 3:
			return "March";
		case 4:
			return "April";
		case 5:
			return "May";
		case 6:
			return "June";
		case 7:
			return "July";
		case 8:
			return "August";
		case 9:
			return "September";
		case 10:
			return "October";
		case 11:
			return "November";
		case 12:
			return "December";
		default:
			return "Invalid";
	}
}

char *weekday_name(int wday) {
	switch(wday) {
		case 1:
			return "Sunday";
		case 2:
			return "Monday";
		case 3:
			return "Tuesday";
		case 4:
			return "Wednesday";
		case 5:
			return "Thursday";
		case 6:
			return "Friday";
		case 7:
			return "Saturday";
		default:
			return "Invalid";
	}
}

URL *url_parse(const char *url) {
	int x, customport = FALSE;
	char username[128], password[128];
	char *ptr, *ptr2;
	URL *ret;

	ret = (URL*)xmalloc(sizeof(URL));
	memset(ret, 0, sizeof(URL));

	ptr = strchr(url, ':');
	if (ptr == NULL) goto error;
	
	ret->proto = xstrndup(url, ptr - url);
	string_tolower(ret->proto);

	if (ptr[1] == '\0' || ptr[2] == '\0') goto error;
	ptr += 3;

	for (ptr2 = ptr; *ptr2 && *ptr2 != '/' && *ptr2 != '@'; ptr2++);
	if (ptr2 != NULL && *ptr2 == '@') {
		x = sscanf(ptr, "%128[^:]:%128[^@]", username, password);
		if (x != 2) goto error;

		ret->username = xstrdup(username);
		ret->password = xstrdup(password);

		ptr = ++ptr2;
	} else {
		ret->username = NULL;
		ret->password = NULL;
	}

	ptr2 = strchr(ptr, ':');
	if (ptr2 != NULL) {
		ret->host = xstrndup(ptr, ptr2 - ptr);

		ret->port = strtol(ptr2 + 1, &ptr2, 10);
		customport = TRUE;

		if (*ptr2 == '\0')
			ret->file = xstrdup("/");
		else if (*ptr2 == '/')
			ret->file = url_path_fix(ptr2);
		else
			goto error;
	} else {
		ptr2 = strchr(ptr, '/');
		if (ptr2 == NULL) {
			ret->host = xstrdup(ptr);
			ret->file = xstrdup("/");
		} else {
			ret->host = xstrndup(ptr, ptr2 - ptr);
			ret->file = url_path_fix(ptr2);
		}
	}

	url_strip(ret->host);
	string_tolower(ret->host);

	if (customport == FALSE) {
		if (!strcasecmp(ret->proto, "http"))
			ret->port = HTTP_PORT;
		else if (!strcasecmp(ret->proto, "ftp"))
			ret->port = FTP_PORT;
	}

	return ret;	
	
	error:
	url_free(ret);
	return NULL;	
}
	
char *url_create(URL *url) {
	char buf[1024];

	snprintf(buf, sizeof(buf), "%s://%s:%d%s", url->proto, url->host, url->port, url->file);

	return xstrdup(buf);
}

void url_free(URL *url) {
	FREE_AND_NULL(url->username);
	FREE_AND_NULL(url->password);
	FREE_AND_NULL(url->proto);
	FREE_AND_NULL(url->host);
	FREE_AND_NULL(url->file);

	xfree(url);
}

int isempty(char *s, size_t len) {
	for (; *s && len; s++, len--)
		if (*s != ' ' && isprint(*s))
			return FALSE;

	return TRUE;
}

int urlcmp(URL *u1, URL *u2) {
	if (u1->username != NULL && u2->username != NULL && strcmp(u1->username, u2->username))
		return FALSE;
	if (u1->password != NULL && u2->password != NULL && strcmp(u1->password, u2->password))
		return FALSE;
	if (u1->port != u2->port)
		return FALSE;
	if (strcasecmp(u1->proto, u2->proto))
		return FALSE;
	if (strcasecmp(u1->host, u2->host))
		return FALSE;
	if (strcmp(u1->file, u2->file))
		return FALSE;

	return TRUE;
}

char *url_path_fix(char *url) {
	char *ptr, *ptr2, buf[2048], *ret;

	/* this will fix the path portion of a URL so that we don't end up caching the same
	   file more than once if it's requested using various combinations of  
           ".."'s, "."'s , multiple "/"'s, or anchors in the URL */

	s_strncpy(buf, url, sizeof(buf));

	/* don't process after the '?' which denotes a CGI argument */
	ptr = strchr(buf, '?');
	if (ptr != NULL) *ptr++ = '\0';

	/* get rid of anchors */
	ptr2 = strchr(buf, '#');
	if (ptr2 != NULL) *ptr2 = '\0';	

	ret = resolve_path(buf);
	if (ptr != NULL) {
		ret = string_append(ret, "?");
		ret = string_append(ret, ptr);
	}

	url_strip(ret);

	return ret;
}

char *resolve_path(char *path) {
	int i, x;
	char **dirs, *ret, *ptr, *ptr2;

	dirs = string_break(path, '/');

	for (i = 0; dirs[i]; i++) {
		if (!strcmp(dirs[i], "..")) {
			xfree(dirs[i]);
			if (i == 0) {
				for (x = 1; dirs[x]; x++)
					dirs[x - 1] = dirs[x];
				dirs[x - 1] = dirs[x];

				i--;
			} else {
				xfree(dirs[i - 1]);
				for (x = i + 1; dirs[x]; x++)
					dirs[x - 2] = dirs[x];
				dirs[x - 2] = dirs[x];

				i -= 2;
			}
		} else if (!strcmp(dirs[i], ".")) {
			xfree(dirs[i]);
			for (x = i + 1; dirs[x]; x++)
				dirs[x - 1] = dirs[x];
			dirs[x - 1] = dirs[x];

			i--;
		}
	}

	ptr = array_merge(dirs, '/');

	ptr2 = strrchr(path, '/');
	ret = (char*)xmalloc(strlen(ptr) + ((ptr2[1] == '\0' && ptr2 != path) ? 4 : 3));
	sprintf(ret, "/%s%s", ptr, (ptr2[1] == '\0' && ptr2 != path) ? "/" : "");

	xfree(ptr);

	array_free(dirs);

	return ret;
}	

void url_strip(char *str) {
	/* strip invalid characters from a url */
	for (; *str; str++) {
		if (*str <= 32) {
			memmove(str, str + 1, strlen(str + 1) + 1);
			str--;
		}
	}
}

int protocol_port(char *proto) {
	if (!strcasecmp(proto, "http"))
		return HTTP_PORT;
	else if (!strcasecmp(proto, "ftp"))
		return FTP_PORT;

	return 0;
}

char *timeval_to_string(struct timeval *tv) {
	int hours, minutes, seconds, fracsecond;
	char buf[16];

	hours = tv->tv_sec / (3600);
	tv->tv_sec -= hours * (3600);

	minutes = tv->tv_sec / 60;
	tv->tv_sec -= minutes * 60;

	seconds = tv->tv_sec;

	fracsecond = (int)(((float)tv->tv_usec / 1000000.0) * 100.0);

	snprintf(buf, sizeof(buf), "%.2d:%.2d:%.2d.%.2d",hours, minutes, seconds, fracsecond);

	return xstrdup(buf); 
}
