/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include "proto.h"

/*
Parse directory listing in this format:
-------------------------
total 9 <- only on some servers
drwx------    3 jasonmc  jasonmc      4096 Feb  4 04:34 Desktop
drwx--S---    2 jasonmc  jasonmc      4096 Apr 28  2001 Documents
drwxr-xr-x    2 jasonmc  jasonmc     12288 Feb  4 15:07 Incomplete
drwx------    2 jasonmc  jasonmc      4096 Feb  5 09:55 Mail
-rw-------    1 jasonmc  jasonmc       378 Jan 19  2002 Makefile
drwxr-xr-x    3 jasonmc  jasonmc      4096 Aug  4  2002 Projects
drwxr-xr-x    2 jasonmc  jasonmc      4096 Jan 20 18:17 Shared
-rw-r--r--    1 jasonmc  jasonmc      2101 Feb  5 08:13 TODO
-rw-r--r--    1 jasonmc  jasonmc      2099 Feb  5 08:13 TODO~
lrwxrwxrwx    1 root     root            7 Feb 11 18:11 ls -> /bin/ls
-rw-r--r--    1 999      999             0 Feb 11 18:28 a
-------------------------
Alternative format:
-------------------------
total 26842
drwxr-sr-x   2 1            512 Jun 29  2001 .forward
-rw-r--r--   1 11             0 Jun 29  2001 .hushlogin
-rw-r--r--   1 100           59 Jun 29  2001 .kermrc
-rw-r--r--   1 100            0 Jun 29  2001 .notar
d--x--x--x   3 1            512 Jun 29  2001 .tmppriv
drwxr-sr-x  13 1            512 Jun 29  2001 .vol
drwxr-sr-x   5 1            512 Jun 29  2001 admin
lrwxrwxrwx   1 0              1 Jun 29  2001 archive -> .

XXX: todo: support for msdos directory lists.
-------------------------
*/

struct dir_t *dir_list_parse(Filebuf * filebuf)
{
	int i, columns = 0;
	struct dir_t *dir = NULL, *start = NULL;
	char buf[256], *ptr;
	char **args;
	struct tm t, lt;
	time_t ltt;
	struct STACK_T *lines, *lstart;

	ltt = time(NULL);
	localtime_r(&ltt, &lt);

	lstart = lines = filebuf->to_stack();

	for (; lines; lines = lines->next) {
		if (!strncasecmp(lines->data, "total", 5))
			continue;

		args = string_break(lines->data, ' ');

		if (columns == 0)
			for (; args[columns] && strcmp(args[columns], "->"); columns++);

		memset(&t, 0, sizeof(struct tm));

		if (columns >= 9) {
			snprintf(buf, sizeof(buf), "%s %s %s", args[5], args[6], args[7]);

			if (strchr(args[7], ':') != NULL) {
				/* in MM DD HH:MM format */
				strptime(buf, "%b %d %H:%M", &t);
				t.tm_year = lt.tm_year;
			} else
				/* in MM DD YYYY format */
				strptime(buf, "%b %d %Y", &t);

			ptr = buf;
			for (i = 8; args[i]; i++)
				ptr += snprintf(ptr, sizeof(buf) - (ptr - buf), "%s%s", args[i], (args[i + 1] != '\0') ? " " : "");

			if (args[0][0] == 'l') {
				/* remove the -> in a symlink filename */
				ptr = strstr(buf, "->");
				if (ptr != NULL && ptr != buf)
					*(ptr - 1) = '\0';
			}

			STRIP_CRLF(buf);

			dir = dir_list_add(dir, (args[0][0] == 'd') ? TRUE : FALSE, mode_parse(args[0]), args[2], args[3], strtoul(args[4], NULL, 10), t, buf);
		} else if (columns == 8) {
			snprintf(buf, sizeof(buf), "%s %s %s", args[4], args[5], args[6]);

			if (strchr(args[6], ':') != NULL) {
				/* in MM DD HH:MM format */
				strptime(buf, "%b %d %H:%M", &t);
				t.tm_year = lt.tm_year;
			} else
				/* in MM DD YYYY format */
				strptime(buf, "%b %d %Y", &t);

			ptr = buf;
			for (i = 7; args[i]; i++)
				ptr += snprintf(ptr, sizeof(buf) - (ptr - buf), "%s%s", args[i], (args[i + 1] != '\0') ? " " : "");

			if (args[0][0] == 'l') {
				ptr = strstr(buf, "->");
				if (ptr != NULL && ptr != buf)
					*(ptr - 1) = '\0';
			}

			STRIP_CRLF(buf);

			dir = dir_list_add(dir, (args[0][0] == 'd') ? TRUE : FALSE, mode_parse(args[0]), args[2], "unknown", strtoul(args[3], NULL, 10), t, buf);
		} else {
			putlog(MMLOG_DEBUG, "unparsable ftp dir entry: %s", lines->data);

			continue;
		}

		if (start == NULL)
			start = dir;

		array_free(args);
	}

	stack_free(lstart);

	return start;
}

struct dir_t *dir_list_add(struct dir_t *dir, int isdir, mode_t mode, char *user, char *group, size_t size, struct tm time, char *name)
{
	if (dir == NULL) {
		dir = (dir_t*)xmalloc(sizeof(struct dir_t));
		dir->prev = NULL;
	} else {
		while (dir->next != NULL)
			dir = dir->next;
		dir->next = (dir_t*)xmalloc(sizeof(struct dir_t));
		dir->next->prev = dir;
		dir = dir->next;
	}

	dir->next = NULL;
	dir->isdir = isdir;
	dir->mode = mode;
	dir->user = xstrdup(user);
	dir->group = xstrdup(group);
	dir->size = size;
	dir->time = time;
	dir->name = xstrdup(name);
	dir->t = mktime(&time);

	if (dir->isdir == TRUE && dir->name[strlen(dir->name) - 1] != '/')
		dir->name = string_append(dir->name, "/");

	return dir;
}

void dir_list_free(struct dir_t *dir)
{
	struct dir_t *tmp;

	if (dir == NULL)
		return;

	while (dir->prev != NULL)
		dir = dir->prev;

	while (dir != NULL) {
		tmp = dir->next;

		xfree(dir->user);
		xfree(dir->group);
		xfree(dir->name);

		xfree(dir);

		dir = tmp;
	}
}

struct dir_t *dir_list_filter(struct dir_t *dir, const char *pattern) {
	struct dir_t *head = dir, *tmp;
	regex_t *pe;

	pe = reg_compile(pattern, REGFLAGS);
	if (pe == NULL) return dir;

	while (dir != NULL) {
		tmp = dir->next;

		if (reg_exec(pe, dir->name)) {
			if (dir->next != NULL)
				dir->next->prev = dir->prev;
			if (dir->prev != NULL)
				dir->prev->next = dir->next;
			else
				head = tmp;

			dir->next = dir->prev = NULL;

			dir_list_free(dir);
		}

		dir = tmp;
	}

	reg_free(pe);

	return head;
}

struct dir_t *dir_list_sort(struct dir_t *dir, int direction, int field)
{
	int done, swap = FALSE;
	struct dir_t *head = dir;

	do {
		done = TRUE;

		for (dir = head; dir && dir->next; dir = dir->next) {
			switch (field) {
			case SORT_NAME:
				if (direction == SORT_ASCENDING) {
					if (strcasecmp(dir->name, dir->next->name) > 0)
						swap = TRUE;
				} else {
					if (strcasecmp(dir->name, dir->next->name) < 0)
						swap = TRUE;
				}
				break;
			case SORT_SIZE:
				if (direction == SORT_ASCENDING) {
					if (dir->isdir == FALSE && dir->next->isdir == TRUE)
						swap = TRUE;
					else if (dir->isdir == dir->next->isdir && dir->size > dir->next->size)
						swap = TRUE;
				} else {
					if (dir->isdir == TRUE && dir->next->isdir == FALSE)
						swap = TRUE;
					else if (dir->isdir == dir->next->isdir && dir->size < dir->next->size)
						swap = TRUE;
				}
				break;
			case SORT_DATE:
				if (direction == SORT_ASCENDING) {
					if (dir->t > dir->next->t)
						swap = TRUE;
				} else {
					if (dir->t < dir->next->t)
						swap = TRUE;
				}
				break;
			}

			if (swap == TRUE) {
				done = swap = FALSE;

				SHIFTNODE(struct dir_t *, head, dir, DOWN);
			}
		}
	} while (done == FALSE);

	return head;
}
