/*
    MiddleMan filtering proxy server
    Copyright (C) 2002-2004  Jason McLaughlin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include "proto.h"

CookieSection::CookieSection():
	Section("cookies", RWLOCK), 
	enabled (field_vec[0].int_value),
	policy  (field_vec[1].int_value)
{
}


void CookieSection::update()
{
	allow_list.clear();
	deny_list.clear();

	ItemList::iterator item;
	for (item = sub_vec[0].item_list.begin(); item != sub_vec[0].item_list.end(); item++)
		allow_list.push_back(Cookie(*item));

	for (item = sub_vec[1].item_list.begin(); item != sub_vec[1].item_list.end(); item++)
		deny_list.push_back(Cookie(*item));

}


Cookie::Cookie(const Item & item):
   enabled(item.field_vec[0].int_value), 
   comment(item.field_vec[1].string_value), 
   profiles(item.field_vec[2].string_list_value), 
   limityears(item.field_vec[3].active),
   year1(item.field_vec[3].int_value),
   year2(item.field_vec[3].int_value2),
   limitmonths(item.field_vec[4].active), 
   month1(item.field_vec[4].int_value), 
   month2(item.field_vec[4].int_value2), 
   limitdays(item.field_vec[5].active), 
   day1(item.field_vec[5].int_value), 
   day2(item.field_vec[5].int_value2), 
   limitweekdays(item.field_vec[6].active), 
   weekday1(item.field_vec[6].int_value), 
   weekday2(item.field_vec[6].int_value2), 
   limithours(item.field_vec[7].active), 
   hour1(item.field_vec[7].int_value), 
   hour2(item.field_vec[7].int_value2), 
   limitminutes(item.field_vec[8].active), 
   minute1(item.field_vec[8].int_value), 
   minute2(item.field_vec[8].int_value2), 
   domain(item.field_vec[9].string_value), 
   path(item.field_vec[10].string_value), 
   direction(item.field_vec[11].int_value), 
   matchmode(item.field_vec[12].int_value)
{
	dp = (domain != "") ? reg_compile(domain.c_str(), REGFLAGS) : NULL;
	pp = (path != "") ? reg_compile(path.c_str(), REGFLAGS) : NULL;

	if (limityears) {
		from.m_year = to.m_year = TRUE;
		from.tm_year = year1 - 1900;
		to.tm_year = year2 - 1900;
	} else from.m_year = to.m_year = FALSE;

	if (limitmonths) {
		from.m_mon = to.m_mon = TRUE;
		from.tm_mon = month1;
		to.tm_mon = month2;
	} else from.m_mon = to.m_mon = FALSE;

	if (limitdays) {
		from.m_mday = to.m_mday = TRUE;
		from.tm_mday = day1;
		to.tm_mday = day2;
	} else from.m_mday = to.m_mday = FALSE;

	if (limitweekdays) {
		from.m_wday = to.m_wday = TRUE;
		from.tm_wday = weekday1;
		to.tm_wday = weekday2;
	} else from.m_wday = to.m_wday = FALSE;

	if (limithours) {
		from.m_hour = to.m_hour = TRUE;
		from.tm_hour = hour1;
		to.tm_hour = hour2;
	} else from.m_hour = to.m_hour = FALSE;

	if (limitminutes) {
		from.m_min = to.m_min = TRUE;
		from.tm_min = minute1;
		to.tm_min = minute2;
	} else from.m_min = to.m_min = FALSE;

}

Cookie::Cookie(const Cookie & cookie):
   enabled (cookie.enabled),
   comment (cookie.comment),
   profiles (cookie.profiles),
   limityears (cookie.limityears),
   year1 (cookie.year1),
   year2 (cookie.year2),
   limitmonths (cookie.limitmonths),
   month1 (cookie.month1),
   month2 (cookie.month2),
   limitdays (cookie.limitdays),
   day1 (cookie.day1),
   day2 (cookie.day2),
   limitweekdays (cookie.limitweekdays),
   weekday1 (cookie.weekday1),
   weekday2 (cookie.weekday2),
   limithours (cookie.limithours),
   hour1 (cookie.hour1),
   hour2 (cookie.hour2),
   limitminutes (cookie.limitminutes),
   minute1 (cookie.minute1),
   minute2 (cookie.minute2),
   domain (cookie.domain),
   path (cookie.path),
   direction (cookie.direction),
   matchmode (cookie.matchmode),
   from(cookie.from),
   to(cookie.to)
{
	dp = (domain != "") ? reg_compile(domain.c_str(), REGFLAGS) : NULL;
	pp = (path != "") ? reg_compile(path.c_str(), REGFLAGS) : NULL;
}

Cookie::~Cookie()
{
	if (dp != NULL)
		reg_free(dp);
	if (pp != NULL)
		reg_free(pp);
}

/*
check if a cookie is allowed to pass in a given direction for a host
*/
bool CookieSection::check(int direction, CONNECTION * connection, CookieHeader &cookie_header)
{
	int ret, action = FALSE, result = TRUE, i;
	CookieList *cookie_list;

	if (connection->bypass & FEATURE_COOKIES)
		return TRUE;

	read_lock();

	if (this->enabled == FALSE) {
		unlock();

		return TRUE;
	}

	for (i = 0; i < 2; i++) {
		if (i == 0) {
			if (this->policy == POLICY_ALLOW) {
				cookie_list = &this->deny_list;
				action = FALSE;
				result = TRUE;
			} else {
				cookie_list = &this->allow_list;
				action = TRUE;
				result = FALSE;
			}
		} else if (action == result) {
			if (this->policy == POLICY_ALLOW) {
				cookie_list = &this->allow_list;
				action = TRUE;
			} else {
				cookie_list = &this->deny_list;
				action = FALSE;
			}
		} else
			break;

		CookieList::const_iterator current;
		for (current = cookie_list->begin(); current != cookie_list->end(); current++) {
			if (current->enabled == FALSE)
				continue;

			if (current->direction != COOKIE_BOTH && current->direction != direction)
				continue;

			if (!profile_find(connection->profiles, current->profiles))
				continue;

			if (direction == COOKIE_IN) {
				if (current->matchmode == TIMEMATCH_ABSOLUTE) {
					if (!in_absolutetimerange(cookie_header.expires, current->from, current->to))
						continue;
				} else {
					if (!in_alltimerange(cookie_header.expires, current->from, current->to))
						continue;
				}

				if (current->dp != NULL) {
					ret = reg_exec(current->dp, cookie_header.domain.c_str());
					if (ret) continue;
				}

				if (current->pp != NULL) {
					ret = reg_exec(current->pp, cookie_header.path.c_str());
					if (ret) continue;
				}
			}

			result = action;
			break;
		}
	}

	unlock();

	return result;
}

CookieHeader::CookieHeader(string cookie)
{
	int i;
	char *start, *equal, **ce = string_break(cookie.c_str(), ';');

	version = 0;
	has_expires = 0;

	for (i = 0; ce[i]; i++) {
		for (start = ce[i]; *start == ' ' || *start == '$'; start++);

		equal = strchr(start, '=');
		if (equal) (*equal++) = '\0';

		if (equal && *equal) {
			if (!strcasecmp(start, "version"))
				version = atoi(equal);
			else if (!strcasecmp(start, "domain"))
				domain = equal;
			else if (!strcasecmp(start, "path"))
				path = equal;
			else if (!strcasecmp(start, "expires")) {
				if (strptime(equal, COOKIETIMEFORMAT, &expires))
					has_expires = TRUE;
				else if (strptime(equal, COOKIETIMEFORMAT2, &expires)) {
					expires.tm_year += 1900;
					has_expires = TRUE;
				} else {
					putlog(MMLOG_WARN, "failed to parse cookie expiry time: %s", equal); 
					/* might as well pass it along */
					name_pairs.push_back(StringPair(start, equal));
				}
			} else
				/* this will handle the Cookie header returned by the browser (multiple key=value pairs)
				   as well as any unsupported attributes. */
				name_pairs.push_back(StringPair(start, equal));
		}

		xfree(ce[i]);
	}
	xfree(ce);

}

string CookieHeader::Build()
{
	string ret;
	char buf[64];
	StringPairList::const_iterator cookie_item;

	if (name_pairs.empty()) return string("");
	
	for (cookie_item = name_pairs.begin(); cookie_item != name_pairs.end(); cookie_item++) {
		if (cookie_item != name_pairs.begin()) ret.append("; ");

		ret.append(cookie_item->first);
		ret.append("=");
		ret.append(cookie_item->second);
	}

	if (domain != "") {
		ret.append("; domain=");
		ret.append(domain);
	}

	if (path != "") {
		ret.append("; path=");
		ret.append(path);
	}

	if (has_expires) {
		strftime(buf, sizeof(buf), COOKIETIMEFORMAT, &expires);
		ret.append("; expires=");
		ret.append(buf);
	}

	if (version != 0) {
		ret.append("; version=");
		snprintf(buf, sizeof(buf), "%d", version);
		ret.append(buf);
	}

	return ret;
}

void interface_cookies_show(CONNECTION *connection) {
	char *ptr1, *ptr2;
	Filebuf *filebuf = xnew Filebuf();
	HttpHeaderList::const_iterator header_list;

	filebuf->Addf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
        filebuf->Addf("<html><head><title>Cookies for %s%s</title></head>", connection->header->host, connection->header->file);
        filebuf->Addf("<body>\n");

	interface_stylesheet(connection, filebuf);

	filebuf->Addf("<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" width=\"100%\">\n");

	for (header_list = connection->header->header.begin(); header_list != connection->header->header.end(); header_list++) {
		if (header_list->type == "Cookie" || header_list->type == "Cookie2") {
			CookieHeader ch(header_list->value);
			filebuf->Addf("<tr><td align=\"center\"><table class=\"dialog\">\n");

			filebuf->Addf("<tr><td colspan=\"2\" class=\"listhead\" align=\"center\">Sent cookie</td></tr>\n");


			StringPairList::iterator cp;
			for (cp = ch.name_pairs.begin(); cp != ch.name_pairs.end(); cp++) {
				ptr1 = string_to_html(cp->first.c_str(), HTML_NEWLINES);
				ptr2 = string_to_html(cp->second.c_str(), HTML_NEWLINES);

				filebuf->Addf("<tr><td>%s</td><td>%s</td></tr>\n", ptr1, ptr2);

				xfree(ptr1);
				xfree(ptr2);
			}

			filebuf->Addf("</table></td></tr> <tr><td><br></td></tr>\n");
		}
	}

	for (header_list = connection->rheader->header.begin(); header_list != connection->rheader->header.end(); header_list++) {
		if (header_list->type == "Set-Cookie" || header_list->type == "Set-Cookie2") {
			CookieHeader ch(header_list->value);
			filebuf->Addf("<tr><td align=\"center\"><table class=\"dialog\">\n");

			filebuf->Addf("<tr><td colspan=\"2\" class=\"listhead\" align=\"center\">Received cookie</td></tr>\n");

			if (ch.has_expires) {
				char buf[128];
				strftime(buf, sizeof(buf), COOKIETIMEFORMAT, &ch.expires);
				filebuf->Addf("<tr><td>Expires</td><td>%s</td></tr>\n", buf);
			}

			if (ch.domain != "") {
				ptr1 = string_to_html(ch.domain.c_str(), HTML_NEWLINES);

				filebuf->Addf("<tr><td>Domain</td><td>%s</td></tr>\n", ptr1);

				xfree(ptr1);
			}

			if (ch.path != "") {
				ptr1 = string_to_html(ch.path.c_str(), HTML_NEWLINES);

				filebuf->Addf("<tr><td>Path</td><td>%s</td></tr>\n", ptr1);

				xfree(ptr1);
			}

			StringPairList::iterator cp;
			for (cp = ch.name_pairs.begin(); cp != ch.name_pairs.end(); cp++) {
				ptr1 = string_to_html(cp->first.c_str(), HTML_NEWLINES);
				ptr2 = string_to_html(cp->second.c_str(), HTML_NEWLINES);

				filebuf->Addf("<tr><td>%s</td><td>%s</td></tr>\n", ptr1, ptr2);

				xfree(ptr1);
				xfree(ptr2);
			}

			filebuf->Addf("</table></td></tr> <tr><td><br></td></tr>\n");
		}
	}

	filebuf->Addf("</table></body></html>\n");

	interface_send_response(connection, filebuf);

	xdelete filebuf;
}
