;;; mew-summary3.el --- Summary mode for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Subfunctions
;;;

(defmacro mew-summary-prepare-draft (&rest body)
  "Common procedure to prepare a draft."
  `(progn
     (unwind-protect
	 (let ((inhibit-quit t))
	   ,@body
	   ;; XEmacs does not draw attachments unless sit for 0...
	   (mew-redraw)
	   ;; XEmacs does not draw toolbar, so...
	   (when (and mew-xemacs-p mew-icon-p
		      (specifier-instance default-toolbar-visible-p))
	     (set-specifier default-toolbar-visible-p nil)
	     (set-specifier default-toolbar-visible-p t)))
       (save-buffer)) ;; to make sure not to use this draft again
     (mew-set-file-modes (buffer-file-name))
     (mew-touch-folder mew-draft-folder)
     (message "Draft is prepared")))

(defsubst mew-summary-prepare-three-windows ()
  "Prepare three windows: Summary mode, Message mode, and Draft mode"
  (unless mew-use-other-frame-for-draft
    (if (get-buffer (mew-buffer-message))
	(delete-windows-on (mew-buffer-message)))
    (cond
     ((< (window-height) 25)
      (delete-other-windows))
     (mew-use-full-window
      (mew-delete-other-window)))
    (let ((split-window-keep-point t))
      (split-window-vertically))))

(defun mew-draft-multi-copy (draft files)
  (let* ((attach (mew-draft-to-attach draft))
	 (attachdir (mew-expand-folder attach)))
    (mew-check-directory attachdir)
    (while files
      (if mew-use-symbolic-link-for-forwarding
	  (mew-symbolic-link (car files) (mew-folder-new-message attach))
	(copy-file (car files) (mew-folder-new-message attach)))
      (setq files (cdr files)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sending
;;;

(defun mew-summary-postp ()
  (cond
   ((mew-summary-p)
    (mew-folder-nntpp (mew-summary-folder-name 'ext)))
   ((mew-virtual-p)
    (mew-folder-nntpp (mew-virtual-folder-name)))
   (t
    (mew-folder-nntpp (mew-proto mew-case-output)))))

(defun mew-summary-send (&optional to cc subject newsgroups)
  "Write a message. A new draft is prepared in Draft mode."
  (interactive)
  (let ((draft (mew-folder-new-message mew-draft-folder)))
    (if (mew-summary-postp)
	(if (null newsgroups) (setq newsgroups ""))
      (when (and mew-ask-to (null to))
        (setq to (mew-input-address (concat mew-to: " "))))
      (when (and mew-ask-cc (null cc))
        (setq cc (mew-input-address (concat mew-cc: " ")))))
    (mew-current-set-window-config)
    (mew-window-configure 'draft)
    (mew-summary-prepare-draft
     (mew-draft-find-and-switch draft)
     (mew-delete-directory-recursively (mew-attachdir draft))
     (mew-draft-header subject nil to cc newsgroups)
     (mew-draft-mode)
     (run-hooks 'mew-draft-mode-newdraft-hook))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Replying
;;;

(defun mew-subject-simplify (str &optional action-list no-replace)
  "A function to simplify a value of Subject: according to
'mew-subject-simplify-replace-alist'."
  (let ((case-fold-search t)
       regexp replace)
    (unless action-list (setq action-list mew-subject-simplify-replace-alist))
    (while action-list
      (setq regexp  (car (car action-list))
	    replace (if no-replace nil (cdr (car action-list)))
	    action-list (cdr action-list))
      (if (string-match regexp str)
	  (setq str (replace-match (if replace (eval replace) "") nil t str))))
    str))

(defun mew-to-cc-newsgroups (replysender)
  (let (alist ent key tcn-flds tcn flds to cc newsgroups fromme func)
    (cond 
     (replysender
      (setq alist mew-reply-sender-alist)
      (setq func 'mew-header-parse-address-list2))
     ((mew-is-my-address mew-regex-my-address-list
			 (mew-header-parse-address mew-from:))
      (setq fromme t)
      (setq alist mew-reply-fromme-alist)
      ;; If from me, let's leave the "anonymous:;" keyword.
      (setq func 'mew-header-parse-address-list))
     (t ;; reply all
      (setq alist mew-reply-all-alist)
      (setq func 'mew-header-parse-address-list2)))
    (catch 'loop
      (while alist
	(setq ent (car alist))
	(setq alist (cdr alist))
	(setq key (car ent))
	(setq ent (cdr ent))
	(when (or (eq key t)
		  (and (stringp key) (mew-header-get-value key))
		  (and (listp key)
		       (string= (mew-header-get-value (nth 0 key))
				(nth 1 key))))
	  (while ent
	    (setq tcn-flds (car ent))
	    (setq ent (cdr ent))
	    (setq tcn (car tcn-flds))
	    (setq flds (cdr tcn-flds))
	    (cond
	     ((mew-case-equal tcn mew-to:)
	      (setq to (funcall func flds)))
	     ((mew-case-equal tcn mew-cc:)
	      (setq cc (funcall func flds)))
	     ((mew-case-equal tcn mew-newsgroups:)
	      (setq newsgroups (funcall func flds)))))
	  (throw 'loop nil))))
    (list to cc newsgroups fromme)))

(defun mew-in-reply-to-references ()
  (let ((old-message-id  (mew-header-get-value mew-message-id:))
	(old-in-reply-to (mew-header-get-value mew-in-reply-to:))
	(old-references  (mew-header-get-value mew-references:))
	(regex "<[^>]+>")
	(start 0) tmp-ref skip in-reply-to references)
    (if (and old-message-id (string-match regex old-message-id))
	(setq old-message-id (match-string 0 old-message-id))
      (setq old-message-id nil))
    ;; Assuming that In-Reply-To: contains one ID.
    (if (and old-in-reply-to (string-match regex old-in-reply-to))
	(setq old-in-reply-to (match-string 0 old-in-reply-to))
      (setq old-in-reply-to nil))
    (if (null old-message-id)
	() ;; we do not care even if old-references exist.
      (setq in-reply-to old-message-id)
      (if (null old-references)
	  (setq tmp-ref (if old-in-reply-to 
			    (list old-in-reply-to old-message-id)
			  (list old-message-id)))
	(while (string-match "<[^>]+>" old-references start)
	  (setq start (match-end 0))
	  (setq tmp-ref (cons (match-string 0 old-references) tmp-ref)))
	;; described in old drums but not in RFC2822
	(mew-addq tmp-ref old-in-reply-to)
	(setq tmp-ref (nreverse (cons old-message-id tmp-ref))))
      (if (integerp mew-references-max-count)
	  (setq skip (- (length tmp-ref) mew-references-max-count)))
      (if (and (numberp skip) (> skip 0))
	  (setq tmp-ref (nthcdr skip tmp-ref)))
      (setq references (mew-join "\n\t" tmp-ref)))
    (list in-reply-to references)))

(defun mew-summary-reply (&optional replysender)
  "Reply to this message. A new draft is prepared in Draft mode. 
Values of To:, Cc:, and Newsgroups: are prepared accoring to
three alists.

(1) If called with '\\[universal-argument]', replying to the
    sender/poster only. In this case, 'mew-reply-sender-alist' is used.

(2) If this message is sent by ME, you probably intend to reply with
    the original header. In this case, 'mew-reply-fromme-alist' is used.

(3) Otherwise, replying to all people listed. In this case, 
    'mew-reply-all-alist' is used.

The default value of 'mew-reply-sender-alist' is as follows:

	'((\"Reply-To:\"
	   (\"To:\" \"Reply-To:\" \"From:\"))
	  (t
	   (\"To:\" \"From:\")))

This is read as follows:

	(1.1) If Reply-To: exists, copy the values of Reply-To:
              and From: to new To:.
	(1.2) Otherwise, copy the value of From: to To:.

If you would like to reply only to the address on Reply-To: (if any),
set 'mew-reply-sender-alist' to:

	'((\"Reply-To:\"
	   (\"To:\" \"Reply-To:\"))
	  (t
	   (\"To:\" \"From:\")))

The default value of 'mew-reply-fromme-alist' is as follows:

	'((t
	   (\"To:\" \"To:\")
	   (\"Cc:\" \"Cc:\")
	   (\"Newsgroups:\" \"Newsgroups:\"))))

This is read as follows:

	(2.1) Copying the value of To: to new To: and 
              copying the value of Cc: to new Cc: and
              copying the value of Newsgroups: to new Newsgroups:.

The default value of 'mew-reply-all-alist' is as follows:

	'(((\"Followup-To:\" \"poster\")
	   (\"To:\" \"From:\"))
	  (\"Followup-To:\"
	   (\"Newsgroups:\" \"Followup-To:\" \"Newsgroups:\"))
	  (\"Newsgroups:\"
	   (\"Newsgroups:\" \"Newsgroups:\"))
	  (\"Reply-To:\"
	   (\"To:\" \"Reply-To:\" \"From:\")
	   (\"Cc:\" \"To:\" \"Cc:\" \"Apparently-To:\"))
	  (t
	   (\"To:\" \"From:\")
	   (\"Cc:\" \"To:\" \"Cc:\" \"Apparently-To:\")))

This is read as follows:

	(3.1) If the value of Followup-To: is \"poster\", copying the
              value of From: to new To:.
	(3.2) If Followup-To: exists, copying the values of
              Followup-To: and Newsgroups: to new Newsgroups:.
	(3.3) If Newsgroups: exists, copying the value of Newsgroups:
              to Newsgroups:.
	(3.4) If Reply-To: exists, copying the values of Reply-To: and
              From: to new To:. And copying the values of To:, Cc: and
              Apparently-To: to new Cc:.

	(3.5) Otherwise, copying the value of From: to new To:. And
              copying the values of To:, Cc: and Apparently-To: to
              new Cc:.

You may want to set 'mew-reply-all-alist' to:

	'(((\"Followup-To:\" \"poster\")
	   (\"To:\" \"From:\"))
	  (\"Followup-To:\"
	   (\"Newsgroups:\" \"Followup-To:\"))
	  (\"Newsgroups:\"
	   (\"Newsgroups:\" \"Newsgroups:\"))
	  (\"Reply-To:\"
	   (\"To:\" \"Reply-To:\"))
	  (t
	   (\"To:\" \"From:\")
	   (\"Cc:\" \"To:\" \"Cc:\" \"Apparently-To:\")))
"
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-summary-toggle-disp-msg 'on)
    (mew-current-set-window-config)
    (let ((owin (selected-window))
	  (fld (mew-summary-folder-name))
	  (msg (mew-summary-message-number2))
	  cwin cbuf draft case
	  subject to cc newsgroups in-reply-to references
	  encrypted fromme ret)
      (if (string= (mew-summary-folder-name) mew-draft-folder)
	  (message "Cannot reply to draft message")
	(setq draft (mew-folder-new-message mew-draft-folder))
	(mew-summary-prepare-draft
	 (mew-summary-prepare-three-windows)
	 (mew-draft-find-and-switch draft t)
	 (mew-delete-directory-recursively (mew-attachdir draft))
	 (setq cwin (selected-window)) ;; draft
	 (setq cbuf (window-buffer cwin))
	 (select-window owin)
	 ;; need to make a cache or a message buffer.
	 (let ((mew-use-full-window nil))
	   (mew-summary-display nil))
	 ;; see also mew-draft-cite
	 (set-buffer (or (save-excursion
			   (set-buffer (mew-buffer-message))
			   (if (mew-header-p) (current-buffer)))
			 ;; header exists only in cache if multipart
			 (mew-cache-hit fld msg)))
	 (when mew-case-guess-when-replied
	   (setq case (mew-draft-get-case-by-guess
		       mew-case-guess-when-replied-alist)))
	 (setq encrypted (mew-syntax-encrypted-p mew-decode-syntax))
	 (save-restriction
	   ;; if body contains ^L, header is not accessible.
	   ;; mew-header-* cannot widen essentially. So widen here.
	   (widen)
	   ;; now cache buffer
	   ;; 
	   ;; Subject:
	   (setq subject (mew-header-get-value mew-subj:))
	   (when subject
	     (setq subject (concat mew-reply-string subject))
	     (setq subject (mew-subject-simplify subject)))
	   ;;
	   ;; To:, Cc:, Newsgroups:
	   (setq ret (mew-to-cc-newsgroups replysender))
	   (setq to (nth 0 ret))
	   (setq cc (nth 1 ret))
	   (setq newsgroups (nth 2 ret))
	   (setq fromme (nth 3 ret))
	   ;;
	   ;; In-Reply-To:, References:
	   (setq ret (mew-in-reply-to-references))
	   (setq in-reply-to (nth 0 ret))
	   (setq references (nth 1 ret)))
	 ;;
	 (if (window-live-p cwin)
	     (select-window cwin) ;; draft
	   (pop-to-buffer cbuf))
	 (when case
	   (if mew-case-guess-addition
	       (setq case (mew-draft-add-case (mew-tinfo-get-case) case)))
	   (mew-tinfo-set-case case))
	 (mew-draft-header subject nil to cc newsgroups in-reply-to references
			   nil fromme)
	 (when (eq mew-summary-reply-position 'body)
	   (goto-char (mew-header-end))
	   (forward-line))
	 (mew-draft-mode encrypted)
	 (run-hooks 'mew-draft-mode-newdraft-hook)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Replaying with citation
;;;

(defun mew-summary-reply-with-citation (&optional replysender)
  "Answer to this message. A new draft is prepared in Draft mode. 
And this message is automatically cited. See also 'mew-summary-reply'."
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (let ((mew-summary-reply-position nil)
	  (mew-message-citation-frame-id (mew-frame-id)))
      (mew-summary-reply replysender)
      ;; mew-draft-mode-hook may insert text.
      (save-excursion
	(goto-char (point-max))
	(run-hooks 'mew-before-cite-hook)
	(mew-draft-cite)))
    ;; the cursor is after To:
    (cond
     ((eq mew-summary-reply-with-citation-position 'body)
      (goto-char (mew-header-end))
      (forward-line))
     ((eq mew-summary-reply-with-citation-position 'end)
      (goto-char (point-max)))))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Forwarding
;;;

(defun mew-summary-forward ()
  "Forward this message to a third person. A new draft is prepared in 
Draft mode and this message is automatically attached."
  (interactive)
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((owin (selected-window))
	   (fld (mew-summary-folder-name))
	   (msg (mew-summary-message-number2))
	   (file (mew-expand-folder fld msg))
	   (draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft))
	   (to (and mew-ask-to (mew-input-address (concat mew-to: " "))))
	   (cc (and mew-ask-cc (mew-input-address (concat mew-cc: " "))))
	   subject fwsubject cwin)
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (setq cwin (selected-window)) ;; draft
       (select-window owin)
       ;; need to make a cache or a message buffer.
       (let ((mew-use-full-window nil))
	 (mew-summary-display 'redisplay))
       ;;
       (set-buffer (or (save-excursion
			 (set-buffer (mew-buffer-message))
			 (if (mew-header-p) (current-buffer)))
		       ;; header exists only in cache if multipart
		       (mew-cache-hit fld msg)))
       (setq subject (mew-header-get-value mew-subj:))
       (if subject
	   (setq fwsubject (mew-subject-simplify (concat mew-forward-string subject))))
       (select-window cwin) ;; draft
       ;;
       (mew-draft-header fwsubject 'nl to cc)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft (list file))
       (setq mew-encode-syntax (mew-encode-syntax-initial-multi draftdir 1))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Multi forwarding
;;;

(defun mew-summary-multi-forward ()
  "Forward messages marked with '@' to a third person. A new draft 
is prepared in Draft mode and this message is automatically 
attached."
  (interactive)
  (mew-summary-multi-msgs
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft))
	   (to (and mew-ask-to (mew-input-address (concat mew-to: " "))))
	   (cc (and mew-ask-cc (mew-input-address (concat mew-cc: " ")))))
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (mew-draft-header nil 'nl to cc)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft FILES)
       (setq mew-encode-syntax
	     (mew-encode-syntax-initial-multi draftdir (length FILES)))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

(provide 'mew-summary3)

;;; Copyright Notice:

;; Copyright (C) 1996-2003 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-summary3.el ends here
