/***************************** LICENSE START ***********************************

 Copyright 2014 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/


#include <QDebug>
#include <QMessageBox>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QFile>

#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/json_parser.hpp>
#include <boost/foreach.hpp>

#include "MvQWeatherRoomManager.h"
#include "mars.h"

MvQVideoWallUploader::MvQVideoWallUploader(QObject* parent) : QObject(parent)
{
   // Create a network access manager to upload the file
   fileUploadManager_ = new QNetworkAccessManager(this);
   connect(fileUploadManager_, SIGNAL(finished(QNetworkReply*)),
           this, SLOT(slotFileUpload(QNetworkReply*)));

   // Create a network access manager to update the wall
   wallUpdateManager_ = new QNetworkAccessManager(this);
   connect(wallUpdateManager_, SIGNAL(finished(QNetworkReply*)),
           this, SLOT(slotWallUpdate(QNetworkReply*)));

   // Define URL
   //URL_   = "http://eccharts-test.ecmwf.int/wall";  // default, should be in preferences
   const char *URL = getenv ("METVIEW_WALL_URL");     // override if env var exists
   URL_ = URL ? URL : "http://localhost:8001/wall";

   // Define token
   //token_ = "metview";                              // default, should be in preferences
   const char *token = getenv ("METVIEW_WALL_TOKEN"); // override if env var exists
   token_ = token ? token : "key0";
}

void MvQVideoWallUploader::wrUpLoader(QString fileName)
{
   // Create a network request. Currently, it is hardcoded to "image/png"
   QString req = QString("%1/upload/?key=%2").arg(URL_).arg(token_);
   QNetworkRequest netReq(QUrl::fromPercentEncoding(req.toUtf8()));
   netReq.setHeader(QNetworkRequest::ContentTypeHeader,"image/png");

   // Open input file
   QFile f(fileName);
   if ( !f.open(QIODevice::ReadOnly) )
   {
      QMessageBox::warning(0, QString("Metview"),
                           tr("Error, unable to open file : %1").
                           arg(fileName));

      return;
   }

   // Upload file
   fileUploadManager_->post(netReq,f.readAll());
}

void MvQVideoWallUploader::slotFileUpload(QNetworkReply* reply)
{
   // Check for errors
   if(reply->error() != QNetworkReply::NoError)
   {
      QMessageBox::warning(0, QString("Metview"),
                           tr("Error, unable to get server response : %1").
                           arg(reply->errorString()));

      //marslog(LOG_EROR,"Logo download failed!\nError: %s ",reply->errorString().toStdString().c_str());
      //log->error(reply->errorString().toStdString() + "\n");
      //QApplication::exit();
      return;
   }

   // Read network reply
   QByteArray ba = reply->readAll();
   QString res(ba);

   // Get Id from the reply
   QString sId = this->getId(res);

   // Create a network request
   QString url = QString("%1/update-cell/?key=%2").arg(URL_).arg(token_);
   QNetworkRequest netReq(QUrl::fromPercentEncoding(url.toUtf8()));

   // Create the request
   QString req = QString("{\"wall\" : \"localhost\", \"cell\" : 0, \"id\": \"%1\"}").arg(sId);

//qDebug() << "reply" << res;
//qDebug() << "command" << req;

   // Post the network request
   wallUpdateManager_->post(netReq,req.toUtf8());

   reply->deleteLater();
}

void MvQVideoWallUploader::slotWallUpdate(QNetworkReply* reply)
{
   // Check for errors
   if(reply->error() != QNetworkReply::NoError)
   {
      QMessageBox::warning(0, QString("Metview"),
                           tr("Error, unable to get server response : %1").
                           arg(reply->errorString()));

      //marslog(LOG_EROR,"Logo download failed!\nError: %s ",reply->errorString().toStdString().c_str());
      //log->error(reply->errorString().toStdString() + "\n");
      //QApplication::exit();
      return;
   }

   reply->deleteLater();
}

QString MvQVideoWallUploader::getId( QString& res )
{
   // Create a text file to write the server response
   QString tmpFilename(marstmp());
   QFile tmpFile(tmpFilename);
   tmpFile.open(QIODevice::WriteOnly);

   // Did we manage to open the tmp file for writing?
   if (!tmpFile.isOpen())
   {
      QMessageBox::warning(0, QString("Metview"),
                      tr("Error, unable to open temporary file %1 to write server response")
                      .arg(tmpFilename));

      return QString("");
   }

   // Write the server response to the file
   QTextStream outStream(&tmpFile);
   outStream << res;
   tmpFile.close();

   // Parse the response using the boost JSON property tree parser
   using boost::property_tree::ptree;
   ptree pt;
   std::string tmpFilenameStd = tmpFilename.toUtf8().constData();

   try
   {
      read_json(tmpFilenameStd, pt);
   }
   catch (const boost::property_tree::json_parser::json_parser_error& e)
   {
      QMessageBox::warning(0, QString("Metview"),
                      tr("Error, unable to parse JSON response from server"));

      return QString("");
   }

   return QString(pt.get<std::string>("id").c_str());
}
