#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <messaging/qt/tp/interfaces/base_channel_roles.h>
#include <messaging/qt/tp/interfaces/types.h>
#include <messaging/qt/tp/interfaces/constants.h>

#include <QSignalSpy>

namespace ifaces = messaging::qt::tp::interfaces;

namespace testing
{

class RolesInterfaceTest : public Test
{
protected:

    RolesInterfaceTest()
        : interface_{ifaces::BaseChannelRolesInterface::create()}
    {
    }

    virtual void SetUp()
    {
        using namespace ifaces;

        uint handle_one = 1;
        uint handle_two = 2;
        ChannelRoles roles_one = QFlags<ChannelRole>(ChannelRoleMember);
        ChannelRoles roles_two = QFlags<ChannelRole>(ChannelRoleMember | ChannelRoleAdmin);
        current_[handle_one] = roles_one;
        current_[handle_two] = roles_two;
        interface_->setRoles(current_);
    }

    virtual void TearDown()
    {
    }

    ifaces::HandleRolesMap current_;
    ifaces::BaseChannelRolesInterfacePtr interface_;
};

TEST_F(RolesInterfaceTest, test_set_roles)
{
    using namespace ifaces;

    HandleRolesMap retrieved = interface_->roles();

    for (uint handle : retrieved.keys())
    {
        const uint retrieved_roles = retrieved.value(handle);
        const uint current_roles = current_.value(handle);
        EXPECT_EQ(retrieved_roles, current_roles);
        // remove from current this entry. At the end, current should be empty if they are equals
        current_.remove(handle);
    }

    EXPECT_EQ(current_.size(), 0);
}


TEST_F(RolesInterfaceTest, test_only_adding_roles)
{
    using namespace ifaces;

    uint handle_one = 1;
    uint handle_two = 2;
    uint handle_twenty_seven = 27;
    ChannelRoles roles_one = QFlags<ChannelRole>(ChannelRoleMember | ChannelRoleAdmin);
    ChannelRoles roles_two = QFlags<ChannelRole>(ChannelRoleMember | ChannelRoleAdmin);
    ChannelRoles roles_twenty_seven = QFlags<ChannelRole>(ChannelRoleMember | ChannelRoleAdmin);

    HandleRolesMap modification;
    modification[handle_one] = roles_one;
    modification[handle_two] = roles_two;
    modification[handle_twenty_seven] = roles_twenty_seven;

    QSignalSpy spy(interface_.data(), SIGNAL(rolesChanged(const HandleRolesMap &, const HandleRolesMap&)));

    interface_->setRoles(modification);

    EXPECT_EQ(spy.count(), 1);
    QList<QVariant> arguments = spy.takeFirst();
    EXPECT_EQ(arguments.size(), 2);
    HandleRolesMap added = qvariant_cast<HandleRolesMap>(arguments[0]);
    HandleRolesMap removed = qvariant_cast<HandleRolesMap>(arguments[1]);

    //added
    EXPECT_EQ(added.size(), 2);
    // there should be one key for handle_one and another for handle_twenty_seven
    EXPECT_TRUE(added.contains(handle_one));
    EXPECT_FALSE(added.contains(handle_two));
    EXPECT_TRUE(added.contains(handle_twenty_seven));
    // evaluate handle_one
    const ChannelRoles expected_one(ChannelRoleAdmin);
    EXPECT_EQ(added.value(handle_one), expected_one);
    // evaluate handle_twenty_seven
    const ChannelRoles expected_twenty_seven(ChannelRoleMember | ChannelRoleAdmin);
    EXPECT_EQ(added.value(handle_twenty_seven), expected_twenty_seven);

    //removed
    EXPECT_EQ(removed.size(), 0);
}

TEST_F(RolesInterfaceTest, test_only_removing_roles)
{
    using namespace ifaces;

    uint handle_one = 1;
    uint handle_two = 2;
    ChannelRoles roles_two = QFlags<ChannelRole>(ChannelRoleAdmin);

    HandleRolesMap modification;
    modification[handle_two] = roles_two;

    QSignalSpy spy(interface_.data(), SIGNAL(rolesChanged(const HandleRolesMap &, const HandleRolesMap&)));

    interface_->setRoles(modification);

    EXPECT_EQ(spy.count(), 1);
    QList<QVariant> arguments = spy.takeFirst();
    EXPECT_EQ(arguments.size(), 2);
    HandleRolesMap added = qvariant_cast<HandleRolesMap>(arguments[0]);
    HandleRolesMap removed = qvariant_cast<HandleRolesMap>(arguments[1]);

    //added
    EXPECT_EQ(added.size(), 0);

    //removed
    EXPECT_EQ(removed.size(), 2);
    // there should be one key for handle_one and another for handle_twenty_seven
    EXPECT_TRUE(removed.contains(handle_one));
    EXPECT_TRUE(removed.contains(handle_two));
    // evaluate handle_one
    const ChannelRoles expected_one(ChannelRoleMember);
    EXPECT_EQ(removed.value(handle_one), expected_one);
    // evaluate handle_two
    const ChannelRoles expected_two(ChannelRoleMember);
    EXPECT_EQ(removed.value(handle_two), expected_two);
}


TEST_F(RolesInterfaceTest, test_adding_and_removing_roles)
{
    using namespace ifaces;

    uint handle_one = 1;
    uint handle_two = 2;
    uint handle_twenty_seven = 27;
    ChannelRoles roles_two = QFlags<ChannelRole>(ChannelRoleMember);
    ChannelRoles roles_twenty_seven = QFlags<ChannelRole>(ChannelRoleMember | ChannelRoleAdmin);

    HandleRolesMap modification;
    modification[handle_two] = roles_two;
    modification[handle_twenty_seven] = roles_twenty_seven;

    QSignalSpy spy(interface_.data(), SIGNAL(rolesChanged(const HandleRolesMap &, const HandleRolesMap&)));

    interface_->setRoles(modification);

    EXPECT_EQ(spy.count(), 1);
    QList<QVariant> arguments = spy.takeFirst();
    EXPECT_EQ(arguments.size(), 2);
    HandleRolesMap added = qvariant_cast<HandleRolesMap>(arguments[0]);
    HandleRolesMap removed = qvariant_cast<HandleRolesMap>(arguments[1]);

    //added
    EXPECT_EQ(added.size(), 1);
    uint handle = added.keys()[0];
    // there should be only one handle
    EXPECT_EQ(handle, handle_twenty_seven);
    const ChannelRoles expected(ChannelRoleMember | ChannelRoleAdmin);
    EXPECT_EQ(added.value(handle), expected);

    //removed
    EXPECT_EQ(removed.size(), 2);
    // there should be one key for handle_one and another for handle_two
    EXPECT_TRUE(removed.contains(handle_one));
    EXPECT_TRUE(removed.contains(handle_two));
    EXPECT_FALSE(removed.contains(handle_twenty_seven));
    // evaluate handle_one
    const ChannelRoles expected_one(ChannelRoleMember);
    EXPECT_EQ(removed.value(handle_one), expected_one);
    // evaluate handle_two
    const ChannelRoles expected_two(ChannelRoleAdmin);
    EXPECT_EQ(removed.value(handle_two), expected_two);
}

}
