/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/runner.h>

#include <messaging/associative_dictionary.h>
#include <messaging/connector.h>
#include <messaging/connector_factory.h>
#include <messaging/raii.h>

#include <messaging/qt/tp/adapter.h>

#include <core/posix/signal.h>

#include <QtCore/QCoreApplication>

#include <boost/bimap.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/program_options.hpp>

#include <map>
#include <stdexcept>
#include <thread>

namespace po = boost::program_options;

namespace
{
template <typename T, typename U>
struct make_bimap
{
    typedef typename boost::bimap<T, U>::value_type Position;

    make_bimap(const T& t, const U& u)
    {
        map.insert(Position{t, u});
    }

    boost::bimap<T, U> operator()() const
    {
        return map;
    }

    make_bimap& operator()(const T& t, const U& u)
    {
        map.insert(Position{t, u});
        return *this;
    }

    boost::bimap<T, U> map;
};

// Make the individual keys for parsing configuration data known here. Good practice to
// collect them in exactly one place.
constexpr const char* key_bus{"bus"};
constexpr const char* key_bus_address{"bus-address"};
constexpr const char* key_connection_manager_name{"connection-manager-name"};
constexpr const char* key_protocol_name{"protocol-name"};
constexpr const char* key_connector{"connector"};
constexpr const char* key_help{"help"};

enum class BusType
{
    session,
    system,
    activation,
    with_address
};

const auto bus_type_lut = make_bimap<BusType, std::string>(BusType::session, "session")(BusType::system, "system")(
    BusType::activation, "activation")(BusType::with_address, "with_address")();

std::istream& operator>>(std::istream& in, BusType& bus_type)
{
    std::string s;
    in >> s;

    bus_type = bus_type_lut.right.at(s);

    return in;
}

std::ostream& operator<<(std::ostream& out, BusType bus_type)
{
    return out << bus_type_lut.left.at(bus_type);
}

std::shared_ptr<messaging::qt::Runtime> qt_runtime_from_args(BusType bus_type, const std::string& address)
{
    switch (bus_type)
    {
        case BusType::session:
            return messaging::qt::Runtime::create_once_or_throw(QDBusConnection::SessionBus);
        case BusType::system:
            return messaging::qt::Runtime::create_once_or_throw(QDBusConnection::SystemBus);
        case BusType::activation:
            return messaging::qt::Runtime::create_once_or_throw(QDBusConnection::ActivationBus);
        case BusType::with_address:
            return messaging::qt::Runtime::create_once_or_throw(QString::fromStdString(address));
    }

    throw std::logic_error{"Could not determine connection type."};
}

// Returns a pre-populated descriptions object.
po::options_description known_options()
{
    po::options_description desc("known options");
    desc.add_options()(key_bus, po::value<BusType>()->default_value(BusType::session), "The bus type to connect to.")(
        key_bus_address,
        po::value<std::string>()->default_value(""),
        "The address referring to a dbus server instance.")(
        key_connection_manager_name, po::value<std::string>(), "The name of the resulting connection manager.")(
        key_protocol_name, po::value<std::string>(), "The name of the protocol implemented by the connection manager.")(
        key_connector,
        po::value<std::string>(),
        "The name of the Connector implementation that should be exposed by this runner.")(key_help,
                                                                                           "Prints a help message");

    return desc;
}
}

messaging::Runner::HelpRequired::HelpRequired(const std::string& help_text)
    : std::runtime_error("Help is required for using the command line.")
    , help_text(help_text)
{
}

messaging::Runner::Configuration messaging::Runner::Configuration::from_command_line_args(
    int argc, const char** argv, const messaging::ConnectorFactory& factory)
{
    auto desc = known_options();
    po::variables_map vm;
    po::parsed_options parsed_options{&desc};

    try
    {
        parsed_options = po::command_line_parser(argc, argv).options(desc).allow_unregistered().run();
        po::store(parsed_options, vm);
        po::notify(vm);
    }
    catch (const po::error& e)
    {
        std::stringstream ss;
        ss << "Problem parsing command line arguments with: " << e.what() << "\n" << desc;
        throw messaging::Runner::HelpRequired(ss.str());
    }

    if (vm.count(key_help) > 0)
    {
        std::stringstream ss;
        ss << desc;
        throw messaging::Runner::HelpRequired(ss.str());
    };

    if (vm.count(key_connector) == 0)
    {
        std::stringstream ss;
        ss << "Missing mandator argument: " << key_connector << "\n" << desc;
        throw messaging::Runner::HelpRequired(ss.str());
    }

    if (vm.count(key_connection_manager_name) == 0)
    {
        std::stringstream ss;
        ss << "Missing mandator argument: " << key_connection_manager_name << "\n" << desc;
        throw messaging::Runner::HelpRequired(ss.str());
    }

    if (vm.count(key_protocol_name) == 0)
    {
        std::stringstream ss;
        ss << "Missing mandator argument: " << key_protocol_name << "\n" << desc;
        throw messaging::Runner::HelpRequired(ss.str());
    }

    messaging::AssociativeDictionary<std::string, std::string> dict;
    for (auto key : po::collect_unrecognized(parsed_options.options, po::include_positional))
    {
        boost::algorithm::erase_all(key, "--");
        std::vector<std::string> splits;
        boost::algorithm::split(splits, key, boost::is_any_of("="), boost::algorithm::token_compress_on);
        dict.set_value_for_key(splits.at(0), splits.at(1));
    }

    return messaging::Runner::Configuration{
        argc,
        argv,
        vm[key_connection_manager_name].as<std::string>(),
        vm[key_protocol_name].as<std::string>(),
        factory.create_instance_for_name(vm[key_connector].as<std::string>())(dict),
        qt_runtime_from_args(vm[key_bus].as<BusType>(), vm[key_bus_address].as<std::string>())};
}

int messaging::Runner::main(const messaging::Runner::Configuration& configuration)
{
    // We shutdown gracefully on sig_term and sig_quit.
    auto trap = core::posix::trap_signals_for_process({core::posix::Signal::sig_term, core::posix::Signal::sig_quit});
    trap->signal_raised().connect([trap, &configuration](core::posix::Signal)
                                  {
                                      configuration.qt_runtime->stop(EXIT_SUCCESS);
                                  });

    // Qt does not provide means for handling posix signals by default.
    // Bummer, use core::posix::SignalTrap to sanely handle posix signals.
    raii::Executor<core::posix::SignalTrap> te{trap};

    // We hand a thread of execution to the connector.
    raii::Executor<messaging::Connector> ce{configuration.connector};

    // Start up the telepathy adapter and expose the connector instance to the bus.
    messaging::qt::tp::Adapter ta{configuration.qt_runtime,
                                  configuration.connector,
                                  configuration.connection_manager_name,
                                  configuration.protocol_name};

    return configuration.qt_runtime->run();
}
