/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <TelepathyQt/DBusObject>
#include <messaging/qt/tp/interfaces/base_channel_destroyable.h>
#include <messaging/qt/tp/interfaces/base_channel_destroyable_internal.h>

namespace mqt = messaging::qt;

// Chan.I.Destroyable
mqt::tp::interfaces::BaseChannelDestroyableInterface::Adaptee::Adaptee(BaseChannelDestroyableInterface *interface)
    : QObject(interface),
      mInterface(interface)
{
}

mqt::tp::interfaces::BaseChannelDestroyableInterface::Adaptee::~Adaptee()
{
}

struct TP_QT_NO_EXPORT mqt::tp::interfaces::BaseChannelDestroyableInterface::Private {
    Private(BaseChannelDestroyableInterface *parent)
        :adaptee(new BaseChannelDestroyableInterface::Adaptee(parent)) {
    }

    DestroyCallback destroyCB;
    BaseChannelDestroyableInterface::Adaptee *adaptee;
};

void mqt::tp::interfaces::BaseChannelDestroyableInterface::Adaptee::destroy(const Tp::Service::ChannelInterfaceDestroyableAdaptor::DestroyContextPtr &context)
{
    if (!mInterface->mPriv->destroyCB.isValid()) {
        context->setFinishedWithError(TP_QT_ERROR_NOT_IMPLEMENTED, QLatin1String("Not implemented"));
        return;
    }

    Tp::DBusError error;
    mInterface->mPriv->destroyCB(&error);
    if (error.isValid()) {
        context->setFinishedWithError(error.name(), error.message());
        return;
    }
    context->setFinished();
}

/**
 * \class BaseChannelDestroyableInterface
 *
 * \brief Base class for implementations of Channel.Interface.Destroyable
 *
 */

/**
 * Class constructor.
 */
mqt::tp::interfaces::BaseChannelDestroyableInterface::BaseChannelDestroyableInterface()
    : Tp::AbstractChannelInterface(TP_QT_IFACE_CHANNEL_INTERFACE_DESTROYABLE),
      mPriv(new Private(this))
{
}

void mqt::tp::interfaces::BaseChannelDestroyableInterface::setDestroyCallback(const DestroyCallback &cb)
{
    mPriv->destroyCB = cb;
}

/**
 * Class destructor.
 */
mqt::tp::interfaces::BaseChannelDestroyableInterface::~BaseChannelDestroyableInterface()
{
    delete mPriv;
}

/**
 * Return the immutable properties of this interface.
 *
 * Immutable properties cannot change after the interface has been registered
 * on a service on the bus with registerInterface().
 *
 * \return The immutable properties of this interface.
 */
QVariantMap mqt::tp::interfaces::BaseChannelDestroyableInterface::immutableProperties() const
{
    QVariantMap map;
    return map;
}

void mqt::tp::interfaces::BaseChannelDestroyableInterface::createAdaptor()
{
    (void) new Tp::Service::ChannelInterfaceDestroyableAdaptor(dbusObject()->dbusConnection(),
            mPriv->adaptee, dbusObject());
}

