/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/plugin_connector.h>

#include <messaging/dictionary.h>
#include <messaging/dynamic_library.h>
#include <messaging/named_symbol_loader.h>

#include <boost/lexical_cast.hpp>

#include <iostream>

namespace
{
namespace symbol
{
template <typename T>
T as(void* p)
{
    return reinterpret_cast<T>(p);
}
}

template <typename T>
struct Option
{
    Option(const std::string& name)
        : name(name)
    {
    }

    T from_dict(const messaging::Dictionary<std::string, std::string>& dict) const
    {
        return boost::lexical_cast<T>(dict.value_for_key(name));
    }

    std::string name;
};

namespace options
{
const Option<std::string>& path()
{
    static const Option<std::string> opt{"PluginConnector::path"};
    return opt;
}
}

std::shared_ptr<messaging::Connector> load_connector_instance(
    const std::shared_ptr<messaging::NamedSymbolLoader>& named_symbol_loader)
{
    typedef messaging::PluginConnector::Create Create;
    typedef messaging::PluginConnector::Destroy Destroy;

    Create create{symbol::as<Create::Type>(named_symbol_loader->load_symbol_for_name(Create::name))};
    Destroy destroy{symbol::as<Destroy::Type>(named_symbol_loader->load_symbol_for_name(Destroy::name))};

    return std::shared_ptr<messaging::Connector>(create.call(),
                                                 [destroy](messaging::Connector* c)
                                                 {
        if (c)
        {
            destroy.call(c);
        }
    });
}
}

messaging::ConnectorFactory::Generator messaging::PluginConnector::Descriptor::generator()
{
    return [](const messaging::Dictionary<std::string, std::string>& dict)
    {
        return std::make_shared<messaging::PluginConnector>(
            std::make_shared<messaging::DynamicLibrary>(options::path().from_dict(dict)));
    };
}

messaging::PluginConnector::PluginConnector(const std::shared_ptr<messaging::NamedSymbolLoader>& named_symbol_loader)
    : named_symbol_loader(named_symbol_loader)
    , connector(load_connector_instance(named_symbol_loader))
{
}

const messaging::Enumerator<messaging::Parameter>& messaging::PluginConnector::parameters() const
{
    return connector->parameters();
}

std::shared_ptr<messaging::Connection> messaging::PluginConnector::request_connection(
    const std::shared_ptr<messaging::Connection::Observer>& connection_observer,
    const std::shared_ptr<messaging::Messenger::Observer>& messenger_observer,
    const std::shared_ptr<messaging::PresenceManager::Observer>& presence_observer,
    const messaging::Dictionary<std::string, messaging::Variant>& dict)
{
    return connector->request_connection(connection_observer, messenger_observer, presence_observer, dict);
}

void messaging::PluginConnector::run()
{
    connector->run();
}

void messaging::PluginConnector::stop()
{
    connector->stop();
}
