/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/group_manager.h>

#include <glog/logging.h>

struct messaging::GroupManager::Private
{
    std::shared_ptr<messaging::GroupManager::Observer> observer;
    messaging::Flags<messaging::GroupPermissions> permissions;
};

messaging::GroupManager::GroupManager()
        : impl{new Private{}}
{
}

void messaging::GroupManager::set_observer(const std::shared_ptr<Observer> &observer)
{
    impl->observer = observer;
}

void messaging::GroupManager::validate_observer()
{
    if (!impl->observer)
    {
        throw std::runtime_error{"Could not call event back. Group manager observer is null and must be set"};
    }
}

messaging::Flags<messaging::GroupPermissions> &messaging::GroupManager::permissions()
{
    return impl->permissions;
}

void messaging::GroupManager::set_permissions(const Flags<GroupPermissions> &permisssions)
{
    impl->permissions = permisssions;
}

void messaging::GroupManager::announce_group_created()
{
    validate_observer();
    impl->observer->on_group_created();
}

void messaging::GroupManager::announce_group_cancelled(CancelGroupReason reason, const std::string &message)
{
    validate_observer();
    impl->observer->on_group_cancelled(reason, message);
}

void messaging::GroupManager::announce_group_title_changed(const std::string &new_title)
{
    validate_observer();
    impl->observer->on_group_title_changed(new_title);
}

void messaging::GroupManager::announce_group_subject_changed(const std::string &new_subject,
                                                             const std::shared_ptr<Member> &actor,
                                                             const std::chrono::system_clock::time_point &when)
{
    validate_observer();
    impl->observer->on_group_subject_changed(new_subject, actor, when);
}

void messaging::GroupManager::announce_members_updated(const Members& members)
{
    validate_observer();
    impl->observer->on_members_updated(members);
}

void messaging::GroupManager::announce_group_permissions_changed(const Flags<GroupPermissions> &permissions)
{
    validate_observer();
    impl->observer->on_group_permissions_changed(permissions);
}
