/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_CONNECTION_H
#define MESSAGING_CONNECTION_H

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>
#include <messaging/visibility.h>

#include <memory>

namespace messaging
{
class Messenger;
class PresenceManager;
class GroupStarter;

/// @brief Connection models an active connection with a messaging service.
class MESSAGING_FW_PUBLIC Connection : NonCopyable, NonMovable
{
public:
    /// @brief Status enumerates all known statii of a Connection instance.
    enum class Status
    {
        connected = 0,
        connecting = 1,
        disconnected = 2
    };

    /// @brief StatusChangedReason enumerates all known reasons for a status change.
    enum class StatusChangedReason
    {
        not_specified = 0, ///< No further reason for the state change.
        requested = 1, ///< The status change has been requested by the user.
        network_error = 2,
        authentication_failed = 3,
        encryption_error = 4,
        name_in_use = 5,
        cert_not_provided = 6,
        cert_untrusted = 7,
        cert_expired = 8,
        not_activated = 9,
        cert_hostname_mismatch = 10,
        cert_fingerprint_mismatch = 11,
        cert_self_signed = 12,
        cert_other_error = 13,
        cert_revoked = 14,
        cert_insecure = 15,
        cert_limit_exceeded = 16
    };

    /// @brief Not much to say here, observe key characteristics of a Connection.
    class Observer : NonCopyable, NonMovable
    {
      public:
        /// @brief on_status_changed is invoked whenever the status of the connection changes.
        /// @param status The new status of the connection.
        /// @param reason The reason why the connection status changed.
        virtual void on_status_changed(Status status, StatusChangedReason reason) = 0;

      protected:
        Observer() = default;
        
    };

    /// @brief self_identifier returns the current registered user identifier
    virtual std::string self_identifier() = 0;

    /// @brief presence_manager returns a shared pointer to a PresenceManager object
    virtual std::shared_ptr<PresenceManager> presence_manager() = 0;

    /// @brief messenger returns a shared pointer to a Messenger object.
    virtual std::shared_ptr<Messenger> messenger() = 0;

    /// @brief group_starter returns a shared pointer to a GroupStarter object
    virtual std::shared_ptr<GroupStarter> group_starter() = 0;

    /// @brief login signs in server if not logged in already
    virtual void connect() = 0;

    /// @brief login signs out from server if logged in
    virtual void disconnect(StatusChangedReason reason) = 0;

    /// @brief normalize identifier
    virtual std::string normalize_identifier(const std::string &id) = 0;

    /// @brief checks if an identifier is valid
    virtual bool is_valid_identifier(const std::string &id) = 0;

protected:
    /// @brief Instances can only be created by subclasses.
    /// @param observer Must be a valid observer instance.
    explicit Connection(const std::shared_ptr<Observer>& observer);

    /// @brief announce_status_changed should be called by implementations to notify
    /// the world that the overall status of this connection instance has changed.
    ///
    /// A note to integrators: It is safe to call this function from any thread.
    void announce_status_changed(Status status, StatusChangedReason reason);

private:
    /// @cond
    struct Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};
}

#endif  // MESSAGING_CONNECTION_H
