/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

  $Id: context_id.cpp,v 1.4 2006/07/09 11:26:27 taku-ku Exp $;

  Copyright (C) 2001-2006 Taku Kudo <taku@chasen.org>
  Copyright (C) 2004-2006 Nippon Telegraph and Telephone Corporation

*/
#include <fstream>
#include "context_id.h"
#include "utils.h"

namespace MeCab {

  bool ContextID::open_(const char *filename,
                        std::map<std::string, int> &cmap)
  {
    std::ifstream ifs(filename);
    CHECK_DIE(ifs) << "no such file or directory: " << filename;
    cmap.clear();
    char buf[BUF_SIZE];
    char *col[2];
    while (ifs.getline(buf, sizeof(buf))) {
      CHECK_DIE( 2 == tokenize2(buf, " \t", col, 2) )
        << "format error: " << buf;
      cmap.insert(std::make_pair<std::string, int>(col[1], std::atoi(col[0])));
    }
    return true;
  }

  bool ContextID::build_(std::map<std::string, int> &cmap,
                         const std::string &bos)
  {
    int id = 1; // for BOS/EOS
    for (std::map<std::string, int>::iterator it = cmap.begin();
         it != cmap.end();
         ++it) it->second = id++;
    cmap.insert(std::make_pair(bos, 0));
    return true;
  }

  bool ContextID::save_(const char* filename,
                        std::map<std::string, int> &cmap)
  {
    std::ofstream ofs(filename);
    CHECK_DIE(ofs) << "permission denied: " << filename;
    for (std::map<std::string, int>::iterator it = cmap.begin();
         it != cmap.end(); ++it) {
      ofs << it->second << " " << it->first << std::endl;
    }
    return true;
  }

  void ContextID::clear()
  {
    left_.clear();
    right_.clear();
    left_bos_.clear();
    right_bos_.clear();
  }

  void ContextID::add(const char *l, const char *r)
  {
    left_.insert(std::make_pair(std::string(l), 1));
    right_.insert(std::make_pair(std::string(r), 1));
  }

  void ContextID::addBOS(const char *l, const char *r)
  {
    left_bos_ = l;
    right_bos_ = r;
  }

  bool ContextID::save(const char* lfile,
                       const char* rfile)
  {
    return (save_(lfile, left_) && save_(rfile, right_));
  }

  bool ContextID::open(const char *lfile,
                       const char *rfile)
  {
    return (open_(lfile, left_) && open_(rfile, right_));
  }

  bool ContextID::build()
  {
    return (build_(left_, left_bos_) && build_(right_, right_bos_));
  }

  int ContextID::lid(const char *l)
  {
    std::map<std::string, int>::iterator it = left_.find(l);
    CHECK_DIE(it != left_.end())
      << "cannot find LEFT-ID  for " << l;
    return it->second;
  }

  int ContextID::rid(const char *r)
  {
    std::map<std::string, int>::iterator it = right_.find(r);
    CHECK_DIE(it != left_.end())
      << "cannot find RIGHT-ID  for " << r;
    return it->second;
  }
};
